"""
Author:      www.tropofy.com

Copyright 2013 Tropofy Pty Ltd, all rights reserved.

This source file is part of Tropofy and govered by the Tropofy terms of service
available at: http://www.tropofy.com/terms_of_service.html

This source file is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE. See the license files for details.
"""

from tropofy.widgets import Form
from tropofy.widgets import grid_widget
from sqlalchemy.sql.expression import desc, or_, cast
from sqlalchemy import String
from sqlalchemy import orm as sqla_orm
from sqlalchemy.schema import CheckConstraint
from tropofy.database import DBSession
from tropofy.database.tropofy_orm import encryption

from sqlalchemy.orm import sessionmaker


class SimpleGrid(grid_widget.GridWidget):
    """Basic Grid. This widget is special. It is not used like the other widgets. You do not need to create a class
    which implements function to use an instance of this widget.

    :param source_class: A user defined SQLAlchemy Python class
    :param editable: (Optional - default True). Enables grid to be edited.
    :type editable: bool
    :param widget_subscriptions: (Optional) Widget to subscribe to a triggered event of
    :type widget_subscriptions: list of tuples of (tropofy.widget, str of trigger_event_name, str of subscribed_action)

    .. note::
        The source_class provided to a SimpleGrid must have the following properties:
            - The database column names must be the same as the member variable names, i.e. you cannot use code like ``some_name = Column('some_other_name', String(50))``),
            - The initialisers parameter names must be the same as the member variable names. Note that SQLAlchemy provides a default initialiser with keyword arguments equal to the names of the column members defined for your class.

    The following class satisfies the above requirements

    .. literalinclude:: ../../../tropofy/widgets/examples/widget_examples.py
        :pyobject: ConformingClass
    """
    def __init__(self, source_class, editable=True, title=None, embedded_form_class=None, desc_sort_cols=None, widget_subscriptions=None, default_ordered_col_name_sort_direction_tuples=None):
        self.source_class = source_class
        self.editable = editable
        self.default_ordered_col_name_sort_direction_tuples = default_ordered_col_name_sort_direction_tuples
        super(SimpleGrid, self).__init__(
            title=title,
            embedded_form_class=embedded_form_class if embedded_form_class else SimpleGridCreateUpdateForm,
            desc_sort_cols=desc_sort_cols,
            widget_subscriptions=widget_subscriptions
        )

    def _get_type(self):
        return 'SimpleGrid'

    def add_new_row(self, data, data_set_id=None, **kwargs):
        """Assumes the data keys are a super set of the constructor arguments for the class type you are making
        i.e. that the grids column headers are a super set of the constructor arguments"""
        data.pop("data_set_id", None)
        data.pop("id", None)
        new_object = self.source_class(**data)
        new_object.data_set_id = data_set_id
        DBSession().add(new_object)
        DBSession().flush()

        return new_object.as_json_data_row(self.get_grids_column_names())

    def edit_row(self, data, **kwargs):
        """Assumption that the names of the members of a class are equal to the keys you get in the
        data object when editing, which in turn are the names of the columns in the grid, which
        are a subset of the columns in the table hierachy for this object"""
        existing_object = DBSession().query(self.source_class).filter_by(id=data['id']).one()

        for k, v in data.iteritems():
            if k not in ['id', 'data_set_id']:
                setattr(existing_object, k, v)

        DBSession().flush()
        return existing_object.as_json_data_row(self.get_grids_column_names())

    def delete_row(self, obj_id, **kwargs):
        DBSession().delete(DBSession().query(self.source_class).filter_by(id=obj_id).one())

    def get_table_hierachy_from_inheritance_chain(self):
        full_class_hierachy = list(self.source_class.__bases__) + [self.source_class]
        return [class_type for class_type in full_class_hierachy if hasattr(class_type, "__table__")]

    def get_grids_column_names(self, **kwargs):
        return [col.name for class_type in self.get_table_hierachy_from_inheritance_chain() for col in class_type.__table__.columns]

    def get_column_name_to_form_input_types(self, **kwargs):
        return {col.name: Form._get_input_type_from_python_type(col.type.python_type) for class_type in self.get_table_hierachy_from_inheritance_chain() for col in class_type.__table__.columns}

    def get_filtered_rows(self, data_set, display_start, display_length, global_search_field, ordered_col_name_sort_direction_tuples, **kwargs):
        try:
            source_class, qry = self._get_qry_of_non_filtered_refresh_objects(data_set)
        except AttributeError:
            # TODO: Log that there was an attribute error on the refreshish
            return [], 0, 0
        total_records_before_filtering = qry.count()

        #Filter from search
        if global_search_field:
            search_conditions = []
            do_not_search_on_cols = self.get_col_names_to_not_search_on(data_set)
            for col_name in _sqla_cls_defined_column_names(source_class):
                if col_name not in do_not_search_on_cols:
                    search_conditions.append(
                        cast(getattr(source_class, col_name), String).ilike('%' + global_search_field + '%')  # .ilike is case insensitive .like
                    )
            qry = qry.filter(or_(*search_conditions))
                    
        total_records_after_filtering = qry.count()
        
        #Order cols
        if self.default_ordered_col_name_sort_direction_tuples and not ordered_col_name_sort_direction_tuples:
            ordered_col_name_sort_direction_tuples = self.default_ordered_col_name_sort_direction_tuples
        for col_name, direction in ordered_col_name_sort_direction_tuples:
            if direction == grid_widget.GridWidget.ASCENDING:
                qry = qry.order_by(getattr(source_class, col_name))
            elif direction == grid_widget.GridWidget.DESCENDING:
                qry = qry.order_by(desc(getattr(source_class, col_name)))

        #Reduce to num rows on grid page
        qry = qry.offset(display_start).limit(display_length)
        rows = [obj.as_json_data_row(self.get_grids_column_names()) for obj in qry]

        qry.session.close()  # Required, as if using encryption, qry uses a session not management by zope transaction manager.
        return rows, total_records_before_filtering, total_records_after_filtering

    def _get_qry_of_non_filtered_refresh_objects(self, data_set, **kwargs):
        result = self.get_qry_of_non_filtered_refresh_objects(data_set, **kwargs)
        try:
            if len(result) == 2:  # result is a tuple, expected to be (source_class, qry)
                return result
        except Exception as e:  # Only qry was returned, create tuple with source_class.
            return self.source_class, result

    def get_qry_of_non_filtered_refresh_objects(self, data_set, **kwargs):
        """Get the sqla query that will return sqla orm objects unfiltered by grid.

        :returns: Either:
         - sqla qry object - that if executed would return all mixins in the grid - unfiltered and unsorted
         - tuple of (<sqla_class>, sqla_qry_object (as in prev)). Use if sqla_class that qry is built off of is different to `self.source_class`.

        Assumes data_set exists.
        """
        if data_set and hasattr(self.source_class, 'data_set_id'):
            if hasattr(self.source_class, '__decrypted_cls__') and self.source_class.__decrypted_cls__:
                # Note: This only works for tables with FK's because SQLite doesn't enforce FK's by default.
                # If it did, we'd need to delete the constraints or copy all data from joined table (for a heavily constraint db, this could recursively pull the whole db for every grid refresh).
                # Can't drop the FK constraint, as SQLite doesn't support ALTER TABLE DROP CONSTRAINT.
                # Easiest thing to do (even though somewhat implicit), is to rely on FK's not being enforced.
                # Tropofy framework turns on FK enforcement in SQLite if it is the main DB being used. This is fine though, because this encryption process wouldn't be compatible with SQLite anyway.
                in_memory_db_engine = encryption.orm_decrypt_data_to_new_in_memory_db(data_set, [self.source_class])

                Session = sessionmaker(bind=in_memory_db_engine)
                session = Session()
                return self.source_class.d, session.query(self.source_class.d).filter_by(data_set_id=data_set.id)

            return data_set.query(self.source_class)
        raise AttributeError('SimpleGrid for source_class `%s` has no `data_set_id` column. SimpleGrid expects this. If not using data sets, extend SimpleGrid and overide `get_qry_of_non_filtered_refresh_objects`' % self.source_class.__name__)

    def get_col_names_to_not_search_on(self, data_set):
        return self.get_hidden_column_names()

    def grid_is_editable(self, **kwargs):
        return self.editable


class SimpleGridCreateUpdateForm(grid_widget.CreateUpdateForm):
    def get_select_options_for_column(self, data_set, col_name, hidden_col_names):
        """For a column, get the select options for the form element. For a SimpleGrid - base off of foreign keys on tables."""
        options = None

        col_name_to_foreign_key_col_objects = {name: col for (name, col) in self._get_query_column_objects().items() if col.foreign_keys and col.name not in hidden_col_names}
        col_name_to_check_constraint_allowable_values = self.get_check_constraints_where_column_must_be_in_list_of_values(data_set)

        if col_name in col_name_to_foreign_key_col_objects.keys():
            column = col_name_to_foreign_key_col_objects[col_name]
            foreign_key_col = list(column.foreign_keys)[0].column
            mapper = _get_mapper_for_table(foreign_key_col.table)

            if data_set:
                options_qry = data_set.query(getattr(mapper.class_, foreign_key_col.name))
            else:
                options_qry = DBSession().query(getattr(mapper.class_, foreign_key_col.name))
            options = [Form.Element.Option(v[0]) for v in options_qry]

            if column.nullable:
                options = [Form.Element.Option(value=None)] + options

        elif col_name in col_name_to_check_constraint_allowable_values.keys() and col_name not in hidden_col_names:
            options = [Form.Element.Option(v) for v in col_name_to_check_constraint_allowable_values[col_name]]

        return options

    def get_check_constraints_where_column_must_be_in_list_of_values(self, data_set):
        dict_cons = {}
        col_names = self._get_query_column_objects().keys()
        for con in self.parent_grid_widget.source_class.__table__.constraints:
            if type(con) is CheckConstraint and hasattr(con, 'sqltext') and hasattr(con.sqltext, 'text'):
                con_text = con.sqltext.text
                col_name = con_text.split()[0] if con_text.split() and con_text.split()[0] in col_names else None
                if col_name and con_text.find("in") != -1:
                    values_text = con_text[con_text.find("(")+1:con_text.find(")")]
                    dict_cons[col_name] = [v.replace("'", "") for v in values_text.split(",")]
        return dict_cons

    def _get_query_column_objects(self):
        return {col.name: col for class_type in self.parent_grid_widget.get_table_hierachy_from_inheritance_chain() for col in class_type.__table__.columns}


def _get_mapper_for_table(table):
    for mapper in list(sqla_orm._mapper_registry):
        if table in mapper._all_tables:
            return mapper
    return None


def _sqla_cls_defined_column_names(cls):
    return [prop.key for prop in sqla_orm.class_mapper(cls).iterate_properties if isinstance(prop, sqla_orm.ColumnProperty)]