# Trosnoth (UberTweak Platform Game)
# Copyright (C) 2006-2009 Joshua D Bartlett
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# version 2 as published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
# 02110-1301, USA.

'''universe.py - defines anything that has to do with the running of the
universe. This includes players, shots, zones, and the level itself.'''

from math import sin, cos

from trosnoth.src.utils.utils import timeNow
import trosnoth.src.utils.logging as logging
from trosnoth.src.utils.checkpoint import checkpoint
import trosnoth.src.model.modes as modes
from trosnoth.src.model.universe_base import GameState
from trosnoth.src.model.upgrades import upgradeOfType

# TODO: remove isinstances so we can remove some of these:
from trosnoth.src.model.map import MapLayout, MapState
from trosnoth.src.model.shot import Shot, GrenadeShot
from trosnoth.src.model.player import Player
from trosnoth.src.model.team import Team


# Component message passing
from trosnoth.src.utils.components import Component, handler, Plug
from trosnoth.src.network.utils import expand_boolean
from trosnoth.src.messages import (ChatMsg, TaggingZoneMsg, ShotFiredMsg,
        PlayerUpdateMsg, KillShotMsg, CannotRespawnMsg, RespawnMsg,
        PlayerKilledMsg, PlayerHitMsg, GameStartMsg, GameOverMsg, SetCaptainMsg,
        TeamIsReadyMsg, StartingSoonMsg, SetTeamNameMsg, SetGameModeMsg,
        AddPlayerMsg, CannotJoinMsg, JoinSuccessfulMsg, UpdatePlayerStateMsg,
        AimPlayerAtMsg, RemovePlayerMsg, PlayerHasUpgradeMsg,
        PlayerStarsSpentMsg, DeleteUpgradeMsg, CannotBuyUpgradeMsg,
        ConnectionLostMsg, ServerShutdownMsg)

from twisted.internet import reactor, task

DIRTY_DELAY = 10.5
UPDATE_DELAY = 0.3        # PlayerUpdateMsg is sent for each player this often.
TICK_PERIOD = 0.02

class Abort(Exception):
    pass

class Universe(Component):
    '''Universe(halfMapWidth, mapHeight)
    Keeps track of where everything is in the level, including the locations
    and states of every alien, the terrain positions, and who owns the
    various territories and orbs.'''

    # The size of players and shots.
    halfPlayerSize = (10, 19)
    halfShotSize = (5, 5)

    # eventPlug is a sending plug
    # Events which are generated by the tick() method of the universe
    # should be sent to this plug. This indicates that the universe thinks
    # that the event should happen.
    eventPlug = Plug()

    # orderPlug is a receiving plug
    # When an object wants to order the universe to do something,
    # they should send a message to this plug
    orderPlug = Plug()

    def __init__(self, halfMapWidth=3, mapHeight=2, gameDuration=0):
        '''
        halfMapWidth:   is the number of columns of zones in each team's
                        territory at the start of the game. There will always
                        be a single column of neutral zones between the two
                        territories at the start of the game.
        mapHeight:      is the number of zones in every second column of
                        zones. Every other column will have mapHeight + 1
                        zones in it. This is subject to the constraints that
                        (a) the columns at the extreme ends of the map will
                        have mapHeight zones; (b) the central (initially
                        neutral) column of zones will never have fewer zones
                        in it than the two bordering it - this will sometimes
                        mean that the column has mapHeight + 2 zones in it.'''

        super(Universe, self).__init__()
        self.gameDuration = gameDuration

        # Initialise
        self.gameState = GameState.PreGame
        self.winningTeam = None
        self.startTime = 0
        self.zonesToReset = []
        self.playerWithId = {}
        self.teamWithId = {'\x00' : None}

        # Create Teams:
        self.teams = (
            Team(self, 'A'),
            Team(self, 'B'),
        )
        Team.setOpposition(self.teams[0], self.teams[1])

        for t in self.teams:
            self.teamWithId[t.id] = t

        # Set up zones
        self.zoneWithDef = {}
        layout = MapLayout(halfMapWidth, mapHeight)
        self.map = MapState(self, layout)
        self.zoneWithId = self.map.zoneWithId
        self.zones = self.map.zones
        self.zoneBlocks = self.map.zoneBlocks

        self.players = set()
        self.playerUpdateTimes = {}
        self.grenades = set()
        # TODO: could make shots iterate directly over the player objects rather than storing twice
        # (redundancy - potential inconsistencies)
        self.shots = set()
        self.gameOverTime = None
        self.gameModeController = modes.GameMode(Shot, Player, GrenadeShot)
        self.gameMode = 'Normal'

        self._lastTime = timeNow()
        self._loop = None
        self._startClock()

    @handler(CannotBuyUpgradeMsg, orderPlug)
    @handler(JoinSuccessfulMsg, orderPlug)
    @handler(CannotJoinMsg, orderPlug)
    @handler(CannotRespawnMsg, orderPlug)
    @handler(ConnectionLostMsg, orderPlug)
    @handler(ServerShutdownMsg, orderPlug)
    @handler(ChatMsg, orderPlug)
    def ignore(self, msg):
        pass

    def getTeam(self, teamId):
        try:
            return self.teamWithId[teamId]
        except KeyError:
            raise Abort()

    def getPlayer(self, playerId):
        try:
            return self.playerWithId[playerId]
        except KeyError:
            raise Abort()

    def getUpgradeType(self, upgradeTypeId):
        try:
            return upgradeOfType[upgradeTypeId]
        except KeyError:
            raise Abort()

    def getZone(self, zoneId):
        try:
            return self.map.zoneWithId[zoneId]
        except KeyError:
            raise Abort()

    def getShot(self, pId, sId):
        try:
            return self.playerWithId[pId].shots[sId]
        except KeyError:
            raise Abort()

    def playerIsDirty(self, pId):
        '''
        Registers the given players as "dirty" in that its location details
        should be sent to all clients at some stage in the near future. This is
        called when, for example, the player releases the jump key before the
        top of the jump, or collides with an obstacle, or such things.
        '''
        self.playerUpdateTimes[pId] = min(self.playerUpdateTimes.get(pId, 1),
                DIRTY_DELAY)

    def _sendPlayerUpdate(self, pId):
        '''
        Sends state information of a dirty player to all clients.
        '''
        self.playerUpdateTimes[pId] = timeNow() + UPDATE_DELAY

        try:
            player = self.playerWithId[pId]
        except KeyError:
            return

        self.eventPlug.send(player.makePlayerUpdate())

    @handler(UpdatePlayerStateMsg, orderPlug)
    def updatePlayerState(self, msg):
        try:
            player = self.getPlayer(msg.playerId)
            player.updateState(msg.stateKey, msg.value)
        except Abort:
            pass

    @handler(AimPlayerAtMsg, orderPlug)
    def aimPlayerAt(self, msg):
        try:
            player = self.getPlayer(msg.playerId)
            player.lookAt(msg.angle, msg.thrust)
        except Abort:
            pass

    @handler(SetGameModeMsg, orderPlug)
    def setGameMode(self, msg):
        mode = msg.gameMode.decode()
        try:
            fn = getattr(self.gameModeController, mode)
        except (AttributeError, TypeError):
            pass
        else:
            fn()
            self.gameMode = mode
            print 'Client: GameMode is set to ' + mode

    @handler(SetTeamNameMsg, orderPlug)
    def setTeamName(self, msg):
        try:
            team = self.getTeam(msg.teamId)
            team.teamName = msg.name
        except Abort():
            pass

    def getTimeLeft(self):
        if self.gameState in (GameState.PreGame, GameState.Starting):
            return None
        elif self.gameState == GameState.InProgress:
            return self.startTime + self.gameDuration - timeNow()
        elif self.gameState == GameState.Ended:
            return self.startTime + self.gameDuration - self.gameOverTime

    @handler(GameOverMsg, orderPlug)
    def gameOver(self, msg):
        self.gameOverTime = timeNow()
        self.gameState = GameState.Ended
        self.winningTeam = self.teamWithId.get(msg.teamId, None)

    @handler(GameStartMsg, orderPlug)
    def gameStart(self, msg):
        time = msg.timeLimit
        self.gameDuration = time
        self.startTime = timeNow()
        self.gameState = GameState.InProgress

    @handler(AddPlayerMsg, orderPlug)
    def addPlayer(self, msg):
        team = self.teamWithId[msg.teamId]
        zone = self.zoneWithId[msg.zoneId]

        # Create the player.
        nick = msg.nick.decode()
        self._makePlayer(nick, team, msg.playerId, zone, msg.dead)

    def _makePlayer(self, nick, team, playerId, zone, dead):
        player = Player(self, nick, team, playerId, zone, dead)

        #Add this player to this universe.
        self.players.add(player)
        self.playerWithId[playerId] = player

        # Make sure player knows its zone
        i, j = MapLayout.getMapBlockIndices(*player.pos)
        try:
            player.currentMapBlock = self.zoneBlocks[i][j]
        except IndexError:
            logging.writeException()
            raise IndexError, 'player start position is off the map'

    @handler(RemovePlayerMsg, orderPlug)
    def delPlayer(self, msg):
        try:
            player = self.getPlayer(msg.playerId)
            player.removeFromGame()
            self.players.remove(player)
            del self.playerWithId[player.id]
        except Abort:
            pass

    def _startClock(self):
        if self._loop is not None:
            self._loop.stop()
        self._loop = task.LoopingCall(self.tick)
        self._loop.start(TICK_PERIOD, False)

    def stopClock(self):
        if self._loop is not None:
            self._loop.stop()
            self._loop = None

    def tick(self):
        '''Advances all players and shots to their new positions.'''
        t = timeNow()
        deltaT = t - self._lastTime
        self.frameRate = 1. / deltaT
        self._lastTime = t
        
        # Update the player and shot positions.
        for s in self.shots:
            s.registeredHit = False
        for s in list(self.shots) + list(self.players) + list(self.grenades):
            s.update(deltaT)

        for zone in self.zonesToReset:
            zone.taggedThisTime = False
        self.zonesToReset = []

        self.checkForResult()
        self.processPlayerUpdates()

    def processPlayerUpdates(self):
        playerIds = set(self.playerWithId.keys())
        now = timeNow()
        for pId, time in list(self.playerUpdateTimes.iteritems()):
            if pId not in playerIds:
                del self.playerUpdateTimes[pId]
                continue
            playerIds.remove(pId)
            if now > time:
                self._sendPlayerUpdate(pId)

        for pId in playerIds:
            # Not yet in dict.
            self.playerUpdateTimes[pId] = UPDATE_DELAY

    def checkForResult(self):
        if not self.gameState == GameState.InProgress:
            return

        # Check first for timeout
        if self.gameDuration > 0 and self.getTimeLeft() <= 0:
            if self.teams[0].numOrbsOwned > self.teams[1].numOrbsOwned:
                self.eventPlug.send(GameOverMsg(self.teams[0].id, True))
                checkpoint('Universe: Out of time, blue wins')
            elif self.teams[1].numOrbsOwned > self.teams[0].numOrbsOwned:
                self.eventPlug.send(GameOverMsg(self.teams[1].id, True))
                checkpoint('Universe: Out of time, red wins')
            else:
                self.eventPlug.send(GameOverMsg('\x00', True))
                checkpoint('Universe: Out of time, game drawn')
            return

        # Now check for an all zones win.
        team2Wins = self.teams[0].isLoser()
        team1Wins = self.teams[1].isLoser()
        if team1Wins and team2Wins:
            # The extraordinarily unlikely situation that all
            # zones have been neutralised in the same tick
            self.eventPlug.send(GameOverMsg('\x00', False))
            checkpoint('Universe: Draw due to all zones neutralised')
        elif team1Wins:
            self.eventPlug.send(GameOverMsg(self.teams[0].id, False))
            checkpoint('Universe: Zones captured, blue wins')
        elif team2Wins:
            self.eventPlug.send(GameOverMsg(self.teams[1].id, False))
            checkpoint('Universe: Zones captured, red wins')

    @handler(ShotFiredMsg, orderPlug)
    def shotFired(self, msg):
        '''A player has fired a shot.'''
        player = self.playerWithId[msg.playerId]
        team = player.team
        pos = (msg.xpos, msg.ypos)
        turret = msg.type in ('T', 'G')
        ricochet = msg.type == 'R'
        shot = self._createFiredShot(msg.shotId, team, player, pos,
                msg.angle, turret, ricochet)
        if shot is None:
            # If you fire a shot while off the map, you die. Just coz.
            self.eventPlug.send(PlayerKilledMsg(player.id, '\x00', '\x00'))

        else:
            self._update_reload_time(player)

            # When a shot's fired, send a player update to make it look less
            # odd.
            self.playerIsDirty(player.id)

    def _update_reload_time(self, player):
        '''
        Updates the player's reload time because the player has just fired a
        shot.
        '''
        if player.turret:
            player.reloadTime = player.turretReloadTime
            player.turretHeat += player.shotHeat
            if player.turretHeat > player.turretHeatCapacity:
                player.turretOverHeated = True
        elif player.currentZone.zoneOwner == player.team:
            player.reloadTime = player.ownReloadTime
        elif player.currentZone.zoneOwner == None:
            player.reloadTime = player.neutralReloadTime
        else:
            player.reloadTime = player.enemyReloadTime

    def _getNewShotId(self, player, _ids={}):
        '''
        Returns an unused shot id for the given player. This is a temporary
        hack that exists only as part of the transitioning from old to new
        universe message behaviour. (talljosh, 2010-06-27)
        '''
        prev = _ids.get(player, 0)
        result = prev + 1
        _ids[player] = result
        return result

    def _createFiredShot(self, id, team, player, pos, angle, turret, ricochet):
        '''
        Factory function for building a Shot object.
        '''
        velocity = (
            Shot.SPEED * sin(angle),
            -Shot.SPEED * cos(angle),
        )
        if turret:
            kind = Shot.TURRET
        elif ricochet:
            kind = Shot.RICOCHET
        else:
            kind = Shot.NORMAL

        lifetime = Shot.LIFETIME

        i, j = MapLayout.getMapBlockIndices(pos[0], pos[1])
        try:
            mapBlock = self.zoneBlocks[i][j]
        except IndexError:
            return None

        shot = self._buildShot(id, team, player, pos, velocity, kind,
                lifetime, mapBlock)

        return shot

    def _buildShot(self, id, team, player, pos, velocity, kind, lifetime,
            mapBlock):
        shot = Shot(self, id, team, player, pos, velocity, kind, lifetime,
                mapBlock)
        mapBlock.addShot(shot)
        if id in player.shots:
            self.removeShot(player.shots[id])
        player.addShot(shot)
        self.shots.add(shot)
        return shot

    def shotExpired(self, shot):
        '''
        Called during the tick method when a shot has run out of lifetime or
        has hit a solid obstacle.
        '''
        msg = KillShotMsg(shot.originatingPlayer.id, shot.id)
        self.removeShot(shot)
        self.eventPlug.send(msg)

    def removeShot(self, shot):
        shot.originatingPlayer.destroyShot(shot.id)
        try:
            self.shots.remove(shot)
            shot.currentMapBlock.removeShot(shot)
        except KeyError:
            pass

    def shotWithId(self, pId, sId):
        try:
            return self.playerWithId[pId].shots[sId]
        except KeyError:
            return None

    def checkTag(self, tagger):
        '''Checks to see if the player has touched the orb of its currentZone
        If so, it fires the onTagged event'''

        # How zone tagging works (more or less)
        # 1. In its _move procedure, if a player is local, it will call checkTag
        # 2. If it is close enough to the orb, and it has the numeric
        #    advantage, and it onws at least one adjacent zone, it has tagged
        #    the zone
        # 3. The zone is checked again to see if the opposing team has also
        #    tagged it
        #    - If so, the zone is rendered neutral (if it's already neutral,
        #    nothing happens).
        #    - If not, the zone is considered to be tagged by the team.
        # 4. If any zone ownership change has been made, the server is informed
        #    however, no zone allocation is performed yet.
        # 5. The server should ensure that the zone hasn't already been tagged
        #    (such as in the situation of two players form the one team tag
        #    a zone simultaneously), as well as checking zone numbers,
        #    before telling all clients of the zone change.
        # 6. The individual clients recalculate zone ownership based on the
        #    zone change

        zone = tagger.currentZone
        if tagger.team == zone.orbOwner:
            return

        # If the tagging player has a phase shift, the zone will not be tagged.
        if tagger.phaseshift:
            return

        # Ensure that the zone has not already been checked in this tick:
        if zone is None or zone.taggedThisTime:
            return


        # Radius from orb (in pixels) that counts as a tag.
        tagDistance = 35
        xCoord1, yCoord1 = tagger.pos
        xCoord2, yCoord2 = zone.defn.pos

        distance = ((xCoord1 - xCoord2) ** 2 + (yCoord1 - yCoord2) ** 2) ** 0.5
        if distance < tagDistance:
            # Check to ensure the team owns an adjacent orb
            found = False
            for adjZoneDef in zone.defn.adjacentZones:
                adjZone = self.zoneWithDef[adjZoneDef]
                if adjZone.orbOwner == tagger.team:
                    found = True
                    break
            if not found:
                return

            # Check to see if the team has sufficient numbers to take the zone.
            numTaggers = 0
            numDefenders = 0
            for player in zone.players:
                if player.turret:
                    # Turreted players do not count as a player for the purpose
                    # of reckoning whether an enemy can capture the orb
                    continue
                if player.team == tagger.team:
                    numTaggers += 1
                else:
                    numDefenders += 1

            if numTaggers > numDefenders or numTaggers > 3:

                if (numDefenders > 3 and
                        zone.checkForOpposingTag(tagger.team)):
                    # The other team has also tagged it
                    if zone.orbOwner != None:
                        self.eventPlug.send(TaggingZoneMsg(zone.id, '\x00',
                                '\x00'))
                        checkpoint('Universe: two teams tag same zone')
                else:
                    # This team is the only to have tagged it
                    self.eventPlug.send(TaggingZoneMsg(zone.id, tagger.id,
                            tagger.team.id))
                    checkpoint('Universe: zone tagged')

                self._killTurret(tagger, zone)
                zone.taggedThisTime = True
                self.zonesToReset.append(zone)
            else:
                checkpoint('Universe: failed attempt to tag zone')

    def _killTurret(self, tagger, zone):
        if zone.turretedPlayer is not None:
            self.eventPlug.send(PlayerKilledMsg(zone.turretedPlayer.id,
                    tagger.id, '\x00'))
            checkpoint('Universe: turret killed by tagging zone')

    @handler(PlayerKilledMsg, orderPlug)
    def playerKilled(self, msg):
        try:
            target = self.getPlayer(msg.targetId)
            shot = self.shotWithId(msg.killerId, msg.shotId)
            try:
                killer = self.playerWithId[msg.killerId]
            except KeyError:
                killer = None
            else:
                if not killer.ghost:
                    killer.incrementStars()

            target.currentZone.removePlayer(target)
            target.die()
            target.currentZone.addPlayer(target)
            if shot is not None:
                self.removeShot(shot)
        except Abort:
            pass

    @handler(TaggingZoneMsg, orderPlug)
    def zoneTagged(self, msg):
        if msg.playerId == '\x00':
            player = None
        else:
            player = self.playerWithId[msg.playerId]
        zone = self.map.zoneWithId[msg.zoneId]
        zone.tag(player)

    @handler(RespawnMsg, orderPlug)
    def respawn(self, msg):
        try:
            player = self.getPlayer(msg.playerId)
            zone = self.getZone(msg.zoneId)
            player.currentZone.removePlayer(player)
            player.currentZone = zone
            player.respawn()
            player.currentZone.addPlayer(player)
        except Abort:
            pass

    @handler(PlayerStarsSpentMsg, orderPlug)
    def starsSpent(self, msg):
        try:
            player = self.getPlayer(msg.playerId)
            player.stars -= msg.count
        except Abort:
            pass

    @handler(PlayerHasUpgradeMsg, orderPlug)
    def gotUpgrade(self, msg):
        try:
            player = self.getPlayer(msg.playerId)
            upgradeKind = self.getUpgradeType(msg.upgradeType)
            player.upgrade = upgradeKind(player)
            player.upgradeGauge = upgradeKind.timeRemaining
            player.upgradeTotal = upgradeKind.timeRemaining
            player.upgrade.use()
        except Abort:
            pass

    @handler(DeleteUpgradeMsg, orderPlug)
    def delUpgrade(self, msg):
        try:
            player = self.getPlayer(msg.playerId)
            player.deleteUpgrade()
        except Abort:
            pass

    def addGrenade(self, grenade):
        self.grenades.add(grenade)

    def removeGrenade(self, grenade):
        self.grenades.remove(grenade)

    @handler(SetCaptainMsg, orderPlug)
    def setCaptain(self, msg):
        try:
            player = self.getPlayer(msg.playerId)
            player.team.captain = player
        except Abort:
            pass

    @handler(TeamIsReadyMsg, orderPlug)
    def teamReady(self, msg):
        try:
            player = self.getPlayer(msg.playerId)
            if not player.team.ready:
                player.team.ready = True
                if self._gameShouldStart():
                    reactor.callLater(0.1, self._startSoon)
        except Abort:
            pass

    def _gameShouldStart(self):
        '''Checks to see if both teams are ready'''
        for team in self.teams:
            if not team.ready:
                return False
        return True

    def _startSoon(self, delay=7):
        self.eventPlug.send(StartingSoonMsg(delay))
        reactor.callLater(delay, self.eventPlug.send,
                GameStartMsg(self.gameDuration))

    @handler(StartingSoonMsg, orderPlug)
    def startingSoon(self, msg):
        self.gameState = GameState.Starting

    @handler(KillShotMsg, orderPlug)
    def shotDestroyed(self, msg):
        shot = self.shotWithId(msg.shooterId, msg.shotId)
        if shot is not None:
            self.removeShot(shot)

    @handler(PlayerHitMsg, orderPlug)
    def shotHit(self, msg):
        shot = self.shotWithId(msg.shooterId, msg.shotId)
        player = self.getPlayer(msg.targetId)
        if player.shielded:
            player.upgrade.protections -= 1
        else:
            player.health -= 1
        if shot is not None:
            self.removeShot(shot)

    @handler(PlayerUpdateMsg, orderPlug)
    def gotPlayerUpdate(self, msg):
        # Receive a player update
        try:
            player = self.playerWithId[msg.playerId]
        except:
            # Mustn't have that info yet
            return

        values = expand_boolean(msg.keys)
        for i, key in enumerate(['left', 'right', 'jump', 'down']):
            player._state[key] = values[i]

        ghost = values[5]

        if ghost != player.ghost:
            if player.ghost:
                player.respawn()
            else:
                player.die()

        player.yVel = msg.yVel
        player.lookAt(msg.angle, msg.ghostThrust)
        player.setPos((msg.xPos, msg.yPos))

    def getTeamStars(self, team):
        total = 0
        for p in self.players:
            if p.team == team:
                total += p.stars
        return total

    def _getWorldParameters(self):
        '''Returns a dict representing the settings which must be sent to
        clients that connect to this server.'''
        result = {
            'teamName0': self.teams[0].teamName,
            'teamName1': self.teams[1].teamName,
        }

        if self.gameState == GameState.PreGame:
            result['gameState'] = "PreGame"
        elif self.gameState == GameState.Starting:
            result['gameState'] = 'Starting'
        elif self.gameState == GameState.InProgress:
            result['gameState'] = "InProgress"
            result['timeRunning'] = timeNow() - self.startTime
            result['timeLimit'] = self.gameDuration
        else:
            result['gameState'] = "Ended"

        return result

    def applyWorldParameters(self, params):
        if 'teamName0' in params:
            self.teams[0].teamName = params['teamName0']
        if 'teamName1' in params:
            self.teams[1].teamName = params['teamName1']
        gs = params.get('gameState', None)
        if gs == 'PreGame':
            self.gameState = GameState.PreGame
        elif gs == 'Starting':
            self.gameState = GameState.Starting
        elif gs == 'InProgress':
            self.gameState = GameState.InProgress
            if 'timeRunning' in params:
                self.startTime = timeNow() - params['timeRunning']
            if 'timeLimit' in params:
                self.gameDuration = params['timeLimit']
        elif gs == 'Ended':
            self.gameState = GameState.Ended

    def setLayout(self, layout):
        for team in self.teams:
            team.numOrbsOwned = 0
        self.map = MapState(self, layout)
        self.zoneWithId = self.map.zoneWithId
        self.zones = self.map.zones
        self.zoneBlocks = self.map.zoneBlocks

    def setTestMode(self):
        Player.STARS_TO_RESET_TO = 20
        for upgradetype in upgradeOfType.itervalues():
            upgradetype.requiredStars = 1
