# Trosnoth (UberTweak Platform Game)
# Copyright (C) 2006-2009 Joshua D Bartlett
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# version 2 as published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
# 02110-1301, USA.

from trosnoth.src.utils.components import Component, handler, Plug
from trosnoth.src.messages import (FireShotMsg, ShotFiredMsg, KillShotMsg,
        JoinApproved, JoinSuccessfulMsg, AddPlayerMsg, RemovePlayerMsg,
        CannotJoinMsg)

class IdManager(Component):
    '''
    Designed to be plugged on the server side. Takes all messages passed in
    (usually generated by the validator and universe) and passes them on after
    assigning shot ids, player ids, etc.
    '''
    inPlug = Plug()
    outPlug = Plug()

    def __init__(self):
        super(IdManager, self).__init__()
        self._usedPlayers = set()
        self._nextPlayer = 1
        self._usedShots = {}
        self._nextShot = {}

    @inPlug.defaultHandler
    def passOn(self, msg):
        self.outPlug.send(msg)

    @handler(FireShotMsg, inPlug)
    def fireShot(self, msg):
        shotId = self._nextShot.get(msg.playerId, 1)
        usedShots = self._usedShots.setdefault(msg.playerId, set())
        if len(usedShots) >= 255:
            return  # Cannot create any more shots for this player.
        while shotId in self._usedShots:
            shotId = shotId % 255 + 1
        usedShots.add(shotId)
        self._nextShot[msg.playerId] = shotId % 255 + 1
        self.outPlug.send(ShotFiredMsg(msg.playerId, chr(shotId), msg.angle,
                msg.xpos, msg.ypos, msg.type))

    @handler(KillShotMsg, inPlug)
    def killShot(self, msg):
        self._usedShots.get(msg.shooterId, set()).remove(ord(msg.shotId))
        self.outPlug.send(msg)

    @handler(JoinApproved, inPlug)
    def joinGame(self, msg):
        playerId = self._nextPlayer
        if len(self._usedPlayers) >= 255:
            # Cannot join: hard max on player ids.
            self.outPlug.send(CannotJoinMsg('F', msg.teamId, msg.tag))
            return

        while playerId in self._usedPlayers:
            playerId = playerId % 255 + 1
        self._usedPlayers.add(playerId)
        self._nextPlayer = playerId % 255 + 1
        self.outPlug.send(AddPlayerMsg(chr(playerId), msg.teamId, msg.zoneId,
                True, msg.nick))
        self.outPlug.send(JoinSuccessfulMsg(chr(playerId), msg.tag))

    @handler(RemovePlayerMsg, inPlug)
    def removePlayer(self, msg):
        try:
            self._usedPlayers.remove(ord(msg.playerId))
        except KeyError:
            pass
        else:
            self.outPlug.send(msg)

