# Trosnoth (UberTweak Platform Game)
# Copyright (C) 2006-2010 Joshua D Bartlett
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# version 2 as published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
# 02110-1301, USA.

import random

from trosnoth.gui.framework import framework
from trosnoth.gui.framework.elements import PictureElement
from trosnoth.gui.common import (Location, FullScreenAttachedPoint,
        ScaledSize, Area, Canvas)
from trosnoth.gui.framework.unobtrusiveValueGetter import YesNoGetter

from trosnoth.trosnothgui.ingame.messagebank import MessageBank
from trosnoth.trosnothgui.ingame.stars import StarGroup
from trosnoth.trosnothgui.ingame.udpnotify import UDPNotificationBar
from trosnoth.trosnothgui.ingame import mainMenu
from trosnoth.trosnothgui.ingame.gamevote import GameVoteMenu, NicknameBox
from trosnoth.trosnothgui.ingame.gauges import TurretGauge, RespawnGauge, \
                                                   UpgradeGauge
from trosnoth.trosnothgui.ingame.achievementBox import AchievementBox
from trosnoth.gui.framework.dialogbox import DialogResult
from trosnoth.trosnothgui.pregame.settingsMenu import SettingsMenu

from trosnoth.base import MenuError
from trosnoth.model.upgrades import Turret, Shield, MinimapDisruption, \
                                        PhaseShift, Grenade, Ricochet
from trosnoth.model.universe_base import GameState
from trosnoth.model.team import Team
from trosnoth.model.player import Player

from trosnoth.messages import (BuyUpgradeMsg, DeleteUpgradeMsg, SetCaptainMsg,
        TeamIsReadyMsg, RespawnRequestMsg, ChangeNicknameMsg)

class DetailsInterface(framework.CompoundElement):
    '''Interface containing all the overlays onto the screen:
    chat messages, player lists, gauges, stars, etc.'''
    def __init__(self, app, gameInterface):
        super(DetailsInterface, self).__init__(app)
        world = gameInterface.world
        self.gameInterface = gameInterface

        # Maximum number of messages viewable at any one time
        maxView = 8

        self.world = world
        self.player = None
        self.currentMessages = MessageBank(self.app, maxView, 50,
                                           Location(FullScreenAttachedPoint(ScaledSize(-40,-40),'bottomright'), 'bottomright'),
                                           'right', 'bottom',
                                           self.app.screenManager.fonts.messageFont)
        self.currentChats = MessageBank(self.app, maxView, 50, Location(FullScreenAttachedPoint(ScaledSize(40,256),'topleft'), 'topleft'), 'left', 'top',
                                        self.app.screenManager.fonts.messageFont)
        # If we want to keep a record of all messages and their senders
        self.input = None
        self.inputText = None
        self.unobtrusiveGetter = None
        self.turretGauge = None
        self.respawnGauge = None
        self.upgradeGauge = None
        self.achievementBox = None
        self.currentUpgrade = None
        self.starGroup = StarGroup(self.app)
        self.udpNotifier = UDPNotificationBar(self.app)
        self.settingsMenu = SettingsMenu(app, onClose=self.hideSettings, showThemes=False)

        menuloc = Location(FullScreenAttachedPoint((0,0), 'bottomleft'), 'bottomleft')
        self.menuManager = mainMenu.MainMenu(self.app, menuloc, self, self.gameInterface.keyMapping)
        self.upgradeDisplay = UpgradeDisplay(app)
        self.gameVoteMenu = GameVoteMenu(app, world, onChange=self._castGameVote)
        self._gameVoteUpdateCounter = 0
        self.elements = [self.currentMessages, self.currentChats, self.upgradeDisplay,
                         self.starGroup, self.gameVoteMenu, self.udpNotifier]

    @property
    def controller(self):
        return self.gameInterface.controller

    def tick(self, deltaT):
        super(DetailsInterface, self).tick(deltaT)
        if self.player is None:
            self.gameVoteMenu.active = False
            return
        else:
            if self._gameVoteUpdateCounter <= 0:
                self.gameVoteMenu.update(self.player)
                self._gameVoteUpdateCounter = 25
            else:
                self._gameVoteUpdateCounter -= 1

        self._updateTurretGauge()
        self._updateRespawnGauge()
        self._updateUpgradeGauge()
        self._updateAchievementBox()
        self._updateGameVoteMenu()

    def _castGameVote(self, msg):
        self.controller.send(msg)

    def _updateGameVoteMenu(self):
        if self.world.gameState == GameState.Ended:
            self.gameVoteMenu.active = True
        else:
            self.gameVoteMenu.active = False

    def localAchievement(self, achievementId):
        if self.achievementBox is None:
            self.achievementBox = AchievementBox(self.app, self.player,
                                                 achievementId)
            self.elements.append(self.achievementBox)
        else:
            self.achievementBox.addAchievement(achievementId)

    def _updateAchievementBox(self):
        if self.achievementBox is not None \
           and len(self.achievementBox.achievements) == 0:
            self.elements.remove(self.achievementBox)
            self.achievementBox = None

    def _updateTurretGauge(self):
        player = self.player
        if self.turretGauge is None:
            if player.turret:
                self.turretGauge = TurretGauge(self.app, Area(
                        FullScreenAttachedPoint(ScaledSize(0,-60),
                        'midbottom'), ScaledSize(100,30), 'midbottom'),
                        player)
                self.elements.append(self.turretGauge)
        elif not player.turret:
            self.elements.remove(self.turretGauge)
            self.turretGauge = None
            
    def _updateRespawnGauge(self):
        player = self.player
        if self.respawnGauge is None:
            if player.ghost:
                self.respawnGauge = RespawnGauge(self.app, Area(
                        FullScreenAttachedPoint(ScaledSize(0,-20),
                        'midbottom'), ScaledSize(100,30), 'midbottom'),
                        player)
                self.elements.append(self.respawnGauge)
        elif not player.ghost:
            self.elements.remove(self.respawnGauge)
            self.respawnGauge = None
            
    def _updateUpgradeGauge(self):
        player = self.player
        if self.upgradeGauge is None:
            if player.upgrade is not None:
                self.upgradeGauge = UpgradeGauge(self.app, Area(
                        FullScreenAttachedPoint(ScaledSize(0,-20),
                        'midbottom'), ScaledSize(100,30), 'midbottom'),
                        player)
                self.elements.append(self.upgradeGauge)
        elif player.upgrade is None:
            self.elements.remove(self.upgradeGauge)
            self.upgradeGauge = None

    def gameOver(self, winningTeam):
        self.gameInterface.gameOver(winningTeam)

    def setPlayer(self, player):
        self.player = player
        if self.menuManager not in self.elements:
            self.elements.append(self.menuManager)

    def showSettings(self):
        if self.settingsMenu not in self.elements:
            self.elements.append(self.settingsMenu)

    def hideSettings(self):
        if self.settingsMenu in self.elements:
            self.elements.remove(self.settingsMenu)
            self.gameInterface.updateKeyMapping()

    def _setCurrentUpgrade(self, upgradeType):
        self.currentUpgrade = upgradeType
        self.upgradeDisplay.setUpgrade(self.currentUpgrade)
        
    def _requestUpgrade(self):
        if self.currentUpgrade is not None:
            self.controller.send(BuyUpgradeMsg(self.player.id,
                    self.currentUpgrade.upgradeType, random.randrange(1<<32)))

    def _changeNickname(self):
        if self.world.gameState == GameState.InProgress:
            self.newMessage('Cannot change nickname after game has started',
                    self.app.theme.colours.errorMessageColour)
            return

        prompt = NicknameBox(self.app)
        @prompt.onClose.addListener
        def _customEntered():
            if prompt.result == DialogResult.OK:
                nickname = prompt.value
                self.controller.send(ChangeNicknameMsg(self.player.id,
                        nickname.encode()))
            
        prompt.Show()

    def doAction(self, action):
        '''
        Activated by hotkey or menu.
        action corresponds to the action name in the keyMapping.
        '''
        if action == 'leaderboard':
            self.showPlayerDetails()
            self.menuManager.manager.reset()
        elif action == 'toggle interface':
            self.toggleInterface()
            self.menuManager.manager.reset()
        elif action == 'more actions':
            if self.menuManager is not None:
                self.menuManager.showMoreMenu()
        elif action == 'settings':
            self.showSettings()
        elif action == 'maybeLeave':
            if self.menuManager is not None:
                self.menuManager.showQuitMenu()
        elif action == 'leave':
            # Disconnect from the server.
            self.gameInterface.disconnect()
        elif action == 'menu':
            # Return to main menu and show or hide the menu.
            if self.menuManager is not None:
                self.menuManager.escape()
        elif action == 'follow':
            if self.gameInterface.gameViewer.replay:
                # Replay-specific: follow the action.
                self.gameInterface.gameViewer.setTarget(None)
        elif action == 'toggle terminal':
            self.gameInterface.toggleTerminal()
        else:
            # All actions after this line should require a player.
            if self.player is None:
                return
            if action == 'respawn':
                self.controller.send(RespawnRequestMsg(self.player.id,
                        random.randrange(1<<32)))
            elif action == 'turret':
                self._setCurrentUpgrade(Turret)
                self.menuManager.manager.reset()
            elif action == 'shield':
                self._setCurrentUpgrade(Shield)
                self.menuManager.manager.reset()
            elif action == 'minimap disruption':
                self._setCurrentUpgrade(MinimapDisruption)
                self.menuManager.manager.reset()
            elif action == 'phase shift':
                self._setCurrentUpgrade(PhaseShift)
                self.menuManager.manager.reset()
            elif action == 'grenade':
                self._setCurrentUpgrade(Grenade)
                self.menuManager.manager.reset()
            elif action == 'ricochet':
                self._setCurrentUpgrade(Ricochet)
                self.menuManager.manager.reset()
            elif action == 'no upgrade':
                self._setCurrentUpgrade(None)
                self.menuManager.manager.reset()
            elif action == 'abandon':
                self.abandon(self.player)
                self.menuManager.manager.reset()
            elif action == 'chat':
                self.chat()
                self.menuManager.manager.reset()
            elif action == 'captain':
                self.becomeCaptain()
                self.menuManager.manager.reset()
            elif action == 'team ready':
                self.teamReady()
                self.menuManager.manager.reset()
            elif action == 'buy upgrade':
                if self.menuManager is not None:
                    self.menuManager.showBuyMenu()
            elif action == 'activate upgrade':
                self._requestUpgrade()
            elif action == 'change nickname':
                self._changeNickname()
            elif action not in ('jump', 'right', 'left', 'down'):
                print 'Unknown action: %r' % (action,)

    def newMessage(self, text, colour=None):
        if colour is None:
            colour = self.app.theme.colours.grey
        self.currentMessages.newMessage(text, colour)

    def newChat(self, text, sender, private):
        if sender is self.player:
            # Don't bother receiving messages from myself.
            return

        nick = sender.nick
        if not private:
            # Message for everyone
            extra = ': '

        elif isinstance(private, Player) and self.player is private:
            # Destined for the one player
            extra = ' (private): '

        elif isinstance(private, Team) and self.player is not None and \
                self.player.team == private:
            # Destined for the one team.
            extra = ': '

        else:
            # May not have been destined for our player after all.
            print "Don't want this text"
            return
        allText = nick + extra + text
        self.currentChats.newMessage(allText,
                self.app.theme.colours.chatColour(sender.team))

    def sentPublicChat(self, team, text):
        self.currentChats.newMessage('Me (open): %s' % (text,),
                self.app.theme.colours.chatColour(team))

    def sentPrivateChat(self, team, nick, text):
        self.currentChats.newMessage('Me (to_%s): %s' % (nick, text),
                self.app.theme.colours.chatColour(team))

    def sentTeamChat(self, team, text):
        self.currentChats.newMessage('Me: %s' % (text,),
                self.app.theme.colours.chatColour(team))

    def processEvent(self, event):
        '''Event processing works in the following way:
        1. If there is a prompt on screen, the prompt will either use the
        event, or pass it on.
        2. If passed on, the event will be sent back to the main class, for it
        to process whether player movement uses this event. If it doesn't use
        the event, it will pass it back.
        3. If so, the hotkey manager will see if the event means anything to it.
        If not, that's the end, the event is ignored.'''

        try:
            return super(DetailsInterface, self).processEvent(event)
        except MenuError, err:
            self.newMessage(err.value, self.app.theme.colours.errorMessageColour)
            self.endInput()
            return None


    def endInput(self):
        if self.input:
            self.elements.remove(self.input)
            self.input = None
        if self.inputText:
            self.elements.remove(self.inputText)
            self.inputText = None
        if self.unobtrusiveGetter:
            self.elements.remove(self.unobtrusiveGetter)
            self.unobtrusiveGetter = None
        if self.menuManager is not None and self.menuManager not in self.elements:
            self.elements.append(self.menuManager)
        self.input = self.inputText = None


    def inputStarted(self):
        self.elements.append(self.input)
        self.elements.append(self.inputText)
        self.input.onEsc.addListener(lambda sender: self.endInput())
        self.input.onEnter.addListener(lambda sender: self.endInput())
        if self.menuManager is not None:
            try:
                self.elements.remove(self.menuManager)
            except ValueError:
                pass



    def chat(self):
        if not self.player:
            return

        self.gameInterface.gameViewer.openChat(self.player)


    def abandon(self, player):
        '''
        Called when a player says they wish to abandon their upgrade.
        '''
        if player.upgrade:
            addOn = 'upgrade'
            if type(player.upgrade) == Grenade:
                self.newMessage('Cannot abandon an active grenade!',
                        self.app.theme.colours.errorMessageColour)
                raise MenuError, "Cannot abandon an active grenade!"

        else:
            return

        message = 'Really abandon your ' + addOn + ' (Y/N)'
        self.endInput()
        self.unobtrusiveGetter = YesNoGetter(self.app, Location(
                FullScreenAttachedPoint((0,0), 'center'), 'center'), message,
                self.app.screenManager.fonts.unobtrusivePromptFont,
                self.app.theme.colours.unobtrusivePromptColour, 3)
        self.elements.append(self.unobtrusiveGetter)

        def gotValue(abandon):
            if self.unobtrusiveGetter:
                self.elements.remove(self.unobtrusiveGetter)
                self.unobtrusiveGetter = None
            if abandon:
                self.controller.send(DeleteUpgradeMsg(player.id, player.upgrade.upgradeType, 'A'))

        self.unobtrusiveGetter.onGotValue.addListener(gotValue)


    def upgradeUsed(self, player):
        upgrade = player.upgrade
        if upgrade is None:
            upgrade = 'an upgrade'
        else:
            upgrade = repr(upgrade)
        message = '%s is using %s' % (player.nick, player.upgrade)
        self.newMessage(message)

    def upgradeDestroyed(self, player, upgrade):
        if upgrade is None:
            upgrade = 'upgrade'
        else:
            upgrade = repr(upgrade)
        message = "%s's %s is gone" % (player.nick, upgrade)
        self.newMessage(message)

    def becomeCaptain(self):
        if self.player:
            self.controller.send(SetCaptainMsg(self.player.id))

    def teamReady(self):
        if self.player:
            self.controller.send(TeamIsReadyMsg(self.player.id))

    def showPlayerDetails(self):
        self.gameInterface.gameViewer.toggleLeaderBoard()

    def toggleInterface(self):
        self.gameInterface.gameViewer.toggleInterface()

class UpgradeDisplay(framework.CompoundElement):

    def __init__(self, app):
        super(UpgradeDisplay, self).__init__(app)

    def setUpgrade(self, upgradeType):
        if upgradeType is None:
            self.elements = []
        else:
            pos = Location(Canvas(620, 0), "midtop")
            image = self.app.theme.sprites.upgradeImage(upgradeType)
            self.elements = [PictureElement(self.app, image, pos)]
