# Trosnoth (UberTweak Platform Game)
# Copyright (C) 2006-2010 Joshua D Bartlett
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# version 2 as published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
# 02110-1301, USA.

from trosnoth.utils.components import Component, handler, Plug
from trosnoth.messages import (FireShotMsg, ShotFiredMsg, KillShotMsg,
        JoinApproved, JoinSuccessfulMsg, AddPlayerMsg, RemovePlayerMsg,
        CannotJoinMsg, CreateCollectableStarMsg, RemoveCollectableStarMsg,
        ShotAbsorbedMsg)

MAX_SHOTS = 1 << 32 - 1
MAX_STARS = 255

class IdManager(Component):
    '''
    Designed to be plugged on the server side. Takes all messages passed in
    (usually generated by the validator and universe) and passes them on after
    assigning shot ids, player ids, etc.
    '''
    inPlug = Plug()
    outPlug = Plug()

    def __init__(self):
        super(IdManager, self).__init__()
        self._usedPlayers = set()
        self._nextPlayer = 1
        self._usedShots = set()
        self._nextShot = 1
        self._usedStars = set()
        self._nextStar = 1

    @inPlug.defaultHandler
    def passOn(self, msg):
        self.outPlug.send(msg)

    @handler(FireShotMsg, inPlug)
    def fireShot(self, msg):
        shotId = self._nextShot
        if len(self._usedShots) >= MAX_SHOTS:
            return  # Cannot create any more shots.
        while shotId in self._usedShots:
            shotId = shotId % MAX_SHOTS + 1
        self._usedShots.add(shotId)
        self._nextShot = shotId % MAX_SHOTS + 1
        self.outPlug.send(ShotFiredMsg(msg.playerId, shotId, msg.angle,
                msg.xpos, msg.ypos, msg.type))

    @handler(KillShotMsg, inPlug)
    @handler(ShotAbsorbedMsg, inPlug)
    def killShot(self, msg):
        self._usedShots.remove(msg.shotId)
        self.outPlug.send(msg)

    @handler(CreateCollectableStarMsg, inPlug)
    def createCollectableStar(self, msg):
        starId = self._nextStar
        if len(self._usedStars) >= MAX_STARS:
            return  # Cannot create any more stars.
        while starId in self._usedStars:
            starId = starId % MAX_STARS + 1
        self._usedStars.add(starId)
        self._nextStar = starId % MAX_STARS + 1
        self.outPlug.send(CreateCollectableStarMsg(chr(starId), msg.teamId,
                msg.xPos, msg.yPos))

    @handler(RemoveCollectableStarMsg, inPlug)
    def removeCollectableStar(self, msg):
        self._usedStars.remove(ord(msg.starId))
        self.outPlug.send(msg)

    @handler(JoinApproved, inPlug)
    def joinGame(self, msg):
        playerId = self._nextPlayer
        if len(self._usedPlayers) >= 255:
            # Cannot join: hard max on player ids.
            self.outPlug.send(CannotJoinMsg('F', msg.teamId, msg.tag))
            return

        while playerId in self._usedPlayers:
            playerId = playerId % 255 + 1
        self._usedPlayers.add(playerId)
        self._nextPlayer = playerId % 255 + 1
        self.outPlug.send(AddPlayerMsg(chr(playerId), msg.teamId, msg.zoneId,
                True, msg.bot, msg.nick))
        self.outPlug.send(JoinSuccessfulMsg(chr(playerId), msg.tag, msg.username))

    @handler(RemovePlayerMsg, inPlug)
    def removePlayer(self, msg):
        try:
            self._usedPlayers.remove(ord(msg.playerId))
        except KeyError:
            pass
        else:
            self.outPlug.send(msg)

