# Trosnoth (UberTweak Platform Game)
# Copyright (C) 2006-2011 Joshua D Bartlett
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# version 2 as published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
# 02110-1301, USA.

'''universe.py - defines anything that has to do with the running of the
universe. This includes players, shots, zones, and the level itself.'''

import logging
from math import sin, cos
import random

from trosnoth.utils.utils import timeNow
from trosnoth.utils.math import distance
from trosnoth.utils.checkpoint import checkpoint
from trosnoth.model.universe_base import GameState, GameRules
from trosnoth.model.upgrades import upgradeOfType, allUpgrades
from trosnoth.model.physics import WorldPhysics

from trosnoth.model.map import MapLayout, MapState
from trosnoth.model.shot import Shot
from trosnoth.model.player import Player
from trosnoth.model.star import CollectableStar
from trosnoth.model.team import Team

# Component message passing
from trosnoth.utils.components import Component, handler, Plug
from trosnoth.utils.network import expand_boolean
from trosnoth.utils.twist import WeakCallLater, WeakLoopingCall
from trosnoth.messages import (ChatMsg, TaggingZoneMsg, ShotFiredMsg,
        PlayerUpdateMsg, KillShotMsg, CannotRespawnMsg, RespawnMsg,
        PlayerKilledMsg, PlayerHitMsg, GameStartMsg, GameOverMsg, SetCaptainMsg,
        TeamIsReadyMsg, StartingSoonMsg, SetTeamNameMsg, SetGameModeMsg,
        AddPlayerMsg, CannotJoinMsg, JoinSuccessfulMsg, UpdatePlayerStateMsg,
        AimPlayerAtMsg, RemovePlayerMsg, PlayerHasUpgradeMsg, ShotAbsorbedMsg,
        PlayerStarsSpentMsg, DeleteUpgradeMsg, CannotBuyUpgradeMsg,
        SetPlayerTeamMsg, ConnectionLostMsg, GrenadeExplosionMsg,
        AchievementUnlockedMsg, ZoneStateMsg, WorldResetMsg,
        QueryWorldParametersMsg, PlayerIsReadyMsg, SetPreferredTeamMsg,
        SetPreferredDurationMsg, SetPreferredSizeMsg, CreateCollectableStarMsg,
        RemoveCollectableStarMsg, PlayerHasElephantMsg, ChangeNicknameMsg,
        FireShoxwaveMsg, MarkZoneMsg, SetGameSpeedMsg)

DIRTY_DELAY = 0.05
UPDATE_DELAY = 1        # PlayerUpdateMsg is sent for each player this often.
TICK_PERIOD = 0.02

DEFAULT_TEAM_NAME_1 = 'Defaulticons'
DEFAULT_TEAM_NAME_2 = 'Standardators'

GAME_STATE_CHECK_PERIOD = 5
STALEMATE_CHECK_PERIOD = 1
COLLECTABLE_STAR_LIFETIME = 12
NO_COLLECTABLE_STARS = False        # For debugging.

# The time after a zone tag after which there is 50% of a new
# collectable star appearing every STALEMATE_CHECK_PERIOD.
ZONE_TAG_STAR_HALF_LIFE = 60

log = logging.getLogger('universe')

class Abort(Exception):
    pass

class Universe(Component):
    '''Universe(halfMapWidth, mapHeight)
    Keeps track of where everything is in the level, including the locations
    and states of every alien, the terrain positions, and who owns the
    various territories and orbs.'''

    # eventPlug is a sending plug
    # Events which are generated by the tick() method of the universe
    # should be sent to this plug. This indicates that the universe thinks
    # that the event should happen.
    eventPlug = Plug()

    # orderPlug is a receiving plug
    # When an object wants to order the universe to do something,
    # they should send a message to this plug
    orderPlug = Plug()

    PLAYER_RESET_STARS = 0

    def __init__(self, halfMapWidth=3, mapHeight=2, gameDuration=0,
            authTagManager=None, voting=False, layoutDatabase=None,
            clientOptimised=False, gameName=None, onceOnly=False):
        '''
        halfMapWidth:   is the number of columns of zones in each team's
                        territory at the start of the game. There will always
                        be a single column of neutral zones between the two
                        territories at the start of the game.
        mapHeight:      is the number of zones in every second column of
                        zones. Every other column will have mapHeight + 1
                        zones in it. This is subject to the constraints that
                        (a) the columns at the extreme ends of the map will
                        have mapHeight zones; (b) the central (initially
                        neutral) column of zones will never have fewer zones
                        in it than the two bordering it - this will sometimes
                        mean that the column has mapHeight + 2 zones in it.
        clientOptimised: if True, this universe is not guaranteed to detect all
            events which occur and send them along the eventPlug.
        '''
        super(Universe, self).__init__()

        self.playerWithElephant = None
        self.gameName = gameName
        self.clientOptimised = clientOptimised
        self.physics = WorldPhysics(self)
        self.gameDuration = gameDuration
        self.layoutDatabase = layoutDatabase
        self.onceOnly = onceOnly
        if authTagManager is None:
            self.authManager = None
        else:
            self.authManager = authTagManager.authManager

        # Initialise
        if voting:
            self.gameState = GameState.Ended
        else:
            self.gameState = GameState.PreGame
        self.gameRules = GameRules.Normal
        self.winningTeam = None
        self.zonesToReset = []
        self.playerWithId = {}
        self.huntTeamWithZones = None
        self.teamWithId = {'\x00' : None}

        # Create Teams:
        self.teams = (
            Team(self, 'A'),
            Team(self, 'B'),
        )
        Team.setOpposition(self.teams[0], self.teams[1])

        for t in self.teams:
            self.teamWithId[t.id] = t

        # Set up zones
        self.zoneWithDef = {}
        layout = MapLayout(halfMapWidth, mapHeight)
        self.map = MapState(self, layout)
        self.zoneWithId = self.map.zoneWithId
        self.zones = self.map.zones
        self.zoneBlocks = self.map.zoneBlocks

        self.players = set()
        self.playerUpdateTimes = {}
        self.grenades = set()
        self.collectableStars = {}      # starId -> CollectableStar
        self.shots = set()
        self.gameOverTime = None
        self.gameMode = 'Normal'
        self.rogueTeamName = 'Rogue'

        self._lastTime = timeNow()
        self._gameTime = 0.0
        self._gameSpeed = 1.0
        self._lastStalemateCheck = self._gameTime
        self._lastGameStateCheck = self._gameTime
        self.lastZoneTagged = self._gameTime
        self._loop = None
        self._startClock()

    @handler(CannotBuyUpgradeMsg, orderPlug)
    @handler(CannotJoinMsg, orderPlug)
    @handler(CannotRespawnMsg, orderPlug)
    @handler(ConnectionLostMsg, orderPlug)
    @handler(ChatMsg, orderPlug)
    @handler(GrenadeExplosionMsg, orderPlug)
    @handler(QueryWorldParametersMsg, orderPlug)
    def ignore(self, msg):
        pass

    @handler(MarkZoneMsg, orderPlug)
    def markZone(self, msg):
        try:
            zone = self.getZone(msg.zoneId)
            player = self.getPlayer(msg.playerId)

            zone.markedBy[player.team] = msg.value
        except Abort:
            pass

    @handler(AchievementUnlockedMsg, orderPlug)
    def achievementUnlocked(self, msg):
        try:
            player = self.getPlayer(msg.playerId)
            if player.user is not None:
                player.user.achievementUnlocked(msg)
        except Abort:
            pass

    def reset(self, layout, gameDuration=None):
        if gameDuration is not None:
            self.gameDuration = gameDuration
        self.setLayout(layout)

        for player in self.players:
            self._resetPlayer(player)

        self.gameState = GameState.PreGame
        self.eventPlug.send(WorldResetMsg())

    def _resetPlayer(self, player):
        zone = self.selectZone(player.teamId)
        player.pos = zone.defn.pos
        player.changeZone(zone)
        player.die()
        player.respawnGauge = 0.0

        # Make sure player knows its map block.
        i, j = MapLayout.getMapBlockIndices(*player.pos)
        try:
            block = self.zoneBlocks[i][j]
        except IndexError, e:
            log.exception(str(e))
            raise IndexError, 'player start position is off the map'
        player.setMapBlock(block)

    def selectZone(self, teamId):
        allTeamZones = [z for z in self.map.zones if z.orbOwner is not None and
                z.orbOwner.id == teamId]
        actionZones = []
        for zone in list(allTeamZones):
            for adj in zone.defn.adjacentZones.iterkeys():
                if (adj is not None and (self.zoneWithDef[adj].orbOwner is None
                        or self.zoneWithDef[adj].orbOwner.id != teamId)):
                    actionZones.append(zone)
        if len(actionZones) > 0:
            return random.choice(actionZones)
        elif len(allTeamZones) > 0:
            return random.choice(allTeamZones)
        return random.choice(list(self.map.zones))

    @handler(WorldResetMsg, orderPlug)
    def gotWorldReset(self, msg):
        self.grenades = set()
        self.shots = set()
        self.gameOverTime = None
        self._gameTime = 0.0
        self._lastStalemateCheck = self._gameTime
        self._lastGameStateCheck = self._gameTime
        self.lastZoneTagged = self._gameTime
        self.gameState = GameState.PreGame

    @handler(JoinSuccessfulMsg, orderPlug)
    def joinSuccessful(self, msg):
        try:
            player = self.getPlayer(msg.playerId)
            if self.authManager is not None:
                player.user = self.authManager.getUserByName(msg.username)
        except Abort:
            pass

        if player.isElephantOwner():
            self.playerWithElephant = player
            self.eventPlug.send(PlayerHasElephantMsg(player.id))

    def getTeam(self, teamId):
        if teamId == '\x00':
            return None
        try:
            return self.teamWithId[teamId]
        except KeyError:
            raise Abort()

    def getPlayer(self, playerId):
        try:
            return self.playerWithId[playerId]
        except KeyError:
            raise Abort()

    def getUpgradeType(self, upgradeTypeId):
        try:
            return upgradeOfType[upgradeTypeId]
        except KeyError:
            raise Abort()

    def getZone(self, zoneId):
        try:
            return self.map.zoneWithId[zoneId]
        except KeyError:
            raise Abort()

    def getShot(self, pId, sId):
        try:
            return self.playerWithId[pId].shots[sId]
        except KeyError:
            raise Abort()

    def playerIsDirty(self, pId):
        '''
        Registers the given players as "dirty" in that its location details
        should be sent to all clients at some stage in the near future. This is
        called when, for example, the player releases the jump key before the
        top of the jump, or collides with an obstacle, or such things.
        '''
        self.playerUpdateTimes[pId] = min(self.playerUpdateTimes.get(pId, 1),
                timeNow() + DIRTY_DELAY)

    def _sendPlayerUpdate(self, pId):
        '''
        Sends state information of a dirty player to all clients.
        '''
        self.playerUpdateTimes[pId] = timeNow() + UPDATE_DELAY

        try:
            player = self.playerWithId[pId]
        except KeyError:
            return

        self.eventPlug.send(player.makePlayerUpdate())

    @handler(UpdatePlayerStateMsg, orderPlug)
    def updatePlayerState(self, msg):
        try:
            player = self.getPlayer(msg.playerId)
            player.updateState(msg.stateKey, msg.value)
        except Abort:
            pass

    @handler(AimPlayerAtMsg, orderPlug)
    def aimPlayerAt(self, msg):
        try:
            player = self.getPlayer(msg.playerId)
            player.lookAt(msg.angle, msg.thrust)
        except Abort:
            pass

    @handler(SetGameModeMsg, orderPlug)
    def setGameMode(self, msg):
        mode = msg.gameMode.decode()
        if self.physics.setMode(mode):
            self.gameMode = mode
            log.debug('Client: GameMode is set to ' + mode)

    @handler(SetGameSpeedMsg, orderPlug)
    def setGameSpeed(self, msg):
        '''Sets the speed of the game to a proportion of normal speed.
        That is, speed=2.0 is twice as fast a game as normal
        '''
        speed = msg.gameSpeed
        self._gameSpeed = speed

    @handler(SetTeamNameMsg, orderPlug)
    def setTeamName(self, msg):
        if msg.teamId == '\x00':
            self.rogueTeamName = msg.name
        else:
            try:
                team = self.getTeam(msg.teamId)
                team.teamName = msg.name
            except Abort():
                pass

    def getTimeLeft(self):
        if self.gameState in (GameState.PreGame, GameState.Starting):
            return None
        elif self.gameState == GameState.InProgress:
            return self.gameDuration - self._gameTime
        elif self.gameState == GameState.Ended:
            return self.gameDuration - self.gameOverTime

    @handler(GameOverMsg, orderPlug)
    def gameOver(self, msg):
        self.gameOverTime = self._gameTime
        self.gameState = GameState.Ended
        self.winningTeam = self.teamWithId.get(msg.teamId, None)
        for player in self.players:
            player.readyToStart = False
            if player.ghost and player.team is not None:
                # Give a chance for stats to save
                WeakCallLater(0.5, self.eventPlug, 'send',
                        SetPlayerTeamMsg(player.id, '\x00'))

    def setHuntedMode(self, winningTeam):
        if winningTeam == None:
            huntTeamWithZones = random.choice(self.teams)
        else:
            huntTeamWithZones = winningTeam
        self.setHuntTeamWithZones(huntTeamWithZones)

    def setHuntTeamWithZones(self, huntTeamWithZones):
        log.debug('Trying to hunt with %s' % (huntTeamWithZones,))
        # Firstly, if somehow all players are on the one zone, swap one to the
        # other team
        if len(self.players) < 4:
            return
        allTeams = set([])
        for player in self.players:
            allTeams.add(player.team)
        if len(allTeams) == 0:
            # No players in the game
            pass
        elif len(allTeams) == 1:
            player = random.choice(self.players)
            team = list(allTeams)[0].opposingTeam
            self.eventPlug.send(SetPlayerTeamMsg(player.id, team.id))
            WeakCallLater(3, self, 'setHuntedMode', team)
        else:
            # Now, check by who is alive and dead on each team
            playersWithoutZones = 0
            playersWithZones = 0
            playersToSpawn = []
            playersToSwapTeam = []
            for player in self.players:
                if player.ghost:
                    playersToSpawn.append(player)
                    if player.team == huntTeamWithZones:
                        playersToSwapTeam.append(player)
                else:
                    if player.team != huntTeamWithZones:
                        playersWithoutZones += 1
                    else:
                        playersWithZones += 1
            if playersWithoutZones < 2: # Having only one player won't work
                WeakCallLater(3, self, 'setHuntedMode',
                        huntTeamWithZones.opposingTeam)
            else:
                log.debug('Hunting!')
                self.huntTeamWithZones = huntTeamWithZones
                self.gameRules = GameRules.Hunting
                if playersWithZones == 0:
                    # We're about to swap everyone onto the one team. Stop this!
                    playersToSwapTeam.remove(random.choice(playersToSwapTeam))

                for p in playersToSwapTeam:
                    self.eventPlug.send(SetPlayerTeamMsg(p.id,
                            huntTeamWithZones.opposingTeam.id))
                for p in playersToSpawn:
                    self.eventPlug.send(RespawnMsg(p.id, player.currentZone.id))
                for zone in self.zones:
                    self.eventPlug.send(ZoneStateMsg(zone.id,
                            self.huntTeamWithZones.id, True, ''))



    @handler(GameStartMsg, orderPlug)
    def gameStart(self, msg):
        time = msg.timeLimit
        self.gameDuration = time
        self.lastZoneTagged = self.startTime = self._gameTime
        self.gameState = GameState.InProgress
        self._gameTime = 0

    @handler(AddPlayerMsg, orderPlug)
    def addPlayer(self, msg):
        team = self.teamWithId[msg.teamId]
        zone = self.zoneWithId[msg.zoneId]

        # Create the player.
        nick = msg.nick.decode()
        self._makePlayer(nick, team, msg.playerId, zone, msg.dead, msg.bot)

    @handler(SetPlayerTeamMsg, orderPlug)
    def setPlayerTeam(self, msg):
        try:
            player = self.getPlayer(msg.playerId)
            team = self.getTeam(msg.teamId)

            player.team = team
        except Abort:
            pass


    def _makePlayer(self, nick, team, playerId, zone, dead, bot):
        # Check if this is a duplicate msg.
        if playerId in self.playerWithId:
            player = self.playerWithId[playerId]
            player.changeZone(zone)
        else:
            player = Player(self, nick, team, playerId, zone, dead, bot)

            # Add this player to this universe.
            self.players.add(player)
            self.playerWithId[playerId] = player

        # Make sure player knows its zone
        i, j = MapLayout.getMapBlockIndices(*player.pos)
        try:
            block = self.zoneBlocks[i][j]
        except IndexError, e:
            log.exception(str(e))
            raise IndexError, 'player start position is off the map'
        player.setMapBlock(block)

    @handler(PlayerHasElephantMsg, orderPlug)
    def gotElephantMsg(self, msg):
        if msg.playerId is None:
            self.playerWithId = None
            return

        try:
            player = self.getPlayer(msg.playerId)
            self.playerWithElephant = player
        except Abort:
            pass


    @handler(RemovePlayerMsg, orderPlug)
    def delPlayer(self, msg):
        try:
            player = self.getPlayer(msg.playerId)
            player.removeFromGame()
            self.players.remove(player)
            del self.playerWithId[player.id]
            # In case anyone else keeps a reference to it
            player.id = -1
        except Abort:
            pass

    def _startClock(self):
        if self._loop is not None:
            self._loop.stop()
        self._loop = WeakLoopingCall(self, 'tick')
        self._loop.start(TICK_PERIOD, False)

    def stopClock(self):
        if self._loop is not None:
            self._loop.stop()
            self._loop = None

    def tick(self):
        '''Advances all players and shots to their new positions.'''
        t = timeNow()
        deltaT = t - self._lastTime
        try:
            self.frameRate = 1. / deltaT
        except ZeroDivisionError:
            self.frameRate = 0
        self._lastTime = t
        gameDeltaT = (deltaT * self._gameSpeed)
        self._gameTime += gameDeltaT

        # Update the player and shot positions.
        for s in (list(self.shots) + list(self.players) + list(self.grenades) +
                self.collectableStars.values()):
            s.reset()
            s.update(gameDeltaT)

        for zone in self.zonesToReset:
            zone.taggedThisTime = False
        self.zonesToReset = []

        if not self.clientOptimised:
            self.checkForResult()
            self.processPlayerUpdates()
            self.processGameState()
            self.checkStalemate()
            self.updateCollectableStars()

    def getGameTime(self):
        '''The amount of time since the start of the game.
        This is influnced by any game speed factors
        '''
        return self._gameTime

    def updateCollectableStars(self):
        for star in self.collectableStars.values():
            if star.creationTime < self._gameTime - COLLECTABLE_STAR_LIFETIME:
                self.eventPlug.send(RemoveCollectableStarMsg(star.id, '\x00'))

    def checkStalemate(self):
        if NO_COLLECTABLE_STARS:
            return
        if self._gameTime < self._lastStalemateCheck + STALEMATE_CHECK_PERIOD:
            return
        self._lastStalemateCheck = self._gameTime

        ePlayerId = '\x00'
        if self.playerWithElephant:
            ePlayerId = self.playerWithElephant.id
        if self.gameState != GameState.InProgress:
            return

        self.eventPlug.send(PlayerHasElephantMsg(ePlayerId))
        prob = 0.5 ** ((self._gameTime - self.lastZoneTagged) /
                ZONE_TAG_STAR_HALF_LIFE)
        if random.random() > prob:
            team = random.choice(self.teams)
            zone = self.selectZone(team.id)
            x, y = zone.defn.randomPosition()
            self.eventPlug.send(CreateCollectableStarMsg('0', team.id, x, y))

    @handler(CreateCollectableStarMsg, orderPlug)
    def createCollectableStar(self, msg):
        self.addCollectableStar(CollectableStar(self, msg.starId, (msg.xPos,
                msg.yPos), self.getTeam(msg.teamId)))

    @handler(RemoveCollectableStarMsg, orderPlug)
    def gotRemoveCollectableStarMsg(self, msg):
        try:
            star = self.collectableStars[msg.starId]
            star.currentMapBlock.removeCollectableStar(star)
            del self.collectableStars[msg.starId]
            try:
                player = self.playerWithId[msg.playerId]
            except KeyError:
                pass
            else:
                if star.team is None or player.team == star.team:
                    player.incrementStars()
        except KeyError:
            pass

    def processPlayerUpdates(self):
        playerIds = set(self.playerWithId.keys())
        now = timeNow()
        for pId, time in list(self.playerUpdateTimes.iteritems()):
            if pId not in playerIds:
                del self.playerUpdateTimes[pId]
                continue
            playerIds.remove(pId)
            if now > time:
                self._sendPlayerUpdate(pId)

        for pId in playerIds:
            # Not yet in dict.
            self.playerUpdateTimes[pId] = now + UPDATE_DELAY

    def checkForResult(self):
        if not self.gameState == GameState.InProgress:
            return

        # Check first for timeout
        if self.gameDuration > 0 and self.getTimeLeft() <= 0:
            if self.teams[0].numOrbsOwned > self.teams[1].numOrbsOwned:
                self.eventPlug.send(GameOverMsg(self.teams[0].id, True))
                checkpoint('Universe: Out of time, blue wins')
            elif self.teams[1].numOrbsOwned > self.teams[0].numOrbsOwned:
                self.eventPlug.send(GameOverMsg(self.teams[1].id, True))
                checkpoint('Universe: Out of time, red wins')
            else:
                self.eventPlug.send(GameOverMsg('\x00', True))
                checkpoint('Universe: Out of time, game drawn')
            return

        # Now check for an all zones win.
        team2Wins = self.teams[0].isLoser()
        team1Wins = self.teams[1].isLoser()
        if team1Wins and team2Wins:
            # The extraordinarily unlikely situation that all
            # zones have been neutralised in the same tick
            self.eventPlug.send(GameOverMsg('\x00', False))
            checkpoint('Universe: Draw due to all zones neutralised')
        elif team1Wins:
            self.eventPlug.send(GameOverMsg(self.teams[0].id, False))
            checkpoint('Universe: Zones captured, blue wins')
        elif team2Wins:
            self.eventPlug.send(GameOverMsg(self.teams[1].id, False))
            checkpoint('Universe: Zones captured, red wins')

    def processGameState(self):
        if self.gameState != GameState.Ended or self.onceOnly:
            return
        if self._gameTime < self._lastGameStateCheck + GAME_STATE_CHECK_PERIOD:
            return
        self._lastGameStateCheck = self._gameTime
        if self.layoutDatabase is None:
            return

        readyPlayerCount = 0
        actualPlayerNum = 0
        for player in self.players:
            if not player.bot:
                actualPlayerNum += 1
                if player.readyToStart:
                    readyPlayerCount += 1
        if actualPlayerNum <= 1:
            return
        if not readyPlayerCount >= 0.8 * actualPlayerNum:
            return

        result = self._getNewTeams()
        if result is None:
            return
        teamName1, players1, teamName2, players2 = result

        size = self._getNewSize(min(len(players1), len(players2)))
        duration = self._getNewDuration(size)

        # Set team names.
        self.teams[0].teamName = teamName1
        self.teams[1].teamName = teamName2

        # Set player teams.
        for player in players1:
            team = self.teams[0]
            self.eventPlug.send(SetPlayerTeamMsg(player.id, team.id))
            player.team = team
        for player in players2:
            team = self.teams[1]
            self.eventPlug.send(SetPlayerTeamMsg(player.id, team.id))
            player.team = team

        layout = self.layoutDatabase.generateRandomMapLayout(size[0], size[1])
        self.reset(layout, duration)
        WeakCallLater(0.1, self, '_startSoon', 12)

    def _getNewDuration(self, mapSize):
        results = {}
        for player in self.players:
            duration = player.preferredDuration
            results[duration] = results.get(duration, 0) + 1
        items = results.items()
        items.sort(key=lambda (duration, count): count)
        items.reverse()
        duration, count = items[0]
        if results.get(0, 0) != count:
            return duration

        # Decide default based on map size.
        if mapSize == (3, 2):
            return 45 * 60
        elif mapSize == (1, 1):
            return 10 * 60
        elif mapSize == (5, 1):
            return 20 * 60
        else:
            return min(7200, 2 * 60 * (mapSize[0]*2+1) * (mapSize[1]*1.5))

    def _getNewSize(self, teamSize):
        '''
        Returns a new map size based on what players vote for, with the
        defaults (if most people select Auto) being determined by the size of
        the smaller team.
        '''
        results = {}
        for player in self.players:
            size = player.preferredSize
            results[size] = results.get(size, 0) + 1
        items = results.items()
        items.sort(key=lambda (size, count): count)
        items.reverse()
        size, count = items[0]
        if results.get((0, 0), 0) != count:
            return size

        # Decide size based on player count.
        if teamSize <= 3:
            return (1, 1)
        elif teamSize <= 4:
            return (5, 1)
        else:
            return (3, 2)

    def _getNewTeams(self):
        '''
        Returns (teamName1, players1, teamName2, players2) based on what teams
        people have selected as their preferred teams.
        '''
        desiredTeams = self._getDesiredTeams()
        if len(desiredTeams) == 1:
            if desiredTeams[0][0] != '':
                return # Everyone is on one team.
            teamName1 = ''
            teamName2 = ''
            players1 = []
            players2 = []
            others = list(self.players)
        else:
            fairLimit = (len(self.players) + 1) / 2
            teamName1, players1 = desiredTeams[0]
            if len(players1) > fairLimit:
                # Require every player on the disadvantaged team to be ready.
                for player in self.players:
                    if player.preferredTeam != teamName1:
                        if not player.readyToStart:
                            return
            teamName2, players2 = desiredTeams[1]
            others = []
            for teamName, players in desiredTeams[2:]:
                others.extend(players)

        if teamName1 == '':
            if teamName2 != DEFAULT_TEAM_NAME_1:
                teamName1 = DEFAULT_TEAM_NAME_1
            else:
                teamName1 = DEFAULT_TEAM_NAME_2
        if teamName2 == '':
            if teamName1 != DEFAULT_TEAM_NAME_1:
                teamName2 = DEFAULT_TEAM_NAME_1
            else:
                teamName2 = DEFAULT_TEAM_NAME_2

        random.shuffle(others)
        for player in others:
            count1 = len(players1)
            count2 = len(players2)
            if count1 > count2:
                players2.append(player)
            elif count2 > count1:
                players1.append(player)
            else:
                random.choice([players1, players2]).append(player)

        return teamName1, players1, teamName2, players2

    def _getDesiredTeams(self):
        '''
        Returns a sorted sequence of duples of the form (teamName, players)
        where teamName is a unicode/string and players is a list of players. The
        sequence will be sorted from most popular to least popular.
        '''
        results = {}
        for player in self.players:
            teamName = player.preferredTeam
            results.setdefault(teamName, []).append(player)
        items = results.items()
        items.sort(key=lambda (teamName, players): len(players))
        items.reverse()
        return items

    def getTeamName(self, id):
        if id == '\x00':
            return self.rogueTeamName
        return self.getTeam(id).teamName

    @handler(FireShoxwaveMsg, orderPlug)
    def shoxwaveExplosion(self, msg):
    	radius = 128
    	# Get the player who fired this shoxwave
    	shoxPlayer = self.getPlayer(msg.playerId)
        self._update_reload_time(shoxPlayer)

        # Loop through all the players in the game
        for player in self.players:
            if (not player.isFriendsWith(shoxPlayer) and distance(player.pos,
                    shoxPlayer.pos) <= radius and not player.ghost):
                player.deathDetected(shoxPlayer.id, 0, 'W')

        for shot in list(self.shots):
            if (not shot.originatingPlayer.isFriendsWith(shoxPlayer) and
                    distance(shot.pos, shoxPlayer.pos) <= radius):
                self.shotExpired(shot)

        for star in list(self.collectableStars.itervalues()):
            if (not shoxPlayer.isFriendsWithTeam(star.team) and
                    distance(star.pos, shoxPlayer.pos) <= radius):
               	star.delete()


    @handler(ShotFiredMsg, orderPlug)
    def shotFired(self, msg):
        '''A player has fired a shot.'''
        try:
            player = self.playerWithId[msg.playerId]
        except KeyError:
            return

        team = player.team
        pos = (msg.xpos, msg.ypos)
        turret = msg.type in ('T', 'G')
        ricochet = msg.type == 'R'
        shot = self._createFiredShot(msg.shotId, team, player, pos,
                msg.angle, turret, ricochet)
        if shot is None:
            return

        self._update_reload_time(player)

        # When a shot's fired, send a player update to make it look less
        # odd.
        self.playerIsDirty(player.id)

    def _update_reload_time(self, player):
        '''
        Updates the player's reload time because the player has just fired a
        shot.
        '''
        if player.turret:
            reloadTime = self.physics.playerTurretReloadTime
            player.turretHeat += self.physics.playerShotHeat
            if player.turretHeat > self.physics.playerTurretHeatCapacity:
                player.turretOverHeated = True
        elif player.machineGunner:
            player.mgBulletsRemaining -= 1
            if player.mgBulletsRemaining > 0:
                reloadTime = self.physics.playerMachineGunFireRate
            elif player.mgBulletsRemaining == 0:
                reloadTime = self.physics.playerMachineGunReloadTime
            else:
                player.mgBulletsRemaining = 15
                reloadTime = 0
        elif player.shoxwave:
            reloadTime = self.physics.playerShoxwaveReloadTime
        elif player.team is None:
            reloadTime = self.physics.playerNeutralReloadTime
        elif player.currentZone.zoneOwner == player.team:
            reloadTime = self.physics.playerOwnReloadTime
        elif player.currentZone.zoneOwner is None:
            reloadTime = self.physics.playerNeutralReloadTime
        else:
            reloadTime = self.physics.playerEnemyReloadTime
        player.reloadTime = player.reloadFrom = reloadTime

    def _getNewShotId(self, player, _ids={}):
        '''
        Returns an unused shot id for the given player. This is a temporary
        hack that exists only as part of the transitioning from old to new
        universe message behaviour. (talljosh, 2010-06-27)
        '''
        prev = _ids.get(player, 0)
        result = prev + 1
        _ids[player] = result
        return result

    def _createFiredShot(self, id, team, player, pos, angle, turret, ricochet):
        '''
        Factory function for building a Shot object.
        '''
        velocity = (
            self.physics.shotSpeed * sin(angle),
            -self.physics.shotSpeed * cos(angle),
        )
        if turret:
            kind = Shot.TURRET
        elif ricochet:
            kind = Shot.RICOCHET
        else:
            kind = Shot.NORMAL

        lifetime = self.physics.shotLifetime

        i, j = MapLayout.getMapBlockIndices(pos[0], pos[1])
        try:
            mapBlock = self.zoneBlocks[i][j]
        except IndexError:
            return None

        shot = self._buildShot(id, team, player, pos, velocity, kind,
                lifetime, mapBlock)

        return shot

    def _buildShot(self, id, team, player, pos, velocity, kind, lifetime,
            mapBlock):
        shot = Shot(self, id, team, player, pos, velocity, kind, lifetime,
                mapBlock)
        mapBlock.addShot(shot)
        if id in player.shots:
            self.removeShot(player.shots[id])
        player.addShot(shot)
        self.shots.add(shot)
        return shot

    def shotExpired(self, shot):
        '''
        Called during the tick method when a shot has run out of lifetime or
        has hit a solid obstacle.
        '''
        msg = KillShotMsg(shot.originatingPlayer.id, shot.id)
        self.removeShot(shot)
        self.eventPlug.send(msg)

    def removeShot(self, shot):
        shot.originatingPlayer.destroyShot(shot.id)
        try:
            self.shots.remove(shot)
            shot.currentMapBlock.removeShot(shot)
        except KeyError:
            pass

    def shotWithId(self, pId, sId):
        try:
            return self.playerWithId[pId].shots[sId]
        except KeyError:
            return None

    def checkTag(self, tagger):
        '''Checks to see if the player has touched the orb of its currentZone
        If so, it sends a TaggingZoneMsg'''
        if self.clientOptimised:
            return
        if tagger.team is None:
            return

        # How zone tagging works (more or less)
        # 1. In its _move procedure, if a player is local, it will call checkTag
        # 2. If it is close enough to the orb, and it has the numeric
        #    advantage, and it onws at least one adjacent zone, it has tagged
        #    the zone
        # 3. The zone is checked again to see if the opposing team has also
        #    tagged it
        #    - If so, the zone is rendered neutral (if it's already neutral,
        #    nothing happens).
        #    - If not, the zone is considered to be tagged by the team.
        # 4. If any zone ownership change has been made, the server is informed
        #    however, no zone allocation is performed yet.
        # 5. The server should ensure that the zone hasn't already been tagged
        #    (such as in the situation of two players form the one team tag
        #    a zone simultaneously), as well as checking zone numbers,
        #    before telling all clients of the zone change.
        # 6. The individual clients recalculate zone ownership based on the
        #    zone change

        zone = tagger.currentZone
        if tagger.team == zone.orbOwner:
            return

        # If the tagging player has a phase shift, the zone will not be tagged.
        if tagger.phaseshift:
            return

        # If the game is over, zone tagging is not allowed.
        if self.gameState == GameState.Ended:
            return

        # Ensure that the zone has not already been checked in this tick:
        if zone is None or zone.taggedThisTime:
            return


        # Radius from orb (in pixels) that counts as a tag.
        tagDistance = 35
        xCoord1, yCoord1 = tagger.pos
        xCoord2, yCoord2 = zone.defn.pos

        dist = ((xCoord1 - xCoord2) ** 2 + (yCoord1 - yCoord2) ** 2) ** 0.5
        if dist < tagDistance:
            attackingTeams = zone.getAttackingTeams()
            if tagger.team in attackingTeams:
                if (len(attackingTeams) > 1 and
                        zone.checkForOpposingTag(tagger.team)):
                    # The other team has also tagged it
                    if zone.orbOwner != None:
                        self.eventPlug.send(TaggingZoneMsg(zone.id, '\x00',
                                '\x00'))
                        checkpoint('Universe: two teams tag same zone')
                else:
                    # This team is the only to have tagged it
                    self.eventPlug.send(TaggingZoneMsg(zone.id, tagger.id,
                            tagger.team.id))
                    checkpoint('Universe: zone tagged')

                self._killTurret(tagger, zone)
                zone.taggedThisTime = True
                self.zonesToReset.append(zone)
            else:
                checkpoint('Universe: failed attempt to tag zone')

    def _killTurret(self, tagger, zone):
        if zone.turretedPlayer is not None:
            zone.turretedPlayer.deathDetected(tagger.id, '\x00', 'T')
            checkpoint('Universe: turret killed by tagging zone')

    @handler(PlayerKilledMsg, orderPlug)
    def playerKilled(self, msg):
        try:
            target = self.getPlayer(msg.targetId)
            shot = self.shotWithId(msg.killerId, msg.shotId)
            try:
                killer = self.playerWithId[msg.killerId]
            except KeyError:
                killer = None
            else:
                if not killer.ghost:
                    killer.incrementStars()

            target.die()

            if self.playerWithElephant == target:
                if not killer.ghost:
                    self.playerWithElephant = killer
                    self.eventPlug.send(PlayerHasElephantMsg(killer.id))
                else:
                    for p in self.players:
                        if p.isElephantOwner():
                            self.playerWithElephant = p
                            self.eventPlug.send(PlayerHasElephantMsg(p.id))
                            break
                    else:
                        self.playerWithElephant = None
                        self.eventPlug.send(PlayerHasElephantMsg('\x00'))

            if self.gameRules == GameRules.Hunting:
                # Hunt mode
                if target.team != self.huntTeamWithZones:
                    i = 0
                    # Check number of remaining players on that team
                    for player in self.players:
                        if (player.team != self.huntTeamWithZones and player is
                                not target):
                            i += 1
                    if i > 1:
                        self.eventPlug.send(SetPlayerTeamMsg(target.id,
                                self.huntTeamWithZones.id))
                    else:
                        # The second last player on that team just died; change
                        # all zones now
                        self.setHuntTeamWithZones(
                                self.huntTeamWithZones.opposingTeam)
            if shot is not None:
                self.removeShot(shot)
        except Abort:
            pass

    @handler(TaggingZoneMsg, orderPlug)
    def zoneTagged(self, msg):
        if msg.playerId == '\x00':
            player = None
        else:
            player = self.playerWithId[msg.playerId]
        zone = self.map.zoneWithId[msg.zoneId]
        zone.tag(player)
        self.lastZoneTagged = self._gameTime

    @handler(ZoneStateMsg, orderPlug)
    def zoneOwned(self, msg):
        zone = self.map.zoneWithId[msg.zoneId]
        team = self.teamWithId[msg.teamId]
        zone.setOwnership(team, msg.dark, msg.marks)

    @handler(RespawnMsg, orderPlug)
    def respawn(self, msg):
        try:
            player = self.getPlayer(msg.playerId)
            zone = self.getZone(msg.zoneId)
            player.currentZone.removePlayer(player)
            player.currentZone = zone
            player.respawn()
            player.currentZone.addPlayer(player)
        except Abort:
            pass

    @handler(PlayerStarsSpentMsg, orderPlug)
    def starsSpent(self, msg):
        try:
            player = self.getPlayer(msg.playerId)
            player.stars -= msg.count
        except Abort:
            pass

    @handler(PlayerHasUpgradeMsg, orderPlug)
    def gotUpgrade(self, msg):
        try:
            player = self.getPlayer(msg.playerId)
            upgradeKind = self.getUpgradeType(msg.upgradeType)
            player.upgrade = upgradeKind(player)
            player.upgradeGauge = upgradeKind.timeRemaining
            player.upgradeTotal = upgradeKind.timeRemaining
            player.upgrade.use()
        except Abort:
            log.debug('Upgrade unable to be given to player %r', msg.playerId)

    @handler(DeleteUpgradeMsg, orderPlug)
    def delUpgrade(self, msg):
        try:
            player = self.getPlayer(msg.playerId)
            player.deleteUpgrade()
        except Abort:
            pass

    def addGrenade(self, grenade):
        self.grenades.add(grenade)

    def removeGrenade(self, grenade):
        self.grenades.remove(grenade)

    def addCollectableStar(self, star):
        self.collectableStars[star.id] = star

    def removeCollectableStar(self, star):
        del self.collectableStars[star.id]

    @handler(SetCaptainMsg, orderPlug)
    def setCaptain(self, msg):
        try:
            player = self.getPlayer(msg.playerId)
            if player.team is None:
                return
            player.team.captain = player
        except Abort:
            pass

    @handler(TeamIsReadyMsg, orderPlug)
    def teamReady(self, msg):
        try:
            player = self.getPlayer(msg.playerId)
            if player.team is not None and not player.team.ready:
                player.team.ready = True
                if self._gameShouldStart():
                    WeakCallLater(0.1, self, '_startSoon')
        except Abort:
            pass

    def _gameShouldStart(self):
        '''Checks to see if both teams are ready'''
        if self.gameState != GameState.PreGame:
            return False

        for team in self.teams:
            if not team.ready:
                return False
        return True

    def _startSoon(self, delay=7):
        self.eventPlug.send(StartingSoonMsg(delay))
        WeakCallLater(delay, self.eventPlug, 'send',
                GameStartMsg(self.gameDuration))

    @handler(StartingSoonMsg, orderPlug)
    def startingSoon(self, msg):
        self.gameState = GameState.Starting

        # Kick all AI players now, the game is about to begin.
        for p in list(self.players):
            if p.bot:
                self.eventPlug.send(RemovePlayerMsg(p.id))

    @handler(KillShotMsg, orderPlug)
    @handler(ShotAbsorbedMsg, orderPlug)
    def shotDestroyed(self, msg):
        shot = self.shotWithId(msg.shooterId, msg.shotId)
        if shot is not None:
            self.removeShot(shot)

    @handler(PlayerHitMsg, orderPlug)
    def shotHit(self, msg):
        shot = self.shotWithId(msg.shooterId, msg.shotId)
        player = self.getPlayer(msg.targetId)
        if player.shielded:
            player.upgrade.protections -= 1
        else:
            player.health -= 1
        if shot is not None:
            self.removeShot(shot)

    @handler(PlayerUpdateMsg, orderPlug)
    def gotPlayerUpdate(self, msg):
        # Receive a player update
        try:
            player = self.playerWithId[msg.playerId]
        except:
            # Mustn't have that info yet
            return

        values = expand_boolean(msg.keys)
        for i, key in enumerate(['left', 'right', 'jump', 'down']):
            player._state[key] = values[i]

        ghost = values[5]

        if ghost != player.ghost:
            if player.ghost:
                player.respawn()
            else:
                player.die()

        player.yVel = msg.yVel
        player.lookAt(msg.angle, msg.ghostThrust)
        player.setPos((msg.xPos, msg.yPos), msg.attached)

    def getTeamStars(self, team):
        if team is None:
            return 0
        total = 0
        for p in self.players:
            if p.team == team:
                total += p.stars
        return total

    def _getWorldParameters(self):
        '''Returns a dict representing the settings which must be sent to
        clients that connect to this server.'''
        result = {
            'teamName0': self.teams[0].teamName,
            'teamName1': self.teams[1].teamName,
        }

        if self.gameState == GameState.PreGame:
            result['gameState'] = "PreGame"
        elif self.gameState == GameState.Starting:
            result['gameState'] = 'Starting'
        elif self.gameState == GameState.InProgress:
            result['gameState'] = "InProgress"
            result['timeRunning'] = self._gameTime
            result['timeLimit'] = self.gameDuration
        else:
            result['gameState'] = "Ended"

        return result

    def applyWorldParameters(self, params):
        if 'teamName0' in params:
            self.teams[0].teamName = params['teamName0']
        if 'teamName1' in params:
            self.teams[1].teamName = params['teamName1']
        gs = params.get('gameState', None)
        if gs == 'PreGame':
            self.gameState = GameState.PreGame
        elif gs == 'Starting':
            self.gameState = GameState.Starting
        elif gs == 'InProgress':
            self.gameState = GameState.InProgress
            if 'timeRunning' in params:
                self._gameTime = params['timeRunning']
            if 'timeLimit' in params:
                self.gameDuration = params['timeLimit']
        elif gs == 'Ended':
            self.gameState = GameState.Ended

    def setLayout(self, layout):
        self.zoneWithDef = {}
        for team in self.teams:
            team.numOrbsOwned = 0
        self.map = MapState(self, layout)
        self.zoneWithId = self.map.zoneWithId
        self.zones = self.map.zones
        self.zoneBlocks = self.map.zoneBlocks

    def setTestMode(self):
        self.PLAYER_RESET_STARS = 20
        for upgradetype in allUpgrades:
            upgradetype.requiredStars = 1

    @handler(PlayerIsReadyMsg, orderPlug)
    def gotPlayerIsReady(self, msg):
        try:
            player = self.getPlayer(msg.playerId)
            player.readyToStart = msg.ready
        except Abort:
            pass

    @handler(SetPreferredTeamMsg, orderPlug)
    def gotPreferredTeam(self, msg):
        try:
            player = self.getPlayer(msg.playerId)
            player.preferredTeam = msg.name.decode()
        except Abort:
            pass

    @handler(SetPreferredSizeMsg, orderPlug)
    def gotPreferredSize(self, msg):
        try:
            player = self.getPlayer(msg.playerId)
            player.preferredSize = (msg.halfMapWidth, msg.mapHeight)
        except Abort:
            pass

    @handler(SetPreferredDurationMsg, orderPlug)
    def gotPreferredDuration(self, msg):
        try:
            player = self.getPlayer(msg.playerId)
            player.preferredDuration = msg.duration
        except Abort:
            pass

    @handler(ChangeNicknameMsg, orderPlug)
    def changeNickname(self, msg):
        try:
            player = self.getPlayer(msg.playerId)
            player.nick = msg.nickname.decode()
        except Abort:
            pass
