# Trosnoth (UberTweak Platform Game)
# Copyright (C) 2006-2011 Joshua D Bartlett
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# version 2 as published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
# 02110-1301, USA.

'''universe.py - defines anything that has to do with the running of the
universe. This includes players, shots, zones, and the level itself.'''

import logging
from math import sin, cos
import random

from trosnoth.utils.utils import timeNow
from trosnoth.utils.math import distance
from trosnoth.utils.checkpoint import checkpoint
from trosnoth.model.upgrades import upgradeOfType, allUpgrades
from trosnoth.model.physics import WorldPhysics

from trosnoth.model.map import MapLayout, MapState
from trosnoth.model.shot import Shot
from trosnoth.model.player import Player
from trosnoth.model.star import CollectableStar
from trosnoth.model.team import Team

# Component message passing
from trosnoth.utils.components import Component, handler, Plug
from trosnoth.utils.network import expand_boolean
from trosnoth.utils.twist import WeakCallLater
from trosnoth.messages import (TaggingZoneMsg, ShotFiredMsg, PlayerUpdateMsg,
        KillShotMsg, RespawnMsg, PlayerKilledMsg, PlayerHitMsg, GameStartMsg,
        GameOverMsg, SetTeamNameMsg, SetGameModeMsg, AddPlayerMsg,
        JoinSuccessfulMsg, UpdatePlayerStateMsg, AimPlayerAtMsg,
        RemovePlayerMsg, PlayerHasUpgradeMsg, ShotAbsorbedMsg,
        PlayerStarsSpentMsg, DeleteUpgradeMsg, SetPlayerTeamMsg,
        AchievementUnlockedMsg, ZoneStateMsg, WorldResetMsg,
        CreateCollectableStarMsg, RemoveCollectableStarMsg,
        PlayerHasElephantMsg, ChangeNicknameMsg, FireShoxwaveMsg, MarkZoneMsg,
        SetGameSpeedMsg, GrenadeReboundedMsg, StarReboundedMsg,
        UpgradeChangedMsg)

DIRTY_DELAY = 0.05
UPDATE_DELAY = 1        # PlayerUpdateMsg is sent for each player this often.

DEFAULT_TEAM_NAME_1 = 'Defaulticons'
DEFAULT_TEAM_NAME_2 = 'Standardators'
STALEMATE_CHECK_PERIOD = 1
COLLECTABLE_STAR_LIFETIME = 12
NO_COLLECTABLE_STARS = False        # For debugging.

# The time after a zone tag after which there is 50% of a new
# collectable star appearing every STALEMATE_CHECK_PERIOD.
ZONE_TAG_STAR_HALF_LIFE = 60

log = logging.getLogger('universe')

class Abort(Exception):
    pass

class Universe(Component):
    '''Universe(halfMapWidth, mapHeight)
    Keeps track of where everything is in the level, including the locations
    and states of every alien, the terrain positions, and who owns the
    various territories and orbs.'''

    # eventPlug is a sending plug
    # Events which are generated by the tick() method of the universe
    # should be sent to this plug. This indicates that the universe thinks
    # that the event should happen.
    eventPlug = Plug()

    # orderPlug is a receiving plug
    # When an object wants to order the universe to do something,
    # they should send a message to this plug
    orderPlug = Plug()

    PLAYER_RESET_STARS = 0

    def __init__(self, halfMapWidth=3, mapHeight=2, authTagManager=None,
            clientOptimised=False, gameName=None):
        '''
        halfMapWidth:   is the number of columns of zones in each team's
                        territory at the start of the game. There will always
                        be a single column of neutral zones between the two
                        territories at the start of the game.
        mapHeight:      is the number of zones in every second column of
                        zones. Every other column will have mapHeight + 1
                        zones in it. This is subject to the constraints that
                        (a) the columns at the extreme ends of the map will
                        have mapHeight zones; (b) the central (initially
                        neutral) column of zones will never have fewer zones
                        in it than the two bordering it - this will sometimes
                        mean that the column has mapHeight + 2 zones in it.
        clientOptimised: if True, this universe is not guaranteed to detect all
            events which occur and send them along the eventPlug.
        '''
        super(Universe, self).__init__()

        self.frameRate = 0
        self.playerWithElephant = None
        self.gameName = gameName
        self.clientOptimised = clientOptimised
        self.physics = WorldPhysics(self)
        if authTagManager is None:
            self.authManager = None
        else:
            self.authManager = authTagManager.authManager

        self.playerWithId = {}
        self.teamWithId = {'\x00' : None}

        # Create Teams:
        self.teams = (
            Team(self, 'A'),
            Team(self, 'B'),
        )
        Team.setOpposition(self.teams[0], self.teams[1])

        for t in self.teams:
            self.teamWithId[t.id] = t

        # Set up zones
        self.zoneWithDef = {}
        layout = MapLayout(halfMapWidth, mapHeight)
        self.map = MapState(self, layout)
        self.zoneWithId = self.map.zoneWithId
        self.zones = self.map.zones
        self.zoneBlocks = self.map.zoneBlocks

        self._state = None

        self.players = set()
        self.playerUpdateTimes = {}
        self.grenades = set()
        self.collectableStars = {}      # starId -> CollectableStar
        self.shots = set()
        self.gameMode = 'Normal'
        self.rogueTeamName = 'Rogue'
        self._elapsedTime = 0.0
        self._gameSpeed = 1.0
        self._lastStalemateCheck = self._elapsedTime
        self.lastZoneTagged = self._elapsedTime
        self._gameInProgress = False
        self._winningTeam = None

    @orderPlug.defaultHandler
    def ignore(self, msg):
        pass

    def getElapsedTime(self):
        return self._elapsedTime

    # Called when we receive parameters of the game
    # from the server
    def _setElapsedTime(self, time):
        self._elapsedTime = time

    # TODO: this seems a bit out of place?
    # Marking zones doesn't seem to be related to
    # positions of everyone on the map.
    @handler(MarkZoneMsg, orderPlug)
    def markZone(self, msg):
        try:
            zone = self.getZone(msg.zoneId)
            player = self.getPlayer(msg.playerId)

            zone.markedBy[player.team] = msg.value
        except Abort:
            pass

    # TODO: this seems out-of-place
    @handler(AchievementUnlockedMsg, orderPlug)
    def achievementUnlocked(self, msg):
        try:
            player = self.getPlayer(msg.playerId)
            if player.user is not None:
                player.user.achievementUnlocked(msg)
        except Abort:
            pass

    def _resetPlayer(self, player):
        zone = self.selectZone(player.teamId)
        player.pos = zone.defn.pos
        player.changeZone(zone)
        player.die()
        player.respawnGauge = 0.0

        # Make sure player knows its map block.
        i, j = MapLayout.getMapBlockIndices(*player.pos)
        try:
            block = self.zoneBlocks[i][j]
        except IndexError, e:
            log.exception(str(e))
            return
        player.setMapBlock(block)

    def selectZone(self, teamId):
        allTeamZones = [z for z in self.map.zones if z.orbOwner is not None and
                z.orbOwner.id == teamId]
        actionZones = []
        for zone in list(allTeamZones):
            for adj in zone.defn.adjacentZones.iterkeys():
                if (adj is not None and (self.zoneWithDef[adj].orbOwner is None
                        or self.zoneWithDef[adj].orbOwner.id != teamId)):
                    actionZones.append(zone)
        if len(actionZones) > 0:
            return random.choice(actionZones)
        elif len(allTeamZones) > 0:
            return random.choice(allTeamZones)
        return random.choice(list(self.map.zones))

    @handler(WorldResetMsg, orderPlug)
    def gotWorldReset(self, msg):
        self.grenades = set()
        self.shots = set()
        self._elapsedTime = 0.0
        self._lastStalemateCheck = self._elapsedTime
        self.lastZoneTagged = self._elapsedTime

    @handler(JoinSuccessfulMsg, orderPlug)
    def joinSuccessful(self, msg):
        try:
            player = self.getPlayer(msg.playerId)
            if self.authManager is not None and not player.bot:
                player.user = self.authManager.getUserByName(msg.username)
        except Abort:
            pass

        if player.isElephantOwner():
            self.playerWithElephant = player
            self.eventPlug.send(PlayerHasElephantMsg(player.id))

    def getTeam(self, teamId):
        if teamId == '\x00':
            return None
        try:
            return self.teamWithId[teamId]
        except KeyError:
            raise Abort()

    def getPlayer(self, playerId):
        try:
            return self.playerWithId[playerId]
        except KeyError:
            raise Abort()

    def getUpgradeType(self, upgradeTypeId):
        try:
            return upgradeOfType[upgradeTypeId]
        except KeyError:
            raise Abort()

    def getZone(self, zoneId):
        try:
            return self.map.zoneWithId[zoneId]
        except KeyError:
            raise Abort()

    def getShot(self, pId, sId):
        try:
            return self.playerWithId[pId].shots[sId]
        except KeyError:
            raise Abort()

    def playerIsDirty(self, pId):
        '''
        Registers the given players as "dirty" in that its location details
        should be sent to all clients at some stage in the near future. This is
        called when, for example, the player releases the jump key before the
        top of the jump, or collides with an obstacle, or such things.
        '''
        self.playerUpdateTimes[pId] = min(self.playerUpdateTimes.get(pId, 1),
                timeNow() + DIRTY_DELAY)

    def _sendPlayerUpdate(self, pId):
        '''
        Sends state information of a dirty player to all clients.
        '''
        self.playerUpdateTimes[pId] = timeNow() + UPDATE_DELAY

        try:
            player = self.playerWithId[pId]
        except KeyError:
            return

        self.eventPlug.send(player.makePlayerUpdate())

    @handler(UpdatePlayerStateMsg, orderPlug)
    def updatePlayerState(self, msg):
        try:
            player = self.getPlayer(msg.playerId)
            player.updateState(msg.stateKey, msg.value)
        except Abort:
            pass

    @handler(AimPlayerAtMsg, orderPlug)
    def aimPlayerAt(self, msg):
        try:
            player = self.getPlayer(msg.playerId)
            player.lookAt(msg.angle, msg.thrust)
        except Abort:
            pass

    @handler(SetGameModeMsg, orderPlug)
    def setGameMode(self, msg):
        mode = msg.gameMode.decode()
        if self.physics.setMode(mode):
            self.gameMode = mode
            log.debug('Client: GameMode is set to ' + mode)

    @handler(SetGameSpeedMsg, orderPlug)
    def setGameSpeed(self, msg):
        '''Sets the speed of the game to a proportion of normal speed.
        That is, speed=2.0 is twice as fast a game as normal
        '''
        speed = msg.gameSpeed
        self._gameSpeed = speed

    @handler(SetTeamNameMsg, orderPlug)
    def setTeamName(self, msg):
        if msg.teamId == '\x00':
            self.rogueTeamName = msg.name
        else:
            try:
                team = self.getTeam(msg.teamId)
                team.teamName = msg.name
            except Abort():
                pass

    @handler(GameOverMsg, orderPlug)
    def gameOver(self, msg):
        for player in self.players:
            player.readyToStart = False
            player.readyForTournament = False
            if player.ghost and player.team is not None:
                # Give a chance for stats to save
                WeakCallLater(0.5, self.eventPlug, 'send',
                        SetPlayerTeamMsg(player.id, '\x00'))

        self._gameInProgress = False

    def gameIsInProgress(self):
        return self._gameInProgress

    @handler(GameStartMsg, orderPlug)
    def gameStart(self, msg):
        self.lastZoneTagged = self.startTime = self._elapsedTime
        self._elapsedTime = 0
        self._gameInProgress = True

    @handler(AddPlayerMsg, orderPlug)
    def addPlayer(self, msg):
        team = self.teamWithId[msg.teamId]
        zone = self.zoneWithId[msg.zoneId]

        # Create the player.
        nick = msg.nick.decode()
        self._makePlayer(nick, team, msg.playerId, zone, msg.dead, msg.bot)

    @handler(SetPlayerTeamMsg, orderPlug)
    def setPlayerTeam(self, msg):
        try:
            player = self.getPlayer(msg.playerId)
            team = self.getTeam(msg.teamId)

            player.team = team
        except Abort:
            pass


    def _makePlayer(self, nick, team, playerId, zone, dead, bot):
        # Check if this is a duplicate msg.
        if playerId in self.playerWithId:
            player = self.playerWithId[playerId]
            player.changeZone(zone)
        else:
            player = Player(self, nick, team, playerId, zone, dead, bot)

            # Add this player to this universe.
            self.players.add(player)
            self.playerWithId[playerId] = player

        # Make sure player knows its zone
        i, j = MapLayout.getMapBlockIndices(*player.pos)
        try:
            block = self.zoneBlocks[i][j]
        except IndexError, e:
            log.exception(str(e))
            raise IndexError, 'player start position is off the map'
        player.setMapBlock(block)

    @handler(PlayerHasElephantMsg, orderPlug)
    def gotElephantMsg(self, msg):
        if msg.playerId is None:
            self.playerWithId = None
            return

        try:
            player = self.getPlayer(msg.playerId)
            self.playerWithElephant = player
        except Abort:
            pass


    @handler(RemovePlayerMsg, orderPlug)
    def delPlayer(self, msg):
        try:
            player = self.getPlayer(msg.playerId)
            player.removeFromGame()
            self.players.remove(player)
            del self.playerWithId[player.id]
            if player == self.playerWithElephant:
                self.returnElephantToOwner()

            # In case anyone else keeps a reference to it
            player.id = -1
        except Abort:
            pass

    def advanceEverything(self, deltaT):
        '''Advance the state of the game by deltaT seconds'''
        try:
            self.frameRate = 1. / deltaT
        except ZeroDivisionError:
            self.frameRate = 0
        gameDeltaT = (deltaT * self._gameSpeed)
        self._elapsedTime += gameDeltaT

        # Update the player and shot positions.
        for s in (list(self.shots) + list(self.players) + list(self.grenades) +
                self.collectableStars.values()):
            s.reset()
            s.update(gameDeltaT)

        if not self.clientOptimised:
            self.updateZoneOwnership()
            self.processPlayerUpdates()
            self.checkStalemate()
            self.updateElephant()
            self.updateCollectableStars()


    def updateCollectableStars(self):
        try:
            for star in self.collectableStars.values():
                if (star.creationTime < self._elapsedTime -
                        COLLECTABLE_STAR_LIFETIME):
                    self.eventPlug.send(RemoveCollectableStarMsg(star.id, '\x00'))
        except:
            log.error('Error updating collectable stars', exc_info=True)

    def collectableStarsCanAppear(self):
        return not NO_COLLECTABLE_STARS and self._gameInProgress

    def canBuyUpgrades(self):
        return self.state.canBuyUpgrades()

    def canRespawn(self):
        return self.state.canRespawn()

    def becomeNeutralWhenJoinOrDie(self):
        return self.state.becomeNeutralWhenJoinOrDie()

    def canShoot(self):
        return self.state.canShoot()

    def canRename(self):
        return self.state.canRename()

    def setWinningTeamById(self, teamId):
        self._winningTeam = self.teamWithId.get(teamId, None)

    def getWinningTeam(self):
        return self._winningTeam

    def checkStalemate(self):
        try:
            if NO_COLLECTABLE_STARS:
                return
            if (self._elapsedTime < self._lastStalemateCheck +
                    STALEMATE_CHECK_PERIOD):
                return
            self._lastStalemateCheck = self._elapsedTime

            if not self.collectableStarsCanAppear():
                return

            prob = 0.5 ** ((self._elapsedTime - self.lastZoneTagged) /
                    ZONE_TAG_STAR_HALF_LIFE)
            if random.random() > prob:
                team = random.choice(self.teams)
                zone = self.selectZone(team.id)
                x, y = zone.defn.randomPosition()
                self.eventPlug.send(CreateCollectableStarMsg('0', team.id, x, y))
        except:
            log.error('Error checking for stalemate', exc_info=True)

    def updateElephant(self):
        try:
            ePlayerId = '\x00'
            if self.playerWithElephant:
                ePlayerId = self.playerWithElephant.id
            if isinstance(ePlayerId, str) and len(ePlayerId) == 1:
                self.eventPlug.send(PlayerHasElephantMsg(ePlayerId))
            else:
                log.error('ePlayerId is %r but should be single char', ePlayerId)
        except:
            log.error('Error updating elephant', exc_info=True)

    @handler(CreateCollectableStarMsg, orderPlug)
    def createCollectableStar(self, msg):
        self.addCollectableStar(CollectableStar(self, msg.starId, (msg.xPos,
                msg.yPos), self.getTeam(msg.teamId)))

    @handler(RemoveCollectableStarMsg, orderPlug)
    def gotRemoveCollectableStarMsg(self, msg):
        try:
            star = self.collectableStars[msg.starId]
            star.currentMapBlock.removeCollectableStar(star)
            del self.collectableStars[msg.starId]
            try:
                player = self.playerWithId[msg.playerId]
            except KeyError:
                pass
            else:
                if star.team is None or player.team == star.team:
                    player.incrementStars()
        except KeyError:
            pass

    def processPlayerUpdates(self):
        try:
            playerIds = set(self.playerWithId.keys())
            now = timeNow()
            for pId, time in list(self.playerUpdateTimes.iteritems()):
                if pId not in playerIds:
                    del self.playerUpdateTimes[pId]
                    continue
                playerIds.remove(pId)
                if now > time:
                    self._sendPlayerUpdate(pId)

            for pId in playerIds:
                # Not yet in dict.
                self.playerUpdateTimes[pId] = now + UPDATE_DELAY
        except:
            log.error('Error in player updates', exc_info=True)

    def updateZoneOwnership(self):
        try:
            for zone in self.zones:
                result = zone.playerWhoTaggedThisZone()
                if result != None:
                    tagger, team = result
                    if tagger == None:
                        assert team == None
                        self.eventPlug.send(TaggingZoneMsg(zone.id, '\x00', '\x00'))
                    else:
                        self.eventPlug.send(TaggingZoneMsg(zone.id, tagger.id,
                                tagger.team.id))
                    self._killTurret(tagger, zone)
        except:
            log.error('Error in zone ownership check', exc_info=True)

    def teamWithAllZones(self):
        # Now check for an all zones win.
        team2Wins = self.teams[0].isLoser()
        team1Wins = self.teams[1].isLoser()
        if team1Wins and team2Wins:
            # The extraordinarily unlikely situation that all
            # zones have been neutralised in the same tick
            return 'Draw'
            checkpoint('Universe: Draw due to all zones neutralised')
        elif team1Wins:
            return self.teams[0]
            checkpoint('Universe: Zones captured, blue wins')
        elif team2Wins:
            return self.teams[1]
            checkpoint('Universe: Zones captured, red wins')
        else:
            return None

    def teamWithMoreZones(self):
        if self.teams[0].numOrbsOwned > self.teams[1].numOrbsOwned:
            return self.teams[0]
        elif self.teams[1].numOrbsOwned > self.teams[0].numOrbsOwned:
            return self.teams[1]
        else:
            return None

    def getTeamPlayerCounts(self):
        '''
        Returns a mapping from team id to number of players currently on that
        team.
        '''
        playerCounts = {}
        for player in self.players:
            playerCounts[player.teamId] = playerCounts.get(player.teamId, 0
                    ) + 1
        return playerCounts

    def getTeamToJoin(self, preferredTeamId):
        return self.state.getTeamToJoin(preferredTeamId, self)

    def getTeamName(self, id):
        if id == '\x00':
            return self.rogueTeamName
        return self.getTeam(id).teamName

    @handler(FireShoxwaveMsg, orderPlug)
    def shoxwaveExplosion(self, msg):
        radius = 128
        # Get the player who fired this shoxwave
        shoxPlayer = self.getPlayer(msg.playerId)
        self._update_reload_time(shoxPlayer)

        # Loop through all the players in the game
        for player in self.players:
            if (not player.isFriendsWith(shoxPlayer) and distance(player.pos,
                    shoxPlayer.pos) <= radius and not player.ghost):
                player.deathDetected(shoxPlayer.id, 0, 'W')

        for shot in list(self.shots):
            if (not shot.originatingPlayer.isFriendsWith(shoxPlayer) and
                    distance(shot.pos, shoxPlayer.pos) <= radius):
                self.shotExpired(shot)

        for star in list(self.collectableStars.itervalues()):
            if (not shoxPlayer.isFriendsWithTeam(star.team) and
                    distance(star.pos, shoxPlayer.pos) <= radius):
                star.delete()


    @handler(ShotFiredMsg, orderPlug)
    def shotFired(self, msg):
        '''A player has fired a shot.'''
        try:
            player = self.playerWithId[msg.playerId]
        except KeyError:
            return

        team = player.team
        pos = (msg.xpos, msg.ypos)
        turret = msg.type in ('T', 'G')
        ricochet = msg.type == 'R'
        shot = self._createFiredShot(msg.shotId, team, player, pos,
                msg.angle, turret, ricochet)
        if shot is None:
            return

        self._update_reload_time(player)

        # When a shot's fired, send a player update to make it look less
        # odd.
        self.playerIsDirty(player.id)

    def _update_reload_time(self, player):
        '''
        Updates the player's reload time because the player has just fired a
        shot.
        '''
        if player.turret:
            reloadTime = self.physics.playerTurretReloadTime
            player.turretHeat += self.physics.playerShotHeat
            if player.turretHeat > self.physics.playerTurretHeatCapacity:
                player.turretOverHeated = True
        elif player.machineGunner:
            player.mgBulletsRemaining -= 1
            if player.mgBulletsRemaining > 0:
                reloadTime = self.physics.playerMachineGunFireRate
            elif player.mgBulletsRemaining == 0:
                reloadTime = self.physics.playerMachineGunReloadTime
            else:
                player.mgBulletsRemaining = 15
                reloadTime = 0
        elif player.shoxwave:
            reloadTime = self.physics.playerShoxwaveReloadTime
        elif player.team is None:
            reloadTime = self.physics.playerNeutralReloadTime
        elif player.currentZone.zoneOwner == player.team:
            reloadTime = self.physics.playerOwnReloadTime
        elif player.currentZone.zoneOwner is None:
            reloadTime = self.physics.playerNeutralReloadTime
        else:
            reloadTime = self.physics.playerEnemyReloadTime
        player.reloadTime = player.reloadFrom = reloadTime

    def _getNewShotId(self, player, _ids={}):
        '''
        Returns an unused shot id for the given player. This is a temporary
        hack that exists only as part of the transitioning from old to new
        universe message behaviour. (talljosh, 2010-06-27)
        '''
        prev = _ids.get(player, 0)
        result = prev + 1
        _ids[player] = result
        return result

    def _createFiredShot(self, id, team, player, pos, angle, turret, ricochet):
        '''
        Factory function for building a Shot object.
        '''
        velocity = (
            self.physics.shotSpeed * sin(angle),
            -self.physics.shotSpeed * cos(angle),
        )
        if turret:
            kind = Shot.TURRET
        elif ricochet:
            kind = Shot.RICOCHET
        else:
            kind = Shot.NORMAL

        lifetime = self.physics.shotLifetime

        i, j = MapLayout.getMapBlockIndices(pos[0], pos[1])
        try:
            mapBlock = self.zoneBlocks[i][j]
        except IndexError:
            return None

        shot = self._buildShot(id, team, player, pos, velocity, kind,
                lifetime, mapBlock)

        return shot

    def _buildShot(self, id, team, player, pos, velocity, kind, lifetime,
            mapBlock):
        shot = Shot(self, id, team, player, pos, velocity, kind, lifetime,
                mapBlock)
        mapBlock.addShot(shot)
        if id in player.shots:
            self.removeShot(player.shots[id])
        player.addShot(shot)
        self.shots.add(shot)
        return shot

    def shotExpired(self, shot):
        '''
        Called during the tick method when a shot has run out of lifetime or
        has hit a solid obstacle.
        '''
        msg = KillShotMsg(shot.originatingPlayer.id, shot.id)
        self.removeShot(shot)
        self.eventPlug.send(msg)

    def removeShot(self, shot):
        shot.originatingPlayer.destroyShot(shot.id)
        try:
            self.shots.remove(shot)
            shot.currentMapBlock.removeShot(shot)
        except KeyError:
            pass

    def shotWithId(self, pId, sId):
        try:
            return self.playerWithId[pId].shots[sId]
        except KeyError:
            return None

    def _killTurret(self, tagger, zone):
        if zone.turretedPlayer is not None:
            zone.turretedPlayer.deathDetected(tagger.id, 0, 'T')
            checkpoint('Universe: turret killed by tagging zone')

    @handler(PlayerKilledMsg, orderPlug)
    def playerKilled(self, msg):
        try:
            target = self.getPlayer(msg.targetId)
            shot = self.shotWithId(msg.killerId, msg.shotId)
            try:
                killer = self.playerWithId[msg.killerId]
            except KeyError:
                killer = None
            else:
                if not killer.ghost:
                    killer.incrementStars()

            target.die()

            if self.playerWithElephant == target:
                if not killer.ghost:
                    self.playerWithElephant = killer
                    self.eventPlug.send(PlayerHasElephantMsg(killer.id))
                else:
                    self.returnElephantToOwner()
            if shot is not None:
                self.removeShot(shot)
        except Abort:
            pass

    def returnElephantToOwner(self):
        for p in self.players:
            if p.isElephantOwner():
                self.playerWithElephant = p
                self.eventPlug.send(PlayerHasElephantMsg(p.id))
                return

        self.playerWithElephant = None
        self.eventPlug.send(PlayerHasElephantMsg('\x00'))

    @handler(TaggingZoneMsg, orderPlug)
    def zoneTagged(self, msg):
        if msg.playerId == '\x00':
            player = None
        else:
            player = self.playerWithId[msg.playerId]
        zone = self.map.zoneWithId[msg.zoneId]
        zone.tag(player)
        self.lastZoneTagged = self._elapsedTime

    @handler(ZoneStateMsg, orderPlug)
    def zoneOwned(self, msg):
        zone = self.map.zoneWithId[msg.zoneId]
        team = self.teamWithId[msg.teamId]
        zone.setOwnership(team, msg.dark, msg.marks)

    @handler(RespawnMsg, orderPlug)
    def respawn(self, msg):
        try:
            player = self.getPlayer(msg.playerId)
            zone = self.getZone(msg.zoneId)
            player.currentZone.removePlayer(player)
            player.currentZone = zone
            player.respawn()
            player.currentZone.addPlayer(player)
        except Abort:
            pass

    @handler(PlayerStarsSpentMsg, orderPlug)
    def starsSpent(self, msg):
        try:
            player = self.getPlayer(msg.playerId)
            player.stars -= msg.count
        except Abort:
            pass

    @handler(PlayerHasUpgradeMsg, orderPlug)
    def gotUpgrade(self, msg):
        try:
            player = self.getPlayer(msg.playerId)
            upgradeKind = self.getUpgradeType(msg.upgradeType)
            player.upgrade = upgradeKind(player)
            player.upgradeGauge = upgradeKind.timeRemaining
            player.upgradeTotal = upgradeKind.timeRemaining
            player.upgrade.use()
        except Abort:
            log.debug('Upgrade unable to be given to player %r', msg.playerId)

    @handler(DeleteUpgradeMsg, orderPlug)
    def delUpgrade(self, msg):
        try:
            player = self.getPlayer(msg.playerId)
            player.deleteUpgrade()
        except Abort:
            pass

    @handler(UpgradeChangedMsg, orderPlug)
    def changeUpgrade(self, msg):
        for upgradeClass in allUpgrades:
            if upgradeClass.upgradeType == msg.upgradeType:
                if msg.statType == 'S':
                    upgradeClass.requiredStars = msg.newValue
                elif msg.statType == 'T':
                    upgradeClass.timeRemaining = msg.newValue

    def addGrenade(self, grenade):
        self.grenades.add(grenade)

    def removeGrenade(self, grenade):
        self.grenades.remove(grenade)

    def addCollectableStar(self, star):
        self.collectableStars[star.id] = star

    def removeCollectableStar(self, star):
        del self.collectableStars[star.id]

    def setGameState(self, state):
        self.state = state

    @handler(KillShotMsg, orderPlug)
    @handler(ShotAbsorbedMsg, orderPlug)
    def shotDestroyed(self, msg):
        shot = self.shotWithId(msg.shooterId, msg.shotId)
        if shot is not None:
            self.removeShot(shot)

    @handler(PlayerHitMsg, orderPlug)
    def shotHit(self, msg):
        shot = self.shotWithId(msg.shooterId, msg.shotId)
        player = self.getPlayer(msg.targetId)
        if player.shielded:
            player.upgrade.protections -= 1
        else:
            player.health -= 1
        if shot is not None:
            self.removeShot(shot)

    @handler(PlayerUpdateMsg, orderPlug)
    def gotPlayerUpdate(self, msg):
        # Receive a player update
        try:
            player = self.playerWithId[msg.playerId]
        except:
            # Mustn't have that info yet
            return

        values = expand_boolean(msg.keys)
        for i, key in enumerate(['left', 'right', 'jump', 'down']):
            player._state[key] = values[i]

        ghost = values[5]

        if ghost != player.ghost:
            if player.ghost:
                player.respawn()
            else:
                player.die()

        player.yVel = msg.yVel
        player.lookAt(msg.angle, msg.ghostThrust)
        player.setPos((msg.xPos, msg.yPos), msg.attached)

    def getTeamStars(self, team):
        if team is None:
            return 0
        total = 0
        for p in self.players:
            if p.team == team:
                total += p.stars
        return total

    def _getWorldParameters(self):
        '''Returns a dict representing the settings which must be sent to
        clients that connect to this server.'''
        result = {
            'teamName0': self.teams[0].teamName,
            'teamName1': self.teams[1].teamName,
        }

        return result

    def applyWorldParameters(self, params):
        if 'teamName0' in params:
            self.teams[0].teamName = params['teamName0']
        if 'teamName1' in params:
            self.teams[1].teamName = params['teamName1']

    def sendGrenadeRebound(self, owningPlayer, pos, xVel, yVel):
        msg = GrenadeReboundedMsg(owningPlayer.id, pos[0], pos[1], xVel, yVel)
        self.eventPlug.send(msg)

    @handler(GrenadeReboundedMsg, orderPlug)
    def gotGrenadeRebounded(self, msg):
        player = self.getPlayer(msg.playerId)
        if player.upgrade is not None:
            if player.upgrade.upgradeType == 'g':
                player.upgrade.gr.doRebound(msg)

    def sendStarRebound(self, star, xVel, yVel):
        msg = StarReboundedMsg(star.id, star.pos[0], star.pos[1], xVel, yVel)
        self.eventPlug.send(msg)

    @handler(StarReboundedMsg, orderPlug)
    def gotStarRebounded(self, msg):
        try:
            star = self.collectableStars[msg.starId]
        except KeyError:
            return
        else:
            star.gotRebound(msg)

    def setLayout(self, layout):
        self.zoneWithDef = {}
        for team in self.teams:
            team.numOrbsOwned = 0
        self.map = MapState(self, layout)
        self.zoneWithId = self.map.zoneWithId
        self.zones = self.map.zones
        self.zoneBlocks = self.map.zoneBlocks

    def setTestMode(self):
        self.PLAYER_RESET_STARS = 20
        for upgradetype in allUpgrades:
            upgradetype.requiredStars = 1

    @handler(ChangeNicknameMsg, orderPlug)
    def changeNickname(self, msg):
        try:
            player = self.getPlayer(msg.playerId)
            player.nick = msg.nickname.decode()
        except Abort:
            pass

    def isTournamentTeam(self, teamName):
        if not self.authManager:
            return False
        return teamName in self.authManager.tournamentSettings.teamNames

