#This file is part of Tryton.  The COPYRIGHT file at the top level of
#this repository contains the full copyright notices and license terms.
from operator import attrgetter
from itertools import groupby, chain

from trytond.model import fields
from trytond.pool import PoolMeta
from trytond.wizard import StateAction
from trytond.modules.company import CompanyReport
from trytond.pool import Pool
from trytond.transaction import Transaction


__all__ = ['Level', 'ProcessDunning', 'Letter']
__metaclass__ = PoolMeta


class Level:
    __name__ = 'account.dunning.level'
    print_on_letter = fields.Boolean('Print on Letter')


class ProcessDunning:
    __name__ = 'account.dunning.process'
    print_letter = StateAction('account_dunning_letter.report_letter')

    @classmethod
    def __setup__(cls):
        super(ProcessDunning, cls).__setup__()
        cls._actions.append('print_letter')

    def do_print_letter(self, action):
        # TODO return None if nothing to print
        return action, {
            'id': Transaction().context['active_id'],
            'ids': Transaction().context['active_ids'],
            }

    def transition_print_letter(self):
        return self.next_state('print_letter')


class Letter(CompanyReport):
    __name__ = 'account.dunning.letter'

    @classmethod
    def parse(cls, report, records, data, localcontext):
        pool = Pool()
        Date = pool.get('ir.date')

        dunnings = [d for d in records
            if d.state == 'done'
            and not d.blocked
            and d.party
            and d.level.print_on_letter]
        parties = list(set((d.party for d in dunnings)))
        payments = cls.get_pending_payments(parties)
        key = attrgetter('party')
        dunnings.sort(key=key)
        dunnings = groupby(dunnings, key)

        PartyLetter = cls.get_party_letter()
        letters = {}
        for party, current_dunnings in dunnings:
            current_dunnings = list(current_dunnings)
            dunning_amount = sum((d.amount for d in current_dunnings))
            current_payments = list(payments.get(party, []))
            payment_amount = sum((l.credit - l.debit
                    for l in current_payments))
            if dunning_amount < payment_amount:
                continue
            letters[party] = PartyLetter(dunnings=current_dunnings,
                payments=current_payments)
        localcontext['letters'] = letters
        localcontext['today'] = Date.today()
        localcontext['get_payment_amount'] = cls.get_payment_amount
        localcontext['get_payment_currency'] = cls.get_payment_currency
        return super(Letter, cls).parse(report, records, data, localcontext)

    @staticmethod
    def get_party_letter():

        class PartyLetter(object):

            def __init__(self, dunnings, payments):
                self.dunnings = dunnings
                self.payments = payments

            def highest_levels(self):
                'Yield each procedure and the highest level'
                key = attrgetter('procedure')
                dunnings = sorted(self.dunnings, key=key)
                for procedure, dunnings in groupby(dunnings, key):
                    i = 0
                    for dunning in dunnings:
                        i = max(i, procedure.levels.index(dunning.level))
                    yield procedure, procedure.levels[i]

        return PartyLetter

    @staticmethod
    def get_pending_payments(parties):
        """
        Return a dictionary with party as key and the list of pending payments
        as value.
        """
        pool = Pool()
        Line = pool.get('account.move.line')
        in_max = Transaction().cursor.IN_MAX
        payments = []
        for i in range(0, len(parties), in_max):
            sub_parties = parties[i:i + in_max]
            payments.append(Line.search([
                        ('account.kind', '=', 'receivable'),
                        ['OR',
                            ('debit', '<', 0),
                            ('credit', '>', 0),
                            ],
                        ('party', 'in', [p.id for p in sub_parties]),
                        ('reconciliation', '=', None),
                        ],
                    order=[('party', 'ASC'), ('id', 'ASC')]))
        payments = list(chain(*payments))
        return dict((party, list(payments))
            for party, payments in groupby(payments, attrgetter('party')))

    @staticmethod
    def get_payment_amount(payment):
        if payment.amount_second_currency:
            return payment.amount_second_currency.copy_sign(
                payment.credit - payment.debit)
        else:
            return payment.credit - payment.debit

    @staticmethod
    def get_payment_currency(payment):
        if payment.amount_second_currency:
            return payment.second_currency
        else:
            return payment.account.company.currency
