#This file is part of Tryton.  The COPYRIGHT file at the top level of
#this repository contains the full copyright notices and license terms.
import datetime
from trytond.pool import PoolMeta, Pool
from trytond.model import ModelView, ModelSQL, Workflow, fields
from trytond.transaction import Transaction
from trytond.pyson import Eval

__all__ = ['Reception']
__metaclass__ = PoolMeta

_STATES = {
    'readonly': Eval('state') != 'draft',
}


class Reception(Workflow, ModelSQL, ModelView):
    'Reception'
    __name__ = 'document.reception'
    _rec_name = 'reference'
    reference = fields.Char("Reference", readonly=True, select=True)
    description = fields.Char('Reception', required=True, states=_STATES)
    party = fields.Char('Party', required=True, states=_STATES)
    party_id = fields.Char('Party Id', states=_STATES)
    user_target = fields.Many2One('res.user', 'User Target',
            states=_STATES)
    date_reception = fields.DateTime('Reception Date-Time', readonly=True)
    comment = fields.Text('Comment', states=_STATES)
    date_checked = fields.Date('Date Checked', states={
            'readonly': True,
            })
    state = fields.Selection([
        ('draft', 'Draft'),
        ('signed', 'Signed'),
        ('checked', 'Checked'),
        ], 'State', readonly=True, select=True)
    priority = fields.Selection([
        ('low', 'Low'),
        ('middle', 'Middle'),
        ('high', 'High'),
        ], 'Priority', required=True, select=True, states=_STATES)

    @classmethod
    def __setup__(cls):
        super(Reception, cls).__setup__()
        cls._buttons.update({
                'checked': {
                    'invisible': Eval('state') != 'signed',
                    },
                'signed': {
                    'invisible': Eval('state') != 'draft',
                    },
                })
        cls._error_messages.update({
            'sequence_missing': ('Sequence Reception Documents is missing!'),
            })
        cls._order.insert(0, ('date_reception', 'DESC'))
        cls._transitions |= set((
                ('draft', 'signed'),
                ('signed', 'checked'),
                ))

    @staticmethod
    def default_date_reception():
        return datetime.datetime.now()

    @staticmethod
    def default_state():
        return 'draft'

    @staticmethod
    def default_priority():
        return 'middle'

    @classmethod
    @ModelView.button
    @Workflow.transition('signed')
    def signed(cls, docs):
        for doc in docs:
            doc.set_number()

    @classmethod
    @ModelView.button
    @Workflow.transition('checked')
    def checked(cls, docs):
        for doc in docs:
            doc.write([doc], {'date_checked': datetime.date.today()})

    def set_number(self):
        if self.reference:
            return
        pool = Pool()
        Sequence = pool.get('ir.sequence')
        Configuration = pool.get('document.configuration')
        configuration = Configuration(1)
        if not configuration.document_reception_sequence:
            self.raise_user_error('sequence_missing',)
        seq = configuration.document_reception_sequence.id
        self.write([self], {'reference': Sequence.get_id(seq)})
