#This file is part of Tryton.  The COPYRIGHT file at the top level of
#this repository contains the full copyright notices and license terms.
from trytond.model import ModelView, ModelSQL, fields
from trytond.pyson import Eval


__all__ = ['CategoryDocument']

STATES = {
    'invisible': Eval('kind') == 'base',
    }


def fmt_pad(val, pad):
    tmp = '{0:0%dd}' % pad 
    return tmp.format(val)


class CategoryDocument(ModelSQL, ModelView):
    "Category Document"
    __name__ = "document.category"
    name = fields.Char('Name', required=True, translate=True)
    parent = fields.Many2One('document.category', 'Parent', select=True,
            states=STATES)
    childs = fields.One2Many('document.category', 'parent',
            string='Children')
    code = fields.Char('Code', select=True, states={
            'required': Eval('kind') != 'base',
            })
    responsible = fields.Many2One('company.employee', 'Responsible',
            states={
                'required': Eval('kind') == 'section',
                'invisible': Eval('kind') != 'section',
            })
    retention_time = fields.Integer('Retention Time', select=True,
            states={
                'required': Eval('kind') == 'series',
                'invisible': Eval('kind') != 'series',
                })
    kind = fields.Selection([
            ('base', 'Base'),
            ('section', 'Section'),
            ('Subsection', 'Subsection'),
            ('series', 'Series'),
            ('subseries', 'Subseries'),
            ], 'Kind', required=True)
    next_number = fields.Integer('Next Number', select=True,
            states={
                'required': Eval('kind') == 'series',
                'invisible': Eval('kind') != 'series',
            })
    padding = fields.Integer('Padding', select=True,
            states={
                'required': Eval('kind') == 'base',
                'invisible': Eval('kind') != 'base',
            })

    @classmethod
    def __setup__(cls):
        super(CategoryDocument, cls).__setup__()
        cls._order.insert(0, ('name', 'ASC'))
        cls._error_messages.update({
                'missing_padding': 'Missing to define padding on base!',
                })

    @staticmethod
    def default_next_number():
        return 1

    @classmethod
    def validate(cls, categories):
        super(CategoryDocument, cls).validate(categories)
        cls.check_recursion(categories, rec_name='name')

    def get_rec_name(self, name):
        if self.parent:
            return self.parent.get_rec_name(name) + ' / ' + self.name
        else:
            return self.name

    def get_next_number(self):
        padding = self.get_base_padding()
        res = fmt_pad(self.next_number, padding)
        self.write([self], {
                'next_number': (self.next_number + 1),
                })
        return res

    def get_base_padding(self):
        parent = self.parent
        while parent.kind != 'base':
            parent = parent.parent
        if not parent.padding:
            self.raise_user_error('missing_padding',)
        return parent.padding

    @classmethod
    def search_rec_name(cls, name, clause):
        if isinstance(clause[2], basestring):
            values = clause[2].split('/')
            values.reverse()
            domain = []
            field = 'name'
            for name in values:
                domain.append((field, clause[1], name.strip()))
                field = 'parent.' + field
        else:
            domain = [('name',) + tuple(clause[1:])]
        ids = [w.id for w in cls.search(domain, order=[])]
        return [('parent', 'child_of', ids)]
