#This file is part of Tryton.  The COPYRIGHT file at the top level of
#this repository contains the full copyright notices and license terms.
import time
import datetime
import subprocess
from os import path as os_path, stat
from trytond.pool import Pool
from trytond.model import ModelView, ModelSQL, Workflow, fields
from trytond.pyson import Eval
from trytond.report import Report
from trytond.transaction import Transaction
from trytond.wizard import Wizard, StateView, Button, StateAction

__all__ = ['Document', 'DocumentRecord', 'TypeDocument',
    'DocumentRecordInventory', 'DocumentRecordInventoryStart',
    'DocumentRecordInventoryReport']

_STATES = {'readonly': Eval('state') != 'draft'}


def fmt_time(val):
    res = time.strftime("%a, %d %b %Y, %H:%M:%S", time.localtime(val))
    return res


class DocumentRecord(Workflow, ModelSQL, ModelView):
    'Document Record'
    __name__ = 'document.record'
    sequence = fields.Char('Sequence', readonly=True, states=_STATES)
    name = fields.Char('Name', required=True, states=_STATES)
    code = fields.Char('Code', readonly=True, select=True)
    catalog_number = fields.Char('Catalog Number', select=True,
            states=_STATES)
    folios_num = fields.Function(fields.Integer("Folios Number"),
            'get_folios_number')
    series = fields.Many2One('document.category', 'Series', domain=[
            ('kind', 'in', ['series', 'subserie']),
            ], states=_STATES, required=True)
    documents = fields.One2Many('document.document', 'record',
            'Documents', states=_STATES)
    location = fields.Many2One('document.location', 'Location',
            states=_STATES)
    state = fields.Selection([
            ('draft', 'Draft'),
            ('paperwork', 'Paperwork'),
            ('central', 'Central'),
            ('historic', 'Historic'),
            ], 'State', required=True, readonly=True)
    # Conservation Place
    place_box = fields.Char('Box', select=True, states=_STATES)
    place_folder = fields.Char('Folder', select=True, states=_STATES)
    place_volume = fields.Char('Volume', select=True, states=_STATES)
    place_other = fields.Char('Other', select=True, states=_STATES)
    notes = fields.Text('Notes', states=_STATES)
    storage_medium = fields.Selection([
        ('paper', 'Paper'),
        ('cd', 'CD'),
        ('other', 'Other'),
        ], 'Storage Mediums', required=True, states=_STATES)

    @classmethod
    def __setup__(cls):
        super(DocumentRecord, cls).__setup__()
        cls._order.insert(0, ('code', 'ASC'))
        cls._error_messages.update({
                'documents_in_draft': 'There are documents in draft state!',
                })
        cls._buttons.update({
                'draft': {
                    'invisible': Eval('state') != 'paperwork',
                    },
                'paperwork': {
                    'invisible': Eval('state').in_(['paperwork', 'historic']),
                    },
                'central': {
                    'invisible': Eval('state') != 'paperwork',
                    },
                'historic': {
                    'invisible': Eval('state') != 'central',
                    },
                })
        cls._transitions |= set((
                ('draft', 'paperwork'),
                ('paperwork', 'central'),
                ('paperwork', 'draft'),
                ('central', 'paperwork'),
                ('central', 'historic'),
                ))

    def get_folios_number(self, name):
        return len(self.documents)

    @staticmethod
    def default_state():
        return 'draft'

    @staticmethod
    def default_storage_medium():
        return 'paper'

    @classmethod
    @ModelView.button
    @Workflow.transition('draft')
    def draft(cls, records):
        pass

    @classmethod
    @ModelView.button
    @Workflow.transition('paperwork')
    def paperwork(cls, records):
        for record in records:
            record.set_sequence()
            record.set_code()
            for doc in record.documents:
                if doc.state == 'draft':
                    cls.raise_user_error('documents_in_draft',)
                doc.set_code()

    @classmethod
    @ModelView.button
    @Workflow.transition('central')
    def central(cls, records):
        pass

    @classmethod
    @ModelView.button
    @Workflow.transition('historic')
    def historic(cls, records):
        pass

    def set_sequence(self):
        if self.sequence or not self.series or not self.series.code:
            return
        next_seq = self.series.get_next_number()
        self.write([self], {'sequence': next_seq})

    def set_code(self):
        if not self.series or not self.series.code or not self.sequence:
            return
        code = self.series.code + '.' + self.sequence
        self.write([self], {'code': code})


class Document(Workflow, ModelSQL, ModelView):
    'Document'
    __name__ = 'document.document'
    name = fields.Char('Name', required=True, states=_STATES)
    code = fields.Char("Code", readonly=True, select=True)
    sequence = fields.Char('Sequence', readonly=True)
    record = fields.Many2One('document.record', 'Record', select=True,
            ondelete='CASCADE', states=_STATES)
    type_document = fields.Many2One('document.type_document', 'Type',
            select=True, states=_STATES)
    state = fields.Selection([
            ('draft', 'Draft'),
            ('posted', 'Posted'),
            ], 'State', required=True, readonly=True)
    date_post = fields.Date('Date Post', select=True)
    document_url = fields.Char('Document Url', states=_STATES)
    description = fields.Text('Description', states=_STATES)
    is_url_valid = fields.Function(fields.Boolean('Is Url Valid'),
        'get_is_url_valid')
    mtd_type = fields.Function(fields.Char('Type'), 'get_mtd')
    mtd_size = fields.Function(fields.Char('Size (Kb)'), 'get_mtd')
    mtd_accessed = fields.Function(fields.Char('Accessed'), 'get_mtd')
    mtd_modified = fields.Function(fields.Char('Modified'), 'get_mtd')
    lending = fields.Many2One('document.lending', 'Lending', select=True,
            states=_STATES)

    @classmethod
    def __setup__(cls):
        super(Document, cls).__setup__()
        cls._buttons.update({
                'draft': {
                    'invisible': True,
                    },
                'post': {
                    'invisible': Eval('state') == 'posted',
                    },
                })
        cls._transitions |= set((
                ('draft', 'posted'),
            ))

    @staticmethod
    def default_state():
        return 'draft'

    @staticmethod
    def default_date_post():
        return datetime.date.today()

    def get_is_url_valid(self, name):
        if self.document_url and os_path.isfile(self.document_url):
            return True
        return False

    def get_mtd(self, name):
        res = None
        if self.is_url_valid:
            file_ = stat(self.document_url)
            if name == 'mtd_type':
                res = subprocess.check_output(['file', '-b', self.document_url])
                res = res.replace('\n', '')
            elif name == 'mtd_size':
                # Convert bytes to kilobytes
                res = str(round((file_.st_size / 1024.0), 2))
            elif name == 'mtd_accessed':
                res = fmt_time(file_.st_atime)
            elif name == 'mtd_modified':
                res = fmt_time(file_.st_mtime)
        return res

    @classmethod
    @ModelView.button
    @Workflow.transition('posted')
    def post(cls, documents):
        for doc in documents:
            doc.set_sequence()

    def set_sequence(self):
        next_seq = 1
        sequences = [doc for doc in self.record.documents if doc.state == 'posted']
        
        if sequences:
            next_seq = len(sequences) + 1
        self.write([self], {'sequence': str(next_seq)})

    def set_code(self):
        if self.code:
            return
        if self.record:
            code = self.record.code + '.' + str(self.sequence)
            self.write([self], {'code': code})


class TypeDocument(ModelSQL, ModelView):
    'Type Document'
    __name__ = 'document.type_document'
    name = fields.Char('Name', required=True)


class DocumentRecordInventoryStart(ModelView):
    'Document Record Inventory Start'
    __name__ = 'document.record_inventory.start'
    company = fields.Many2One('company.company', 'Company', required=True)
    section = fields.Many2One('document.category', 'Section', domain=[
            ('kind', 'in', ['section', 'subsection']),
            ], required=True)

    @staticmethod
    def default_company():
        return Transaction().context.get('company')


class DocumentRecordInventory(Wizard):
    'Document Record Inventory'
    __name__ = 'document.record.inventory'
    start = StateView('document.record_inventory.start',
        'document.record_inventory_start_view_form', [
            Button('Cancel', 'end', 'tryton-cancel'),
            Button('Print', 'print_', 'tryton-ok', default=True),
            ])
    print_ = StateAction('document.report_document_record_inventory')

    def do_print_(self, action):
        data = {
            'company': self.start.company.id,
            'section': self.start.section.id,
            }
        return action, data

    def transition_print_(self):
        return 'end'


class DocumentRecordInventoryReport(Report):
    'Document Record Inventory'
    __name__ = 'document.record.inventory_report'

    @classmethod
    def parse(cls, report, objects, data, localcontext):
        DocumentRecord = Pool().get('document.record')
        if data['section']:
            print data['section']
        objects = DocumentRecord.search([])
        return super(DocumentRecordInventoryReport, cls).parse(report,
                objects, data, localcontext)
