#This file is part of Tryton.  The COPYRIGHT file at the top level of
#this repository contains the full copyright notices and license terms.
from os import path as os_path
import datetime
from trytond.pool import PoolMeta, Pool
from trytond.model import ModelView, ModelSQL, Workflow, fields
from trytond.pyson import Eval
from trytond.modules.company import CompanyReport
from trytond.wizard import Wizard, StateTransition, StateView, Button

__all__ = ['Communication', 'CommunicationAttachment',
    'DocumentCommunicationEmployee', 'CommunicationLetter',
    'CommunicationLetterReport', 'SearchDocumentFileStart',
    'SearchDocumentFile']
__metaclass__ = PoolMeta

_STATES = {
    'readonly': Eval('state') != 'draft',
}



class Communication(Workflow, ModelSQL, ModelView):
    'Communication'
    __name__ = 'document.communication'
    _rec_name = 'reference'
    reference = fields.Char("Reference", readonly=True, select=True)
    description = fields.Char('Description', required=True, states=_STATES)
    party = fields.Many2One('party.party', 'Party', required=False,
        states=_STATES)
    user_target = fields.Many2Many('company.employee-document.communication',
        'document_communication', 'employee', 'Employee Target',
        states=_STATES)
    date_communication = fields.DateTime('Date',
            readonly=True)
    comment = fields.Text('Comment', states=_STATES)
    date_checked = fields.Date('Date Checked', states={
            'readonly': True,
            })
    response_time = fields.Date('Response Date', states=_STATES)
    kind = fields.Selection([
        ('incoming', 'Incoming'),
        ('outgoing', 'Outgoing'),
        ('internal', 'Internal'),
        (None, ''),
        ], 'Kind', select=True, states=_STATES)
    state = fields.Selection([
        ('draft', 'Draft'),
        ('signed', 'Signed'),
        ('checked', 'Checked'),
        ('cancelled', 'Cancelled'),
        ], 'State', readonly=True, select=True)
    priority = fields.Selection([
        ('low', 'Low'),
        ('middle', 'Middle'),
        ('high', 'High'),
        ], 'Priority', required=True, select=True, states=_STATES)
    attachments = fields.One2Many('document.communication.attachment', 'communication', 
            'Attachments', states=_STATES)

    @classmethod
    def __setup__(cls):
        super(Communication, cls).__setup__()
        cls._error_messages.update({
            'sequence_missing': ('Sequence Communication Documents is missing!'),
            })
        cls._order.insert(0, ('date_communication', 'DESC'))
        cls._transitions |= set((
                ('draft', 'cancelled'),
                ('draft', 'signed'),
                ('signed', 'checked'),
                ))
        cls._buttons.update({
                'draft': {
                    'invisible': True,
                    },
                'checked': {
                    'invisible': Eval('state') != 'signed',
                    },
                'signed': {
                    'invisible': Eval('state') != 'draft',
                    },
                })

    @staticmethod
    def default_date_communication():
        return datetime.datetime.now()

    @staticmethod
    def default_state():
        return 'draft'

    @staticmethod
    def default_priority():
        return 'middle'

    @staticmethod
    def default_kind():
        return 'incoming'

    @classmethod
    @ModelView.button
    @Workflow.transition('draft')
    def draft(cls, docs):
        pass

    @classmethod
    @ModelView.button
    @Workflow.transition('signed')
    def signed(cls, docs):
        for doc in docs:
            doc.set_number()

    @classmethod
    @ModelView.button
    @Workflow.transition('checked')
    def checked(cls, docs):
        for doc in docs:
            doc.write([doc], {'date_checked': datetime.date.today()})

    def set_number(self):
        if self.reference:
            return
        pool = Pool()
        Sequence = pool.get('ir.sequence')
        Configuration = pool.get('document.configuration')
        configuration = Configuration(1)
        if not configuration.document_communication_sequence:
            self.raise_user_error('sequence_missing',)
        seq = configuration.document_communication_sequence.id
        self.write([self], {'reference': Sequence.get_id(seq)})


class CommunicationAttachment(ModelSQL, ModelView):
    'Communication Attachment'
    __name__ = 'document.communication.attachment'
    _rec_name = 'description'
    name = fields.Char('Name', required=False)
    communication = fields.Many2One('document.communication', 'Communication',
            required=False)
    document_url = fields.Char('Document Url', required=True)
    date = fields.Date('Date', required=False)
    employee = fields.Many2One('company.employee', 'Employee',
            required=False)
    description = fields.Text('Description', required=False)
    is_url_valid = fields.Function(fields.Boolean('Is Url Valid'),
        'get_is_url_valid')
    kind = fields.Selection([
            ('incoming', 'Incoming'),
            ('outgoing', 'Outgoing'),
            ('internal', 'Internal'),
            ], 'Kind', select=True)
    responses = fields.One2Many('document.communication.attachment',
            'parent', 'Responses')
    parent = fields.Many2One('document.communication.attachment', 'Parent',
            select=True, ondelete="RESTRICT")

    @classmethod
    def __setup__(cls):
        super(CommunicationAttachment, cls).__setup__()
        cls._buttons.update({
                'search_file': {
                    'invisible': False,
                    },})

    def get_is_url_valid(self, name):
        if self.document_url and os_path.isfile(self.document_url):
            return True
        return False

    @staticmethod
    def default_kind():
        return 'incoming'

    @staticmethod
    def default_date():
        return datetime.date.today()

    @classmethod
    @ModelView.button_action('document.act_search_file')
    def search_file(cls, docs):
        print "se ejecuta search"
        pass


class SearchDocumentFileStart(ModelView):
    'Search Document File'
    __name__ = 'document.communication.attachment_search_file.start'
    filepath = fields.Binary('Filepath', required=True)


class SearchDocumentFile(Wizard):
    'Search Document File'
    __name__ = 'document.communication.attachment_search_file'

    start = StateView('document.communication.attachment_search_file.start',
        'document.communication_attachment_search_file_view_form', [
                Button('Cancel', 'end', 'tryton-cancel'),
                Button('Add', 'search_', 'tryton-ok', default=True),
            ])
    search_ = StateTransition()

    def transition_search_(self):
        #print self.start.filepath
        print type(self.start.filepath)
        return 'end'


class DocumentCommunicationEmployee(ModelSQL):
    'Document Communication Employee'
    __name__ = 'company.employee-document.communication'
    _table = 'company_employee_document_communication'
    document_communication = fields.Many2One('document.communication',
        'Document Communication', ondelete='CASCADE', select=True, 
        required=True)
    employee = fields.Many2One('company.employee', 'Employee', select=True, 
        required=True, ondelete='RESTRICT')


class CommunicationLetter(Workflow, ModelSQL, ModelView):
    'Communication Letter'
    __name__ = 'document.communication.letter'
    date = fields.Date('Date', select=True, states=_STATES)
    number = fields.Char('Reference', readonly=True, select=True)    
    subject = fields.Char('Subject', required=True, states=_STATES)
    party = fields.Many2One('party.party', 'Party', required=True,
            select=True, states=_STATES)
    content = fields.Text("Content", select=True, states=_STATES)
    state = fields.Selection([
            ('draft', 'Draft'),
            ('check', 'Check'),
            ('done', 'Done'),
            ], 'State', required=True, readonly=True)

    @classmethod
    def __setup__(cls):
        super(CommunicationLetter, cls).__setup__()
        cls._transitions |= set((
                ('draft', 'check'),
                ('check', 'done'),
                ('check', 'draft'),
                ))
        cls._buttons.update({
                'draft': {
                    'invisible': Eval('state') != 'check',
                    },
                'check': {
                    'invisible': Eval('state') != 'draft',
                    },
                'done': {
                    'invisible': Eval('state') != 'check',
                    },
                })

    @staticmethod
    def default_state():
        return 'draft'

    @staticmethod
    def default_date():
        return datetime.date.today()

    @classmethod
    @ModelView.button
    @Workflow.transition('draft')
    def draft(cls, letters):
        pass

    @classmethod
    @ModelView.button
    @Workflow.transition('check')
    def check(cls, letters):
        pass

    @classmethod
    @ModelView.button
    @Workflow.transition('done')
    def done(cls, letters):
        for letter in letters:
            letter.set_sequence()

    def set_sequence(self):
        return
        next_seq = 1
        sequences = [doc for doc in self.record.documents if doc.state == 'posted']
        
        if sequences:
            next_seq = len(sequences) + 1
        self.write([self], {'number': str(next_seq)})


class CommunicationLetterReport(CompanyReport):
    __name__ = 'document.communication.letter'
