#This file is part of Tryton.  The COPYRIGHT file at the top level of
#this repository contains the full copyright notices and license terms.
try:
    from cStringIO import StringIO
except:
    from StringIO import StringIO

import os
from datetime import datetime, date
from io import BufferedReader, BytesIO
from pyPdf import PdfFileWriter, PdfFileReader
from reportlab.pdfgen import canvas
from reportlab.lib.pagesizes import letter
from reportlab.lib.colors import HexColor

from trytond.pool import PoolMeta, Pool
from trytond.model import ModelView, ModelSQL, Workflow, fields
from trytond.pyson import Eval, And, In, Not
from trytond.modules.company import CompanyReport
from trytond.wizard import Wizard, StateTransition, StateView, Button
from trytond.transaction import Transaction
from trytond.config import config


__all__ = ['Communication', 'CommunicationAttachment',
    'DocumentCommunicationUser', 'CommunicationLetter',
    'CommunicationLetterReport', 'AddAttachmentStart',
    'AddAttachment']
__metaclass__ = PoolMeta

_STATES = {
    'readonly': Eval('state') != 'draft',
}
EXTENSION = '.pdf'

class Communication(Workflow, ModelSQL, ModelView):
    'Communication'
    __name__ = 'document.communication'
    _rec_name = 'reference'
    reference = fields.Char("Reference", readonly=True, select=True)
    description = fields.Char('Description', required=True, states=_STATES)
    party = fields.Many2One('party.party', 'Party', required=False,
        states=_STATES)
    users_target = fields.Many2Many('res.user-document.communication',
        'document_communication', 'user', 'User Target',
        states=_STATES)
    date_communication = fields.DateTime('Date',
            readonly=True)
    comment = fields.Text('Comment', states=_STATES)
    date_checked = fields.Date('Date Checked', states={
            'readonly': True,
            })
    response_time = fields.Date('Response Date', states=_STATES)
    kind = fields.Selection([
        ('incoming', 'Incoming'),
        ('outgoing', 'Outgoing'),
        ('internal', 'Internal'),
        ], 'Kind', select=True, readonly=True, states=_STATES)
    state = fields.Selection([
        ('draft', 'Draft'),
        ('signed', 'Signed'),
        ('checked', 'Checked'),
        ('cancelled', 'Cancelled'),
        ], 'State', readonly=True, select=True)
    priority = fields.Selection([
        ('low', 'Low'),
        ('middle', 'Middle'),
        ('high', 'High'),
        ], 'Priority', required=True, select=True, states=_STATES)
    attachments = fields.One2Many('document.communication.attachment', 'communication', 
            'Attachments', states={
                'readonly': Eval('state') != 'draft',
                'invisible': And(
                    Not(In(Eval('_user', '0'), Eval('users_target', []))),
                    Eval('create_uid') != Eval('_user', '0'),
                    )
            })

    @classmethod
    def __setup__(cls):
        super(Communication, cls).__setup__()
        cls._error_messages.update({
            'sequence_missing': ('Sequence Communication Documents is missing!'),
            })
        cls._order.insert(0, ('date_communication', 'DESC'))
        cls._transitions |= set((
                ('draft', 'cancelled'),
                ('draft', 'signed'),
                ('signed', 'checked'),
                ))
        cls._buttons.update({
                'draft': {
                    'invisible': True,
                    },
                'checked': {
                    'invisible': Eval('state') != 'signed',
                    },
                'signed': {
                    'invisible': Eval('state') != 'draft',
                    },
                'add_attachment': {
                    'invisible': Eval('state') != 'draft',
                    },
                })

    @staticmethod
    def default_date_communication():
        return datetime.now()

    @staticmethod
    def default_state():
        return 'draft'

    @staticmethod
    def default_priority():
        return 'middle'

    @staticmethod
    def default_kind():
        return 'incoming'

    @classmethod
    @ModelView.button
    @Workflow.transition('draft')
    def draft(cls, docs):
        pass

    @classmethod
    @ModelView.button
    @Workflow.transition('signed')
    def signed(cls, docs):
        for doc in docs:
            doc.set_number()

    @classmethod
    @ModelView.button
    @Workflow.transition('checked')
    def checked(cls, docs):
        for doc in docs:
            doc.write([doc], {'date_checked': date.today()})

    @classmethod
    @ModelView.button_action('document.act_add_attachment')
    def add_attachment(cls, docs):
        for doc in docs:
            doc.set_number()

    def set_number(self):
        if self.reference:
            return
        pool = Pool()
        Sequence = pool.get('ir.sequence')
        Configuration = pool.get('document.configuration')
        configuration = Configuration(1)
        if not configuration.document_communication_sequence:
            self.raise_user_error('sequence_missing',)
        seq = configuration.document_communication_sequence.id
        self.write([self], {'reference': Sequence.get_id(seq)})

    def add_new_attachment(self, vals):
        Attachment = Pool().get('document.communication.attachment')
        attachment, = Attachment.create([{
            'name': vals['name'],
            'communication': self.id,
            'date': vals['date'],
            'path': vals['path'],
            'description': vals['description'],
            'kind': vals['kind'],
            'document_url': vals['document_url'],
        }])
        return attachment


class CommunicationAttachment(ModelSQL, ModelView):
    'Communication Attachment'
    __name__ = 'document.communication.attachment'
    _rec_name = 'description'
    name = fields.Char('Name', required=True, states={
            'readonly': True,
            })
    communication = fields.Many2One('document.communication', 'Communication',
            required=True)
    document_url = fields.Char('Document Url', states={
            'required': True,
            'readonly': True,
            })
    date = fields.Date('Date', required=True)
    employee = fields.Many2One('company.employee', 'Employee',
            required=False)
    path = fields.Char('Path', states={
            'required': False,
            'readonly': True,
            })
    description = fields.Text('Description', required=False)
    is_url_valid = fields.Function(fields.Boolean('Is Url Valid'),
            'get_is_url_valid')
    kind = fields.Selection([
            ('incoming', 'Incoming'),
            ('outgoing', 'Outgoing'),
            ('internal', 'Internal'),
            ], 'Kind', select=True, readonly=True)
    responses = fields.One2Many('document.communication.attachment',
            'parent', 'Responses')
    parent = fields.Many2One('document.communication.attachment', 'Parent',
            select=True, ondelete="RESTRICT")

    @classmethod
    def __setup__(cls):
        super(CommunicationAttachment, cls).__setup__()

    def get_is_url_valid(self, name):
        if self.document_url and os.path.isfile(self.document_url):
            return True
        return False

    @staticmethod
    def default_kind():
        return 'incoming'

    @staticmethod
    def default_date():
        return date.today()


class AddAttachmentStart(ModelView):
    'Add Attachment Start'
    __name__ = 'document.communication.add_attachment.start'
    name = fields.Char('Name Document', required=True)
    buff_file = fields.Binary('Filepath', filename='name_document', required=True)
    date = fields.Date('Date', required=False)
    description = fields.Text('Description', required=False)
    kind = fields.Selection([
            ('incoming', 'Incoming'),
            ('outgoing', 'Outgoing'),
            ('internal', 'Internal'),
            ], 'Kind', select=True)

    @staticmethod
    def default_date():
        return date.today()

    @staticmethod
    def default_kind():
        return 'incoming'


class AddAttachment(Wizard):
    'Add Attachment'
    __name__ = 'document.communication.add_attachment'

    start = StateView('document.communication.add_attachment.start',
        'document.communication_add_attachment_view_form', [
                Button('Cancel', 'end', 'tryton-cancel'),
                Button('Add', 'search_', 'tryton-ok', default=True),
            ])
    search_ = StateTransition()

    @classmethod
    def __setup__(cls):
        super(AddAttachment, cls).__setup__()
        cls._error_messages.update({
                'document_pdf_invalid': 'Document PDF invalid!',
                'missing_path_fs': 'Missing path to Filesystem!',
                'filename_duplicated': 'File name duplicated!',
                })

    def transition_search_(self):
        dest_path = config.get('path_sgd', 'data_path')
        if not dest_path or not os.path.exists(dest_path):
            self.raise_user_error('missing_path_fs')
        year = str(date.today().year)
        full_path = os.path.join(dest_path, year)
        if not os.path.exists(full_path):
            os.mkdir(full_path, 0755)
        pool = Pool()
        Communication = pool.get('document.communication')
        Company = pool.get('company.company')
        communication = Communication(Transaction().context.get('active_id'))
        company = Company(Transaction().context.get('company'))

        name_doc = communication.reference + '_' + self.start.name + EXTENSION
        new_doc = os.path.join(full_path, name_doc)
        if os.path.exists(new_doc):
            self.raise_user_error('filename_duplicated')
        self.set_document(
            new_doc, 
            self.start.buff_file,
            company.party.name,
            communication.reference,
            communication.create_uid.name,
            communication.create_date,
        )
        if not self.start.description:
            description = name_doc
        else:
            description = self.start.description

        data = {
            'name': name_doc,
            'date': self.start.date,
            'path': new_doc,
            'description': description,
            'kind': self.start.kind,
            'document_url': new_doc,
        }
        communication.add_new_attachment(data)
        return 'end'

    def set_document(self, name_doc, buff_file, company, reference, 
            create_uid, create_date):
        if not reference:
            reference = 'RAD-00000001'
        ref = u'Radicado No. ' + reference
        date_ = 'Fecha: ' + str(create_date)[:19]
        user = 'Recibio: ' + create_uid

        # read your existing PDF
        bytes_io = BytesIO(buff_file)
        reader = BufferedReader(bytes_io)
        try:
            existing_pdf = PdfFileReader(reader)
        except:
            self.raise_user_error('document_pdf_invalid')
        page_zero = existing_pdf.getPage(0)
        x_dim = int(page_zero.mediaBox.getUpperRight_x())
        y_dim = int(page_zero.mediaBox.getUpperRight_y())

        # create a new PDF with Reportlab
        # Frame
        string_io = StringIO()
        can = canvas.Canvas(string_io, pagesize=letter)
        can.setLineWidth(0.5)
        can.setFillColor(HexColor(0xFCFCFF), alpha=0.65)
        can.setDash([3, 2, 1, 2], phase=0)
        can.setStrokeColor(HexColor(0xB3B7BA))
        can.rect((x_dim - 145), (y_dim - 52), 135, 45, fill=1)

        # Text Frame
        can.setFont("Helvetica-Oblique", 8)
        can.setFillColorRGB(0.2, 0.2, 0.2)
        can.drawString((x_dim - 140), (y_dim - 18), company)
        can.setFont("Helvetica-Bold", 8)
        can.drawString((x_dim - 140), (y_dim - 28), ref)
        can.setFont("Helvetica", 8)
        can.drawString((x_dim - 140), (y_dim - 38), date_)
        can.drawString((x_dim - 140), (y_dim - 48), user)
        can.save()

        #move to the beginning of the StringIO buffer
        string_io.seek(0)
        new_pdf = PdfFileReader(string_io)

        output = PdfFileWriter()
        # add the "watermark" (which is the new pdf) on the existing page
        for i in range(existing_pdf.getNumPages()):
            page = existing_pdf.getPage(i)
            if i == 0:
                try:
                    page.mergePage(new_pdf.getPage(0))
                except:
                    self.raise_user_error('document_pdf_invalid')
            output.addPage(page)

        # finally, write "output" to a real file
        outputStream = open(name_doc, "wb")
        output.write(outputStream)
        outputStream.close()


class DocumentCommunicationUser(ModelSQL):
    'Document Communication User'
    __name__ = 'res.user-document.communication'
    _table = 'res_user_document_communication'
    document_communication = fields.Many2One('document.communication',
        'Document Communication', ondelete='CASCADE', select=True, 
        required=True)
    user = fields.Many2One('res.user', 'User', select=True, 
        required=True, ondelete='RESTRICT')


class CommunicationLetter(Workflow, ModelSQL, ModelView):
    'Communication Letter'
    __name__ = 'document.communication.letter'
    date = fields.Date('Date', select=True, states=_STATES)
    number = fields.Char('Reference', readonly=True, select=True)    
    subject = fields.Char('Subject', required=True, states=_STATES)
    party = fields.Many2One('party.party', 'Party', required=True,
            select=True, states=_STATES)
    content = fields.Text("Content", select=True, states=_STATES)
    state = fields.Selection([
            ('draft', 'Draft'),
            ('check', 'Check'),
            ('done', 'Done'),
            ], 'State', required=True, readonly=True)

    @classmethod
    def __setup__(cls):
        super(CommunicationLetter, cls).__setup__()
        cls._transitions |= set((
                ('draft', 'check'),
                ('check', 'done'),
                ('check', 'draft'),
                ))
        cls._buttons.update({
                'draft': {
                    'invisible': Eval('state') != 'check',
                    },
                'check': {
                    'invisible': Eval('state') != 'draft',
                    },
                'done': {
                    'invisible': Eval('state') != 'check',
                    },
                })

    @staticmethod
    def default_state():
        return 'draft'

    @staticmethod
    def default_date():
        return date.today()

    @classmethod
    @ModelView.button
    @Workflow.transition('draft')
    def draft(cls, docs):
        pass

    @classmethod
    @ModelView.button
    @Workflow.transition('check')
    def check(cls, docs):
        pass

    @classmethod
    @ModelView.button
    @Workflow.transition('done')
    def done(cls, docs):
        for doc in docs:
            doc.set_sequence()

    def set_sequence(self):
        return
        next_seq = 1
        sequences = [doc for doc in self.record.documents if doc.state == 'posted']
        
        if sequences:
            next_seq = len(sequences) + 1
        self.write([self], {'number': str(next_seq)})


class CommunicationLetterReport(CompanyReport):
    __name__ = 'document.communication.letter'
