#This file is part of Tryton.  The COPYRIGHT file at the top level of
#this repository contains the full copyright notices and license terms.

from trytond.model import ModelView, ModelSQL, fields
from trytond.pyson import Eval
from trytond.transaction import Transaction

__all__ = ['Equipment', 'MaintenanceEquipmentPart', 'CheckListEquipment',
        'EquipmentOperationCard']

# Size photo default
WIDTH = 250
HEIGHT = 250

_STATES_VEHICLE = {
        'invisible': Eval('type_device') != 'vehicle',
        'readonly': ~Eval('active', True),
        }

STATES = {
    'readonly': ~Eval('active', True),
    }

#Pasar Request Service, Planning, Repuestos a Relacionado quitar de la vista directa, agregar Costos Analiticas
class Equipment(ModelSQL, ModelView):
    "Equipment and Instrument"
    __name__ = 'maintenance.equipment'
    _rec_name = 'product'
    product = fields.Many2One('product.template', 'Product',
            domain=[('type', '=', 'assets')], required=True)
    active = fields.Boolean('Active')
    model_num = fields.Char('Model Number', states=STATES)
    code = fields.Char('Code', depends=['product'])
    external_code = fields.Char('External Code', states=STATES)
    brand = fields.Char('Brand', states=STATES)
    serial = fields.Char('Serial', states=STATES)
    criticality = fields.Selection([
            ('low', 'Low'),
            ('middle', 'Middle'),
            ('high', 'High')
            ], 'Criticality', states=STATES)
    purchase_date = fields.Date('Purchase Date', states=STATES)
    capacity = fields.Char('Capacity', states=STATES)
    location = fields.Many2One('maintenance.location', 'Location',
            states=STATES)
    type_device = fields.Selection([
            ('', ''),
            ('equipment', 'Equipment'),
            ('instrument', 'Instrument'),
            ('vehicle', 'Vehicle'),
            ], 'Type Device', required=True, states=STATES)
    parent = fields.Many2One('maintenance.equipment','Parent',
            select=True)
    tolerance = fields.Char('Tolerance', states={
            'invisible': (Eval('type_device') != 'instrument'),
            'readonly': ~Eval('active', True),
            })
    instruments = fields.One2Many('maintenance.equipment', 'parent', 
            string='Equipments Associated', 
            states={
                'invisible': (Eval('type_device') == 'instrument'),
                'readonly': ~Eval('active', True),
                })
    parts = fields.Many2Many('maintenance.equipment.part', 'equipment',
            'part', 'Parts',  states=STATES)
    planning = fields.One2Many('maintenance.planning', 'equipment', 
            'Planning',  states=STATES)
    request = fields.One2Many('maintenance.request_service', 'equipment', 
            'Request Service', states=STATES)
    check_list = fields.One2Many('maintenance.check_list_equipment', 'equipment', 
            'Check List Equipment', states=STATES)
    photo = fields.Binary('Photo', states=STATES)
    party = fields.Many2One('party.party', 'Supplier', select=True,
         states=STATES)
    mnt_supplier = fields.Many2One('party.party', 'Maintenance Supplier', 
            select=True,  states=STATES)
    notes = fields.Text('Notes', states=STATES)
    guarantee = fields.Date('Guarantee',  states=STATES)
    #-----------------------  Vehicle Attributes -----------------------
    motor = fields.Char('Motor', states=_STATES_VEHICLE)
    color = fields.Char('Color', states=_STATES_VEHICLE)
    operation_card = fields.One2Many('maintenance.equipment.operation_card', 
            'equipment', 'Operation Card', states=_STATES_VEHICLE)
    traffic_license_driver = fields.Many2One('party.party', 
            'Traffic License Driver', states=_STATES_VEHICLE)
    traffic_license_number = fields.Char('Traffic License Number', states=_STATES_VEHICLE)
    traffic_office_city = fields.Char('Traffic License City', states=_STATES_VEHICLE)
    insurer_company = fields.Many2One('party.party', 'SOAT Insurer Company', 
            required=False, ondelete="RESTRICT", states=_STATES_VEHICLE)
    soat_number = fields.Char('SOAT Number', states=_STATES_VEHICLE)
    soat_date = fields.Date('SOAT Date', states=_STATES_VEHICLE)
    checktechmec = fields.Date('Check Tech-Mechanical', states=_STATES_VEHICLE)
    license_date = fields.Date('License Date', states=_STATES_VEHICLE)

    @classmethod
    def __setup__(cls):
        super(Equipment, cls).__setup__()

    def get_rec_name(self, name):
        rec_name = self.product.rec_name
        if self.code:
            rec_name = '[' + self.code + ']' + ' ' + rec_name
        return (rec_name)

    @staticmethod
    def default_criticality():
        return 'low'

    @staticmethod
    def default_company():
        return Transaction().context.get('company')

    @classmethod
    def create(cls, vals):
        if not vals:
            return
        res = []
        for val in vals:
            if val.get('photo'):
                val['photo'] = cls._resize_photo(val.get('photo'))
            res.append(val)
        return super(Equipment, cls).create(res)

    @classmethod
    def write(cls, equipments, vals):
        if 'photo' in vals:
            img_buff = vals.get('photo')
            vals['photo'] = cls._resize_photo(img_buff)
        super(Equipment, cls).write(equipments, vals)

    @classmethod
    def _resize_photo(cls, img_buff):
        import Image
        from StringIO import StringIO

        io_image = StringIO(img_buff)
        if io_image.len == 0:
            return None
        img = Image.open(io_image)
        width, height = img.size
        # To try center original photo for resize it
        factor_resize = float(WIDTH)/width
        height_new = int(height * factor_resize)
        
        # To resize photo to standar size (WIDTH, HEIGHT)
        img_resized = img.resize((WIDTH, height_new),)
        img_resized = img_resized.crop((0, 0, WIDTH, HEIGHT))
        #img_resized.show()
        
        # Convert format and save photo.
        io_obj = StringIO()
        img_resized.save(io_obj, 'JPEG')
        img_data_buff  = io_obj.getvalue()
        io_obj.close()
        return img_data_buff

    @fields.depends('product', 'code', 'active')
    def on_change_product(self):
        if self.product:
            return {
                'code' : self.product.products[0].code,
                'active' : self.product.active,
                }
        else:
            return {'code' : None}

    
class MaintenanceEquipmentPart(ModelSQL):
    'Maintenance Equipment - Part'
    __name__ = 'maintenance.equipment.part'
    _table = 'maintenance_equipment_part'
    equipment = fields.Many2One('maintenance.equipment', 'Maintenance Equipment', 
            ondelete='CASCADE', select=True, required=True)
    part = fields.Many2One('product.template', 'Product', ondelete='RESTRICT',
            domain=[('type', '=', 'goods')], select=True, required=True)


class EquipmentOperationCard(ModelSQL, ModelView):
    'Equipment Operation Card'
    __name__ = 'maintenance.equipment.operation_card'
    equipment = fields.Many2One('maintenance.equipment', 'Equipment', 
            ondelete='CASCADE', required=True)
    year = fields.Char('Year', required=True)
    card_number = fields.Char('Card Number', required=True)

    @classmethod
    def __setup__(cls):
        super(EquipmentOperationCard, cls).__setup__()
        cls._order.insert(0, ('year', 'DESC'))

class CheckListEquipment(ModelSQL, ModelView):
    'Check List Equipment'
    __name__ = 'maintenance.check_list_equipment'
    element = fields.Char('Element', required=True)
    quantity = fields.Numeric('Quantity', required=True)
    checked = fields.Boolean('Checked')
    notes = fields.Char('Notes')
    equipment = fields.Many2One('maintenance.equipment', 'Equipment')

    @classmethod
    def __setup__(cls):
        super(CheckListEquipment, cls).__setup__()
