#This file is part of Tryton.  The COPYRIGHT file at the top level of
#this repository contains the full copyright notices and license terms.
from datetime import datetime, timedelta, time
from trytond.model import Workflow, ModelView, ModelSQL, fields
from trytond.pyson import Eval, Id
from trytond.pool import Pool

__all__ = ['PlanningPeriod', 'Planning', 'PlanningLines', 'ForecastPlanning']

STATES = {'readonly': (Eval('state') != 'draft')}


class PlanningPeriod(Workflow, ModelSQL, ModelView):
    'Maintenance Planning Period'
    __name__ = 'maintenance.planning_period'
    name = fields.Char('Name Period', required=True, states=STATES)
    reference = fields.Char('Reference', required=True, states=STATES)
    start_date = fields.Date('Start Date', required=True, states=STATES)
    end_date = fields.Date('End Date', required=True, states=STATES)
    # FIXME: change name draft, process, closed
    state = fields.Selection([
        ('draft', 'Draft'),
        ('cancel', 'Cancel'),
        ('done', 'Done'),
        ], 'State', readonly=True, required=True)

    @classmethod
    def __setup__(cls):
        super(PlanningPeriod, cls).__setup__()
        cls._constraints += [
                ('overlap_date', 'periods_overlaps'),
                ('check_dates', 'wrong_date'),
                ]
        cls._order.insert(0, ('start_date', 'ASC'))
        cls._error_messages.update({
                'periods_overlaps': 'You can not have two overlapping periods!',
                'wrong_date': 'The start date is greater or equal than end date \n'
                              'or start date is smaller that today!'
                })
        cls._transitions |= set((
                ('draft', 'done'),
                ('draft', 'cancel'),
                ))
        cls._buttons.update({
                'draft': {
                    'invisible': Eval('state') != 'draft',
                    },        
                'cancel': {
                    'invisible': Eval('state') != 'draft',
                    },
                'done': {
                    'invisible': Eval('state') != 'draft',
                    },
                })

        # The states where amounts are cached
        cls._states_cached = ['draft', 'done', 'cancel']

    @staticmethod
    def default_state():
        return 'draft'

    @classmethod
    @ModelView.button
    @Workflow.transition('draft')
    def draft(cls, records):
        pass

    @classmethod
    @ModelView.button
    @Workflow.transition('cancel')
    def cancel(cls, records):
        pass

    @classmethod
    @ModelView.button
    @Workflow.transition('done')
    def done(cls, records):
        for record in records:
            cls.create_forecast(record)

    def _get_last_date(self, equipment, code):
        pool = Pool()
        last_date = None
        last_plan_date = None
        Request = pool.get('maintenance.request_service')
        Forecast = pool.get('maintenance.forecast_planning')

        requests = Request.search([
                ('equipment', '=', equipment),
                ('forecast_plannings.code', '=', code),
                ('state', '=', 'done')
                ])

        forecasts = Forecast.search([
                ('equipment', '=', equipment),
                ('code', '=', code),
                ])

        if requests:
            for req in requests:
                if not last_date or req.effective_date > last_date:
                    last_date = req.effective_date
            if isinstance(last_date, datetime):
                last_date = last_date.date()

        if forecasts:
            for forecast in forecasts:
                if not last_plan_date or forecast.planning_date > last_plan_date:
                    last_plan_date = forecast.planning_date
            if not last_date or last_date < last_plan_date:
                last_date = last_plan_date
        return last_date

    def _get_plan_date(self, period, line, last_date):
        plan_date = []
        frecuency = timedelta(float(line.frecuency))
        if isinstance(last_date, datetime):
            last_date = last_date.date()

        if not last_date or (last_date + frecuency) < self.start_date:
            last_date = self.start_date

        while (last_date + frecuency) <= period.end_date:
            last_date = last_date + frecuency
            plan_date.append(last_date)
        return plan_date

    @classmethod
    def check_dates(cls, plan_periods):
        now = datetime.now().date()
        for period in plan_periods:
            if period.start_date >= period.end_date or period.start_date <= now:
                return False
        return True

    @classmethod
    def overlap_date(cls, plan_periods):
        new_period = plan_periods[0]
        old_periods = cls.search([('id', '!=', new_period.id)])
        for old_period in old_periods:
            if (
                new_period.start_date <= old_period.start_date and \
                new_period.end_date >= old_period.start_date) \
                or (
                new_period.start_date >= old_period.start_date and \
                new_period.start_date <= old_period.end_date \
                ):
                return False
        return True

    @classmethod
    def create_forecast(cls, period):
        pool = Pool()
        Planning = pool.get('maintenance.planning')
        Forecast = pool.get('maintenance.forecast_planning')
        plannings = Planning.search([])
        lines_to_create = []
        for plan in plannings:
            for line in plan.lines:
                last_date = period._get_last_date(plan.equipment, line.code)
                plan_date = period._get_plan_date(period, line, last_date)
                print(line.id, plan_date)
                if not plan_date:
                    continue
                for new_date in plan_date:
                    lines_to_create.append({
                        'plan_period': period.id,
                        'planning': plan.id,
                        'equipment': plan.equipment,
                        'description': line.description,
                        'code': line.id,
                        'piece': line.piece,
                        'last_date': last_date,
                        'work_forecast_hours': line.work_forecast_hours,
                        'planning_date': new_date,
                        'state': 'open',
                        })
        Forecast.create(lines_to_create)


class Planning(ModelSQL, ModelView):
    'Maintenance Planning'
    __name__ = 'maintenance.planning'
    equipment = fields.Many2One('maintenance.equipment', 'Equipment', 
            select=True, required=True)
    type_activity = fields.Many2One('maintenance.type_activity', 
            'Type Activity', required=True)
    notes = fields.Text('Notes')
    party = fields.Many2One('party.party', 'Party', select=True)
    lines = fields.One2Many('maintenance.planning.lines', 'planning', 
            'Lines', required=False)

    @classmethod
    def __setup__(cls):
        super(Planning, cls).__setup__()


class PlanningLines(ModelSQL, ModelView):
    'Maintenance Planning Lines'
    __name__ = 'maintenance.planning.lines'
    _rec_name = 'code'
    planning = fields.Many2One('maintenance.planning', 'Planning', 
            required=True)
    code = fields.Char('Code', required=True)
    piece = fields.Char('Piece', required=True)
    description = fields.Text('Description', required=True)
    frecuency = fields.Numeric('Frecuency')
    work_forecast_hours = fields.Numeric('Work Forecast Hours')
    uom_frecuency = fields.Many2One('product.uom', 'UoM Frecuency', required=True,
            domain=[('category', '=', Id('product', 'uom_cat_time'))])
    fulfillment = fields.Numeric('Fulfillment')
    party = fields.Many2One('party.party', 'Party', select=True)

    @classmethod
    def __setup__(cls):
        super(PlanningLines, cls).__setup__()


class ForecastPlanning(Workflow, ModelSQL, ModelView):
    'Maintenance Forecast Planning'
    __name__ = 'maintenance.forecast_planning'
    plan_period = fields.Many2One('maintenance.planning_period', 'Period')
    planning = fields.Many2One('maintenance.planning', 'Planning')
    equipment = fields.Many2One('maintenance.equipment', 'Equipment', 
            required=True, depends=['code'])
    #FIXME: change name to code_line
    code = fields.Many2One('maintenance.planning.lines', 'Code',
            required=True)
    piece = fields.Char('Piece')
    work_forecast_hours = fields.Numeric('Work Forecast Hours')
    description = fields.Char('Description', required=True)
    last_date = fields.Date('Last Date')
    planning_date = fields.Date('Planning Date', required=True, select=True)
    exec_date = fields.Date('Execution Date')
    request = fields.Many2One('maintenance.request_service', 
            'Reference Request Service', readonly=True)
    party = fields.Many2One('party.party', 'Party', select=True)
    state = fields.Selection([
            ('open', 'Open'),
            ('process', 'Process'),
            ('cancel', 'Cancel'),
            ('done', 'Done'),
            ], 'State', readonly=True, required=True)

    @classmethod
    def __setup__(cls):
        super(ForecastPlanning, cls).__setup__()
        cls._transitions |= set((
                ('open', 'process'),
                ('process', 'done'),
                ('open', 'cancel'),
                ))
        cls._buttons.update({
                'open': {
                    'invisible': Eval('state') != 'open',
                    },
                'cancel': {
                    'invisible': Eval('state') != 'open',
                    },
                'process': {
                    'invisible': Eval('state') != 'open',
                    },
                'done': {
                    'invisible': Eval('state') != 'process',
                    },
                })

    @classmethod
    def write(cls, forecasts, vals):
        if vals.get('planning_date'):
            for forecast in forecasts:
                replanning = cls.search([
                                ('code', '=', forecast.code),
                                ('planning_date', '>', forecast.planning_date),
                                ('state', '=', 'open')
                                ])
                delta_days = vals.get('planning_date') - forecast.planning_date
                for replan_forecast in replanning:
                    super(ForecastPlanning, cls).write(replan_forecast.id, {
                            'planning_date': (replan_forecast.planning_date + delta_days)
                            })
        else:
            print forecasts
            super(ForecastPlanning, cls).write(forecasts, vals)

    @staticmethod
    def default_state():
        return 'open'

    @fields.depends('code', 'equipment', 'piece', 'description')
    def on_change_code(self):
        res = {}
        pool = Pool()
        PlanLines = pool.get('maintenance.planning.lines')
        
        if self.code:
            plan_line = PlanLines.search([
                ('code', '=', self.code.code),
                ])
            if plan_line:
                plan_line = plan_line[0]
                res['equipment'] = plan_line.planning.equipment.id
                res['piece'] = plan_line.piece
                res['description'] = plan_line.description
        return res

    @classmethod
    def create_request_service(cls, forecast):
        Request = Pool().get('maintenance.request_service')
        start_time = datetime.combine(forecast.planning_date, time())
        description = forecast.piece + " - " + forecast.description
        period_name = None
        location_id = None
        if forecast.plan_period:
            period_name = forecast.plan_period.rec_name
        if forecast.equipment.location:
            location_id = forecast.equipment.location.id
        request_id = Request.create([{
            'start_time': start_time,
            'repair_time': forecast.work_forecast_hours,
            'equipment': forecast.equipment.id,
            'location': location_id,
            'forecast_plannings': [('add', [forecast.id])],
            'code': forecast.code.rec_name,
            'period': period_name,
            'description': description,
            'type_activity': forecast.planning.type_activity.id,
            'department': None,
            'priority': 'low',
            'state': 'open',
            }])
        return request_id

    @classmethod
    @ModelView.button
    @Workflow.transition('open')
    def open(cls, forecasts):
        pass

    @classmethod
    @ModelView.button
    @Workflow.transition('cancel')
    def cancel(cls, forecasts):
        pass

    @classmethod
    @ModelView.button
    @Workflow.transition('process')
    def process(cls, forecasts):
        for record in forecasts:
            cls.create_request_service(record)

    @classmethod
    @ModelView.button
    @Workflow.transition('done')
    def done(cls, forecasts):
        pass
