#This file is part of Tryton.  The COPYRIGHT file at the top level of
#this repository contains the full copyright notices and license terms.
from __future__ import with_statement
from datetime import datetime
from trytond.model import Workflow, ModelView, ModelSQL, fields
from trytond.modules.company import CompanyReport
from trytond.pyson import Not, Equal, Eval, Or, If, In, Get, And
from trytond.transaction import Transaction
from trytond.pool import Pool

__all__ = ['RequestService', 'RequestServiceReport', 'CheckListService']


STATES_USER = {
    'readonly': (Eval('state') != 'open'),
}

STATES_ADMIN = {
    'readonly': Not(In(Eval('state'), ['approved', 'assigned', 'waiting_approval'])),
}


class CheckListService(Workflow, ModelSQL, ModelView):
    'Check List Service'
    __name__ = 'maintenance.check_list'
    element = fields.Char('Element', required=True)
    quantity = fields.Numeric('Quantity', required=True)
    checked = fields.Boolean('Checked')
    notes = fields.Char('Notes')
    request = fields.Many2One('maintenance.request_service', 'Request')

    @classmethod
    def __setup__(cls):
        super(CheckListService, cls).__setup__()


class RequestService(Workflow, ModelSQL, ModelView):
    'Request Service'
    __name__ = 'maintenance.request_service'
    _rec_name = 'reference'
    reference = fields.Char('Reference', readonly=True, select=True)
    company = fields.Many2One('company.company', 'Company', required=True,
            states=STATES_USER, domain=[('id', If(In('company',
            Eval('context', {})), '=', '!='), Get(Eval('context', {}),
            'company', 0)), ])
    start_time = fields.DateTime('Start Time', states=STATES_ADMIN)
    repair_time = fields.Numeric('Repair Time', help="In hours",
        states=STATES_ADMIN)
    equipment = fields.Many2One('maintenance.equipment', 'Equipment',
            states={'readonly': Or(
                    (Eval('create_uid') != Eval('_user', '0')),
                    (Eval('state') != 'open')),
                    }, required=True)
    location = fields.Many2One('maintenance.location', 'Location',
            required=True, depends=['equipment'],
            states={'readonly': Or(
                    (Eval('create_uid') != Eval('_user', '0')),
                    (Eval('state') != 'open')),
                    })
    description = fields.Text('Description', required=True,
            states={'readonly': Or(
                    (Eval('create_uid') != Eval('_user', '0')),
                    (Eval('state') != 'open')),
            })
    type_activity = fields.Many2One('maintenance.type_activity',
            'Type Activity', states=STATES_ADMIN)
    action = fields.Text('Action', states=STATES_ADMIN)
    forecast_plannings = fields.One2Many('maintenance.forecast_planning',
            'request', 'Forecast Planning', states=STATES_ADMIN)
    code = fields.Char('Code', readonly=True, states=STATES_ADMIN)
    period = fields.Char('Period', readonly=True, states=STATES_ADMIN)
    assigned_to = fields.Many2One('party.party', 'Party', select=True,
            states=STATES_ADMIN)
    type_service = fields.Selection([
            ('internal', 'Internal'),
            ('external', 'External'),
            ('both', 'Both'),
            ], 'Type Service', states=STATES_ADMIN)
    priority = fields.Selection([
            ('urgent', 'Urgent'),
            ('important', 'Important'),
            ('low', 'Low'),
            ], 'Priority', required=True, states=STATES_USER)
    effective_date = fields.DateTime('Effective Datetime',
            states={'readonly': Not(In(Eval('state'),
                        ['approved', 'assigned', 'waiting_approval'])),
                    'required': Equal(Eval('state'), 'done')})
    notes = fields.Char('Notes', states=STATES_ADMIN)
    efficacy = fields.Selection([
            ('yes', 'Yes'),
            ('no', 'No'),
            ('', ''),
            ], 'Efficacy', states={
            'readonly': Or(
                    (Eval('create_uid') != Eval('_user', '0')),
                    (Eval('state') != 'done')),
            'required': Eval('state') == 'marked',
            })
    department = fields.Many2One('company.department', 'Department',
            states=STATES_USER)
    state = fields.Selection([
        ('open', 'Open'),
        ('assigned', 'Assigned'),
        ('cancel', 'Cancel'),
        ('waiting_approval', 'Waiting Approval'),
        ('approved', 'Approved'),
        ('no_approved', 'No Approved'),
        ('done', 'Done'),
        ('marked', 'Marked'),
    ], 'State', readonly=True, required=True)
    check_list = fields.One2Many('maintenance.check_list',
            'request', 'Check List', states=STATES_ADMIN)

    @classmethod
    def __setup__(cls):
        super(RequestService, cls).__setup__()
        cls._order.insert(0, ('create_date', 'DESC'))
        cls._transitions |= set((
                ('open', 'assigned'),
                ('open', 'cancel'),
                ('assigned', 'done'),
                ('assigned', 'waiting_approval'),
                ('waiting_approval', 'no_approved'),
                ('waiting_approval', 'approved'),
                ('approved', 'done'),
                ('done', 'marked'),
                ))
        cls._buttons.update({
                'open': {
                    'invisible': Eval('state') != 'open',
                    },
                'assign': {
                    'invisible': Eval('state') != 'open',
                    },
                'cancel': {
                    'invisible': Eval('state') != 'open',
                    },
                'waiting_approval': {
                    'invisible': Eval('state') != 'assigned',
                    },
                'approved': {
                    'invisible': Eval('state') != 'waiting_approval',
                    },
                'no_approved': {
                    'invisible': Eval('state') != 'waiting_approval',
                    },
                'done': {
                    'invisible': And(
                                    Eval('state') != 'assigned',
                                    Eval('state') != 'approved',
                                    ),
                    },
                'marked': {
                    'invisible': Eval('state') != 'done',
                    },
                })

    def get_rec_name(self, name):
        return self.reference

    @staticmethod
    def default_company():
        return Transaction().context.get('company') or False

    @staticmethod
    def default_state():
        return 'open'

    @staticmethod
    def default_priority():
        return 'low'

    @staticmethod
    def default_type_service():
        return 'internal'

    @classmethod
    @ModelView.button
    @Workflow.transition('open')
    def open(cls, records):
        pass

    @classmethod
    @ModelView.button
    @Workflow.transition('cancel')
    def cancel(cls, records):
        pass

    @classmethod
    @ModelView.button
    @Workflow.transition('assigned')
    def assign(cls, records):
        cls.set_reference(records)

    @classmethod
    @ModelView.button
    @Workflow.transition('waiting_approval')
    def waiting_approval(cls, records):
        pass

    @classmethod
    @ModelView.button
    @Workflow.transition('approved')
    def approved(cls, records):
        pass

    @classmethod
    @ModelView.button
    @Workflow.transition('no_approved')
    def no_approved(cls, records):
        pass

    @classmethod
    @ModelView.button
    @Workflow.transition('done')
    def done(cls, records):
        for record in records:
            cls._close_forecast(record)

    @classmethod
    @ModelView.button
    @Workflow.transition('marked')
    def marked(cls, records):
        pass


    @classmethod
    def set_reference(cls, requests):
        '''
        Fill the reference field with the request sequence
        '''
        pool = Pool()
        Sequence = pool.get('ir.sequence')
        Config = pool.get('maintenance.configuration')
        config = Config(1)

        for request in requests:
            if request.reference:
                continue
            if not config.maintenance_sequence:
                continue
            reference = Sequence.get_id(config.maintenance_sequence.id)
            cls.write([request], {'reference': reference})

    @classmethod
    def set_created_by(cls, requests):
        '''
        Fill the created_by field with the request
        '''
        pool = Pool()
        User = pool.get('res.user')
        user = User.search(Transaction().user)

        for request in requests:
            cls.write(request.id, {
                'created_by': user.id,
                })

    @fields.depends('equipment', 'location')
    def on_change_equipment(self):
        '''
        Fill the location field with the request
        '''
        res = {'location': None}
        if self.equipment:
            if self.equipment.location:
                res['location'] = self.equipment.location.id
        return res

    @classmethod
    def _close_forecast(cls, record):
        Forecast = Pool().get('maintenance.forecast_planning')
        if record.forecast_plannings:
            for forecast in record.forecast_plannings:
                eff_date = None
                if isinstance(record.effective_date, datetime):
                    eff_date = record.effective_date.date()
                Forecast.write([forecast], {
                        'state': 'done',
                        'exec_date': eff_date
                        })


class RequestServiceReport(CompanyReport):
    __name__ = 'maintenance.request_service'

    @classmethod
    def parse(cls, report, objects, data, localcontext):
        user = Pool().get('res.user')(Transaction().user)
        localcontext['company'] = user.company
        localcontext['user'] = user
        return super(RequestServiceReport, cls).parse(report, objects, 
            data, localcontext)
