#This file is part of Tryton.  The COPYRIGHT file at the top level of
#this repository contains the full copyright notices and license terms.

import locale
from trytond.pool import PoolMeta
from trytond.model import ModelView, ModelSQL, fields
from trytond.report import Report
from trytond.pool import Pool
from trytond.transaction import Transaction

__all__ = ['Uniform', 'MedicalExam', 'EmployeeMedicalExam',
        'ContractingGeneralReport', 'UniformReport', 
        'MedicalExamReport', 'BankLetterReport',
        'FinishContractReport', 'RetirementExamReport',
        'FreeLiabilityReport', 'CertificationReport', 
        'ContractsEmployeesReport', 'LegalContractReport',
        'AdministrativeContractReport',
]
__metaclass__ = PoolMeta

locale.setlocale(locale.LC_ALL, 'es_CO.utf8')

class MedicalExam(ModelSQL, ModelView):
    'Empolyee Medical Exam'
    __name__ = 'staff.medical_exam'
    name = fields.Char('Name', required=True)
    description = fields.Char('Description')
    category = fields.Selection([
            ('laboratory', 'Laboratory'),
            ('occupational_health', 'Occupational Health'),
            ], 'Category', select=True)

    @classmethod
    def __setup__(cls):
        super(MedicalExam, cls).__setup__()


class EmployeeMedicalExam(ModelSQL):
    'Employee - Medical Exams'
    __name__ = 'staff.employee-staff.medical_exam'
    employee = fields.Many2One('company.employee', 'Employee', 
            ondelete='CASCADE', select=True, required=True)
    medical_exam = fields.Many2One('staff.medical_exam', 
            'Medical Exam', ondelete='CASCADE', select=True, 
            required=True)

    @classmethod
    def __setup__(cls):
        super(EmployeeMedicalExam, cls).__setup__()


class Uniform(ModelSQL, ModelView):
    'Staff Uniform'
    __name__ = 'staff.uniform'
    element = fields.Selection([
            ('shirt', 'Shirt'),
            ('pant', 'Pants'),
            ('boots', 'Boots'),
            ('helmet', 'Helmet'),
            ('glass', 'Glass'),
            ('hearing_protectors', 'Hearing Protectors'),
            ('muffler', 'Muffler'),
            ('chest_guard', 'Chest Guard'),
            ('sleeve', 'Sleeve'),
            ('leggings', 'Leggings'),
            ('mask', 'Mask'),
            ('glove', 'Glove'),
            ('coverall', 'Coverall'),
            ('other', 'Other'),
            ('', ''),
            ], 'Element', required=True, select=True)
    quantity = fields.Integer('Quantity')
    description = fields.Char('Description')
    size = fields.Char('Size')
    employee = fields.Many2One('company.employee', 'Employee')
    employee_category = fields.Many2One('staff.employee_category', 
        'Employee Category')
    date = fields.Date('Date', required=False)

    @classmethod
    def __setup__(cls):
        super(Uniform, cls).__setup__()


class ContractingGeneralReport(Report):
    'Contracting Report'
    __name__ = 'staff_contracting.general_report'

    @classmethod
    def parse(cls, report, objects, data, localcontext):
        user = Pool().get('res.user')(Transaction().user)
        localcontext['company'] = user.company
        localcontext['user'] = user
        return super(ContractingGeneralReport, cls).parse(report, objects, data,
            localcontext)


class UniformReport(Report):
    'Uniform Report'
    __name__ = 'staff_contracting.uniform_report'

    @classmethod
    def parse(cls, report, objects, data, localcontext):
        user = Pool().get('res.user')(Transaction().user)
        localcontext['company'] = user.company
        localcontext['user'] = user
        return super(UniformReport, cls).parse(report, objects, 
            data, localcontext)


class MedicalExamReport(Report):
    'Medical Exam Report'
    __name__ = 'staff_contracting.medical_exam_report'

    @classmethod
    def parse(cls, report, objects, data, localcontext):
        user = Pool().get('res.user')(Transaction().user)
        localcontext['company'] = user.company
        localcontext['user'] = user
        for obj in objects:
            if not obj.laboratory:
                obj.raise_user_error('The employee does not have laboratory!')
                return
        return super(MedicalExamReport, cls).parse(report, objects, 
            data, localcontext)


class BankLetterReport(Report):
    "Bank Letter Report"
    __name__ = 'staff_contracting.bank_letter_report'

    @classmethod
    def parse(cls, report, objects, data, localcontext):
        user = Pool().get('res.user')(Transaction().user)
        localcontext['company'] = user.company
        localcontext['user'] = user
        for obj in objects:
            if not obj.party.bank_accounts:
                obj.raise_user_error('The employee does not have bank!')
                return
        return super(BankLetterReport, cls).parse(report, objects,
            data, localcontext)


class FinishContractReport(Report):
    'Finish Contract Report'
    __name__ = 'staff_contracting.finish_contract_report'

    @classmethod
    def parse(cls, report, objects, data, localcontext):
        user = Pool().get('res.user')(Transaction().user)
        localcontext['company'] = user.company
        localcontext['user'] = user
        return super(FinishContractReport, cls).parse(report, 
                objects, data, localcontext)


class RetirementExamReport(Report):
    'Retirement Exam Report'
    __name__ = 'staff_contracting.retirement_exam_report'

    @classmethod
    def parse(cls, report, objects, data, localcontext):
        user = Pool().get('res.user')(Transaction().user)
        localcontext['company'] = user.company
        localcontext['user'] = user
        return super(RetirementExamReport, cls).parse(report, 
                objects, data, localcontext)


class FreeLiabilityReport(Report):
    'Free Liability Report'
    __name__ = 'staff_contracting.free_liability_report'

    @classmethod
    def parse(cls, report, objects, data, localcontext):
        user = Pool().get('res.user')(Transaction().user)
        localcontext['company'] = user.company
        localcontext['user'] = user
        return super(FreeLiabilityReport, cls).parse(report, 
                objects, data, localcontext)


class CertificationReport(Report):
    'Certification Report'
    __name__ = 'staff_contracting.certification_report'

    @classmethod
    def parse(cls, report, objects, data, localcontext):
        user = Pool().get('res.user')(Transaction().user)
        localcontext['company'] = user.company
        localcontext['user'] = user
        return super(CertificationReport, cls).parse(report, 
                objects, data, localcontext)


class ContractsEmployeesReport(Report):
    'Contracts Employees Report'
    __name__ = 'staff_contracting.contracts_employees_report'

    @classmethod
    def parse(cls, report, objects, data, localcontext):
        user = Pool().get('res.user')(Transaction().user)
        localcontext['company'] = user.company
        localcontext['user'] = user
        return super(ContractsEmployeesReport, cls).parse(report, 
                objects, data, localcontext)


class LegalContractReport(Report):
    'Legal Contract'
    __name__ = 'staff_contracting.legal_contract_report'

    @classmethod
    def parse(cls, report, objects, data, localcontext):
        user = Pool().get('res.user')(Transaction().user)
        localcontext['company'] = user.company
        localcontext['user'] = user
        localcontext['locale'] = locale
        return super(LegalContractReport, cls).parse(report, 
                objects, data, localcontext)


class AdministrativeContractReport(Report):
    'Administrative Contract'
    __name__ = 'staff_contracting.administrative_contract_report'

    @classmethod
    def parse(cls, report, objects, data, localcontext):
        user = Pool().get('res.user')(Transaction().user)
        localcontext['company'] = user.company
        localcontext['user'] = user
        localcontext['locale'] = locale
        return super(AdministrativeContractReport, cls).parse(report, 
                objects, data, localcontext)
