#This file is part of Tryton.  The COPYRIGHT file at the top level of
#this repository contains the full copyright notices and license terms.
from trytond.model import fields, ModelSQL
from trytond.pool import PoolMeta, Pool
from trytond.transaction import Transaction
from trytond.wizard import Wizard, StateTransition

__all__ = ['Employee', 'EmployeeCategory', 'CreateContractingAttributes',
    'EmployeeCategoryMedicalExamDefault']

__metaclass__ = PoolMeta

class Employee:
    __name__ = 'company.employee'
    laboratory = fields.Many2One('party.party', 'Laboratory', select=True)
    payment_detail = fields.Char('Payment Detail', select=True)
    examination_medical_date = fields.Date('Examination Medical Date')
    box_family_date = fields.Date('Box Family Date')
    health_input_date = fields.Date('Health Input Date')
    health_output_date = fields.Date('Health Output Date')
    retirement_input_date = fields.Date('Retirement Input Date')
    retirement_output_date = fields.Date('Retirement Output Date')
    risk_input_date = fields.Date('Risk Input Date')
    risk_output_date = fields.Date('Risk Output Date')
    hse_training = fields.Date('HSE Training')
    coverage_start = fields.Date('Coverage Start')
    coverage_end = fields.Date('Coverage End')
    uniform = fields.One2Many('staff.uniform', 'employee', 'Uniform')
    medical_exam = fields.Many2Many('staff.employee-staff.medical_exam',
            'employee', 'medical_exam', 'Employee - Medical Exam')
    specific_labour = fields.Char('Specific Labour')
    syndicate = fields.Selection([
            ('yes', 'Yes'),
            ('no', 'No'),
            ('', ''),
            ], 'Syndicate', select=True)
    party_health = fields.Function(fields.Char('Party Health'),
            'get_party_concept')
    party_retirement = fields.Function(fields.Char('Party Retirement'),
            'get_party_concept')
    party_risk = fields.Function(fields.Char('Party Risk'),
            'get_party_concept')
    party_box_family = fields.Function(fields.Char('Party Box Family'),
            'get_party_concept')
    party_bank = fields.Function(fields.Char('Party Bank'),
            'get_party_concept')

    @classmethod
    def __setup__(cls):
        super(Employee, cls).__setup__()

    def get_party_concept(self, name=None):
        res = ''
        name = name[6:]
        for p in self.mandatory_wages:
            if p.wage_type.type_concept == name and p.party:
                res = p.party.name
                break
        return res

    def set_default_values(self):
        if not self.category:
            return

        pool = Pool()
        Uniform = pool.get('staff.uniform')
        MedicalExam = pool.get('staff.employee-staff.medical_exam')

        values = []
        for exam in self.category.medical_exam_default:
            val = {
                'employee': self.id,
                'medical_exam': exam.id,
                }
            values.append(val)
        if values:
            MedicalExam.create(values)
        values = []
        for uniform in self.category.uniform_default:
            val = {
                'employee': self.id,
                'element': uniform.element,
                'quantity': uniform.quantity,
                'description': uniform.description,
                'size': uniform.size,
                }
            values.append(val)
        if values:
            Uniform.create(values)


class EmployeeCategory:
    __name__ = 'staff.employee_category'
    uniform_default = fields.One2Many('staff.uniform', 'employee_category',
            'Uniform')
    medical_exam_default = fields.Many2Many('staff.employee_category-staff.medical_exam_default',
            'employee_category', 'medical_exam',
            'Employee Category - Medical Exam')

    @classmethod
    def __setup__(cls):
        super(EmployeeCategory, cls).__setup__()


class EmployeeCategoryMedicalExamDefault(ModelSQL):
    'Employee Category - Medical Exam Default'
    __name__ = 'staff.employee_category-staff.medical_exam_default'
    employee_category = fields.Many2One('staff.employee_category', 
            'Employee Category', select=True, required=True, ondelete='CASCADE')
    medical_exam = fields.Many2One('staff.medical_exam', 
            'Medical Exam', ondelete='CASCADE', select=True, 
            required=True)

    @classmethod
    def __setup__(cls):
        super(EmployeeCategoryMedicalExamDefault, cls).__setup__()


class CreateContractingAttributes(Wizard):
    'Create Contracting Attributes'
    __name__ = 'staff.payroll.create_contracting_attributes'
    start_state = 'create_attributes'
    create_attributes = StateTransition()

    def transition_create_attributes(self):
        pool = Pool()
        Employee = pool.get('company.employee')

        employee_id = Transaction().context['active_id']
        employee = Employee(employee_id)
        employee.set_default_values()
        return 'end'
