#This file is part of Tryton.  The COPYRIGHT file at the top level of
#this repository contains the full copyright notices and license terms.
import datetime
from decimal import Decimal
from trytond.model import Workflow, ModelView, ModelSQL, fields
from trytond.pyson import Bool, Eval, If, Id
from trytond.pool import Pool, PoolMeta
from trytond.transaction import Transaction
from trytond.modules.company import CompanyReport
from trytond.wizard import Wizard, StateView, StateTransition, Button

__all__ = ['Payroll', 'PayrollLine', 'PayrollReport', 'Move',
        'PayrollGroupStart', 'PayrollGroup', 'PayrollPreliquidation']
__metaclass__ = PoolMeta

STATES = {'readonly': (Eval('state') != 'draft'),}


def dec(value):
    return Decimal(str(value))

_ZERO = Decimal('0.0')


class Payroll(Workflow, ModelSQL, ModelView):
    "Staff Payroll"
    __name__ = "staff.payroll"
    _rec_name = 'number'
    number = fields.Char('Number', readonly=True, help="Secuence")
    period = fields.Many2One('staff.payroll.period', 'Period',
            required=True, states={
                'readonly': (Eval('state') != 'draft'),
                'required': (Eval('state') == 'normal'),
            })
    employee = fields.Many2One('company.employee', 'Employee',
            states=STATES, required=True, depends=['state'])
    kind = fields.Selection([
            ('normal', 'Normal'),
            ('special', 'Special'),
            ], 'Kind', required=True, select=True, states=STATES,
            help="Special allow overlap dates with another payroll")
    start = fields.Date('Start', states=STATES, required=True)
    end = fields.Date('End', states=STATES, required=True)
    date_effective = fields.Date('Date Effective', states=STATES)
    description = fields.Char('Description', states=STATES)
    lines = fields.One2Many('staff.payroll.line', 'payroll', 'Wage Line',
            states=STATES, depends=['employee', 'state'])
    gross_payments = fields.Function(fields.Property(fields.Numeric(
            'Gross Payments', states=STATES, digits=(16, 2))),
            'get_gross_payments')
    total_deductions = fields.Function(fields.Property(fields.Numeric(
            'Total Deductions', states=STATES, digits=(16, 2))),
            'get_total_deductions')
    net_payment = fields.Function(fields.Property(fields.Numeric(
            'Net Payment', states=STATES, digits=(16, 2))), 
            'get_net_payment')
    total_cost = fields.Function(fields.Numeric('Total Cost', 
            states=STATES, digits=(16, 2)), 'get_total_cost')
    currency = fields.Function(fields.Many2One('currency.currency',
            'Currency'), 'get_currency')
    worked_days = fields.Function(fields.Integer('Worked Days'),
            'get_worked_days')
    state = fields.Selection([
            ('draft', 'Draft'),
            ('processed', 'Processed'),
            ('cancel', 'Cancel'),
            ('done', 'Done'),
            ], 'State', readonly=True)
    journal = fields.Many2One('account.journal', 'Journal', required=True,
            states=STATES)
    company = fields.Many2One('company.company', 'Company', required=True,
        states={
            'readonly': (Eval('state') != 'draft') | Eval('lines', [0]),
            },
        domain=[
            ('id', If(Eval('context', {}).contains('company'), '=', '!='),
                Eval('context', {}).get('company', 0)),
            ],
        depends=['state'], select=True)
    move = fields.Many2One('account.move', 'Move', readonly=True)
    origin = fields.Reference('Origin', selection='get_origin', 
            select=True, depends=['state'],
            states={
                'readonly': Eval('state') != 'draft',
                })

    @classmethod
    def __setup__(cls):
        super(Payroll, cls).__setup__()
        cls._error_messages.update({
            'employee_without_salary': ('The employee does not have salary!'),
            'wrong_start_end': ('The date end can not smaller than date start'),
            'sequence_missing': ('Sequence Payroll is missing!'),
            'period_closed': ('Payroll period is closed!'),
            'wrong_date_consistent': ('The date start/end is repetead \
                        or crossed with other date payroll'),
            })
        cls._transitions |= set((
                ('draft', 'cancel'),
                ('draft', 'processed'),
                ('processed', 'done'),
                ('processed', 'draft'),
                ))
        cls._buttons.update({
                'done': {
                    'invisible': Eval('state') != 'processed',
                    },
                'cancel': {
                    'invisible': Eval('state') != 'draft',
                    },
                'process': {
                    'invisible': Eval('state') != 'draft',
                    },
                })
        cls._sql_constraints += [
            ('period_employee', 'UNIQUE(period, employee)',
                'Payroll must be unique for each employee in period!'),
        ]

    @staticmethod
    def default_company():
        return Transaction().context.get('company')

    @staticmethod
    def default_kind():
        return 'normal'

    @staticmethod
    def default_journal():
        Configuration = Pool().get('staff.configuration')
        configuration = Configuration(1)
        if configuration.default_journal:
            return configuration.default_journal.id

    @classmethod
    def validate(cls, payrolls):
        super(Payroll, cls).validate(payrolls)
        for payroll in payrolls:
            payroll.check_start_end()

    @staticmethod
    def default_state():
        return 'draft'

    @staticmethod
    def _get_origin():
        'Return list of Model names for origin Reference'
        return []

    @classmethod
    @ModelView.button
    @Workflow.transition('draft')
    def draft(cls, payrolls):
        pass

    @classmethod
    @ModelView.button
    @Workflow.transition('cancel')
    def cancel(cls, payrolls):
        pass

    @classmethod
    @ModelView.button
    @Workflow.transition('processed')
    def process(cls, payrolls):
        for payroll in payrolls:
            print('here 1')
            if payroll.period.state == 'closed':
                cls.raise_user_error('period_closed',)
                return
            payroll.set_number()
            payroll.create_move()

    @classmethod
    @ModelView.button
    @Workflow.transition('done')
    def done(cls, payrolls):
        for payroll in payrolls:
            pass

    @fields.depends('start', 'end')
    def on_change_start(self):
        if not self.start:
            return {}
        res = {}
        Configuration = Pool().get('staff.configuration')
        configuration = Configuration(1)
        period_days = configuration.default_liquidation_period
        if period_days and not self.end:
            res['end'] = self.start + datetime.timedelta(period_days - 1)
        return res

    @classmethod
    def get_origin(cls):
        Model = Pool().get('ir.model')
        models = cls._get_origin()
        models = Model.search([
                ('model', 'in', models),
        ])
        return [(None, '')] + [(m.model, m.name) for m in models]

    def set_number(self):
        if self.number:
            return
        pool = Pool()
        Sequence = pool.get('ir.sequence')
        Configuration = pool.get('staff.configuration')
        configuration = Configuration(1)
        if not configuration.staff_payroll_sequence:
            self.raise_user_error('sequence_missing',)
        seq = configuration.staff_payroll_sequence.id
        self.write([self], {'number': Sequence.get_id(seq)})

    def get_worked_days(self, name):
        if self.start and self.end:
            return self.get_days(self.start, self.end)

    def get_currency(self, name):
        return self.company.currency.id

    def get_salary_full(self, wage):
        """
        Return a dict with sum of total amount of all wages defined 
            as salary on context of wage
        """
        wages_ids = [s.id for s in wage.concepts_salary]
        if wages_ids:
            salary_full = sum([line.total_amount for line in self.lines
                if line.wage_type.id in wages_ids])
        else:
            salary_full = self.employee.salary or 0
        return {'salary': salary_full}

    def create_move(self):
        pool = Pool()
        Move = pool.get('account.move')
        Period = pool.get('account.period')

        period_id = Period.find(self.company.id, date=self.date_effective)

        move_lines = self.get_moves_lines()

        move, = Move.create([{
                'journal': self.journal.id,
                'origin': str(self),
                'period': period_id,
                'date': self.date_effective,
                'state': 'draft',
                'description': self.description,
                'lines': [('create', move_lines)],
        }])
        self.write([self], {'move': move.id})

    def get_moves_lines(self):
        lines_moves = {}
        mandatory_wages = dict([(m.wage_type.id, m.party) 
                for m in self.employee.mandatory_wages
        ])

        for line in self.lines:
            if line.total_amount <= 0:
                continue
            if mandatory_wages.get(line.wage_type.id):
                party = mandatory_wages[line.wage_type.id]
            else:
                party = self.employee.party

            description = line.wage_type.name
            expense = Decimal(0)
            if line.wage_type.expense_formula:
                salary_args = self.get_salary_full(line.wage_type)
                expense = line.wage_type.compute_expense(salary_args)

            if line.wage_type.definition == 'payment':
                amount_debit = line.total_amount + expense
            else:
                amount_debit = expense

            amount_credit = line.total_amount + expense

            debit_acc = line.wage_type.debit_account
            if debit_acc and amount_debit > _ZERO:
                if debit_acc.id not in lines_moves.keys():
                    lines_moves[debit_acc.id] = self._prepare_line(
                        description, debit_acc, self.employee.party,
                        )
                lines_moves[debit_acc.id]['debit'] += amount_debit

            credit_acc = line.wage_type.credit_account
            if credit_acc and amount_credit > _ZERO:
                if credit_acc.id not in lines_moves.keys():
                    lines_moves[credit_acc.id] = self._prepare_line(
                        description, credit_acc, party)

                if line.wage_type.definition != 'payment':
                    deduction_acc = line.wage_type.deduction_account
                    if deduction_acc:
                        if deduction_acc.id not in lines_moves.keys():
                            lines_moves[deduction_acc.id] = self._prepare_line(
                                description, deduction_acc, party
                                )
                        lines_moves[deduction_acc.id]['credit'] -= line.total_amount
                lines_moves[credit_acc.id]['credit'] += amount_credit
        return lines_moves.values()

    def _prepare_line(self, description, account, party):
        res = {
            'description': description,
            'debit': _ZERO,
            'credit': _ZERO,
            'account': account.id,
            'party': party.id,
        }
        return res

    def _create_payroll_lines(self, wages, extras):
        PayrollLine = Pool().get('staff.payroll.line')
        values = []
        salary_args = {}
        for wage in wages:
            if self.kind == 'special' and not wage.contract_finish:
                continue
            if wage.salary_constitute:
                salary_args['salary'] = self.employee.salary
            else:
                salary_args = self.get_salary_full(wage)
            unit_value = wage.compute_unit_price(salary_args)
            qty = self.get_line_quantity(wage, self.start, self.end, 
                extras)
            values.append(self._prepare_lines(wage, qty, unit_value))
        PayrollLine.create(values)

    def set_preliquidation(self, extras):
        wage_salary = []
        wage_no_salary = []

        for concept in self.employee.mandatory_wages:
            if concept.wage_type.salary_constitute:
                wage_salary.append(concept.wage_type)
            else:
                wage_no_salary.append(concept.wage_type)

        self._create_payroll_lines(wage_salary, extras)
        self._create_payroll_lines(wage_no_salary, extras)

    def update_preliquidation(self, extras):
        for line in self.lines:
            if not line.wage_type.salary_constitute:
                salary_args = self.get_salary_full(line.wage_type)
                unit_value = line.wage_type.compute_unit_price(salary_args)
                line.write([line], {
                    'unit_value': unit_value,
                })

    def _prepare_lines(self, wage, qty, unit_value):
        res = {
                'sequence': wage.sequence,
                'payroll': self.id,
                'wage_type': wage.id,
                'description': wage.name,
                'quantity': qty,
                'unit_value': unit_value,
                'uom': wage.uom,
                'receipt': wage.receipt,
                }
        return res

    def get_line_quantity(self, wage, start=None, end=None, extras=None):
        Configuration = Pool().get('staff.configuration')
        configuration = Configuration(1)
        default_hour_workday = configuration.default_hour_workday or 8
        quantity = wage.default_quantity or 0

        quantity_days = self.get_days(start, end)

        if wage.uom.id == Id('product', 'uom_day').pyson():
            quantity = quantity_days
        elif wage.uom.id == Id('product', 'uom_hour').pyson():
            if wage.type_concept != 'extras':
                quantity = quantity_days * default_hour_workday
            elif extras.get(wage.name.lower()):
                quantity = extras[wage.name.lower()]
        return quantity

    def get_gross_payments(self, name):
        res = {}
        res.setdefault(self.id, dec(0))
        for line in self.lines:
            if line.wage_type.definition == 'payment' and \
                line.wage_type.receipt:
                res[self.id] += line.total_amount
        gross_payment = sum((self.currency.round(res[key]) for key in res),
            Decimal(0))
        return gross_payment

    def get_total_deductions(self, name):
        res = {}
        res.setdefault(self.id, dec(0))
        for line in self.lines:
            if line.wage_type.definition != 'payment' and \
                line.wage_type.receipt:
                res[self.id] += line.total_amount
        total_deduction = sum((self.currency.round(res[key]) for key in res),
            Decimal(0))
        return total_deduction

    def get_total_cost(self, name):
        res = sum([line.total_amount for line in self.lines
                if line.wage_type.definition == 'payment'])
        return res

    def get_net_payment(self, name):
        return self.currency.round(self.gross_payments - self.total_deductions)

    def check_start_end(self):
        if self.start <= self.end:
            if self.kind != 'normal':
                return
            if self.start >= self.period.start and \
                    self.end <= self.period.end:
                return
        self.raise_user_error('wrong_start_end',)

    @fields.depends('period', 'start', 'end')
    def on_change_period(self):
        res = {}
        if self.period:
            res = {
                'start': self.period.start,
                'end': self.period.end,
                }
        return res

    def get_days(self, start, end):
        adjust = 1
        if end.day == 31 or (end.month == 2 and \
            (end.day == 28 or end.day == 29)):
            adjust = 31 - end.day
        quantity_days = (end - start).days + adjust
        if quantity_days < 0:
            quantity_days = 0
        return quantity_days


class PayrollLine(Workflow, ModelSQL, ModelView):
    "Payroll Line"
    __name__ = "staff.payroll.line"
    sequence = fields.Integer('Sequence')
    payroll = fields.Many2One('staff.payroll', 'Payroll', 
            ondelete='CASCADE', select=True, required=True)
    description = fields.Char('Description', required=True)
    wage_type = fields.Many2One('staff.wage_type', 'Wage Type', 
            required=True, domain=[('active', '=', True)],
            depends=['payroll'])
    uom = fields.Many2One('product.uom', 'Unit', depends=['wage_type'],
            states={'readonly': Bool(Eval('wage_type'))})
    quantity = fields.Numeric('Quantity', digits=(16, 2))
    unit_value = fields.Numeric('Unit Value', digits=(16, 2))
    total_amount = fields.Function(fields.Numeric('Total Amount',
            digits=(16, 2), readonly=True), 'get_total_amount')
    receipt = fields.Boolean('Print Receipt')

    @classmethod
    def __setup__(cls):
        super(PayrollLine, cls).__setup__()
        cls._order.insert(0, ('sequence', 'ASC'))

    @staticmethod
    def default_quantity():
        return dec(1)

    @fields.depends('wage_type', 'uom', 'quantity', 'description'
            'unit_value', 'payroll', 'receipt', 'sequence', 
            '_parent_payroll.employee')
    def on_change_wage_type(self):
        res = {}
        if not self.wage_type:
            return res
        res['uom'] = self.wage_type.uom.id
        res['description'] = self.wage_type.name
        res['quantity'] = self.wage_type.default_quantity
        res['receipt'] = self.wage_type.receipt
        res['sequence'] = self.wage_type.sequence
        if self.wage_type.unit_price_formula:
            salary_args = self.payroll.get_salary_full(self.wage_type)
            res['unit_value'] = self.wage_type.compute_unit_price(
                    salary_args,
            )
        return res

    def get_total_amount(self, name):
        quantity = 0
        unit_value = 0
        if self.quantity:
            quantity = float(self.quantity)
        if self.unit_value:
            unit_value = float(self.unit_value)
        return Decimal(str(round((quantity * unit_value), 2)))


class PayrollReport(CompanyReport):
    __name__ = 'staff.payroll'


class PayrollGroupStart(ModelView):
    'Payroll Group Start'
    __name__ = 'staff.payroll_group.start'
    period = fields.Many2One('staff.payroll.period', 'Period',
        required=True, domain=[('state', '=', 'open')])
    description = fields.Char('Description', required=True)
    company = fields.Many2One('company.company', 'Company',
        required=True)

    @staticmethod
    def default_company():
        return Transaction().context.get('company')


class PayrollGroup(Wizard):
    'Payroll Group'
    __name__ = 'staff.payroll_group'
    start = StateView('staff.payroll_group.start',
        'staff_payroll.payroll_group_start_view_form', [
            Button('Cancel', 'end', 'tryton-cancel'),
            Button('Accept', 'open_', 'tryton-ok', default=True),
            ])
    open_ = StateTransition()

    def transition_open_(self):
        pool = Pool()
        Employee = pool.get('company.employee')
        Payroll = pool.get('staff.payroll')
        Date = Pool().get('ir.date')

        dom_employees = ['OR', [
                    ('end_contract', '>=', self.start.period.start),
                    ('end_contract', '=', None),
                    ('active', '=', True),
                ], [
                    ('start_contract', '<=', self.start.period.end),
                    ('active', '=', True),
                ],]

        payroll_to_create = []
        for employee in Employee.search(dom_employees):
            start = self.start.period.start
            end = self.start.period.end
            if employee.start_contract and \
                employee.start_contract >= start and \
                employee.start_contract <= start:
                start = employee.start_contract
            if employee.end_contract and \
                employee.end_contract >= start and \
                employee.end_contract <= end:
                end = employee.end_contract
            values = {
                'employee': employee.id,
                'period': self.start.period.id,
                'start': start,
                'end': end,
                'description': self.start.description,
                'date_effective': Date.today(),
            }
            payroll_to_create.append(values)
        payrolls = Payroll.create(payroll_to_create)
        for payroll in payrolls:
            payroll.set_preliquidation(extras={})
        return 'end'


class PayrollPreliquidation(Wizard):
    'Payroll Preliquidation'
    __name__ = 'staff.payroll.preliquidation'
    start_state = 'create_preliquidation'
    create_preliquidation = StateTransition()

    def transition_create_preliquidation(self):
        Payroll = Pool().get('staff.payroll')
        ids = Transaction().context['active_ids']
        for payroll in Payroll.browse(ids):
            if payroll.state != 'draft':
                return
            if not payroll.lines:
                payroll.set_preliquidation({})
            else:
                payroll.update_preliquidation({})
        return 'end'


class Move:
    __name__ = 'account.move'

    @classmethod
    def _get_origin(cls):
        return super(Move, cls)._get_origin() + ['staff.payroll']
