#This file is part of Tryton.  The COPYRIGHT file at the top level of
#this repository contains the full copyright notices and license terms.
from decimal import Decimal
from trytond.model import ModelView, ModelSQL, fields
from trytond.pyson import Not, Bool, Eval

__all__ = ['WageType', 'WageTypeSalary']

STATES = {'readonly': Not(Bool(Eval('active'))),}

class WageType(ModelSQL, ModelView):
    "Wage Type"
    __name__ = "staff.wage_type"
    _rec_name = 'name'
    sequence = fields.Integer('Sequence', select=True)
    code = fields.Char('Code', select=True)
    active = fields.Boolean('Active', select=True)
    name = fields.Char('Name', select=True, required=True)
    definition = fields.Selection([
            ('payment', 'Payment'),
            ('discount', 'Discount'),
            ('deduction', 'Deduction'),
            ], 'Definition', required=True)
    unit_price_formula = fields.Char('Unit Price Formula',
            help='Python expression for eval(expr)')
    expense_formula = fields.Char('Expense Formula',
            help='Python expression for eval(expr)')
    uom = fields.Many2One('product.uom', 'UOM', required=True)
    default_quantity = fields.Numeric('Default Quantity', digits=(16, 2))
    preliquidation = fields.Boolean('Preliquidation', select=True)
    type_concept = fields.Selection([
            ('salary', 'Salary'),
            ('extras', 'Extras'),
            ('other', 'Other'),
            ('', ''),
            ], 'Type Concept')
    salary_constitute = fields.Boolean('Salary Constitute', 
            select=True)
    debit_account = fields.Property(fields.Many2One('account.account',
            'Debit Account', domain=[
                ('kind', '!=', 'view'),
                ('company', '=', Eval('context', {}).get('company', 0))],
            states={
                'invisible': ~Eval('context', {}).get('company'),
                }))
    credit_account = fields.Property(fields.Many2One('account.account',
            'Credit Account', domain=[
                ('kind', '!=', 'view'),
                ('company', '=', Eval('context', {}).get('company', 0))],
            states={
                'invisible': ~Eval('context', {}).get('company'),
                }))
    deduction_account = fields.Property(fields.Many2One('account.account',
            'Deduction Account', domain=[
                ('kind', '!=', 'view'),
                ('company', '=', Eval('context', {}).get('company', 0))],
            states={
                'invisible': Eval('definition') != 'deduction',
                'required': Eval('definition') == 'deduction',
            }))
    receipt = fields.Boolean('Receipt', select=True)
    party_payment = fields.Many2One('party.party', 'Party Payment',
            select=True)
    parent = fields.Many2One('staff.wage_type', 'Parent', select=True)
    concepts_salary = fields.Many2Many('staff.wage_type-staff.wage_type',
            'parent',  'child', 'Concepts Salary')
    contract_finish = fields.Boolean('Contract Finish', select=True)

    @classmethod
    def __setup__(cls):
        super(WageType, cls).__setup__()
        cls._error_messages.update({
            'invalid_formula': 'Invalid formula [ %s ] for unit price!',
        })
        cls._order[0] = ('sequence', 'ASC')

    @classmethod
    def validate(cls, wage_types):
        super(WageType, cls).validate(wage_types)
        for wage in wage_types:
            test_salary = {'salary': 1000}
            if wage.unit_price_formula:
                wage.compute_unit_price(test_salary)
            if wage.expense_formula:
                wage.compute_expense(test_salary)

    @staticmethod
    def default_unit_price_formula():
        return 'salary'

    @staticmethod
    def default_active():
        return True

    def get_rec_name(self, name):
        return self.name

    def compute_unit_price(self, args):
        return self.compute_formula('unit_price_formula', args)

    def compute_expense(self, args):
        return self.compute_formula('expense_formula', args)

    def compute_formula(self, formula, args=None):
        '''
        Compute a formula field with a salary
        :return: A decimal
        '''
        formula = getattr(self, formula)
        if not formula:
            return Decimal('0')
        if args.get('salary') != None:
            salary = float(args['salary'])
        try:
            value =  Decimal(str(round(eval(formula), 2)))
            return value
        except Exception:
            self.raise_user_error('invalid_formula', formula)


class WageTypeSalary(ModelSQL):
    "Wage Type Salary"
    __name__ = "staff.wage_type-staff.wage_type"
    _table = 'staff_wage_type_salary_rel'
    parent = fields.Many2One('staff.wage_type', 'Parent', ondelete='CASCADE',
            select=True, required=True)
    child = fields.Many2One('staff.wage_type', 'Child', ondelete='RESTRICT',
            select=True, required=True)
