#This file is part of Tryton.  The COPYRIGHT file at the top level of
#this repository contains the full copyright notices and license terms.
from trytond.model import fields, ModelSQL, ModelView
from trytond.pool import PoolMeta, Pool
from trytond.transaction import Transaction
from trytond.wizard import Wizard, StateTransition

__all__ = ['Employee', 'MandatoryWage', 'CreateMandatoryWages']
__metaclass__ = PoolMeta


class Employee:
    __name__ = 'company.employee'
    category = fields.Many2One('staff.employee_category', 'Category')
    mandatory_wages = fields.One2Many('staff.payroll.mandatory_wage',
            'employee', 'Mandatory Wage')

    @classmethod
    def __setup__(cls):
        super(Employee, cls).__setup__()
        cls._error_messages.update({
            'no_bank_accounts': ('The employee does not have banks!'),
        })

class MandatoryWage(ModelSQL, ModelView):
    "Mandatory Wage"
    __name__ = 'staff.payroll.mandatory_wage'
    employee = fields.Many2One('company.employee', 'Employee')
    wage_type = fields.Many2One('staff.wage_type', 'Wage Type', 
            required=True, domain=[('active', '=', True)])
    party = fields.Many2One('party.party', 'Party', select=True)

    @classmethod
    def __setup__(cls):
        super(MandatoryWage, cls).__setup__()
        cls._order[0] = ('id', 'ASC')


class CreateMandatoryWages(Wizard):
    'Create Mandatory Wages'
    __name__ = 'staff.payroll.create_mandatory_wages'
    start_state = 'create_wages'
    create_wages = StateTransition()

    def transition_create_wages(self):
        pool = Pool()
        Employee = pool.get('company.employee')
        MandatoryWage = pool.get('staff.payroll.mandatory_wage')

        employee_id = Transaction().context['active_id']
        employee = Employee(employee_id)
        values = []
        if employee.category and employee.category.wages_default:
            for wage in employee.category.wages_default:
                val = {
                    'employee': employee_id,
                    'wage_type': wage.id,
                    }
                if wage.party_payment:
                    val.update({'party': wage.party_payment.id})
                values.append(val)
        MandatoryWage.create(values)
        return 'end'
