#This file is part of Tryton.  The COPYRIGHT file at the top level of
#this repository contains the full copyright notices and license terms.
from trytond.model import ModelView, ModelSQL, fields
from trytond.pool import Pool, PoolMeta

__all__ = ['Position', 'WorkdayDefinition']
__metaclass__ = PoolMeta


class Position:
    __name__ = 'staff.position'
    workday_definition = fields.One2Many('staff.workday_definition',
                'position', 'Workday Definition')

    @classmethod
    def __setup__(cls):
        super(Position, cls).__setup__()
        cls._error_messages.update({
                'missing_config_default': ('Missing default values for '
                    'workday or restday on configuration!')})

    @classmethod
    def trigger_create(cls, records):
        for rec in records:
            rec._default_workdays()

    def _default_workdays(self):
        pool = Pool()
        Workday = pool.get('staff.workday_definition')
        Config = pool.get('staff.configuration')
        config = Config(1)
        if not config.default_hour_workday or not config.default_hour_workday:
            self.raise_user_error('missing_config_default')
        for day in Workday.weekday.selection:
            values = {
                'position': self.id, 'weekday': day[0],
                'workday': config.default_hour_workday,
                'restday': config.default_restday
            }
            Workday.create([values])


class WorkdayDefinition(ModelSQL, ModelView):
    'Workday Definition'
    __name__ = 'staff.workday_definition'
    weekday = fields.Selection([
            ('monday', 'Monday'),
            ('tuesday', 'Tuesday'),
            ('wednesday', 'Wednesday'),
            ('thursday', 'Thursday'),
            ('friday', 'Friday'),
            ('saturday', 'Saturday'),
            ('sunday', 'Sunday'),
            ], 'Weekday', required=True, readonly=True)
    workday = fields.Numeric('Workday',  digits=(2, 2), required=True)
    restday = fields.Numeric('Restday',  digits=(2, 2), required=True)
    note = fields.Text('Note')
    position = fields.Many2One('staff.position', 'Position', 
            required=True, ondelete='CASCADE')

    @classmethod
    def __setup__(cls):
        super(WorkdayDefinition, cls).__setup__()
