#This file is part of Tryton.  The COPYRIGHT file at the top level of
#this repository contains the full copyright notices and license terms.
from trytond.model import ModelView, Workflow, ModelSQL, fields
from trytond.pyson import Eval

__all__ = ['Period']

STATES = {'readonly': Eval('state') != 'draft'}


class Period(Workflow, ModelSQL, ModelView):
    "Period"
    __name__ = 'staff.payroll.period'
    _rec_name = 'name'
    name = fields.Char('Name', select=True, states=STATES,
            required=True)
    sequence = fields.Integer('Sequence', select=True, states=STATES,
            required=True)
    start = fields.Date('Start', required=True, states=STATES)
    end = fields.Date('End', required=True, states=STATES)
    state = fields.Selection([
            ('draft', 'Draft'),
            ('open', 'Open'),
            ('closed', 'Closed'),
            ], 'State', readonly=True)

    @classmethod
    def __setup__(cls):
        super(Period, cls).__setup__()
        cls._order[0] = ('sequence', 'ASC')
        cls._error_messages.update({
            'wrong_start_end': ('The date end can not smaller than date start'),
            'wrong_period_overlap': ('The period overlap another period!')
                        })
        cls._transitions |= set((
                ('draft', 'open'),
                ('open', 'closed'),
                ))
        cls._buttons.update({
                'open': {
                    'invisible': Eval('state') != 'draft',
                    },
                'close': {
                    'invisible': Eval('state') != 'open',
                    },
                })

    @staticmethod
    def default_state():
        return 'draft'

    @classmethod
    @ModelView.button
    @Workflow.transition('draft')
    def draft(self, payrolls):
        pass

    @classmethod
    @ModelView.button
    @Workflow.transition('open')
    def open(self, periods):
        pass

    @classmethod
    @ModelView.button
    @Workflow.transition('closed')
    def close(self, periods):
        pass

    @classmethod
    def validate(cls, periods):
        super(Period, cls).validate(periods)
        for period in periods:
            period.check_start_end()
            period.check_date_consistent()

    def check_start_end(self):
        if self.start > self.end:
            self.raise_user_error('wrong_start_end',)

    def check_date_consistent(self):
        periods = self.search([
                ('id', '!=', self.id), ['OR',
                [
                    ('start', '>=', self.start),
                    ('start', '<=', self.end),
                ], [
                    ('end', '>=', self.start),
                    ('end', '<=', self.end),
                ], [
                    ('start', '<=', self.start),
                    ('end', '>=', self.end),
                ]
                ]])
        if periods:
            self.raise_user_error('wrong_period_overlap',)
