 #This file is part of Tryton.  The COPYRIGHT file at the top level of
#this repository contains the full copyright notices and license terms.
from decimal import Decimal
from trytond.model import ModelView, fields
from trytond.pool import Pool, PoolMeta
from trytond.report import Report
from trytond.wizard import Wizard, StateView, Button, StateAction
from trytond.transaction import Transaction


__all__ = ['Payroll', 'PayrollGlobalStart', 'PayrollGlobal',
        'PayrollGlobalReport', 'PayrollPaymentReport', 'PayrollPayment',
        'PayrollPaymentStart']
__metaclass__ = PoolMeta


class Payroll:
    __name__ = "staff.payroll"

    @staticmethod
    def default_date_effective():
        Date = Pool().get('ir.date')
        return Date.today()

    @fields.depends('period', 'start', 'end', 'description')
    def on_change_period(self):
        res = {}
        if self.period:
            res = {
                'start': self.period.start,
                'end': self.period.end,
                }
            if not self.description and self.period:
                res['description'] = self.period.name
        return res

    def get_salary_full(self, wage):
        res = super(Payroll, self).get_salary_full(wage)
        if self.employee.base_salary:
            res.update({'base_salary': self.employee.base_salary})
        return res


class PayrollGlobalStart(ModelView):
    'Payroll Global Start'
    __name__ = 'staff.payroll_global.start'
    period = fields.Many2One('staff.payroll.period', 'Period')
    company = fields.Many2One('company.company', 'Company', required=True)

    @staticmethod
    def default_company():
        return Transaction().context.get('company')


class PayrollGlobal(Wizard):
    'Payroll Global'
    __name__ = 'staff.payroll.global'
    start = StateView('staff.payroll_global.start',
        'staff_payroll_co.payroll_global_start_view_form', [
            Button('Cancel', 'end', 'tryton-cancel'),
            Button('Print', 'print_', 'tryton-ok', default=True),
            ])
    print_ = StateAction('staff_payroll_co.report_payroll_global')

    def do_print_(self, action):
        period = None
        if self.start.period:
            period = self.start.period.id
        data = {
            'company': self.start.company.id,
            'period': period,
            }
        return action, data

    def transition_print_(self):
        return 'end'


class PayrollGlobalReport(Report):
    __name__ = 'staff.payroll.global_report'

    @classmethod
    def parse(cls, report, objects, data, localcontext):
        user = Pool().get('res.user')(Transaction().user)
        Payroll = Pool().get('staff.payroll')
        Period = Pool().get('staff.payroll.period')
        clause = []
        start_date = None
        end_date = None
        period = None
        period_name = None
        if data['period']:
            period = Period(data['period'])
            start_date = period.start
            end_date = period.end
            clause = [('period', '=', period)]
            period_name = period.name
        payrolls = Payroll.search(clause)
        new_objects = []
        default_vals = cls.default_values()
        sum_gross_payments = []
        sum_total_deductions = []
        sum_net_payment = []
        for payroll in payrolls:
            values = default_vals.copy()
            for line in payroll.lines:
                if line.wage_type.type_concept in values.keys():
                    values[line.wage_type.type_concept] += line.total_amount

            values['employee_code'] = payroll.employee.code
            values['employee'] = payroll.employee.party.name
            values['employee_vat_number'] = payroll.employee.party.vat_number
            values['party_health'] = payroll.employee.party_health
            values['party_retirement'] = payroll.employee.party_retirement
            values['employee_salary'] = payroll.employee.salary
            values['worked_days'] = payroll.worked_days
            values['gross_payments'] = payroll.gross_payments
            values['total_deductions'] = payroll.total_deductions
            values['net_payment'] = payroll.net_payment
            sum_gross_payments.append(payroll.gross_payments)
            sum_total_deductions.append(payroll.total_deductions)
            sum_net_payment.append(payroll.net_payment)
            new_objects.append(values)
        localcontext['start_date'] = start_date
        localcontext['end_date'] = end_date
        localcontext['period'] = period_name
        localcontext['company'] = user.company
        localcontext['user'] = user
        localcontext['sum_gross_payments'] = sum(sum_gross_payments)
        localcontext['sum_net_payment'] = sum(sum_net_payment)
        localcontext['sum_total_deductions'] = sum(sum_total_deductions)
        return super(PayrollGlobalReport, cls).parse(report,
                new_objects, data, localcontext)

    @classmethod
    def default_values(cls):
        WageType = Pool().get('staff.wage_type')
        fields_ = ['employee_code', 'employee', 'employee_vat_number',
            'employee_salary', 'gross_payments', 'total_deductions',
            'net_payment', 'days', 'health_company', 'retirement_company',
        ]

        lines_fields = dict(WageType.type_concept.selection).keys()
        if '' in lines_fields:
            lines_fields.remove('')
        default_values = {}
        for field in fields_:
            default_values.setdefault(field, None)
        for field in lines_fields:
            default_values.setdefault(field, Decimal(0))
        return default_values


class PayrollPaymentStart(ModelView):
    'Payroll Payment Start'
    __name__ = 'staff.payroll_payment.start'
    period = fields.Many2One('staff.payroll.period', 'Period')
    company = fields.Many2One('company.company', 'Company', required=True)

    @staticmethod
    def default_company():
        return Transaction().context.get('company')



class PayrollPayment(Wizard):
    'Payroll Payment'
    __name__ = 'staff.payroll.payment'
    start = StateView('staff.payroll_payment.start',
        'staff_payroll_co.payroll_payment_start_view_form', [
            Button('Cancel', 'end', 'tryton-cancel'),
            Button('Print', 'print_', 'tryton-ok', default=True),
            ])
    print_ = StateAction('staff_payroll_co.report_payroll_payment')

    def do_print_(self, action):
        period = None
        if self.start.period:
            period = self.start.period.id
        data = {
            'company': self.start.company.id,
            'period': period,
            }
        return action, data

    def transition_print_(self):
        return 'end'


class PayrollPaymentReport(Report):
    __name__ = 'staff.payroll.payment_report'

    @classmethod
    def parse(cls, report, objects, data, localcontext):
        user = Pool().get('res.user')(Transaction().user)
        Payroll = Pool().get('staff.payroll')
        Period = Pool().get('staff.payroll.period')
        clause = []
        start_date = None
        end_date = None
        period = None
        period_name = None
        if data['period']:
            period = Period(data['period'])
            start_date = period.start
            end_date = period.end
            clause = [('period', '=', period)]
            period_name = period.name
        payrolls = Payroll.search(clause)
        new_objects = []
        sum_net_payment = []
        values = {}
        for payroll in payrolls:
            values = values.copy()
            values['employee'] = payroll.employee.party.name
            values['employee_vat_number'] = payroll.employee.party.vat_number
            type_account = bank_account = bank = '---'
            if payroll.employee.party.bank_accounts:
                bank = payroll.employee.party.bank_accounts[0].bank.party.name
                if payroll.employee.party.bank_accounts[0].numbers:
                    bank_account = payroll.employee.party.bank_accounts[0].numbers[0].number
                    type_account = payroll.employee.party.bank_accounts[0].numbers[0].getTranslated('type')
            values['bank'] = bank
            values['bank_account'] = bank_account
            values['type_account'] = type_account
            values['net_payment'] = payroll.net_payment
            sum_net_payment.append(payroll.net_payment)
            new_objects.append(values)
        localcontext['start_date'] = start_date
        localcontext['end_date'] = end_date
        localcontext['period'] = period_name
        localcontext['company'] = user.company
        localcontext['user'] = user
        localcontext['sum_net_payment'] = sum(sum_net_payment)
        return super(PayrollPaymentReport, cls).parse(report,
                new_objects, data, localcontext)
