#-*- coding: utf8 -*-
'''
Question objects
================

A question object contains data pertinent to render a given question in the 
tutor system. In fact, question objects can be very complex in order to support
various features. This section describes the design decisions and features
of these objects.

Motivation and use cases
------------------------

A typical workflow is illustrated bellow:

1. A template is created for each question. This template should be typeset in 
LaTeX (or some other human friendly format) and the logic to create randomized 
values is written in a programming language such as Python. 
 
2. The tutor creates exams for each student using the questions generated by 
the above template. 

3. After receiving feedback from students, the tutor (or the question creator) 
may correct small (or big) defects and may add additional features to the 
question (such as a more complete student feedback, better graphics, etc). 

4. The tutor may tag a question, or a particular version of a question, as 
invalid or as requiring further work. The whole history of the question should 
be available (the teacher, for instance, may tag an specific version of the 
question to be faulty, but may want to use a correct and updated version of the 
same question in a later exam).

5. The tutor may also collect statistics from the student's responses in order 
to assess the quality of the question. 

The *Question* object must store data pertinent to all of these steps. The data
can be stored in many different ways in the hard drive, but it must be 
presented, loaded and exchanged between all subsystems uniformly as a JSON-like
structure.
  
Desired features
----------------  
  
The root JSON element has 


'''

from pyson.schemas import Schema, Str, Int, Bool, Ref, Number, Array
#from pyson.schemas.ext.datetime import JSONDateTime, JSONDate

__all__ = 'MultipleChoiceItem MultipleChoice TrueFalseItem TrueFalse '\
          'Association AssociationItem Text Node Citation LearningObj'.split()

#===============================================================================
#                 Validators for atomic question types: 
#                e.g., multiple choice, true/false, association, etc
#===============================================================================

# Multiple Choice --------------------------------------------------------------
schema = {
    u'grade': Number(default=0),
    u'text': Str(),
    u'id': Int(),
    u'feedback?': Str(),
    u'comment?': Str(),
}
MultipleChoiceItem = Schema(schema, name='MultipleChoiceItem')

MultipleChoice = Schema({ u'type': 'multiple-choice',
                          u'stem': Str(),
                          u'grade': Number(1),
                          u'shuffle?': Bool(True),
                          u'multiple_answers?': Bool(False),
                          u'columns?': Int(),
                          u'items': Array(MultipleChoiceItem, name='Items'),
                          u'comment?': Str() },
                        name='MultipleChoice')

# True/False -------------------------------------------------------------------
schema = {
    u'answer': Bool(),
    u'text': Str(),
    u'text_other?': Str(),
    u'feedback?': Str(),
    u'comment?': Str(),
}
TrueFalseItem = Schema(schema, name='TrueFalseItem')

schema = {
    u'type': 'true-false',
    u'stem': Str(),
    u'shuffle?': Bool(True),
    u'items': Array(TrueFalseItem, name='Items'),
    u'comment?': Str(),
}
TrueFalse = Schema(schema, name='TrueFalse')

# Association ------------------------------------------------------------------
schema = {
    u'answer': Str(),
    u'text': Str(),
    u'feedback?': Str(),
    u'comment?': Str(),
}
AssociationItem = Schema(schema, name='AssociationItem')

schema = {
    u'type': 'association',
    u'stem': Str(),
    u'shuffle?': Bool(True),
    u'items?': Array(AssociationItem, name='Items'),
    u'answers_ordering?': Array(Int),
    u'comment?': Str(),
}
Association = Schema(schema, name='Association')


# Text Nodes -------------------------------------------------------------------
schema = {
    u'type': 'text',
    u'text': Str(),
}
Text = Schema(schema, name='Text')

# Nodes ------------------------------------------------------------------------
Node = Text | TrueFalse | MultipleChoice | Association

#===============================================================================
#                                    LearningObj
#===============================================================================
Citation = Schema({ u'type': Str(),
                    u'full_ref': Str(),
                    u'cite': Str() },
                  name='Citation')

LearningObj = Schema({
    # Database info and behavior
    u'name': Str(is_ref=True),
    u'is_template': Bool(False),
    u'template_name?': Str(),
    u'parent': Ref('self'),
    u'type?': Str(),
    u'lang?': Str(),
#    u'creation?': JSONDateTime(),
    u'version?': Str(),
    u'hash?': Str(),
    u'is_active?': Bool(True),
    u'comment?': Str(),

    # Meta information
    u'title': Str(),
    u'author': Str(),
#    u'date': JSONDate(),
    u'difficulty?': Int(),

    # Content
    u'content': Array(Node()),
    u'glossary?': Array(Array(Str())),
    u'bibliography?': Array(Citation()),
    u'further_reading?': Array(Citation()),

    # Namespace
    u'namespace_pool?': Schema({ u'type': Str(),
                                 u'code': Str(),
                                 u'var_names': Array(Str())}),
    u'namespace_subs?': Array(Schema({ u'path': Str(),
                                       u'text': Str(),
                                       u'substitutions':
                                           Array(Schema({ u'var_name': Str(),
                                                          u'filters': Array(Str())}))})),

    # Feedback to students
    u'solution?': Str(),
    u'grade_feedback?': Schema({
        u'good': Str(),
        u'regular': Str(),
        u'poor': Str(),
    }),
}, name='LearningObj')
