#-*- coding: utf8 -*-
from __future__ import absolute_import
from tutor.plugins import tests
import collections
import itertools
import numpy as np
import operator
import random
import sympy as sp

# Define constants
SIMPLE_NUMBERS_9 = \
    [1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20,
     21, 22, 23, 24, 25, 26, 27, 28, 30, 32, 33, 34, 35, 36, 38, 39, 40, 42, 44,
     45, 48, 49, 50, 51, 52, 54, 55, 56, 60, 63, 64, 65, 66, 68, 70, 72, 75, 77,
     78, 80, 81, 84, 88, 90, 96, 98, 99, 100, 104, 105, 108, 110, 112, 120, 125,
     126, 128, 132, 135, 140, 144, 150, 160, 162, 168, 176, 180, 192, 200, 216,
     224, 240, 256, 288, 320, 384, 512]

SIMPLE_NUMBERS_8 = \
    [1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20,
     21, 22, 24, 25, 26, 27, 28, 30, 32, 33, 34, 35, 36, 39, 40, 42, 44, 45, 48,
     49, 50, 52, 54, 55, 56, 60, 63, 64, 66, 70, 72, 75, 80, 81, 84, 88, 90, 96,
     100, 108, 112, 120, 128, 144, 160, 192, 256]

SIMPLE_NUMBERS_7 = \
    [1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 20, 21,
     22, 24, 25, 26, 27, 28, 30, 32, 33, 35, 36, 40, 42, 44, 45, 48, 50, 54, 56,
     60, 64, 72, 80, 96, 128]

SIMPLE_NUMBERS_6 = \
    [1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 18, 20, 21, 22,
     24, 25, 27, 28, 30, 32, 36, 40, 48, 64]

SIMPLE_NUMBERS_5 = \
    [1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 14, 15, 16, 18, 20, 24, 32]

SIMPLE_NUMBERS_4 = \
    [1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 12, 16]

SIMPLE_NUMBERS_3 = \
    [1, 2, 3, 4, 5, 6, 8]

SIMPLE_NUMBERS_2 = \
    [1, 2, 3, 4]

SIMPLE_NUMBERS_1 = [1, 2]

SIMPLE_NUMBERS_0 = [1]

SIMPLE_NUMBERS = [SIMPLE_NUMBERS_0, SIMPLE_NUMBERS_1, SIMPLE_NUMBERS_2,
                  SIMPLE_NUMBERS_3, SIMPLE_NUMBERS_4, SIMPLE_NUMBERS_5,
                  SIMPLE_NUMBERS_6, SIMPLE_NUMBERS_7, SIMPLE_NUMBERS_8,
                  SIMPLE_NUMBERS_9]

One = sp.Integer(1)

def rnumber(max_compl=3, force_pos=False):
    '''
    Returns a random number of given maximum complexity 'max_compl'.
    
    Arguments
    ---------
    
    max_compl: int
        Specify the maximum complexity of the random numbers generated by this
        function.
    force_pos: bool, float
        True to force positive numbers. If 'force_pos' is a number between 0 
        and 1 it is interpreted as the probability of yielding positive values.
        The default value of False produces negative and positive numbers with
        the same probability.
        
    Examples
    --------
    
    >>> abs(rnumber(2)) in [1, 2, 3, 4]
    True
    >>> rnumber(force_pos=True) > 0 
    True
    >>> rnumber(force_pos=0.0) < 0
    True
    '''
    try:
        numbers = SIMPLE_NUMBERS[max_compl]
    except IndexError:
        raise ValueError('unsupported maximum complexity')

    number = random.choice(numbers)
    force_pos = float(0.5 if force_pos is False else force_pos)
    if force_pos >= random.random():
        return number * One
    else:
        return -number * One

def rfraction(max_compl=5, force_pos=False):
    '''
    Return a random fraction with given maximum complexity.
    
    Examples
    --------
    >>> frac = rfraction(2, force_pos=True)
    >>> float(frac) in [ 2.0, 1.0, 0.25, 0.5 ]
    True
    '''
    numer = max_compl // 2
    denom = max_compl - numer
    return rnumber(numer, force_pos) / rnumber(denom, force_pos=True)

def rfractions(max_compl=5, force_pos=False, init=None, max_size=None, unique=True):
    '''
    Iterator over a (possibly infinite) sequence of random fractions. 
    
    Examples
    --------
    
    >>> lst = list(rfractions(init=[2, One/2], max_size=4, max_compl=5))
    >>> lst[0] == 2 and lst[1] == One/2
    True
    >>> Integer(lst[3]).complexity() <= 5
    True
    '''

    init = ([] if init is None else map(sp.sympify, init))
    max_size = (2 ** 100 if max_size is None else max_size)
    values = set(init)
    values_lst = list(init) if init else [1]
    if unique and (len(values) != len(init)):
        raise ValueError('Initial elements are not unique')

    # Yield values of the initial list
    iter = 0
    for x in init:
        iter += 1
        if iter > max_size:
            return
        yield x

    while True:
        new = random.choice(values_lst)
        new *= rfraction(max_compl, force_pos)
        if new in values:
            continue
        else:
            numer, denom = new.as_numer_denom()
            compl = Integer(numer).complexity() + Integer(denom).complexity()
            if compl > max_compl:
                continue
            else:
                iter += 1
                if iter > max_size:
                    return
                values.add(new)
                values_lst.append(new)
                yield new

def rnumbers(max_compl=5, force_pos=False, init=None, max_size=None, unique=True):
    '''
    Iterator over a (possibly infinite) sequence of random integers. 
    
    Examples
    --------
    
    >>> lst = list(rnumbers(init=[1, 2], max_size=4, max_compl=5))
    >>> lst[0] == 1 and lst[1] == 2
    True
    >>> Integer(lst[3]).complexity() <= 5
    True
    '''

    init = ([] if init is None else map(sp.sympify, init))
    max_size = (2 ** 100 if max_size is None else max_size)
    values = set(init)
    values_lst = list(init) if init else [1]
    if unique and (len(values) != len(init)):
        raise ValueError('Initial elements are not unique')

    # Yield values of the initial list
    iter = 0
    for x in init:
        iter += 1
        if iter > max_size:
            return
        yield x

    # Create new numbers randomly
    while True:
        new = random.choice(values_lst)
        new *= rnumber(max_compl, force_pos)
        if new in values:
            continue
        else:
            compl = Integer(new).complexity()
            if compl > max_compl:
                continue
            else:
                iter += 1
                if iter > max_size:
                    return
                values.add(new)
                values_lst.append(new)
                yield new

def issimple(obj, max_compl=5):
    '''
    Return True if object's complexity is less then or equal to 'max_compl'
    
    Examples
    --------
    
    >>> issimple(6)
    True
    >>> issimple(6, max_compl=2)
    False
    >>> issimple(sp.Rational(2, 4), max_compl=5)
    True
    >>> issimple(sp.Rational(3, 10), max_compl=3)
    False
    '''
    if isinstance(obj, (int, sp.Integer)):
        return Integer(obj).complexity() <= max_compl
    if isinstance(obj, (sp.Rational)):
        numer, denom = obj.as_numer_denom()
        compl = Integer(numer).complexity() + Integer(denom).complexity()
        return compl <= max_compl


__all__ = [ 'Primes', 'Integer', 'SIMPLE_NUMBERS', 'rnumber', 'rfraction',
            'rnumbers', 'rfractions', 'issimple' ]

#===============================================================================
# Auxiliary class: a dispatcher of python's __magic__ methods
#===============================================================================
class Facade(object):
    '''
    Subclasses of this class offers an façade to some arbitrary Python object.
    The subclasses must be able to be initialized as SubClass(obj) and should 
    hold a copy of obj in the 'value' attribute.
    '''
    def __init__(self, value, tt=None):
        self.value = value
        self.type = tt or type(value)

    # Printing -----------------------------------------------------------------
    def __str__(self):
        return str(self.value)

    def __repr__(self):
        return repr(self.value)

    def __unicode__(self):
        return unicode(self.value)

    # Mathematical operations --------------------------------------------------
    def _get_other(self, other):
        if isinstance(other, Facade):
            return other.value
        else:
            return other

    # Unary operations
    def __neg__(self):
        return Integer(-self.value)

    # Binary operations
    def __mul__(self, other):
        other = self._get_other(other)
        return self.type(self.value * other)

    def __add__(self, other):
        other = self._get_other(other)
        return self.type(self.value + other)

    def __sub__(self, other):
        other = self._get_other(other)
        return self.type(self.value - other)

    def __div__(self, other):
        other = self._get_other(other)
        return self.type(self.value / other)

    # Inplace operations
    def __iadd__(self, other):
        value = self.value
        value += self._get_other(other)

    def __isub__(self, other):
        value = self.value
        value -= self._get_other(other)

    def __imul__(self, other):
        value = self.value
        value *= self._get_other(other)

    def __idiv__(self, other):
        value = self.value
        value /= self._get_other(other)

    # Other --------------------------------------------------------------------
    def __getattr__(self, attr):
        return getattr(self.value, attr)

#===============================================================================
# Iteration over prime factors
#===============================================================================
class Primes(object):
    '''
    Object that behaves as a list of primes. The list is computed on-the-fly as 
    higher prime numbers are requested.
    '''
    def __init__(self, head=None):
        '''
        Object that behaves as a list of primes. The list is computed on-the-fly
        as higher prime numbers are requested.
        
        The 'head' argument can be used to modify Primes() to generate 
        pseudo-primes lists. This can be used to include zero, one, minus one 
        or even non-prime numbers.
        
        Examples
        --------
        
        >>> p = Primes(); p[0:10]
        [2, 3, 5, 7, 11, 13, 17, 19, 23, 29]
        
        >>> p = Primes([-1, 0, 1, 2]); p[0:10]
        [-1, 0, 1, 2, 3, 5, 7, 11, 13, 17]
        
        >>> p = Primes([-1, 0, 1, 2, 10]); p[0:10]
        [-1, 0, 1, 2, 10, 3, 5, 7, 11, 13]
        '''
        self._primes_iter = sp.primerange(0, 2 ** 500)
        head = ([2] if head is None else head)
        self._primes = list(head)
        self._primes_set = set(head)
        self._primes_set.add(None)

        self._order = {}
        for idx, v in enumerate(self._primes):
            self._order[v] = idx

    def __iter__(self):
        '''
        Iterate over Primes sequences.
        
        >>> it = iter(Primes([1, 2, 3]))
        >>> it.next(); it.next(); it.next(); it.next();
        1
        2
        3
        5
        '''
        for i in itertools.count():
            yield self[i]

    def __getitem__(self, idx):
        '''
        x.__getitem__(i) <==> x[i]
        
        >>> Primes()[4]
        11
        >>> Primes([0, 1])[4]
        5
        >>> Primes()[-1]
        Traceback (most recent call last):
        ...
        IndexError: negative indices not supported
        '''
        if idx < 0:
            raise IndexError('negative indices not supported')

        next_prime = None
        while idx > len(self._primes) - 1:
            while next_prime in self._primes_set:
                next_prime = self._primes_iter.next()
            else:
                self._primes_set.add(next_prime)

            self._primes.append(next_prime)
            self._order[next_prime] = len(self._order)
        return self._primes[idx]

    def index(self, p):
        '''
        Return the index of the prime number in the sequence defined in Primes().
        
        Examples
        --------
        
        >>> p = Primes(); p.index(7)
        3
        >>> p = Primes([1, 2]); p.index(5)
        3
        >>> p = Primes(); p.index(6)
        Traceback (most recent call last):
        ...
        ValueError: 6 is not a prime number
        '''
        idx = len(self._primes)
        while p > self._primes[-1]:
            idx += 1
            self[idx]
        try:
            return self._order[p]
        except KeyError:
            raise ValueError('%s is not a prime number' % p)

    def __getslice__(self, a, b):
        '''
        x.__getslice__(a, b) <==> x[a:b]
        
        Return a list of primes with indexes between a and b. Negative indexes
        are *not* supported.
        
        Examples
        --------
        
        >>> Primes()[2:10]
        [5, 7, 11, 13, 17, 19, 23, 29]
        '''
        if a > b:
            raise IndexError('invalid index: %s > %s' % (a, b))
        elif (b < 0) or (a < 0):
            raise IndexError('negative indexes are not supported')
        else:
            if b > len(self._primes):
                _aux_prime = self[b]
            return self._primes[a:b]

    # Tests --------------------------------------------------------------------
    if tests.define_tests():
        def test_pass(self):
            pass

#===============================================================================
# Integers represented internally by a dictionary of its prime factors.
# This representation is useful to compute the complexity of an integer number  
#===============================================================================
class Integer(Facade):
    PRIMES = Primes([-1, 2])

    def __init__(self, number):
        '''
        Integer number represented internally from its prime factors. This
        representation is useful to compute the complexity of a integer, but 
        most operations on regular integers are not supported.
        
        Attributes
        ----------
        
        factors: a dictionary holding all the prime factors and the respective 
            exponents of the given integer
        
        value: the integer value of Integer(). The same as int(obj) 
        
        Initialization
        --------------
        
        Can be initialized in three different ways:
        
        >>> Integer(10)
        Integer(10)
        >>> Integer({2: 1, 5: 1})
        Integer(10)
        >>> Integer([0, 1, 0, 1])
        Integer(10)
        
        
        Examples
        --------
        
        >>> x = Integer(20)
        >>> x.value
        20
        >>> x.full_factors()
        array([0, 2, 0, 1])
        '''
        if isinstance(number, dict):
            self._factors = number
        elif isinstance(number, collections.Sequence):
            factors = {}
            for k, p in zip(number, iter(self.PRIMES)):
                if k:
                    factors[p] = k
            self._factors = factors
        else:
            self._factors = sp.factorint(number)

    # Properties ---------------------------------------------------------------
    @property
    def factors(self):
        return self._factors

    @property
    def value(self):
        if self._factors:
            return reduce(operator.mul, (int(k ** v) for k, v in self._factors.items()))
        else:
            return 1

    # API functions ------------------------------------------------------------
    def full_factors(self, min_length=None):
        '''
        Return an array with the exponents all prime factors up to the exponent
        of the largest prime factor. The first value is equal to zero or one 
        and tell if the number is positive or negative.
        
        Arguments
        ---------
        
        min_length: minimum size of the returning list. If the order of largest
            prime factor is less than 'min_length', the returning list is 
            filled with trailing zeros.
            
            
        Examples
        --------
        
        >>> Integer(10).full_factors()
        array([0, 1, 0, 1])
        
        >>> Integer(-10).full_factors(min_length=5)
        array([1, 1, 0, 1, 0])
        '''
        prime_order = self.PRIMES.index
        factors = self._factors
        if factors:
            largest = max(factors)
            size = max(min_length, prime_order(largest) + 1)
        else:
            return np.zeros(0, dtype=int)

        full_factors = np.zeros(size, dtype=int)
        for k, v in factors.items():
            idx = prime_order(k)
            full_factors[idx] = v

        return full_factors

    def complexity(self):
        '''
        Compute the complexity of the number.
        
        Example
        -------
        
        >>> Integer(10).complexity()
        4.0
        '''

        # Special case zero and one
        if not self._factors:
            return 0.0
        elif 0 in self._factors:
            return 0.0

        if self._factors.get(-1, 0):
            res = 0.5
        else:
            res = 0.0
        res += sum((idx) * k for idx, k in enumerate(self.full_factors()))
        return res

    # Python protocol ----------------------------------------------------------
    def __str__(self):
        return 'Integer(%s)' % self.value

    def __repr__(self):
        return self.__str__()

    def __int__(self):
        return self.value

    # Mathematical operations --------------------------------------------------
    def __neg__(self):
        return Integer(-self.value)

    def __mul__(self, other):
        other = getattr(other, 'value', other)
        return Integer(self.value * other)

    def __add__(self, other):
        other = getattr(other, 'value', other)
        return Integer(self.value + other)

    def __sub__(self, other):
        other = getattr(other, 'value', other)
        return Integer(self.value - other)

    def __div__(self, other):
        other = getattr(other, 'value', other)
        return Integer(self.value / other)

    # Tests --------------------------------------------------------------------
    if tests.define_tests():
        def test_positive(self):
            '''
            Support for small positive numbers
            '''
            for i in [2, 3, 4, 5, 10]:
                x = Integer(i)
                assert x.value == i, i
                assert x.complexity() > 0, i

        def test_zero(self):
            '''
            Support to number zero
            '''
            assert 0 == Integer(0).value

        def test_one(self):
            '''
            Support to number one
            '''
            assert 1 == Integer(1).value

        def test_negative(self):
            '''
            Support to negative numbers
            '''
            for i in [-1, -2, -3, -4, -5, -10]:
                x = Integer(i)
                assert x.value == i, i
                assert x.complexity() > 0, i
                assert x.complexity() == (-x).complexity() + 0.5

        def test_math_ops(self):
            '''
            Basic mathematical operations
            '''
            x, y = map(Integer, [3, 5])
            assert (x + y).value == 8
            assert (x * y).value == 15
            assert (x / y).value == 0
            assert (x - y).value == -2

    if tests.define_examples():
        def example_simple_numbers(self):
            snumbers = []
            for i in xrange(1000):
                x = Integer(i)
                if x.complexity() <= 2.0:
                    snumbers.append(x.value)
            print 'List of all simple numbers up to 2000: \n    %s' % snumbers

#===============================================================================
# Lists set
#===============================================================================
class SetOfLists(object):
    def __init__(self, initial_set):
        '''
        Represent a set of lists. Support modification of list members and 
        inclusion of randomized new members. 
        '''

        # Assert that all lists have the same size
        size = len(iter(initial_set).next())
        if any(len(l) != size for l in initial_set):
            raise ValueError('lists of inconsistent sizes')

        # Create set and assert that elements are unique         
        initial_set = map(tuple, initial_set)
        self._members = set(initial_set)
        if len(initial_set) != len(self._members):
            raise ValueError("elements in 'initial_set' are not unique")

    def expand_size(self, k):
        '''
        Expand the size of all lists by length 'k'. If 'k' is negative, the 
        lists are shrunk. 
        '''
        pass

if __name__ == '__main__':
    tests.main()
