##
# Copyright (c) 2010-2014 Apple Inc. All rights reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
# http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
##

"""
OpenDirectory service tests.
"""

from twisted.trial import unittest

from ...idirectory import QueryNotSupportedError
from ...expression import (
    CompoundExpression, Operand, MatchExpression, MatchType, MatchFlags
)
from .._constants import LDAPOperand
from .._service import DirectoryService
from .._util import (
    ldapQueryStringFromQueryStrings,
    ldapQueryStringFromMatchExpression,
    ldapQueryStringFromCompoundExpression,
    ldapQueryStringFromExpression,
)



class LDAPQueryTestCase(unittest.TestCase):
    """
    Tests for LDAP query generation.
    """

    def service(self):
        # Use intentionally funky conenction info, since we don't expect
        # to connect.
        return DirectoryService(u"ldap://cretin/", u"o=plugh")


    def fieldNameMap(self, service):
        """
        Create a mapping from field names to LDAP attribute names.
        The attribute names returned here are not real LDAP attribute names,
        but we don't care for these tests, since we're not actually connecting
        to LDAP.
        """
        return dict([
            (c, (c.name,))
            for c in service.fieldName.iterconstants()
        ])


    def recordTypeMap(self, service):
        """
        Create a mapping from record types to LDAP object class names.
        The object class names returned here are not real LDAP object class
        names, but we don't care for these tests, since we're not actually
        connecting to LDAP.
        """
        return dict([
            (c, (c.name,))
            for c in service.recordType.iterconstants()
        ])


    def _test_ldapQueryStringFromQueryStrings(self, queryStrings, expected):
        for operand in (LDAPOperand.AND.value, LDAPOperand.OR.value):
            compound = ldapQueryStringFromQueryStrings(operand, queryStrings)
            self.assertEquals(compound, expected.format(operand=operand))


    def test_ldapQueryStringFromQueryStrings_empty(self):
        """
        A single expression should just be returned as-is.
        """
        return self._test_ldapQueryStringFromQueryStrings((), u"")


    def test_ldapQueryStringFromQueryStrings_single(self):
        """
        A single expression should just be returned as-is.
        """
        queryStrings = (u"(x=yzzy)",)
        return self._test_ldapQueryStringFromQueryStrings(
            queryStrings, queryStrings[0]
        )


    def test_ldapQueryStringFromQueryStrings_multiple(self):
        """
        Multiple expressions should just be combined with an operator.
        """
        return self._test_ldapQueryStringFromQueryStrings(
            (u"(x=yzzy)", u"(xy=zzy)"), u"({operand}(x=yzzy)(xy=zzy))"
        )


    def test_queryStringFromMatchExpression_matchTypes(self):
        """
        Match expressions with each match type produces the correct
        operator=value string.
        """
        service = self.service()

        for matchType, expected in (
            (MatchType.equals, u"=xyzzy"),
            (MatchType.startsWith, u"=xyzzy*"),
            (MatchType.endsWith, u"=*xyzzy"),
            (MatchType.contains, u"=*xyzzy*"),
            (MatchType.lessThan, u"<xyzzy"),
            (MatchType.greaterThan, u">xyzzy"),
            (MatchType.lessThanOrEqualTo, u"<=xyzzy"),
            (MatchType.greaterThanOrEqualTo, u">=xyzzy"),
        ):
            expression = MatchExpression(
                service.fieldName.shortNames, u"xyzzy",
                matchType=matchType
            )
            queryString = ldapQueryStringFromMatchExpression(
                expression,
                self.fieldNameMap(service), self.recordTypeMap(service),
            )
            expected = u"({attribute}{expected})".format(
                attribute=u"shortNames", expected=expected
            )
            self.assertEquals(queryString, expected)


    def test_queryStringFromMatchExpression_match_not(self):
        """
        Match expression with the C{NOT} flag adds the C{!} operator.
        """
        service = self.service()

        expression = MatchExpression(
            service.fieldName.shortNames, u"xyzzy",
            flags=MatchFlags.NOT
        )
        queryString = ldapQueryStringFromMatchExpression(
            expression,
            self.fieldNameMap(service), self.recordTypeMap(service),
        )
        expected = u"(!{attribute}=xyzzy)".format(
            attribute=u"shortNames",
        )
        self.assertEquals(queryString, expected)


    def test_queryStringFromMatchExpression_match_caseInsensitive(self):
        """
        Match expression with the C{caseInsensitive} flag adds the C{??????}
        operator.
        """
        service = self.service()

        expression = MatchExpression(
            service.fieldName.shortNames, u"xyzzy",
            flags=MatchFlags.caseInsensitive
        )
        queryString = ldapQueryStringFromMatchExpression(
            expression,
            self.fieldNameMap(service), self.recordTypeMap(service),
        )
        expected = u"???????({attribute}=xyzzy)".format(
            attribute=u"shortNames",
        )
        self.assertEquals(queryString, expected)


    test_queryStringFromMatchExpression_match_caseInsensitive.todo = (
        "unimplemented"
    )


    def test_queryStringFromMatchExpression_match_quoting(self):
        """
        Special characters are quoted properly.
        """
        service = self.service()

        expression = MatchExpression(
            service.fieldName.fullNames,
            u"\\xyzzy: a/b/(c)* ~~ >=< ~~ &| \0!!"
        )
        queryString = ldapQueryStringFromMatchExpression(
            expression,
            self.fieldNameMap(service), self.recordTypeMap(service),
        )
        expected = u"({attribute}={expected})".format(
            attribute=u"fullNames",
            expected=(
                u"\\5Cxyzzy: a\\2Fb\\2F\\28c\\29\\2A "
                "\\7E\\7E \\3E\\3D\\3C \\7E\\7E \\26\\7C \\00!!"
            )
        )
        self.assertEquals(queryString, expected)


    def test_queryStringFromMatchExpression_unknownFieldName(self):
        """
        Unknown expression.
        """
        service = self.service()

        expression = MatchExpression(
            object(), u"xyzzy",
        )

        self.assertRaises(
            QueryNotSupportedError,
            ldapQueryStringFromMatchExpression,
            expression,
            self.fieldNameMap(service), self.recordTypeMap(service),
        )


    def test_queryStringFromMatchExpression_unknownMatchType(self):
        """
        Unknown expression.
        """
        service = self.service()

        expression = MatchExpression(
            service.fieldName.shortNames, u"xyzzy",
            matchType=object()
        )

        self.assertRaises(
            QueryNotSupportedError,
            ldapQueryStringFromMatchExpression,
            expression,
            self.fieldNameMap(service), self.recordTypeMap(service),
        )


    def _test_queryStringFromMatchExpression_multiAttribute(
        self, flags, expected
    ):
        service = self.service()

        expression = MatchExpression(
            service.fieldName.emailAddresses, u"xyzzy",
            flags=flags,
        )

        fieldNameToAttributeMap = {
            service.fieldName.emailAddresses: (u"mail", u"alternateMail"),
        }

        queryString = ldapQueryStringFromMatchExpression(
            expression, fieldNameToAttributeMap, self.recordTypeMap(service)
        )

        self.assertEquals(queryString, expected)


    def test_queryStringFromMatchExpression_multipleAttribute(self):
        """
        Match expression when the queried field name maps to multiple
        attributes.
        """

        # We want a match for either attribute.
        expected = u"(|(mail=xyzzy)(alternateMail=xyzzy))"

        return self._test_queryStringFromMatchExpression_multiAttribute(
            MatchFlags.none, expected
        )


    def test_queryStringFromMatchExpression_multipleAttribute_not(self):
        """
        Match expression when the queried field name maps to multiple
        attributes and the NOT flag is set.
        """

        # We want a NOT match for both attributes.
        expected = u"(&(!mail=xyzzy)(!alternateMail=xyzzy))"

        return self._test_queryStringFromMatchExpression_multiAttribute(
            MatchFlags.NOT, expected
        )


    def _test_queryStringFromMatchExpression_multiRecordType(
        self, flags, expected
    ):
        service = self.service()

        recordTypeField = service.fieldName.recordType

        expression = MatchExpression(
            recordTypeField, service.recordType.user,
            flags=flags,
        )

        fieldNameToAttributeMap = self.fieldNameMap(service)

        recordTypeToObjectClassMap = {
            service.recordType.user: (u"person", u"account"),
        }

        queryString = ldapQueryStringFromMatchExpression(
            expression, fieldNameToAttributeMap, recordTypeToObjectClassMap
        )

        self.assertEquals(
            queryString,
            expected.format(attr=fieldNameToAttributeMap[recordTypeField][0])
        )


    def test_queryStringFromMatchExpression_multipleRecordType(self):
        """
        Match expression when the queried field name is the record type field,
        which maps to multiple attributes.
        """

        # We want a match for both values.
        expected = u"(&({attr}=person)({attr}=account))"

        return self._test_queryStringFromMatchExpression_multiRecordType(
            MatchFlags.none, expected
        )

    def test_queryStringFromMatchExpression_multipleRecordType_not(self):
        """
        Match expression when the queried field name is the record type field,
        which maps to multiple attributes and the NOT flag is set.
        """

        # We want a NOT match for either value.
        expected = u"(|(!{attr}=person)(!{attr}=account))"

        return self._test_queryStringFromMatchExpression_multiRecordType(
            MatchFlags.NOT, expected
        )


    def test_queryStringFromCompoundExpression_single(
        self, queryFunction=ldapQueryStringFromCompoundExpression
    ):
        """
        Compound expression with a single sub-expression.

        Should result in the same query string as just the sub-expression
        would.

        The Operand shouldn't make any difference here, so we test AND and OR,
        expecting the same result.
        """
        service = self.service()

        for operand in (Operand.AND, Operand.OR):
            matchExpression = MatchExpression(
                service.fieldName.shortNames, u"xyzzy"
            )
            compoundExpression = CompoundExpression(
                [matchExpression],
                operand
            )
            queryString = queryFunction(
                compoundExpression,
                self.fieldNameMap(service), self.recordTypeMap(service),
            )
            expected = u"{match}".format(
                match=ldapQueryStringFromMatchExpression(
                    matchExpression,
                    self.fieldNameMap(service), self.recordTypeMap(service),
                )
            )
            self.assertEquals(queryString, expected)


    def test_queryStringFromCompoundExpression_multiple(
        self, queryFunction=ldapQueryStringFromCompoundExpression
    ):
        """
        Compound expression with multiple sub-expressions.

        The sub-expressions should be grouped with the given operand.
        """
        service = self.service()

        for (operand, token) in ((Operand.AND, u"&"), (Operand.OR, u"|")):
            matchExpression1 = MatchExpression(
                service.fieldName.shortNames, u"xyzzy"
            )
            matchExpression2 = MatchExpression(
                service.fieldName.shortNames, u"plugh"
            )
            compoundExpression = CompoundExpression(
                [matchExpression1, matchExpression2],
                operand
            )
            queryString = queryFunction(
                compoundExpression,
                self.fieldNameMap(service), self.recordTypeMap(service),
            )
            expected = u"({op}{match1}{match2})".format(
                op=token,
                match1=ldapQueryStringFromMatchExpression(
                    matchExpression1,
                    self.fieldNameMap(service), self.recordTypeMap(service),
                ),
                match2=ldapQueryStringFromMatchExpression(
                    matchExpression2,
                    self.fieldNameMap(service), self.recordTypeMap(service),
                ),
            )
            self.assertEquals(queryString, expected)


    def test_queryStringFromExpression_match(self):
        """
        Match expression.
        """
        service = self.service()

        matchExpression = MatchExpression(
            service.fieldName.shortNames, u"xyzzy"
        )
        queryString = ldapQueryStringFromExpression(
            matchExpression,
            self.fieldNameMap(service), self.recordTypeMap(service),
        )
        expected = ldapQueryStringFromMatchExpression(
            matchExpression,
            self.fieldNameMap(service), self.recordTypeMap(service),
        )
        self.assertEquals(queryString, expected)


    def test_queryStringFromExpression_compound(self):
        """
        Compound expression.
        """
        self.test_queryStringFromCompoundExpression_single(
            queryFunction=ldapQueryStringFromExpression
        )
        self.test_queryStringFromCompoundExpression_multiple(
            queryFunction=ldapQueryStringFromExpression
        )


    def test_queryStringFromExpression_unknown(self):
        """
        Unknown expression.
        """
        service = self.service()

        self.assertRaises(
            QueryNotSupportedError,
            ldapQueryStringFromExpression,
            object(), self.fieldNameMap(service), self.recordTypeMap(service)
        )
