from requests.auth import HTTPBasicAuth
from requests_oauthlib import OAuth1Session
from . import __version__

TWITTER_API_VERSION = '1.1'
TWITTER_BASE_API_URL = 'https://api.twitter.com'

class Component(object):
    def __init__(self, client, path=None):
        self._client = client
        self._path = path

    def __getitem__(self, path):
        path = '%s/%s' % (self._path, path)
        return Component(self._client, path)

    def __getattr__(self, path):
        return self[path]

    def get(self, **params):
        return self._client.request('GET', self._path, **params).data

    def post(self, **params):
        return self._client.request('POST', self._path, **params).data

    def get_path(self):
        return self._path

class Response(object):
    def __init__(self, response, request_method, json_data):
        self.resource_url = response.url
        self.headers = response.headers
        self.request_method = request_method
        self.data = json_data

class Client(object):
    api_version = TWITTER_API_VERSION
    base_api_url = TWITTER_BASE_API_URL
    user_agent_string = 'Twypy Twitter Client v%s' % __version__

    def __getattr__(self, path):
        return Component(self, path)

    def configure_oauth_session(self, session):
        session.headers = {'User-Agent': self.get_user_agent_string()}
        return session

    def get_user_agent_string(self):
        return self.user_agent_string

    def request(self, method, path, **params):
        method = method.upper()
        url = self.construct_resource_url(path)
        request_kwargs = {}

        if method == 'GET':
            request_kwargs['params'] = params
        elif method == 'POST':
            request_kwargs['data'] = params

        response = self.make_api_call(method, url, **request_kwargs)

        return self.handle_response(method, response)

    def construct_resource_url(self, path):
        paths = path.split('/')
        return '%s/%s/%s.json' % (self.base_api_url, self.api_version, '/'.join(paths[1:]))

    def make_api_call(self, method, url, **request_kwargs):
        return self.session.request(method, url, **request_kwargs)

    def handle_response(self, method, response):
        data = response.json()

        if response.status_code == 200:
            return Response(response, method, data)

class Api(Client):
    def __init__(self, consumer_key, consumer_secret, access_token=None, access_token_secret=None):
        self.request_token_url = '%s/oauth/request_token' % self.base_api_url
        self.access_token_url = '%s/oauth/access_token' % self.base_api_url

        self.consumer_key = consumer_key
        self.consumer_secret = consumer_secret
        self.access_token = access_token
        self.access_token_secret = access_token_secret

        self.session = self.get_oauth_session()

    def get_oauth_session(self):
        return self.configure_oauth_session(OAuth1Session(client_key=self.consumer_key,
                                                          client_secret=self.consumer_secret,
                                                          resource_owner_key=self.access_token,
                                                          resource_owner_secret=self.access_token_secret))
