##############################################################################
#
# Copyright (C) Zenoss, Inc. 2013, all rights reserved.
#
# This content is made available according to terms specified in
# License.zenoss under the directory where your Zenoss product is installed.
#
##############################################################################

import logging
LOG = logging.getLogger('txwinrm.krb5')

import collections
import os
import re

from twisted.internet import defer, reactor
from twisted.internet.protocol import ProcessProtocol


__all__ = [
    'kinit',
    'ccname',
    ]


KRB5_CONF_TEMPLATE = (
    "# DO NOT EDIT THIS FILE!!\n"
    "#\n"
    "# This file is managed by the txwinrm python module.\n"
    "# NOTE: Any changes to this file will be overwritten.\n"
    "#\n"
    "\n"
    "includedir {includedir}\n"
    "[logging]\n"
    " default = FILE:/var/log/krb5libs.log\n"
    " kdc = FILE:/var/log/krb5kdc.log\n"
    " admin_server = FILE:/var/log/kadmind.log\n"
    "\n"
    "[libdefaults]\n"
    " default_realm = EXAMPLE.COM\n"
    " dns_lookup_realm = false\n"
    " dns_lookup_kdc = false\n"
    " ticket_lifetime = 24h\n"
    " renew_lifetime = 7d\n"
    " forwardable = true\n"
    "\n"
    "[realms]\n"
    "{realms_text}"
    "\n"
    "[domain_realm]\n"
    "{domain_realm_text}"
    )

REALM_TEMPLATE = (
    " {realm} = {{\n"
    "  kdc = {kdc}\n"
    "  admin_server = {kdc}\n"
    " }}\n"
    )

DOMAIN_REALM_TEMPLATE = (
    " .{domain} = {realm}\n"
    " {domain} = {realm}\n"
    )


class Config(object):
    '''
    Manages KRB5_CONFIG.
    '''

    def __init__(self):
        '''
        Initialize instance with data from KRB5_CONFIG.
        '''
        self.path = self.get_path()
        self.realms = self.load()

        # For further usage by kerberos python module.
        os.environ['KRB5_CONFIG'] = self.path

    def add_kdc(self, realm, kdc):
        '''
        Add realm and KDC to KRB5_CONFIG.
        '''
        if kdc in self.realms[realm]:
            return

        '''
        Remove any old kdcs ZEN-13244
        '''
        try:
            self.realms[realm].pop()
        except KeyError:
            pass
        self.realms[realm].add(kdc)
        self.save()

    def get_path(self):
        '''
        Return the path to krb5.conf.

        Order of preference:
            1. $KRB5_CONFIG
            2. $ZENHOME/var/krb5.conf
            3. $HOME/.txwinrm/krb5.conf
            4. /etc/krb5.conf
        '''
        if 'KRB5_CONFIG' in os.environ:
            return os.environ['KRB5_CONFIG']

        if 'ZENHOME' in os.environ:
            return os.path.join(os.environ['ZENHOME'], 'var', 'krb5.conf')

        if 'HOME' in os.environ:
            return os.path.join(os.environ['HOME'], '.txwinrm', 'krb5.conf')

        return os.path.join('/etc', 'krb5.conf')

    def get_ccname(self, username):
        '''
        Return KRB5CCNAME environment for username.

        We use a separate credential cache for each username because
        kinit causes all previous credentials to be destroyed when a new
        one is initialized.

        https://groups.google.com/forum/#!topic/comp.protocols.kerberos/IjtK9Mo39qc
        '''
        if 'ZENHOME' in os.environ:
            return os.path.join(
                os.environ['ZENHOME'], 'var', 'krb5cc', username)

        if 'HOME' in os.environ:
            return os.path.join(
                os.environ['HOME'], '.txwinrm', 'krb5cc', username)

        return ''

    def load(self):
        '''
        Load current realms from KRB5_CONFIG file.
        '''
        realm_kdcs = collections.defaultdict(set)

        if not os.path.isfile(self.path):
            return realm_kdcs

        with open(self.path, 'r') as krb5_conf:
            in_realms_section = False
            in_realm = None

            for line in krb5_conf:
                if line.strip().startswith('[realms]'):
                    in_realms_section = True
                elif line.strip().startswith('['):
                    in_realms_section = False
                elif in_realms_section:
                    line = line.strip()
                    if not line:
                        continue

                    match = re.search(r'(\S+)\s+=\s+{', line)
                    if match:
                        in_realm = match.group(1)
                        continue

                    if in_realm:
                        match = re.search(r'kdc\s+=\s+(\S+)', line)
                        if match:
                            realm_kdcs[in_realm].add(match.group(1))

        return realm_kdcs

    def save(self):
        '''
        Save current realm KDCs to KRB5_CONFIG.
        '''
        realms_list = []
        domain_realm_list = []

        for realm, kdcs in self.realms.iteritems():
            if not kdcs:
                continue

            realms_list.append(
                REALM_TEMPLATE.format(
                    realm=realm.upper(), kdc=tuple(kdcs)[0]))

            domain_realm_list.append(
                DOMAIN_REALM_TEMPLATE.format(
                    domain=realm.lower(), realm=realm.upper()))

        dirname = os.path.dirname(self.path)
        if not os.path.isdir(dirname):
            os.makedirs(dirname)

        # create config dir for user supplied options
        includedir = os.path.join(dirname, 'config')
        if not os.path.isdir(includedir):
            os.makedirs(includedir)

        with open(self.path, 'w') as krb5_conf:
            krb5_conf.write(
                KRB5_CONF_TEMPLATE.format(
                    includedir=includedir,
                    realms_text=''.join(realms_list),
                    domain_realm_text=''.join(domain_realm_list)))


# Singleton. Loads from KRB5_CONFIG on import.
config = Config()


class KinitProcessProtocol(ProcessProtocol):
    '''
    Communicates with kinit command.

    The only thing we do is answer the password prompt. We don't even
    care about the output.
    '''

    def __init__(self, password):
        self._password = password
        self.d = defer.Deferred()
        self._data = ''

    def outReceived(self, data):
        self._data += data
        if 'Password for' in self._data and ':' in self._data:
            self.transport.write('{0}\n'.format(self._password))
            self._data = ''

    def processEnded(self, reason):
        self.d.callback(None)


@defer.inlineCallbacks
def kinit(username, password, kdc):
    '''
    Perform kerberos initialization.
    '''
    kinit = None
    for path in ('/usr/bin/kinit', '/usr/kerberos/bin/kinit'):
        if os.path.isfile(path):
            kinit = path
            break

    if not kinit:
        raise Exception("krb5-workstation is not installed")

    try:
        user, realm = username.split('@')
    except ValueError:
        raise Exception("kerberos username must be in user@domain format")

    realm = realm.upper()

    global config

    config.add_kdc(realm, kdc)

    ccname = config.get_ccname(username)
    dirname = os.path.dirname(ccname)
    if not os.path.isdir(dirname):
        os.makedirs(dirname)

    kinit_args = [kinit, '{}@{}'.format(user, realm)]
    kinit_env = {
        'KRB5_CONFIG': config.path,
        'KRB5CCNAME': ccname,
        }

    protocol = KinitProcessProtocol(password)

    reactor.spawnProcess(protocol, kinit, kinit_args, kinit_env)

    yield protocol.d


def ccname(username):
    '''
    Return KRB5CCNAME value for username.
    '''
    return config.get_ccname(username)
