#!/usr/bin/env python
# coding=UTF-8

__author__ = "Pierre-Yves Langlois"
__copyright__ = "https://github.com/pylanglois/uadm/blob/master/LICENCE"
__credits__ = ["Pierre-Yves Langlois"]
__license__ = "BSD"
__version__ = "1.0"
__maintainer__ = "Pierre-Yves Langlois"
__status__ = "Production"

"""
uadm scripting tools core module. Import this module in your script. See
uadm_base.py for an example.

If you have many scripts you can customize the CONF_MAP with a file in 
/etc/uadm.conf. See the example file for the structure.

"""

import sys
from os import path
import socket
import subprocess
import shlex
import logging

def mod_conf(params, override=True):
    """
    This method will modify configuration in the CONF_MAP dictionarie
    """
    for p in params.keys():
        if (p in CONF_MAP and override) or p not in CONF_MAP:
            CONF_MAP[p] = params[p]

def init_logger():
    """
    This method create a logger with an output to a file
    """
    log_name =  u"%s.log" % CONF_MAP["UADM_TOOL_NAME"].split(".")[0]
    logger = logging.getLogger(CONF_MAP["UADM_TOOL_NAME"])
    logger.setLevel(logging.DEBUG)

    fh = logging.FileHandler("%s/%s" % (CONF_MAP["UADM_LOG_PATH"], log_name), mode="a+")
    fh.setLevel(logging.DEBUG)
    formatter = logging.Formatter('%(asctime)s - %(name)s - %(levelname)s - %(message)s')
    fh.setFormatter(formatter)
    logger.addHandler(fh)

    fh = logging.StreamHandler(sys.stdout)
    fh.setLevel(logging.DEBUG)
    formatter = logging.Formatter('%(asctime)s - %(name)s - %(levelname)s - %(message)s')
    fh.setFormatter(formatter)
    logger.addHandler(fh)

    return logger

def get_host_info():
    """
    This method tries to get the IP and the hostname of the computer
    """
    hostname = socket.getfqdn()
    ip = None
    try:
        s = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
        s.connect(('www.google.ca', 0))
        ip = s.getsockname()[0]
    except:
        pass
    
    return {"hostname":unicode(hostname), "ip":unicode(ip)}

def mail(subject, message, image_list = [], smtp_server = None, from_add = None, to_add = None):
    """
    Code from activestate, slightly modified, allowing to send mail
    """
    smtp_server = smtp_server if smtp_server else CONF_MAP["UADM_SMTP_SERVER"]
    from_add = from_add if from_add else CONF_MAP["UADM_SRC_EMAIL"]
    to_add = to_addd if to_add else CONF_MAP["UADM_SRC_EMAIL"]

    ## {{{ http://code.activestate.com/recipes/473810/ (r1)
    # Send an HTML email with an embedded image and a plain text message for
    # email clients that don't want to display the HTML.
    from email.mime.multipart import MIMEMultipart
    from email.mime.text import MIMEText
    from email.mime.image import MIMEImage

    # Define these once; use them twice!
    strFrom = from_add
    strTo = to_add

    # Create the root message and fill in the from, to, and subject headers
    msgRoot = MIMEMultipart('related')
    msgRoot.set_charset('UTF-8')
    msgRoot['Subject'] = subject
    msgRoot['From'] = strFrom
    msgRoot['To'] = strTo
    msgRoot.preamble = 'This is a multi-part message in MIME format.'

    # Encapsulate the plain and HTML versget_host_info()ions of the message body in an
    # 'alternative' part, so message agents can decide which they want to display.
    msgAlternative = MIMEMultipart('alternative')
    msgAlternative.set_charset('UTF-8')
    msgRoot.attach(msgAlternative)

    msgText = MIMEText('This is the alternative plain text message.')
    msgAlternative.attach(msgText)

    # We reference the image in the IMG SRC attribute by the ID we give it below
#    msgText = MIMEText('<b>Some <i>HTML</i> text</b> and an image.<br><img src="cid:image1"><br>Nifty!', 'html')
    img_iter = 0
    for image in image_list:
        if image is not None:
            message += '<img src="cid:image%d">' % img_iter
            img_iter += 1

    msgText = MIMEText(unicode(message), 'html', 'UTF-8')
    msgAlternative.attach(msgText)

    # This example assumes the image is in the current directory
    img_iter = 0
    for image in image_list:
        if image is not None:
            fp = open(image, 'rb')
            msgImage = MIMEImage(fp.read())
            # Define the image's ID as referenced above
            msgImage.add_header('Content-ID', '<image%d>' % img_iter)
            msgRoot.attach(msgImage)
            img_iter += 1
            fp.close()

    # Send the email (this example assumes SMTP authentication is required)
    import smtplib
    smtp = smtplib.SMTP(smtp_server)
    #smtp.connect(smtp,25)
    #smtp.login('exampleuser', 'examplepass')
    smtp.sendmail(strFrom, strTo, msgRoot.as_string())
    smtp.quit()
    ## end of http://code.activestate.com/recipes/473810/ }}}

def send_report(message, subject_prefix = None):
    """
    This method will send an email to the administrator.
    """
    subject_prefix = subject_prefix if subject_prefix else CONF_MAP["UADM_SUCCESS_SUBJECT"]
    subject = u"[%s] %s sur %s (%s)" % (CONF_MAP["UADM_TOOL_NAME"], 
                                        subject_prefix, 
                                        HOST_INFO["hostname"], 
                                        HOST_INFO["ip"])
    if not CONF_MAP["UADM_DISABLE_MAIL"]:
        mail(subject, message)

def send_error_report(message, subject_prefix = None):
    subject_prefix = subject_prefix if subject_prefix else CONF_MAP["UADM_ERROR_SUBJECT"]
    send_report(message, subject_prefix)

def debug_pinfo(pinfo, command, newline='<br>'):
    """
    This method build a readable version of the process output
    """
    msg = unicode(
                "command: %(command)s%(newline)s"
                "return code: %(return_code)s%(newline)s"
                "stderr: %(stderr)s"
                "stdout: %(stdout)s" %
                    {
                        "newline": newline,
                        "command": command,
                        "return_code": pinfo["return_code"],
                        "stderr": split_output(pinfo["stderr"], newline=newline),
                        "stdout": split_output(pinfo["stdout"], newline=newline),
                    }
                )
    return msg

def split_output(output, newline=u'<br>'):
    """
    Format each line with the proper newline definition
    """
    output = unicode(output, 'UTF-8')
    out = ""
    for line in output.split("\n"):
        out += line + newline
    return out

def run_cmd(command):
    """
    This method runs a single command with Popen and return a map of the output
    """
    proc = subprocess.Popen(shlex.split(command),stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    ret = proc.wait()
    out, err = proc.communicate()
    return {"process":proc, "return_code":ret, "stdout":out, "stderr":err}

def cfb(string_to_clean):
    """
    This method format a multiple line command into a single line command
    """
    return ' '.join(string_to_clean.strip("\n").split())

def exec_cmd_list(cmd_list):
    """
    This method take a list of command and execute them with Popen. If a command fails
    a report is send by mail to the administrator
    """    
    output = ""
    completed = True

    for command in cmd_list:

        pinfo = run_cmd(command)
        output += debug_pinfo(pinfo, command)

        if pinfo["return_code"] == 0:
            l().info(u"%s" % debug_pinfo(pinfo, command, newline="\n"))
        else:
            completed = False
            l().error(u"%s" % debug_pinfo(pinfo, command, newline="\n"))
            send_error_report(output)
            break

    if completed and CONF_MAP["UADM_REPORT_ON_SUCCESS"]:
        send_report("%s" % output)

    return completed, pinfo

def l(init=None):
    if init:
        l_instance['l'] = init
    return l_instance['l']

HOST_INFO = get_host_info()

#Configuration map
CONF_MAP = {

    "UADM_TOOL_NAME" : unicode(path.basename(sys.modules['__main__'].__file__)),
    "UADM_LOG_PATH" : u"/var/log/uadm",
    "UADM_SCRIPT_PATH" : u"/etc/uadm/scripts",
    "UADM_SMTP_SERVER" : u"smtp.example.com",
    "UADM_SRC_EMAIL" : u"tech@example.com",
    "UADM_SUCCESS_SUBJECT" : u"Success",
    "UADM_ERROR_SUBJECT" : u"Error",
    "UADM_REPORT_ON_SUCCESS" : False,
    "UADM_DISABLE_MAIL" : False,
    "UADM_ENABLED" : True,
}

l_instance = {'l':''}

if __name__ == "uadm.uadmcore":
    conf_path = "/etc/uadm/uadm.conf"
    if path.exists(conf_path):
        import imp
        conf = imp.load_source('uadm.conf', conf_path)
        mod_conf(conf.CONF_MAP)

    l(init_logger())

    if not CONF_MAP["UADM_ENABLED"]:
        l().info(u'UADM scripts are not enabled. Change your conf file with "UADM_ENABLED" : True, EXITING!!! ')
        exit(1)


         



