"""
    ufl.core.structures.dict
    ~~~~~~~~~~~~~~~~~~~~~~~~

    Provides some useful variations of the dictionary structure.

    :copyright: Copyright 2009-2011 UFL team. See AUTHORS for details.
    :license: GPLv3
"""
from ufl.core.structures.list import flatten


__all__ = [
    "find_key",
    "find_key_with_parents",
    "AttrDictMixin",
    "AttrDict",
    "DeepAttrDictMixin",
    "DictNamespaceMixin"
]


def find_key(name, dictionary):
    """Find the first occurence of *name* (as a key) in a dictionary."""
    stack = [dictionary]
    while stack:
        item = stack.pop()
        if name in item:
            return item[name]
        for key, value in item.iteritems():
            if isinstance(value, dict):
                stack.append(value)


def find_key_with_parents(name, dictionary):
    """Return the first occurence of *name* and its parents in a dictionary."""
    parents = []
    for key, value in dictionary.items():
        if key == name:
            return [key]
        if isinstance(value, dict):
            parents.append(flatten([key, find_key_with_parents(name, value)]))
            if name in parents[-1]: return parents[-1]


class AttrDictMixin(object):
    """Expose attribute style access to any dict-like structure.

        >>> class Foo(AttrDictMixin, dict):
        ...     pass
        >>> f = Foo()
        >>> f['bar'] = 'baz'
        >>> f
        {'bar': 'baz'}
        >>> f.bar
        'baz'
    """

    def copy(self):
        return self.__copy__()

    def __expand(self, key, value):
        if '.' in key:
            self.pop(key, None)
            parts = key.split('.')
            return {parts[0]: self.__expand('.'.join(parts[1:]), value)}
        else:
            return {key: value}

    def __delattr__(self, key):
        try:
            super(AttrDictMixin, self).__delattr__(key)
        except AttributeError:
            self.__delitem__(key)

    def __getattr__(self, key):
        try:
            return self[key]
        except KeyError, k:
            if key.endswith("_"):
                try:
                    return self[key[:-1]]
                except KeyError:
                    pass
            raise AttributeError(k)

    def __setattr__(self, key, value):
        self.__setitem__(key, value)

    def __setitem__(self, key, value):
        if '.' in key:
            parts = key.split('.')
            self.__setitem__(parts[0], self.__expand('.'.join(parts[1:]),
                value))
        else:
            super(AttrDictMixin, self).__setitem__(key, value)

    def __copy__(self):
        return AttrDict(self)


class DeepAttrDictMixin(AttrDictMixin):
    """Recursively create :class:`AttrDict` for nested dictionaries."""

    def __setitem__(self, key, value):
        if isinstance(value, dict):
            value = AttrDict(value)
        super(DeepAttrDictMixin, self).__setitem__(key, value)


class AttrDict(DeepAttrDictMixin, dict):
    """A dictionary with attribute/dotted style access.

    This class is nothing more than a child of :class:`AttrDictMixin` and the
    default *dict*.

    Keys that are dot-delimited are expanded automatically::

        >>> a = AttrDict()
        >>> a['foo.bar'] = 'baz'
        >>> a
        {'foo': {'bar': 'baz'}}
        >>> a.bar.baz = 'foo'
        >>> a
        {'foo': {'bar': 'baz'}, 'bar': {'baz': 'foo'}}
    """

    def __init__(self, *args, **kwargs):
        if args:
            m = args[0]
            if isinstance(m, dict):
                m = m.iteritems()
        elif kwargs:
            m = kwargs.iteritems()
        else:
            m = []
        if m:
            for k, v in m:
                self[k] = v

    def __expand(self, key, value):
        return AttrDict(AttrDictMixin.__expand(self, key, value))


class DictNamespaceMixin(object):
    """Prefix all keys with a specified namespace on item manipulation.

    Set the namespace class property in order for this to work::

        >>> class FooDict(DictNamespaceMixin, dict):
        ...    namespace = 'foo'
        >>> foo = FooDict({'bar': 'baz'})
        >>> foo
        {'foo.bar': 'baz'}
        >>> foo['bar']
        'baz'
    """

    namespace = ''
    delimiter = '.'

    def __init__(self, *args, **kwargs):
        if args:
            m = args[0]
            if type(m) == dict:
                m = m.iteritems()
        elif kwargs:
            m = kwargs.iteritems()
        else:
            m = []
        if m:
            for k, v in m:
                self[k] = v

    def __delitem__(self, key):
        dict.__delitem__(self,
            self.delimiter.join((self.namespace, key)))

    def __getitem__(self, key):
        return dict.__getitem__(self,
            self.delimiter.join((self.namespace, key)))

    def __setitem__(self, key, value):
        dict.__setitem__(self,
            self.delimiter.join((self.namespace, key)), value)
