"""
    ufl.os.xlib.xresources
    ~~~~~~~~~~~~~~~~~~~~~~

    An object-oriented interface to Xlib's resource DB.

    :copyright: Copyright 2009-2011 UFL team. See AUTHORS for details.
    :license: GPLv3
"""
from collections import OrderedDict
from cStringIO import StringIO
import re

from ufl.core.structures.dict import AttrDict
from ufl.core.structures.list import flatten
from Xlib import rdb


class ResourceCollection(object):
    """Thin wrapper around :class:`Xlib.rdb.ResourceDB`."""

    INCLUDE_RE = re.compile(r'^#\s?include\s+(?:\'|")?(.+)"(?:\'|")?')

    def __init__(self, initial_filename=None):
        self.resources = OrderedDict()
        initial_filename and self.add_resource_file(initial_filename)

    def add_resource_file(self, filename):
        """Add a resource file and any includes to the collection."""
        self.resources[filename] = rdb.ResourceDB(file=filename)
        for line in open(filename, "r").readlines():
            match = self.INCLUDE_RE.match(line)
            match and self.add_resource_file(match.groups()[0])


class SimpleResourceCollection(ResourceCollection):
    """A more human friendly approach to a :class:`ResourceCollection`."""

    def __init__(self, initial_filename=None):
        self.db = AttrDict()
        super(SimpleResourceCollection, self).__init__(initial_filename)

    @property
    def modified(self):
        rc = SimpleResourceCollection()
        map(rc.add_resource_file, self.resources.keys())
        diff = map(
            lambda (filename, data):
                (filename, data != rc.resources[filename]),
            self.resources.items()
        )
        return (any(zip(*diff)[1]), [
            (filename, self.resources[filename])
            for filename, is_modified in diff if is_modified])

    def add_resource_file(self, filename):
        """Map a parsed resource file to an accessible dictionary."""
        super(SimpleResourceCollection, self).add_resource_file(filename)

        def collapse((key, value)):
            if value[-1] == {}:
                return (key, AttrDict(map(collapse, value[0].items())))
            else:
                if type(value[-1]) == dict:
                    return (key, AttrDict(map(collapse, value[-1].items())))
                else:
                    return (key, value[-1])

        self.resources[filename] = dict(map(
            collapse,
            self.resources[filename].db.items()
        ))
        self.db.update(self.resources[filename])

    def output(self, filename=None):
        """Convert resource data into resource file format.

        :param filename: Optional, if supplied, only returns keys from the
                         given file.
        """
        if filename is None:
            return "\n".join(map(self.output, self.resources.keys()))
        else:
            def dot((key, value)):
                if isinstance(value, dict):
                    return "\n".join(flatten(map(
                        lambda (k, v):
                            dot((".".join([key or "*", k]), v)),
                        value.items())))
                else:
                    return "{0}: {1}".format(key, value)
            return "\n".join(map(dot, self.resources[filename].items()))

    def save(self, filename=None, save_as=None):
        """Write formatted data to a resource file.

        :param filename: This must be identical to an existing file in the
                         resource db.
        :param save_as: Optional, alternate location to save file.
        """
        assert filename in self.resources, "No such resource file in db"
        return open(save_as or filename, "w").write(self.output(filename))
