"""
    ufl.core.stringutils
    ~~~~~~~~~~~~~~~~~~~~

    Provides string formatters and encapsulates various standard library
    string manipulation procedures.

    :copyright: Copyright 2009-2011 UFL team. See AUTHORS for details.
    :license: GPLv3
"""
import string


class SafeObjectFormatter(string.Formatter):
    """A :class:`string.Formatter` that supports attributes and methods::

        >>> f = SafeObjectFormatter()
        >>> f.format("{foo.upper()}", foo="bar")
        "BAR"
        >>> f.format("{1.upper()}", "foo", "bar")
        "BAR"
    """

    def get_field(self, field, args, kwargs):
        try:
            return super(SafeObjectFormatter, self).get_field(
                field, args, kwargs)
        except AttributeError:
            chunks = field.split(".")
            used_key = chunks[0]
            if used_key.isdigit() and (len(args) - 1) >= int(used_key):
                used_key = long(used_key)
                result = args[used_key]
            else:
                result = kwargs[used_key]
            # Unsafe, swiss cheese, microsoft version:
            # result = eval(field, {"__builtins__": None}, kwargs)
            # Safer, albeit limited version:
            for c in chunks[1:]:
                lp_pos = c.find("(")
                if lp_pos != -1:
                    arguments = ast.literal_eval(c[lp_pos+1:c.find(")")])
                    if type(arguments) != tuple and c[lp_pos+1] != "(":
                        arguments = (arguments, )
                    c = c[:lp_pos]
                else:
                    arguments = None
                assert not c.startswith("__"), "No can do!"
                result = getattr(result, c)
                if arguments:
                    result = result(*arguments)
            return (result, used_key)


def render_template(template_path, output_path=None, numbered_data=[],
    named_data={}):
    """Render a template using `SafeObjectFormatter`.

    :param output_path: If specified, renders template to a file rather
                        returning the rendered string.
    """
    formatter = SafeObjectFormatter()
    with open(template_path, "r+") as f:
        data = formatter.format(f.read(), *numbered_data, **named_data)
        return open(output_path, "w").write(data) if output_path else data
