#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""
    uilayer

    python user interface layer to monitor progress of long-running
    algorithms
"""

from __future__ import division

import os
import sys
import warnings


__all__=["dummy", "ui", "stdout"]


class base_ui():
    """
        This is the main class from which all interfaces are derived.
        It is quiet, meaning it does not output any progress data, but
        it will display warings and errors in the console.
    """
    def __init__(self, pids=None, steps=None, progress=None,
                 subpids=None, show_warnings=True):
        """
            Creates an instance and sets initial parameters for progress
            dialogs (optional).
            
            
            Parameters:
            -----------
            pids, steps, progress, subpids : optional
                See documentation of self.SetProgress
            show_warnings : bool
                Display warnings of/in the interface.

            See Also
            --------
            SetProgress : Sets initial parameters manually
        """
        self.ShowWarnings(show_warnings)
        self.SetProgress(pids, steps, progress, subpids)
    
    
    def Iterate(self, pid=None, subpid=0, **kwargs):
        """
            Increments the "current step" of a progress by one.
            Warns the user if the step is above "total steps".
        """
        if pid is None:
            self.warn("I do not know which progress to iterate."+
                          " Please specify the kwarg pid.")
            pid=0

        pid = int(pid)
        subpid = int(subpid)

        for i in range(len(self._pids)):
            if self._pids[i] == pid and self._subpids[i] == subpid:
                self._progress[i] += 1

        self.Update()


    def Finalize(self, pid=None, subpid=0, **kwargs):
        """
            Triggers output that a process i finished
        """
        pass
    
    
    def SetProgress(self, pids=None, steps=None, progress=None,
                          subpids=None):
        """
            Set the initial parameters of the processes that are to be
            monitored by this class.
            
            
            Parameters:
            -----------
            pids : Array-like dtype(int) or int, length N
                Process identifiers
            steps : Array-like dtype(int) or int, length N
                Maximum number of steps for each process
            progress : Array-like dtype(int) or int, length N
                Current step in the progress counting up until `steps`
            subpids : Array-like dtype(int) or int, length N
                Sub-process identifiers - useful for multithreading

            The following kwargs combinations are possible:
                1. pids, steps, progress, subpids all `None`
                    Nothing will happen
                2. pids, steps, progress, subpids all some int
                    Only one process
                3. pids, steps, progress, subpids all integer `list`s
                    Multiple processes are assumed
                4. pids, progress, subpids all `None` and steps int
                    Only one process of known length
                5. pids, progress, subpids all `None` and steps `list`
                    Multiple processes of known length - internally
                    the processes get `pids` that are enumerated from
                    zero onwards and `progress` is set no zeros.
        """
        if (pids is None and steps is None and progress is None):
            # 1. Nothing will happen
            return
        elif (pids is not None and steps is not None and
                                  progress is not None):
            # 2. Only one process
            # or
            # 3. Multiple processes are assumed
            pass
        elif (pids is None and steps is not None and progress is None):
            if isinstance(steps, (int, long, float, complex)):
                # 4. Only one process of known length
                pids = 0
                progress = 0
            else:
                # 5. Multiple processes of known length - internally
                #    the processes get `pids` that are enumerated from
                #    zero onwards and `progress` is set no zeros.
                pids = range(len(steps))
                progress = [0]*len(steps)
        else:
            raise NameError("pids, steps, and progress must all "+
                            "be set or all be None.")

        # Convert input to list
        if isinstance(pids, (int, long, float, complex)):
            pids = [pids]
        if isinstance(steps, (int, long, float, complex)):
            steps = [steps]
        if isinstance(progress, (int, long, float, complex)):
            progress = [progress]
        if isinstance(subpids, (int, long, float, complex)):
            subpids = [subpids]
            
        self._pids = pids
        # Defines how many steps should be used to split the total
        # computation. The function self.progress will be used to
        # follow the steps of the algorithm.
        self._steps = steps
        self._progress = progress
        if subpids is None:
            self._subpids = [0]*len(self._pids)
        else:
            self._subpids = subpids


    def ShowWarnings(self, show_warnings):
        """
            show_warnings : bool
                enable or disable warning
        """
        self._show_warnings = show_warnings


    def Update(self, **kwargs):
        pass
        

    def warn(self, msg):
        """ Warns the user when there is a problem. """
        if self._show_warnings:
            warnings.warn(msg)


class dummy(object):
    """
        This is a dummy class that can be used to improve speed.
        There will be no output.
    """
    def __init__(self, *args, **kwargs):
        pass
    def Finalize(self, *args, **kwargs):
        pass
    def Iterate(self, *args, **kwargs):
        pass
    def SetProgress(self, *args, **kwargs):
        pass
    def Update(self, *args, **kwargs):
        pass
    def warn(self, *args, **kwargs):
        pass


class stdout(base_ui):
    """
        Writes output into the console.
        I is the standard output.
        Processes are sorted according to how far they did proceed.
    """
    def __init__(self, **kwargs):
        """ 
            Stuff that needs to be done in the beginning.
        """
        base_ui.__init__(self, **kwargs)
        self._maxlength = get_terminal_width()


    def Finalize(self, pid=None, subpid=0, name=None, **kwargs):
        """
            Triggers output that a process i finished
        """
        if pid is None:
            self.warn("I do not know which progress to finalize."+
                          " Please specify the kwarg pid.")

        pid = int(pid)
        subpid = int(subpid)


        msg = "Progress finished: {}-{}".format(pid,subpid)
        if name is not None:
            msg += " ({})".format(name)
        dn = max(self._maxlength-len(msg),0)

        sys.stdout.write("\r{}{}\n".format(msg,dn*" "))
        sys.stdout.flush()

    
    def Update(self, **kwargs):
        """
            The algorithm wants to tell the user something.
            We use the last line of the standard output to write
            the progress of all algorithms.
        """
        startline = u"Progress: "
        msg = startline
        for i in range(len(self._pids)):
            # compute percentage
            if self._pids[i] != -1:
                if self._steps[i] == 0:
                    perc = int(self._progress[i])
                    msg += u"{}-{}@{}|".format(
                        self._pids[i], self._subpids[i], perc)
                else:
                    perc = self._progress[i]/self._steps[i]*100.
                    msg += u"{}-{}@{:0.2f}%|".format(
                        self._pids[i], self._subpids[i], perc)
        msg = msg.strip(" |")

        
        dn = max(self._maxlength-len(msg),0)

        newline = "\r{}{}".format(msg,dn*" ")
        
        if len(newline) > self._maxlength+1:
            warnings.warn("Terminal width too small. Output cropped.")
            newline = newline[:self._maxlength-1]
        
        if newline.strip() == startline.strip():
            newline = self._maxlength*" "+"\r"
        
        sys.stdout.write(newline)
        sys.stdout.flush()


def get_terminal_size(fd=1):
    """
    Returns height and width of current terminal. First tries to get
    size via termios.TIOCGWINSZ, then from environment. Defaults to 25
    lines x 80 columns if both methods fail.

    :param fd: file descriptor (default: 1=stdout)

    Author: cas (http://blog.taz.net.au/author/cas/)
    """
    try:
        import fcntl, termios, struct
        hw = struct.unpack('hh', fcntl.ioctl(fd, termios.TIOCGWINSZ, '1234'))
    except:
        try:
            hw = (os.environ['LINES'], os.environ['COLUMNS'])
        except:  
            hw = (25, 80)

    return hw


def get_terminal_height(fd=1):
    """
    Returns height of terminal if it is a tty, 999 otherwise

    :param fd: file descriptor (default: 1=stdout)
    """
    if os.isatty(fd):
        height = get_terminal_size(fd)[0]
    else:
        height = 999
   
    return height


def get_terminal_width(fd=1):
    """
    Returns width of terminal if it is a tty, 999 otherwise

    :param fd: file descriptor (default: 1=stdout)
    """
    if os.isatty(fd):
        width = get_terminal_size(fd)[1]
    else:
        width = 999

    return width


__version__ = "0.1.0"
__author__ = "Paul Mueller"
__email__ = "paul.mueller@biotec.tu-dresden.de"
__license__ = "BSD (3-Clause)"
