__author__ = 'tyndyll'

import exceptions
import datetime
import geopy
import service
import urllib2


class Crime:
    """An instance of a crime, as provided by the API

    .. note::

       This class should never be initiated directly. It should be returned from :class:`Police.Force`

    """

    def __init__(self, json_object):
        """Create Crime Object.

        .. note::

        This should not be called directly, only from one of the module methods
        """
        self.__data = json_object

    def __gd(self, key, default_value=None):
        return self.__data.get(key, default_value)

    @property
    def category(self):
        """Category of the crime. Returns a crime category as ..."""
        #TODO Enhance this comment
        return self.__gd("category")

    @property
    def crime_id(self):
        """64-character unique identifier for that crime."""
        return self.__gd("persistent_id")

    @property
    def date(self):
        """Year and month of the crime"""
        return self.__gd("month")

    @property
    def point(self):
        """Approximate location of the crime"""
        data = self.__gd("location", None)
        if data is not None and "longitude" in data and "latitude" in data:
            return geopy.Point(latitude=data["latitude"], longitude=data["longitude"])
        else:
            return None

    @property
    def street(self):
        """The approximate street where the crime occurred"""
        data = self.__gd("location", None)
        if data is not None and "street" in data:
            return data["street"]
        else:
            return None

    @property
    def location_type(self):
        """The type of the location. Either Force or BTP: Force indicates a normal police force
        location; BTP indicates a British Transport Police location. BTP locations fall within
        normal police force boundaries."""
        self.__gd("location_type")

    @property
    def btp_location(self):
        """For BTP locations, the type of location at which this crime was recorded. If the
        :mod:`Crime.location_type` is not BTP return None"""
        return self.__gd("location_subtype") if self.location_type == "BTP" else None


    @property
    def context(self):
        """Extra information about the crime (if applicable)"""
        return self.__gd("context")

    @property
    def outcome(self):
        """Return list of outcomes


        The full category name is returned
        IMPORTANT NOTE: This element may not be present for all crimes

        Code 	Name
        awaiting-court-result 	Awaiting court outcome
        unable-to-proceed 	Court case unable to proceed
        local-resolution 	Local resolution
        no-further-action 	No further action at this time
        deprived-of-property 	Offender deprived of property
        fined 	Offender fined
        absolute-discharge 	Offender given absolute discharge
        cautioned 	Offender given a caution
        penalty-notice-issued 	Offender given a penalty notice
        community-penalty 	Offender given community sentence
        conditional-discharge 	Offender given conditional discharge
        suspended-sentence 	Offender given suspended prison sentence
        imprisoned 	Offender sent to prison
        other-court-disposal 	Offender otherwise dealt with
        compensation 	Offender ordered to pay compensation
        sentenced-in-another-case 	Suspect charged as part of another case
        charged 	Suspect charged
        not-guilty 	Defendant found not guilty
        sent-to-crown-court 	Defendant sent to Crown Court
        unable-to-prosecute 	Unable to prosecute suspect
        under-investigation 	Under investigation

        """
        data = self.__gd("outcome_status", None)
        if data is not None:
            return {
                "category": data["category"] if "category" in data else None,
                'date': data["date"]
            }
        else:
            return None


def __format_date(date_to_be_tested):
    """Internal method used to format date to method appropriate for the API"""
    parts = date_to_be_tested.split("-")
    if len(parts) != 2:
        raise exceptions.InvalidApiDateException(date_to_be_tested)
    else:
        if not int(parts[0]) >= 2011 or (int(parts[1]) < 1 or int(parts[1]) > 12):
            raise exceptions.InvalidApiDateException(date_to_be_tested)
    return date_to_be_tested


def __get_poly(coordinates):
    """Internal method to convert list of points to a group of points suitable for defining an area in the API"""
    points = []
    for point in coordinates:
        points.append(point.format_decimal())
    poly = ":".join(points).replace(" ","")
    return poly


def __return_crimes(crime_records):
    """Internal method used to create a list of Crime instances"""
    results = []
    for crime in crime_records:
        results.append(Crime(crime))
    return results


def __get_crimes_by_id(service_url, location_id, results_month):
    """Internal method used to create a get crime JSON by location ID"""
    results_month = datetime.datetime.now().strftime("%Y-%m") if results_month is None else __format_date(results_month)
    data = None
    try:
        data = service.fetch_json(service_url, location_id=location_id, date=results_month)
    except urllib2.HTTPError as E:
        if E.code == 503:
            raise exceptions.TooManyCrimesException()
    return data


def __get_crimes_by_point(service_url, point, results_month):
    """Internal method used to create a get crime JSON by point"""
    results_month = datetime.datetime.now().strftime("%Y-%m") if results_month is None else __format_date(results_month)
    data = service.fetch_json(service_url, lat=point.latitude, lng=point.longitude, date=results_month)
    return data


def __get_crimes_by_area(service_url, poly_points, results_month):
    """Internal method used to create a get crime JSON by area"""
    results_month = datetime.datetime.now().strftime("%Y-%m") if results_month is None else __format_date(results_month)
    data = service.fetch_json(service_url, poly=__get_poly(poly_points), date=results_month)
    return data


def street_by_point(point, results_month=None):
    """Crimes at street-level within a 1 mile radius of a single point

    http://data.police.uk/docs/method/crime-street/

    The street-level crimes returned in the API are only an approximation of
    where the actual crimes occurred, they are NOT the exact locations.

    Arguments:
        point: Instance of geopy.Point
        results_month: string in the format YYYY-MM

    Returns:
        List of instances of Crime objects
    """
    return __return_crimes(__get_crimes_by_point("crimes-street/all-crime", point, results_month))


def street_by_area(poly_points, results_month=None):
    """Crimes at street-level within a custom area.

    http://data.police.uk/docs/method/crime-street/

    The street-level crimes returned in the API are only an approximation of
    where the actual crimes occurred, they are NOT the exact locations.

    Arguments:
        poly_points: List containing instances of Point
        results_month: string in the format YYYY-MM

    Returns:
        List of instances of Crime objects
    """
    return __return_crimes(__get_crimes_by_area("crimes-street/all-crime", poly_points, results_month))


def street_level_availability(reverse_order=False):
    """Return a list of available data sets.

    http://data.police.uk/docs/method/crimes-street-dates/

    Returns:
        List of date data containing Year and month of all available street level crime data in
        ISO format. Ordered from most recent [0] to oldest [n], unless reverse is set to True

    """
    data = service.fetch_json("crimes-street-dates")
    dates = []
    for entry in data:
        dates.append(entry["date"])
    if reverse_order:
        dates.reverse()
    return dates


def street_outcomes_by_id(location_id, results_month=None):
    """Outcomes at street-level at a specific location

    http://data.police.uk/docs/method/outcomes-at-location/

    Note: Outcomes are not available for the Police Service of Northern Ireland.

    Arguments:
        location_id: Valid IDs are returned by other methods which return location information.
        results_month: string in the format YYYY-MM

    Returns:
        List of instances of Crime objects
    """
    return __get_crimes_by_id("outcomes-at-location", location_id, results_month)


def street_outcomes_by_point(point, results_month=None):
    """Outcomes at street-level at a specific location

    http://data.police.uk/docs/method/outcomes-at-location/

    Note: Outcomes are not available for the Police Service of Northern Ireland.

    Arguments:
        location_id: Valid IDs are returned by other methods which return location information.
        results_month: string in the format YYYY-MM

    Returns:
        List of instances of Crime objects
    """
    return __get_crimes_by_point("outcomes-at-location", point, results_month)


def street_outcomes_by_area(poly_points, results_month=None):
    """Outcomes at street-level at a specific location

    http://data.police.uk/docs/method/outcomes-at-location/

    Note: Outcomes are not available for the Police Service of Northern Ireland.

    Arguments:
        location_id: Valid IDs are returned by other methods which return location information.
        results_month: string in the format YYYY-MM

    Returns:
        List of instances of Crime objects
    """
    return __get_crimes_by_area("outcomes-at-location", poly_points, results_month)


def at_location_by_id(location_id, results_month=None):
    """Returns just the crimes which occurred at the specified location

    http://data.police.uk/docs/method/crimes-at-location/

    Note: Returns just the crimes which occurred at the specified location, rather than those
    within a radius. If given latitude and longitude, finds the nearest pre-defined location and
    returns the crimes which occurred there.

    Arguments:
        location_id: Valid IDs are returned by other methods which return location information.
        results_month: string in the format YYYY-MM

    Returns:
        List of instances of Crime objects
    """
    return __return_crimes(__get_crimes_by_id("crimes-at-location", location_id, results_month))


def at_location_by_point(point, results_month=None):
    """Returns just the crimes which occurred at the nearest pre-defined location to the supplied Point

    http://data.police.uk/docs/method/crimes-at-location/

    Arguments:
        location_id: Valid IDs are returned by other methods which return location information.
        results_month: string in the format YYYY-MM

    Returns:
        List of instances of Crime objects
    """
    return __return_crimes(__get_crimes_by_point("crimes-at-location", point, results_month))


def with_no_location(crime_category, force, results_month=None):
    """Returns a list of crimes where the responsible force hasn't specified a location.

    http://data.police.uk/docs/method/crimes-no-location/

    Arguments:
        crime_category: Valid string from categories()
        force: Instance of Force object
        results_month: string in the format YYYY-MM

    Returns:
        List of instances of Crime objects
    """

    results_month = datetime.datetime.now().strftime("%Y-%m") if results_month is None else __format_date(results_month)
    data = service.fetch_json("crimes-no-location", category=crime_category, force=force.id, date=results_month)
    return __return_crimes(data)


def categories(dataset_date=None):
    """Returns a dictionary of valid categories for a given data set date.

    http://data.police.uk/docs/method/crime-categories/

    Arguments:
        dataset_date = Year/Month to provide crime categories for

    Returns:
        Dictionary where the key is the unique identifier for the named
        crime category and the value is name of the crime category
    """
    data = service.fetch_json("crime-categories", date=dataset_date)
    results = {}
    for item in data:
        results[item['url']] = item['name']
    return results


def last_updated():
    """Show when crime data in the API was last updated

    http://data.police.uk/docs/method/crime-last-updated/

    Crime data in the API is updated once a month. Find out when it was last
    updated.

     Returns:
        String with the month of latest crime data in ISO format. The day is
        irrelevant and is only there to keep a standard formatted date.
    """
    data = service.fetch_json("crime-last-updated")
    return data['date']


def available_dates():
    """Return a list of available data sets.

    http://data.police.uk/docs/method/crimes-street-dates/

    Return:
        List of strings indicating what years and months are available to
        query, returned from newest to oldest. String format is YYYY-MM
    """
    data = service.fetch_json("crimes-street-dates")
    results = []
    for i in data:
        results.append(i["date"])
    return results


def crime_detail(crime_identifier):
    """Returns the outcomes (case history) for the specified crime.

    http://data.police.uk/docs/method/outcomes-for-crime/

    Arguments:
        crime_identifier: 64-character identifier, available as the 'id' property of a Crime object.
    Returns:
        Outcomes #TODO - this and others
    """
    data = Crime(service.fetch_json("outcomes-for-crime/%s" % crime_identifier))
    return data