"""
.. module:: police
   :platform: Unix, Windows
   :synopsis: Provides access to Police Force and Neighbourhood data

.. moduleauthor:: Tyndyll <police@tyndyll.net>

"""

import exceptions
import geopy
import service
import urllib2


class Neighbourhood:
    """Provides access to neighbourhood information and contact details

    .. note::

       This class should never be initiated directly. It should be returned from :class:`Police.Force`

    """

    __data = None

    def __init__(self, neighbourhood_name, neighbourhood_id, force_id):
        """
        :param neighbourhood_name: Name of neighbourhood.
        :type neighbourhood_name: str.
        :param neighbourhood_id: Unique ID of neighbourhood.
        :type neighbourhood_id: str.
        :param force_id: Police force associated with this neighbourhood.
        :type force_id: str.

        """
        self.__name = neighbourhood_name
        self.__id = neighbourhood_id
        self.__force_id = force_id

    def __gd(self, key, default_value=""):
        """Internal utility function - GetData"""
        if self.__data is None:
            self.__data = service.fetch_json("%s/%s" % (self.__force_id, self.__id))
        return self.__data.get(key, default_value)

    @property
    def url(self):
        """URL for the neighbourhood on the Force's website"""
        return self.__gd("url_force")

    @property
    def boundary(self):
        """URL for the neighbourhood's boundary in KML format"""
        return self.__gd("url_boundary", None)

    @property
    def email(self):
        """Email address to get in touch with the neighbourhood officers"""
        if "contact" in self.__data:
            return self.__data["contact"].get("email", None)
        else:
            return None

    @property
    def phone(self):
        """Phone number to get in touch with the neighbourhood officers"""
        if "contact" in self.__data:
            return self.__data["contact"].get("telephone", None)
        else:
            return None

    @property
    def name(self):
        """Name of the neighbourhood"""
        return self.__gd("name")

    @property
    def message(self):
        """An introduction message for the neighbourhood"""
        return self.__gd("welcome_message")

    @property
    def links(self):
        """Links relevant to the neighbourhood.

        :returns:  List -- Returns a list of objects with keys 'url', 'title' and 'description' [optional].
        """
        #TODO: Describe Objects
        data = self.__gd("links", None)
        return data if data is not None else []

    @property
    def centre(self):
        """Centre point locator for the neighbourhood.

        .. note::

        This may not be exactly in the centre of the neighbourhood

        :returns: Point -- Returns an instance of :mod:`geopy.Point`
        """
        return geopy.Point(latitude=self.__data["centre"]["latitude"],
                           longitude=self.__data["centre"]["longitude"])

    @property
    def locations(self):
        """Any associated locations with the neighbourhood, e.g. police stations

        :returns:  List -- Returns a list of objects.
        """
        #TODO: Describe Objects
        data = self.__gd("locations", None)
        return data if data is not None else []

    @property
    def population(self):
        """Population for the neighbourhood"""
        return int(self.__data["population"])

    @property
    def id(self):
        """Unique ID for the neighbourhood"""
        return self.__id

    @property
    def description(self):
        """Description of the neighbourhood"""
        return self.__gd("description")

    @property
    def people(self):
        """Names of people involved"""
        #TODO: Return actual class
        try:
            return service.fetch_json("%s/%s/people" % (self.__force_id, self.__id))
        except urllib2.HTTPError as E:
            if E.code == 404:
                return []
            else:
                raise E

    @property
    def events(self):
        """Return police event detail"""
        #TODO: Return actual class
        try:
            return service.fetch_json("%s/%s/events" % (self.__force_id, self.__id))
        except urllib2.HTTPError as E:
            if E.code == 404:
                return []
            else:
                raise E


class Force:

    """Provides access to Police Force information and the neighbourhoods they are responsible for

    .. note::

       This class should never be initiated directly. It should be returned from :mod:`police` modules


    """

    __data = None
    __neighbourhoods = None

    def __init__(self, json_object):
        """

        """
        self.__id = json_object["id"]
        self.__name = json_object["name"]

    def __gd(self, key, default_value=""):
        """Internal utility function - GetData"""
        if self.__data is None:
            self.__data = service.fetch_json("forces/%s" % self.__id)
        return self.__data.get(key, default_value)

    def get_neighbourhood_by_id(self, neighbourhood_id):
        if neighbourhood_id not in self.neighbourhoods:
            raise exceptions.NeighbourhoodNotFoundException(neighbourhood_id)
        return self.neighbourhoods[neighbourhood_id]

    @property
    def description(self):
        """Description of the force"""
        return self.__gd("description")

    @property
    def url(self):
        """Force website URL"""
        return self.__gd("url")

    @property
    def engagement_methods(self):
        """Ways to keep informed

        :returns:  List -- Returns a list of objects.
        """
        #TODO: Describe Objects
        data = self.__gd("engagement-methods", None)
        return data if data is not None else []

    @property
    def phone(self):
        """Force telephone number"""
        return self.__gd("telephone")

    @property
    def name(self):
        """Force name"""
        return self.__name

    @property
    def id(self):
        """Unique force identifier"""
        return self.__id

    @property
    def neighbourhoods(self):
        if self.__neighbourhoods is None:
            data = service.fetch_json("%s/neighbourhoods" % self.__id)
            self.__neighbourhoods = {}
            for item in data:
                self.__neighbourhoods[item["id"]] = Neighbourhood(item["name"], item["id"], self.__id)
        return self.__neighbourhoods


def list_all_force_ids():
    """
    A list of all the police forces available via the API. The function returns a list with the
    "Unique Force identifier". The unique force identifier should be used in requests for
    force-specific data via other methods.
    """
    return get_all_forces()


def get_all_forces():
    """
    A list of all the police forces available via the API. The function returns a dictionary
    with the "Unique Force identifier" as key and a Force object as the value. The unique force
    identifier should be used in requests for force-specific data via other methods.
    """

    global __forces
    if len(__forces) == 0:
        forces = {}
        for force_item in service.fetch_json("forces"):
            __forces[force_item["id"]] = Force(force_item)
    return __forces


def get_force_by_name(force_name):
    """
    Return a Force instance which corresponds to the supplied name. Raise ForceNotFoundException if
    no instance is found

    :param force_name: Name of the police force.
    :type force_name: str.
    """
    cc_force_name = force_name.lower()
    for force in get_all_forces().values():
        if force.name.lower() == cc_force_name:
            return force
    raise exceptions.ForceNotFoundException(force_name)


def get_force_by_id(unique_force_identifier):
    """
    Return a Force instance which corresponds to the supplied unique force identifier. Returns None if no instance is
    found

    :param unique_force_identifier: Unique Force identifier of the police force.
    :type unique_force_identifier: str.
    """
    forces = get_all_forces()
    if unique_force_identifier not in forces:
        raise exceptions.ForceNotFoundException(unique_force_identifier)
    return forces[unique_force_identifier]


def get_force_by_point(point):
    """
    Find the police force for a particular area. Returns an instance of a Force object

    :param point: Point for which the responsible police force should be returned
    :type point: :mod:`geopy.Point` instance
    """
    data = service.fetch_json("locate-neighbourhood", q=point)
    f = None
    if "force" in data:
        f = get_force_by_id(data["force"])
    return f


def get_neighbourhood_by_point(point):
    """
    Find the neighbourhood policing team responsible for a particular area. Returns an instance of a Neighbourhood
    object

    :param point: Point for which the responsible police force should be returned
    :type point: :mod:`geopy.Point` instance
    """
    data = service.fetch_json("locate-neighbourhood", q=point)
    neighbourhood = None
    if "force" in data:
        f = get_force_by_id(data["force"])
        neighbourhood = f.get_neighbourhood_by_id(data["neighbourhood"])
    return neighbourhood


__forces = {}

