#    This file is part of unforward
# 
#    Unforward is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#    
#    Unforward is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#    
#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
#

import email
import email.parser
import email.message
import email.utils
import unforward.emailclients
import quopri
from unforward.errors import NoForwardFoundError
from pprint import pprint

class Extractor:
    """ This is the main class which handles extracting the original email from a chain of forwarded emails."""
    def __init__(self):
        self.parser = email.parser.Parser()
        # Grab the list of clients from the emailclients module
        self.clients = [client() for client in unforward.emailclients.clients]
    
    def get_client(self, name):
        for client in self.clients:
            if client.client_type == name:
                return client
        return None
    
    def extract(self, email_text):
        """ Assuming that email_text is a mime-encoded email message containing a forward,
            this will extract the original body email and sender, and return them in a
            a dictionary. """
        
        parsed_email = self.parser.parsestr(email_text)
        
        def process_payload(payloads):
            if isinstance(payloads, list):
                for payload in payloads:
                    if payload.get('content-type').startswith('text/html'):
                        payload_string = quopri.decodestring(payload.get_payload().encode('utf8'))
                        return self.extract_from_html(payload_string)
                    elif payload.get('content-type').startswith('multipart/alternative'):
                        return process_payload(payload.get_payload())
                raise NotImplementedError("Did not find any payloads with text/html.")
            else:
                payload_string = quopri.decodestring(payloads)
                return self.extract_from_html(payload_string)
        
        return process_payload(parsed_email.get_payload())
        
        
        
    def extract_from_html(self, email_html):
        """ Assuming that email_html is the raw html extracted from an email message,
            this method will attempt to determine if it contains a forwarded message,
            and if so, it will extract the original message. """
        for client in self.clients:
            # Use the first client which indicates that it matches the email html
            try:
                if client.check_match(email_html):
                    extract_result = client.extract_from_html(email_html)
                    # See if there is another layer of forwarding
                    try:
                        return self.extract_from_html(extract_result['body'])
                    except NoForwardFoundError:
                        # Additional layer was not found.
                        return extract_result
            except NotImplementedError:
                pass # Ignore unimplemented clients
        raise NoForwardFoundError()

