'''
Module unicode_tex provides the map between unicode 
characters and latex commands.

To convert a unicode character u to latex commands:

    unicode_tex.unicode_to_tex_map.get(u, u)

To convert a latex command l to unicode characters:

    unicode_tex.tex_to_unicode_map.get(l, l)

Both unicode_to_tex_map and tex_to_unicode_map are dict
objects, generated from http://www.w3.org/Math/characters/unicode.xml .

The code is a python version of http://www.johndcook.com/data.js .
'''

unicode_to_tex_map = {u' ': u'\\space',
    u'#': u'\\#',
    u'$': u'\\textdollar',
    u'%': u'\\%',
    u'&': u'\\&',
    u"'": u'\\textquotesingle',
    u'*': u'\\ast',
    u'\\': u'\\textbackslash',
    u'^': u'\\^{}',
    u'_': u'\\_',
    u'`': u'\\textasciigrave',
    u'{': u'\\lbrace',
    u'|': u'\\vert',
    u'}': u'\\rbrace',
    u'~': u'\\textasciitilde',
    u'\xa0': u'~',
    u'\xa1': u'\\textexclamdown',
    u'\xa2': u'\\textcent',
    u'\xa3': u'\\textsterling',
    u'\xa4': u'\\textcurrency',
    u'\xa5': u'\\textyen',
    u'\xa6': u'\\textbrokenbar',
    u'\xa7': u'\\textsection',
    u'\xa8': u'\\textasciidieresis',
    u'\xa9': u'\\textcopyright',
    u'\xaa': u'\\textordfeminine',
    u'\xab': u'\\guillemotleft',
    u'\xac': u'\\lnot',
    u'\xad': u'\\-',
    u'\xae': u'\\textregistered',
    u'\xaf': u'\\textasciimacron',
    u'\xb0': u'\\textdegree',
    u'\xb1': u'\\pm',
    u'\xb2': u'{^2}',
    u'\xb3': u'{^3}',
    u'\xb4': u'\\textasciiacute',
    u'\xb5': u'\\mathrm{\\mu}',
    u'\xb6': u'\\textparagraph',
    u'\xb7': u'\\cdot',
    u'\xb8': u'\\c{}',
    u'\xb9': u'{^1}',
    u'\xba': u'\\textordmasculine',
    u'\xbb': u'\\guillemotright',
    u'\xbc': u'\\textonequarter',
    u'\xbd': u'\\textonehalf',
    u'\xbe': u'\\textthreequarters',
    u'\xbf': u'\\textquestiondown',
    u'\xc0': u'\\`{A}',
    u'\xc1': u"\\'{A}",
    u'\xc2': u'\\^{A}',
    u'\xc3': u'\\~{A}',
    u'\xc4': u'\\"{A}',
    u'\xc5': u'\\AA',
    u'\xc6': u'\\AE',
    u'\xc7': u'\\c{C}',
    u'\xc8': u'\\`{E}',
    u'\xc9': u"\\'{E}",
    u'\xca': u'\\^{E}',
    u'\xcb': u'\\"{E}',
    u'\xcc': u'\\`{I}',
    u'\xcd': u"\\'{I}",
    u'\xce': u'\\^{I}',
    u'\xcf': u'\\"{I}',
    u'\xd0': u'\\DH',
    u'\xd1': u'\\~{N}',
    u'\xd2': u'\\`{O}',
    u'\xd3': u"\\'{O}",
    u'\xd4': u'\\^{O}',
    u'\xd5': u'\\~{O}',
    u'\xd6': u'\\"{O}',
    u'\xd7': u'\\texttimes',
    u'\xd8': u'\\O',
    u'\xd9': u'\\`{U}',
    u'\xda': u"\\'{U}",
    u'\xdb': u'\\^{U}',
    u'\xdc': u'\\"{U}',
    u'\xdd': u"\\'{Y}",
    u'\xde': u'\\TH',
    u'\xdf': u'\\ss',
    u'\xe0': u'\\`{a}',
    u'\xe1': u"\\'{a}",
    u'\xe2': u'\\^{a}',
    u'\xe3': u'\\~{a}',
    u'\xe4': u'\\"{a}',
    u'\xe5': u'\\aa',
    u'\xe6': u'\\ae',
    u'\xe7': u'\\c{c}',
    u'\xe8': u'\\`{e}',
    u'\xe9': u"\\'{e}",
    u'\xea': u'\\^{e}',
    u'\xeb': u'\\"{e}',
    u'\xec': u'\\`{\\i}',
    u'\xed': u"\\'{\\i}",
    u'\xee': u'\\^{\\i}',
    u'\xef': u'\\"{\\i}',
    u'\xf0': u'\\dh',
    u'\xf1': u'\\~{n}',
    u'\xf2': u'\\`{o}',
    u'\xf3': u"\\'{o}",
    u'\xf4': u'\\^{o}',
    u'\xf5': u'\\~{o}',
    u'\xf6': u'\\"{o}',
    u'\xf7': u'\\div',
    u'\xf8': u'\\o',
    u'\xf9': u'\\`{u}',
    u'\xfa': u"\\'{u}",
    u'\xfb': u'\\^{u}',
    u'\xfc': u'\\"{u}',
    u'\xfd': u"\\'{y}",
    u'\xfe': u'\\th',
    u'\xff': u'\\"{y}',
    u'\u0100': u'\\={A}',
    u'\u0101': u'\\={a}',
    u'\u0102': u'\\u{A}',
    u'\u0103': u'\\u{a}',
    u'\u0104': u'\\k{A}',
    u'\u0105': u'\\k{a}',
    u'\u0106': u"\\'{C}",
    u'\u0107': u"\\'{c}",
    u'\u0108': u'\\^{C}',
    u'\u0109': u'\\^{c}',
    u'\u010a': u'\\.{C}',
    u'\u010b': u'\\.{c}',
    u'\u010c': u'\\v{C}',
    u'\u010d': u'\\v{c}',
    u'\u010e': u'\\v{D}',
    u'\u010f': u'\\v{d}',
    u'\u0110': u'\\DJ',
    u'\u0111': u'\\dj',
    u'\u0112': u'\\={E}',
    u'\u0113': u'\\={e}',
    u'\u0114': u'\\u{E}',
    u'\u0115': u'\\u{e}',
    u'\u0116': u'\\.{E}',
    u'\u0117': u'\\.{e}',
    u'\u0118': u'\\k{E}',
    u'\u0119': u'\\k{e}',
    u'\u011a': u'\\v{E}',
    u'\u011b': u'\\v{e}',
    u'\u011c': u'\\^{G}',
    u'\u011d': u'\\^{g}',
    u'\u011e': u'\\u{G}',
    u'\u011f': u'\\u{g}',
    u'\u0120': u'\\.{G}',
    u'\u0121': u'\\.{g}',
    u'\u0122': u'\\c{G}',
    u'\u0123': u'\\c{g}',
    u'\u0124': u'\\^{H}',
    u'\u0125': u'\\^{h}',
    u'\u0126': u'{\\fontencoding{LELA}\\selectfont\\char40}',
    u'\u0127': u'\\Elzxh',
    u'\u0128': u'\\~{I}',
    u'\u0129': u'\\~{\\i}',
    u'\u012a': u'\\={I}',
    u'\u012b': u'\\={\\i}',
    u'\u012c': u'\\u{I}',
    u'\u012d': u'\\u{\\i}',
    u'\u012e': u'\\k{I}',
    u'\u012f': u'\\k{i}',
    u'\u0130': u'\\.{I}',
    u'\u0131': u'\\i',
    u'\u0132': u'IJ',
    u'\u0133': u'ij',
    u'\u0134': u'\\^{J}',
    u'\u0135': u'\\^{\\j}',
    u'\u0136': u'\\c{K}',
    u'\u0137': u'\\c{k}',
    u'\u0138': u'{\\fontencoding{LELA}\\selectfont\\char91}',
    u'\u0139': u"\\'{L}",
    u'\u013a': u"\\'{l}",
    u'\u013b': u'\\c{L}',
    u'\u013c': u'\\c{l}',
    u'\u013d': u'\\v{L}',
    u'\u013e': u'\\v{l}',
    u'\u013f': u'{\\fontencoding{LELA}\\selectfont\\char201}',
    u'\u0140': u'{\\fontencoding{LELA}\\selectfont\\char202}',
    u'\u0141': u'\\L',
    u'\u0142': u'\\l',
    u'\u0143': u"\\'{N}",
    u'\u0144': u"\\'{n}",
    u'\u0145': u'\\c{N}',
    u'\u0146': u'\\c{n}',
    u'\u0147': u'\\v{N}',
    u'\u0148': u'\\v{n}',
    u'\u0149': u"'n",
    u'\u014a': u'\\NG',
    u'\u014b': u'\\ng',
    u'\u014c': u'\\={O}',
    u'\u014d': u'\\={o}',
    u'\u014e': u'\\u{O}',
    u'\u014f': u'\\u{o}',
    u'\u0150': u'\\H{O}',
    u'\u0151': u'\\H{o}',
    u'\u0152': u'\\OE',
    u'\u0153': u'\\oe',
    u'\u0154': u"\\'{R}",
    u'\u0155': u"\\'{r}",
    u'\u0156': u'\\c{R}',
    u'\u0157': u'\\c{r}',
    u'\u0158': u'\\v{R}',
    u'\u0159': u'\\v{r}',
    u'\u015a': u"\\'{S}",
    u'\u015b': u"\\'{s}",
    u'\u015c': u'\\^{S}',
    u'\u015d': u'\\^{s}',
    u'\u015e': u'\\c{S}',
    u'\u015f': u'\\c{s}',
    u'\u0160': u'\\v{S}',
    u'\u0161': u'\\v{s}',
    u'\u0162': u'\\c{T}',
    u'\u0163': u'\\c{t}',
    u'\u0164': u'\\v{T}',
    u'\u0165': u'\\v{t}',
    u'\u0166': u'{\\fontencoding{LELA}\\selectfont\\char47}',
    u'\u0167': u'{\\fontencoding{LELA}\\selectfont\\char63}',
    u'\u0168': u'\\~{U}',
    u'\u0169': u'\\~{u}',
    u'\u016a': u'\\={U}',
    u'\u016b': u'\\={u}',
    u'\u016c': u'\\u{U}',
    u'\u016d': u'\\u{u}',
    u'\u016e': u'\\r{U}',
    u'\u016f': u'\\r{u}',
    u'\u0170': u'\\H{U}',
    u'\u0171': u'\\H{u}',
    u'\u0172': u'\\k{U}',
    u'\u0173': u'\\k{u}',
    u'\u0174': u'\\^{W}',
    u'\u0175': u'\\^{w}',
    u'\u0176': u'\\^{Y}',
    u'\u0177': u'\\^{y}',
    u'\u0178': u'\\"{Y}',
    u'\u0179': u"\\'{Z}",
    u'\u017a': u"\\'{z}",
    u'\u017b': u'\\.{Z}',
    u'\u017c': u'\\.{z}',
    u'\u017d': u'\\v{Z}',
    u'\u017e': u'\\v{z}',
    u'\u0192': u'f',
    u'\u0195': u'\\texthvlig',
    u'\u019e': u'\\textnrleg',
    u'\u01aa': u'\\eth',
    u'\u01ba': u'{\\fontencoding{LELA}\\selectfont\\char195}',
    u'\u01c2': u'\\textdoublepipe',
    u'\u01f5': u"\\'{g}",
    u'\u0250': u'\\Elztrna',
    u'\u0252': u'\\Elztrnsa',
    u'\u0254': u'\\Elzopeno',
    u'\u0256': u'\\Elzrtld',
    u'\u0258': u'{\\fontencoding{LEIP}\\selectfont\\char61}',
    u'\u0259': u'\\Elzschwa',
    u'\u025b': u'\\varepsilon',
    u'\u0261': u'g',
    u'\u0263': u'\\Elzpgamma',
    u'\u0264': u'\\Elzpbgam',
    u'\u0265': u'\\Elztrnh',
    u'\u026c': u'\\Elzbtdl',
    u'\u026d': u'\\Elzrtll',
    u'\u026f': u'\\Elztrnm',
    u'\u0270': u'\\Elztrnmlr',
    u'\u0271': u'\\Elzltlmr',
    u'\u0272': u'\\Elzltln',
    u'\u0273': u'\\Elzrtln',
    u'\u0277': u'\\Elzclomeg',
    u'\u0278': u'\\textphi',
    u'\u0279': u'\\Elztrnr',
    u'\u027a': u'\\Elztrnrl',
    u'\u027b': u'\\Elzrttrnr',
    u'\u027c': u'\\Elzrl',
    u'\u027d': u'\\Elzrtlr',
    u'\u027e': u'\\Elzfhr',
    u'\u027f': u'{\\fontencoding{LEIP}\\selectfont\\char202}',
    u'\u0282': u'\\Elzrtls',
    u'\u0283': u'\\Elzesh',
    u'\u0287': u'\\Elztrnt',
    u'\u0288': u'\\Elzrtlt',
    u'\u028a': u'\\Elzpupsil',
    u'\u028b': u'\\Elzpscrv',
    u'\u028c': u'\\Elzinvv',
    u'\u028d': u'\\Elzinvw',
    u'\u028e': u'\\Elztrny',
    u'\u0290': u'\\Elzrtlz',
    u'\u0292': u'\\Elzyogh',
    u'\u0294': u'\\Elzglst',
    u'\u0295': u'\\Elzreglst',
    u'\u0296': u'\\Elzinglst',
    u'\u029e': u'\\textturnk',
    u'\u02a4': u'\\Elzdyogh',
    u'\u02a7': u'\\Elztesh',
    u'\u02bc': u"'",
    u'\u02c7': u'\\textasciicaron',
    u'\u02c8': u'\\Elzverts',
    u'\u02cc': u'\\Elzverti',
    u'\u02d0': u'\\Elzlmrk',
    u'\u02d1': u'\\Elzhlmrk',
    u'\u02d2': u'\\Elzsbrhr',
    u'\u02d3': u'\\Elzsblhr',
    u'\u02d4': u'\\Elzrais',
    u'\u02d5': u'\\Elzlow',
    u'\u02d8': u'\\textasciibreve',
    u'\u02d9': u'\\textperiodcentered',
    u'\u02da': u'\\r{}',
    u'\u02db': u'\\k{}',
    u'\u02dc': u'\\texttildelow',
    u'\u02dd': u'\\H{}',
    u'\u02e5': u'\\tone{55}',
    u'\u02e6': u'\\tone{44}',
    u'\u02e7': u'\\tone{33}',
    u'\u02e8': u'\\tone{22}',
    u'\u02e9': u'\\tone{11}',
    u'\u0300': u'\\`',
    u'\u0301': u"\\'",
    u'\u0302': u'\\^',
    u'\u0303': u'\\~',
    u'\u0304': u'\\=',
    u'\u0306': u'\\u',
    u'\u0307': u'\\.',
    u'\u0308': u'\\"',
    u'\u030a': u'\\r',
    u'\u030b': u'\\H',
    u'\u030c': u'\\v',
    u'\u030f': u'\\cyrchar\\C',
    u'\u0311': u'{\\fontencoding{LECO}\\selectfont\\char177}',
    u'\u0318': u'{\\fontencoding{LECO}\\selectfont\\char184}',
    u'\u0319': u'{\\fontencoding{LECO}\\selectfont\\char185}',
    u'\u0321': u'\\Elzpalh',
    u'\u0322': u'\\Elzrh',
    u'\u0327': u'\\c',
    u'\u0328': u'\\k',
    u'\u032a': u'\\Elzsbbrg',
    u'\u032b': u'{\\fontencoding{LECO}\\selectfont\\char203}',
    u'\u032f': u'{\\fontencoding{LECO}\\selectfont\\char207}',
    u'\u0335': u'\\Elzxl',
    u'\u0336': u'\\Elzbar',
    u'\u0337': u'{\\fontencoding{LECO}\\selectfont\\char215}',
    u'\u0338': u'{\\fontencoding{LECO}\\selectfont\\char216}',
    u'\u033a': u'{\\fontencoding{LECO}\\selectfont\\char218}',
    u'\u033b': u'{\\fontencoding{LECO}\\selectfont\\char219}',
    u'\u033c': u'{\\fontencoding{LECO}\\selectfont\\char220}',
    u'\u033d': u'{\\fontencoding{LECO}\\selectfont\\char221}',
    u'\u0361': u'{\\fontencoding{LECO}\\selectfont\\char225}',
    u'\u0386': u"\\'{A}",
    u'\u0388': u"\\'{E}",
    u'\u0389': u"\\'{H}",
    u'\u038a': u"\\'{}{I}",
    u'\u038c': u"\\'{}O",
    u'\u038e': u"\\mathrm{'Y}",
    u'\u038f': u"\\mathrm{'\\Omega}",
    u'\u0390': u'\\acute{\\ddot{\\iota}}',
    u'\u0391': u'\\Alpha',
    u'\u0392': u'\\Beta',
    u'\u0393': u'\\Gamma',
    u'\u0394': u'\\Delta',
    u'\u0395': u'\\Epsilon',
    u'\u0396': u'\\Zeta',
    u'\u0397': u'\\Eta',
    u'\u0398': u'\\Theta',
    u'\u0399': u'\\Iota',
    u'\u039a': u'\\Kappa',
    u'\u039b': u'\\Lambda',
    u'\u039c': u'M',
    u'\u039d': u'N',
    u'\u039e': u'\\Xi',
    u'\u039f': u'O',
    u'\u03a0': u'\\Pi',
    u'\u03a1': u'\\Rho',
    u'\u03a3': u'\\Sigma',
    u'\u03a4': u'\\Tau',
    u'\u03a5': u'\\Upsilon',
    u'\u03a6': u'\\Phi',
    u'\u03a7': u'\\Chi',
    u'\u03a8': u'\\Psi',
    u'\u03a9': u'\\Omega',
    u'\u03aa': u'\\mathrm{\\ddot{I}}',
    u'\u03ab': u'\\mathrm{\\ddot{Y}}',
    u'\u03ac': u"\\'{$\\alpha$}",
    u'\u03ad': u'\\acute{\\epsilon}',
    u'\u03ae': u'\\acute{\\eta}',
    u'\u03af': u'\\acute{\\iota}',
    u'\u03b0': u'\\acute{\\ddot{\\upsilon}}',
    u'\u03b1': u'\\alpha',
    u'\u03b2': u'\\beta',
    u'\u03b3': u'\\gamma',
    u'\u03b4': u'\\delta',
    u'\u03b5': u'\\epsilon',
    u'\u03b6': u'\\zeta',
    u'\u03b7': u'\\eta',
    u'\u03b8': u'\\texttheta',
    u'\u03b9': u'\\iota',
    u'\u03ba': u'\\kappa',
    u'\u03bb': u'\\lambda',
    u'\u03bc': u'\\mu',
    u'\u03bd': u'\\nu',
    u'\u03be': u'\\xi',
    u'\u03bf': u'o',
    u'\u03c0': u'\\pi',
    u'\u03c1': u'\\rho',
    u'\u03c2': u'\\varsigma',
    u'\u03c3': u'\\sigma',
    u'\u03c4': u'\\tau',
    u'\u03c5': u'\\upsilon',
    u'\u03c6': u'\\varphi',
    u'\u03c7': u'\\chi',
    u'\u03c8': u'\\psi',
    u'\u03c9': u'\\omega',
    u'\u03ca': u'\\ddot{\\iota}',
    u'\u03cb': u'\\ddot{\\upsilon}',
    u'\u03cc': u"\\'{o}",
    u'\u03cd': u'\\acute{\\upsilon}',
    u'\u03ce': u'\\acute{\\omega}',
    u'\u03d0': u'\\Pisymbol{ppi022}{87}',
    u'\u03d1': u'\\textvartheta',
    u'\u03d2': u'\\Upsilon',
    u'\u03d5': u'\\phi',
    u'\u03d6': u'\\varpi',
    u'\u03da': u'\\Stigma',
    u'\u03dc': u'\\Digamma',
    u'\u03dd': u'\\digamma',
    u'\u03de': u'\\Koppa',
    u'\u03e0': u'\\Sampi',
    u'\u03f0': u'\\varkappa',
    u'\u03f1': u'\\varrho',
    u'\u03f4': u'\\textTheta',
    u'\u03f6': u'\\backepsilon',
    u'\u0401': u'\\cyrchar\\CYRYO',
    u'\u0402': u'\\cyrchar\\CYRDJE',
    u'\u0403': u"\\cyrchar{\\'\\CYRG}",
    u'\u0404': u'\\cyrchar\\CYRIE',
    u'\u0405': u'\\cyrchar\\CYRDZE',
    u'\u0406': u'\\cyrchar\\CYRII',
    u'\u0407': u'\\cyrchar\\CYRYI',
    u'\u0408': u'\\cyrchar\\CYRJE',
    u'\u0409': u'\\cyrchar\\CYRLJE',
    u'\u040a': u'\\cyrchar\\CYRNJE',
    u'\u040b': u'\\cyrchar\\CYRTSHE',
    u'\u040c': u"\\cyrchar{\\'\\CYRK}",
    u'\u040e': u'\\cyrchar\\CYRUSHRT',
    u'\u040f': u'\\cyrchar\\CYRDZHE',
    u'\u0410': u'\\cyrchar\\CYRA',
    u'\u0411': u'\\cyrchar\\CYRB',
    u'\u0412': u'\\cyrchar\\CYRV',
    u'\u0413': u'\\cyrchar\\CYRG',
    u'\u0414': u'\\cyrchar\\CYRD',
    u'\u0415': u'\\cyrchar\\CYRE',
    u'\u0416': u'\\cyrchar\\CYRZH',
    u'\u0417': u'\\cyrchar\\CYRZ',
    u'\u0418': u'\\cyrchar\\CYRI',
    u'\u0419': u'\\cyrchar\\CYRISHRT',
    u'\u041a': u'\\cyrchar\\CYRK',
    u'\u041b': u'\\cyrchar\\CYRL',
    u'\u041c': u'\\cyrchar\\CYRM',
    u'\u041d': u'\\cyrchar\\CYRN',
    u'\u041e': u'\\cyrchar\\CYRO',
    u'\u041f': u'\\cyrchar\\CYRP',
    u'\u0420': u'\\cyrchar\\CYRR',
    u'\u0421': u'\\cyrchar\\CYRS',
    u'\u0422': u'\\cyrchar\\CYRT',
    u'\u0423': u'\\cyrchar\\CYRU',
    u'\u0424': u'\\cyrchar\\CYRF',
    u'\u0425': u'\\cyrchar\\CYRH',
    u'\u0426': u'\\cyrchar\\CYRC',
    u'\u0427': u'\\cyrchar\\CYRCH',
    u'\u0428': u'\\cyrchar\\CYRSH',
    u'\u0429': u'\\cyrchar\\CYRSHCH',
    u'\u042a': u'\\cyrchar\\CYRHRDSN',
    u'\u042b': u'\\cyrchar\\CYRERY',
    u'\u042c': u'\\cyrchar\\CYRSFTSN',
    u'\u042d': u'\\cyrchar\\CYREREV',
    u'\u042e': u'\\cyrchar\\CYRYU',
    u'\u042f': u'\\cyrchar\\CYRYA',
    u'\u0430': u'\\cyrchar\\cyra',
    u'\u0431': u'\\cyrchar\\cyrb',
    u'\u0432': u'\\cyrchar\\cyrv',
    u'\u0433': u'\\cyrchar\\cyrg',
    u'\u0434': u'\\cyrchar\\cyrd',
    u'\u0435': u'\\cyrchar\\cyre',
    u'\u0436': u'\\cyrchar\\cyrzh',
    u'\u0437': u'\\cyrchar\\cyrz',
    u'\u0438': u'\\cyrchar\\cyri',
    u'\u0439': u'\\cyrchar\\cyrishrt',
    u'\u043a': u'\\cyrchar\\cyrk',
    u'\u043b': u'\\cyrchar\\cyrl',
    u'\u043c': u'\\cyrchar\\cyrm',
    u'\u043d': u'\\cyrchar\\cyrn',
    u'\u043e': u'\\cyrchar\\cyro',
    u'\u043f': u'\\cyrchar\\cyrp',
    u'\u0440': u'\\cyrchar\\cyrr',
    u'\u0441': u'\\cyrchar\\cyrs',
    u'\u0442': u'\\cyrchar\\cyrt',
    u'\u0443': u'\\cyrchar\\cyru',
    u'\u0444': u'\\cyrchar\\cyrf',
    u'\u0445': u'\\cyrchar\\cyrh',
    u'\u0446': u'\\cyrchar\\cyrc',
    u'\u0447': u'\\cyrchar\\cyrch',
    u'\u0448': u'\\cyrchar\\cyrsh',
    u'\u0449': u'\\cyrchar\\cyrshch',
    u'\u044a': u'\\cyrchar\\cyrhrdsn',
    u'\u044b': u'\\cyrchar\\cyrery',
    u'\u044c': u'\\cyrchar\\cyrsftsn',
    u'\u044d': u'\\cyrchar\\cyrerev',
    u'\u044e': u'\\cyrchar\\cyryu',
    u'\u044f': u'\\cyrchar\\cyrya',
    u'\u0451': u'\\cyrchar\\cyryo',
    u'\u0452': u'\\cyrchar\\cyrdje',
    u'\u0453': u"\\cyrchar{\\'\\cyrg}",
    u'\u0454': u'\\cyrchar\\cyrie',
    u'\u0455': u'\\cyrchar\\cyrdze',
    u'\u0456': u'\\cyrchar\\cyrii',
    u'\u0457': u'\\cyrchar\\cyryi',
    u'\u0458': u'\\cyrchar\\cyrje',
    u'\u0459': u'\\cyrchar\\cyrlje',
    u'\u045a': u'\\cyrchar\\cyrnje',
    u'\u045b': u'\\cyrchar\\cyrtshe',
    u'\u045c': u"\\cyrchar{\\'\\cyrk}",
    u'\u045e': u'\\cyrchar\\cyrushrt',
    u'\u045f': u'\\cyrchar\\cyrdzhe',
    u'\u0460': u'\\cyrchar\\CYROMEGA',
    u'\u0461': u'\\cyrchar\\cyromega',
    u'\u0462': u'\\cyrchar\\CYRYAT',
    u'\u0464': u'\\cyrchar\\CYRIOTE',
    u'\u0465': u'\\cyrchar\\cyriote',
    u'\u0466': u'\\cyrchar\\CYRLYUS',
    u'\u0467': u'\\cyrchar\\cyrlyus',
    u'\u0468': u'\\cyrchar\\CYRIOTLYUS',
    u'\u0469': u'\\cyrchar\\cyriotlyus',
    u'\u046a': u'\\cyrchar\\CYRBYUS',
    u'\u046c': u'\\cyrchar\\CYRIOTBYUS',
    u'\u046d': u'\\cyrchar\\cyriotbyus',
    u'\u046e': u'\\cyrchar\\CYRKSI',
    u'\u046f': u'\\cyrchar\\cyrksi',
    u'\u0470': u'\\cyrchar\\CYRPSI',
    u'\u0471': u'\\cyrchar\\cyrpsi',
    u'\u0472': u'\\cyrchar\\CYRFITA',
    u'\u0474': u'\\cyrchar\\CYRIZH',
    u'\u0478': u'\\cyrchar\\CYRUK',
    u'\u0479': u'\\cyrchar\\cyruk',
    u'\u047a': u'\\cyrchar\\CYROMEGARND',
    u'\u047b': u'\\cyrchar\\cyromegarnd',
    u'\u047c': u'\\cyrchar\\CYROMEGATITLO',
    u'\u047d': u'\\cyrchar\\cyromegatitlo',
    u'\u047e': u'\\cyrchar\\CYROT',
    u'\u047f': u'\\cyrchar\\cyrot',
    u'\u0480': u'\\cyrchar\\CYRKOPPA',
    u'\u0481': u'\\cyrchar\\cyrkoppa',
    u'\u0482': u'\\cyrchar\\cyrthousands',
    u'\u0488': u'\\cyrchar\\cyrhundredthousands',
    u'\u0489': u'\\cyrchar\\cyrmillions',
    u'\u048c': u'\\cyrchar\\CYRSEMISFTSN',
    u'\u048d': u'\\cyrchar\\cyrsemisftsn',
    u'\u048e': u'\\cyrchar\\CYRRTICK',
    u'\u048f': u'\\cyrchar\\cyrrtick',
    u'\u0490': u'\\cyrchar\\CYRGUP',
    u'\u0491': u'\\cyrchar\\cyrgup',
    u'\u0492': u'\\cyrchar\\CYRGHCRS',
    u'\u0493': u'\\cyrchar\\cyrghcrs',
    u'\u0494': u'\\cyrchar\\CYRGHK',
    u'\u0495': u'\\cyrchar\\cyrghk',
    u'\u0496': u'\\cyrchar\\CYRZHDSC',
    u'\u0497': u'\\cyrchar\\cyrzhdsc',
    u'\u0498': u'\\cyrchar\\CYRZDSC',
    u'\u0499': u'\\cyrchar\\cyrzdsc',
    u'\u049a': u'\\cyrchar\\CYRKDSC',
    u'\u049b': u'\\cyrchar\\cyrkdsc',
    u'\u049c': u'\\cyrchar\\CYRKVCRS',
    u'\u049d': u'\\cyrchar\\cyrkvcrs',
    u'\u049e': u'\\cyrchar\\CYRKHCRS',
    u'\u049f': u'\\cyrchar\\cyrkhcrs',
    u'\u04a0': u'\\cyrchar\\CYRKBEAK',
    u'\u04a1': u'\\cyrchar\\cyrkbeak',
    u'\u04a2': u'\\cyrchar\\CYRNDSC',
    u'\u04a3': u'\\cyrchar\\cyrndsc',
    u'\u04a4': u'\\cyrchar\\CYRNG',
    u'\u04a5': u'\\cyrchar\\cyrng',
    u'\u04a6': u'\\cyrchar\\CYRPHK',
    u'\u04a7': u'\\cyrchar\\cyrphk',
    u'\u04a8': u'\\cyrchar\\CYRABHHA',
    u'\u04a9': u'\\cyrchar\\cyrabhha',
    u'\u04aa': u'\\cyrchar\\CYRSDSC',
    u'\u04ab': u'\\cyrchar\\cyrsdsc',
    u'\u04ac': u'\\cyrchar\\CYRTDSC',
    u'\u04ad': u'\\cyrchar\\cyrtdsc',
    u'\u04ae': u'\\cyrchar\\CYRY',
    u'\u04af': u'\\cyrchar\\cyry',
    u'\u04b0': u'\\cyrchar\\CYRYHCRS',
    u'\u04b1': u'\\cyrchar\\cyryhcrs',
    u'\u04b2': u'\\cyrchar\\CYRHDSC',
    u'\u04b3': u'\\cyrchar\\cyrhdsc',
    u'\u04b4': u'\\cyrchar\\CYRTETSE',
    u'\u04b5': u'\\cyrchar\\cyrtetse',
    u'\u04b6': u'\\cyrchar\\CYRCHRDSC',
    u'\u04b7': u'\\cyrchar\\cyrchrdsc',
    u'\u04b8': u'\\cyrchar\\CYRCHVCRS',
    u'\u04b9': u'\\cyrchar\\cyrchvcrs',
    u'\u04ba': u'\\cyrchar\\CYRSHHA',
    u'\u04bb': u'\\cyrchar\\cyrshha',
    u'\u04bc': u'\\cyrchar\\CYRABHCH',
    u'\u04bd': u'\\cyrchar\\cyrabhch',
    u'\u04be': u'\\cyrchar\\CYRABHCHDSC',
    u'\u04bf': u'\\cyrchar\\cyrabhchdsc',
    u'\u04c0': u'\\cyrchar\\CYRpalochka',
    u'\u04c3': u'\\cyrchar\\CYRKHK',
    u'\u04c4': u'\\cyrchar\\cyrkhk',
    u'\u04c7': u'\\cyrchar\\CYRNHK',
    u'\u04c8': u'\\cyrchar\\cyrnhk',
    u'\u04cb': u'\\cyrchar\\CYRCHLDSC',
    u'\u04cc': u'\\cyrchar\\cyrchldsc',
    u'\u04d4': u'\\cyrchar\\CYRAE',
    u'\u04d5': u'\\cyrchar\\cyrae',
    u'\u04d8': u'\\cyrchar\\CYRSCHWA',
    u'\u04d9': u'\\cyrchar\\cyrschwa',
    u'\u04e0': u'\\cyrchar\\CYRABHDZE',
    u'\u04e1': u'\\cyrchar\\cyrabhdze',
    u'\u04e8': u'\\cyrchar\\CYROTLD',
    u'\u04e9': u'\\cyrchar\\cyrotld',
    u'\u2002': u'\\hspace{0.6em}',
    u'\u2003': u'\\hspace{1em}',
    u'\u2004': u'\\hspace{0.33em}',
    u'\u2005': u'\\hspace{0.25em}',
    u'\u2006': u'\\hspace{0.166em}',
    u'\u2007': u'\\hphantom{0}',
    u'\u2008': u'\\hphantom{,}',
    u'\u2009': u'\\hspace{0.167em}',
    u'\u2009\u200a\u200a': u'\\;',
    u'\u200a': u'\\mkern1mu',
    u'\u2010': u'-',
    u'\u2013': u'\\textendash',
    u'\u2014': u'\\textemdash',
    u'\u2015': u'\\rule{1em}{1pt}',
    u'\u2016': u'\\Vert',
    u'\u2018': u'`',
    u'\u2019': u"'",
    u'\u201a': u',',
    u'\u201b': u'\\Elzreapos',
    u'\u201c': u'\\textquotedblleft',
    u'\u201d': u'\\textquotedblright',
    u'\u201e': u',,',
    u'\u2020': u'\\textdagger',
    u'\u2021': u'\\textdaggerdbl',
    u'\u2022': u'\\textbullet',
    u'\u2024': u'.',
    u'\u2025': u'..',
    u'\u2026': u'\\ldots',
    u'\u2030': u'\\textperthousand',
    u'\u2031': u'\\textpertenthousand',
    u'\u2032': u"{'}",
    u'\u2033': u"{''}",
    u'\u2034': u"{'''}",
    u'\u2035': u'\\backprime',
    u'\u2039': u'\\guilsinglleft',
    u'\u203a': u'\\guilsinglright',
    u'\u2057': u"''''",
    u'\u205f': u'\\mkern4mu',
    u'\u2060': u'\\nolinebreak',
    u'\u20a7': u'\\ensuremath{\\Elzpes}',
    u'\u20ac': u'\\mbox{\\texteuro}',
    u'\u20db': u'\\dddot',
    u'\u20dc': u'\\ddddot',
    u'\u2102': u'\\mathbb{C}',
    u'\u210a': u'\\mathscr{g}',
    u'\u210b': u'\\mathscr{H}',
    u'\u210c': u'\\mathfrak{H}',
    u'\u210d': u'\\mathbb{H}',
    u'\u210f': u'\\hslash',
    u'\u2110': u'\\mathscr{I}',
    u'\u2111': u'\\mathfrak{I}',
    u'\u2112': u'\\mathscr{L}',
    u'\u2113': u'\\mathscr{l}',
    u'\u2115': u'\\mathbb{N}',
    u'\u2116': u'\\cyrchar\\textnumero',
    u'\u2118': u'\\wp',
    u'\u2119': u'\\mathbb{P}',
    u'\u211a': u'\\mathbb{Q}',
    u'\u211b': u'\\mathscr{R}',
    u'\u211c': u'\\mathfrak{R}',
    u'\u211d': u'\\mathbb{R}',
    u'\u211e': u'\\Elzxrat',
    u'\u2122': u'\\texttrademark',
    u'\u2124': u'\\mathbb{Z}',
    u'\u2126': u'\\Omega',
    u'\u2127': u'\\mho',
    u'\u2128': u'\\mathfrak{Z}',
    u'\u2129': u'\\ElsevierGlyph{2129}',
    u'\u212b': u'\\AA',
    u'\u212c': u'\\mathscr{B}',
    u'\u212d': u'\\mathfrak{C}',
    u'\u212f': u'\\mathscr{e}',
    u'\u2130': u'\\mathscr{E}',
    u'\u2131': u'\\mathscr{F}',
    u'\u2133': u'\\mathscr{M}',
    u'\u2134': u'\\mathscr{o}',
    u'\u2135': u'\\aleph',
    u'\u2136': u'\\beth',
    u'\u2137': u'\\gimel',
    u'\u2138': u'\\daleth',
    u'\u2153': u'\\textfrac{1}{3}',
    u'\u2154': u'\\textfrac{2}{3}',
    u'\u2155': u'\\textfrac{1}{5}',
    u'\u2156': u'\\textfrac{2}{5}',
    u'\u2157': u'\\textfrac{3}{5}',
    u'\u2158': u'\\textfrac{4}{5}',
    u'\u2159': u'\\textfrac{1}{6}',
    u'\u215a': u'\\textfrac{5}{6}',
    u'\u215b': u'\\textfrac{1}{8}',
    u'\u215c': u'\\textfrac{3}{8}',
    u'\u215d': u'\\textfrac{5}{8}',
    u'\u215e': u'\\textfrac{7}{8}',
    u'\u2190': u'\\leftarrow',
    u'\u2191': u'\\uparrow',
    u'\u2192': u'\\rightarrow',
    u'\u2193': u'\\downarrow',
    u'\u2194': u'\\leftrightarrow',
    u'\u2195': u'\\updownarrow',
    u'\u2196': u'\\nwarrow',
    u'\u2197': u'\\nearrow',
    u'\u2198': u'\\searrow',
    u'\u2199': u'\\swarrow',
    u'\u219a': u'\\nleftarrow',
    u'\u219b': u'\\nrightarrow',
    u'\u219c': u'\\arrowwaveright',
    u'\u219d': u'\\arrowwaveright',
    u'\u219e': u'\\twoheadleftarrow',
    u'\u21a0': u'\\twoheadrightarrow',
    u'\u21a2': u'\\leftarrowtail',
    u'\u21a3': u'\\rightarrowtail',
    u'\u21a6': u'\\mapsto',
    u'\u21a9': u'\\hookleftarrow',
    u'\u21aa': u'\\hookrightarrow',
    u'\u21ab': u'\\looparrowleft',
    u'\u21ac': u'\\looparrowright',
    u'\u21ad': u'\\leftrightsquigarrow',
    u'\u21ae': u'\\nleftrightarrow',
    u'\u21b0': u'\\Lsh',
    u'\u21b1': u'\\Rsh',
    u'\u21b3': u'\\ElsevierGlyph{21B3}',
    u'\u21b6': u'\\curvearrowleft',
    u'\u21b7': u'\\curvearrowright',
    u'\u21ba': u'\\circlearrowleft',
    u'\u21bb': u'\\circlearrowright',
    u'\u21bc': u'\\leftharpoonup',
    u'\u21bd': u'\\leftharpoondown',
    u'\u21be': u'\\upharpoonright',
    u'\u21bf': u'\\upharpoonleft',
    u'\u21c0': u'\\rightharpoonup',
    u'\u21c1': u'\\rightharpoondown',
    u'\u21c2': u'\\downharpoonright',
    u'\u21c3': u'\\downharpoonleft',
    u'\u21c4': u'\\rightleftarrows',
    u'\u21c5': u'\\dblarrowupdown',
    u'\u21c6': u'\\leftrightarrows',
    u'\u21c7': u'\\leftleftarrows',
    u'\u21c8': u'\\upuparrows',
    u'\u21c9': u'\\rightrightarrows',
    u'\u21ca': u'\\downdownarrows',
    u'\u21cb': u'\\leftrightharpoons',
    u'\u21cc': u'\\rightleftharpoons',
    u'\u21cd': u'\\nLeftarrow',
    u'\u21ce': u'\\nLeftrightarrow',
    u'\u21cf': u'\\nRightarrow',
    u'\u21d0': u'\\Leftarrow',
    u'\u21d1': u'\\Uparrow',
    u'\u21d2': u'\\Rightarrow',
    u'\u21d3': u'\\Downarrow',
    u'\u21d4': u'\\Leftrightarrow',
    u'\u21d5': u'\\Updownarrow',
    u'\u21da': u'\\Lleftarrow',
    u'\u21db': u'\\Rrightarrow',
    u'\u21dd': u'\\rightsquigarrow',
    u'\u21f5': u'\\DownArrowUpArrow',
    u'\u2200': u'\\forall',
    u'\u2201': u'\\complement',
    u'\u2202': u'\\partial',
    u'\u2203': u'\\exists',
    u'\u2204': u'\\nexists',
    u'\u2205': u'\\varnothing',
    u'\u2207': u'\\nabla',
    u'\u2208': u'\\in',
    u'\u2209': u'\\not\\in',
    u'\u220b': u'\\ni',
    u'\u220c': u'\\not\\ni',
    u'\u220f': u'\\prod',
    u'\u2210': u'\\coprod',
    u'\u2211': u'\\sum',
    u'\u2212': u'-',
    u'\u2213': u'\\mp',
    u'\u2214': u'\\dotplus',
    u'\u2216': u'\\setminus',
    u'\u2217': u'{_\\ast}',
    u'\u2218': u'\\circ',
    u'\u2219': u'\\bullet',
    u'\u221a': u'\\surd',
    u'\u221d': u'\\propto',
    u'\u221e': u'\\infty',
    u'\u221f': u'\\rightangle',
    u'\u2220': u'\\angle',
    u'\u2221': u'\\measuredangle',
    u'\u2222': u'\\sphericalangle',
    u'\u2223': u'\\mid',
    u'\u2224': u'\\nmid',
    u'\u2225': u'\\parallel',
    u'\u2226': u'\\nparallel',
    u'\u2227': u'\\wedge',
    u'\u2228': u'\\vee',
    u'\u2229': u'\\cap',
    u'\u222a': u'\\cup',
    u'\u222b': u'\\int',
    u'\u222c': u'\\int\\!\\int',
    u'\u222d': u'\\int\\!\\int\\!\\int',
    u'\u222e': u'\\oint',
    u'\u222f': u'\\surfintegral',
    u'\u2230': u'\\volintegral',
    u'\u2231': u'\\clwintegral',
    u'\u2232': u'\\ElsevierGlyph{2232}',
    u'\u2233': u'\\ElsevierGlyph{2233}',
    u'\u2234': u'\\therefore',
    u'\u2235': u'\\because',
    u'\u2237': u'\\Colon',
    u'\u2238': u'\\ElsevierGlyph{2238}',
    u'\u223a': u'\\mathbin{{:}\\!\\!{-}\\!\\!{:}}',
    u'\u223b': u'\\homothetic',
    u'\u223c': u'\\sim',
    u'\u223d': u'\\backsim',
    u'\u223e': u'\\lazysinv',
    u'\u2240': u'\\wr',
    u'\u2241': u'\\not\\sim',
    u'\u2242': u'\\ElsevierGlyph{2242}',
    u'\u2242\u0338': u'\\NotEqualTilde',
    u'\u2243': u'\\simeq',
    u'\u2244': u'\\not\\simeq',
    u'\u2245': u'\\cong',
    u'\u2246': u'\\approxnotequal',
    u'\u2247': u'\\not\\cong',
    u'\u2248': u'\\approx',
    u'\u2249': u'\\not\\approx',
    u'\u224a': u'\\approxeq',
    u'\u224b': u'\\tildetrpl',
    u'\u224b\u0338': u'\\not\\apid',
    u'\u224c': u'\\allequal',
    u'\u224d': u'\\asymp',
    u'\u224e': u'\\Bumpeq',
    u'\u224e\u0338': u'\\NotHumpDownHump',
    u'\u224f': u'\\bumpeq',
    u'\u224f\u0338': u'\\NotHumpEqual',
    u'\u2250': u'\\doteq',
    u'\u2250\u0338': u'\\not\\doteq',
    u'\u2251': u'\\doteqdot',
    u'\u2252': u'\\fallingdotseq',
    u'\u2253': u'\\risingdotseq',
    u'\u2254': u':=',
    u'\u2255': u'=:',
    u'\u2256': u'\\eqcirc',
    u'\u2257': u'\\circeq',
    u'\u2259': u'\\estimates',
    u'\u225a': u'\\ElsevierGlyph{225A}',
    u'\u225b': u'\\starequal',
    u'\u225c': u'\\triangleq',
    u'\u225f': u'\\ElsevierGlyph{225F}',
    u'\u2260': u'\\not =',
    u'\u2261': u'\\equiv',
    u'\u2262': u'\\not\\equiv',
    u'\u2264': u'\\leq',
    u'\u2265': u'\\geq',
    u'\u2266': u'\\leqq',
    u'\u2267': u'\\geqq',
    u'\u2268': u'\\lneqq',
    u'\u2268\ufe00': u'\\lvertneqq',
    u'\u2269': u'\\gneqq',
    u'\u2269\ufe00': u'\\gvertneqq',
    u'\u226a': u'\\ll',
    u'\u226a\u0338': u'\\NotLessLess',
    u'\u226b': u'\\gg',
    u'\u226b\u0338': u'\\NotGreaterGreater',
    u'\u226c': u'\\between',
    u'\u226d': u'\\not\\kern-0.3em\\times',
    u'\u226e': u'\\not<',
    u'\u226f': u'\\not>',
    u'\u2270': u'\\not\\leq',
    u'\u2271': u'\\not\\geq',
    u'\u2272': u'\\lessequivlnt',
    u'\u2273': u'\\greaterequivlnt',
    u'\u2274': u'\\ElsevierGlyph{2274}',
    u'\u2275': u'\\ElsevierGlyph{2275}',
    u'\u2276': u'\\lessgtr',
    u'\u2277': u'\\gtrless',
    u'\u2278': u'\\notlessgreater',
    u'\u2279': u'\\notgreaterless',
    u'\u227a': u'\\prec',
    u'\u227b': u'\\succ',
    u'\u227c': u'\\preccurlyeq',
    u'\u227d': u'\\succcurlyeq',
    u'\u227e': u'\\precapprox',
    u'\u227e\u0338': u'\\NotPrecedesTilde',
    u'\u227f': u'\\succapprox',
    u'\u227f\u0338': u'\\NotSucceedsTilde',
    u'\u2280': u'\\not\\prec',
    u'\u2281': u'\\not\\succ',
    u'\u2282': u'\\subset',
    u'\u2283': u'\\supset',
    u'\u2284': u'\\not\\subset',
    u'\u2285': u'\\not\\supset',
    u'\u2286': u'\\subseteq',
    u'\u2287': u'\\supseteq',
    u'\u2288': u'\\not\\subseteq',
    u'\u2289': u'\\not\\supseteq',
    u'\u228a': u'\\subsetneq',
    u'\u228a\ufe00': u'\\varsubsetneqq',
    u'\u228b': u'\\supsetneq',
    u'\u228b\ufe00': u'\\varsupsetneq',
    u'\u228e': u'\\uplus',
    u'\u228f': u'\\sqsubset',
    u'\u228f\u0338': u'\\NotSquareSubset',
    u'\u2290': u'\\sqsupset',
    u'\u2290\u0338': u'\\NotSquareSuperset',
    u'\u2291': u'\\sqsubseteq',
    u'\u2292': u'\\sqsupseteq',
    u'\u2293': u'\\sqcap',
    u'\u2294': u'\\sqcup',
    u'\u2295': u'\\oplus',
    u'\u2296': u'\\ominus',
    u'\u2297': u'\\otimes',
    u'\u2298': u'\\oslash',
    u'\u2299': u'\\odot',
    u'\u229a': u'\\circledcirc',
    u'\u229b': u'\\circledast',
    u'\u229d': u'\\circleddash',
    u'\u229e': u'\\boxplus',
    u'\u229f': u'\\boxminus',
    u'\u22a0': u'\\boxtimes',
    u'\u22a1': u'\\boxdot',
    u'\u22a2': u'\\vdash',
    u'\u22a3': u'\\dashv',
    u'\u22a4': u'\\top',
    u'\u22a5': u'\\perp',
    u'\u22a7': u'\\truestate',
    u'\u22a8': u'\\forcesextra',
    u'\u22a9': u'\\Vdash',
    u'\u22aa': u'\\Vvdash',
    u'\u22ab': u'\\VDash',
    u'\u22ac': u'\\nvdash',
    u'\u22ad': u'\\nvDash',
    u'\u22ae': u'\\nVdash',
    u'\u22af': u'\\nVDash',
    u'\u22b2': u'\\vartriangleleft',
    u'\u22b3': u'\\vartriangleright',
    u'\u22b4': u'\\trianglelefteq',
    u'\u22b5': u'\\trianglerighteq',
    u'\u22b6': u'\\original',
    u'\u22b7': u'\\image',
    u'\u22b8': u'\\multimap',
    u'\u22b9': u'\\hermitconjmatrix',
    u'\u22ba': u'\\intercal',
    u'\u22bb': u'\\veebar',
    u'\u22be': u'\\rightanglearc',
    u'\u22c0': u'\\ElsevierGlyph{22C0}',
    u'\u22c1': u'\\ElsevierGlyph{22C1}',
    u'\u22c2': u'\\bigcap',
    u'\u22c3': u'\\bigcup',
    u'\u22c4': u'\\diamond',
    u'\u22c5': u'\\cdot',
    u'\u22c6': u'\\star',
    u'\u22c7': u'\\divideontimes',
    u'\u22c8': u'\\bowtie',
    u'\u22c9': u'\\ltimes',
    u'\u22ca': u'\\rtimes',
    u'\u22cb': u'\\leftthreetimes',
    u'\u22cc': u'\\rightthreetimes',
    u'\u22cd': u'\\backsimeq',
    u'\u22ce': u'\\curlyvee',
    u'\u22cf': u'\\curlywedge',
    u'\u22d0': u'\\Subset',
    u'\u22d1': u'\\Supset',
    u'\u22d2': u'\\Cap',
    u'\u22d3': u'\\Cup',
    u'\u22d4': u'\\pitchfork',
    u'\u22d6': u'\\lessdot',
    u'\u22d7': u'\\gtrdot',
    u'\u22d8': u'\\verymuchless',
    u'\u22d9': u'\\verymuchgreater',
    u'\u22da': u'\\lesseqgtr',
    u'\u22db': u'\\gtreqless',
    u'\u22de': u'\\curlyeqprec',
    u'\u22df': u'\\curlyeqsucc',
    u'\u22e2': u'\\not\\sqsubseteq',
    u'\u22e3': u'\\not\\sqsupseteq',
    u'\u22e5': u'\\Elzsqspne',
    u'\u22e6': u'\\lnsim',
    u'\u22e7': u'\\gnsim',
    u'\u22e8': u'\\precedesnotsimilar',
    u'\u22e9': u'\\succnsim',
    u'\u22ea': u'\\ntriangleleft',
    u'\u22eb': u'\\ntriangleright',
    u'\u22ec': u'\\ntrianglelefteq',
    u'\u22ed': u'\\ntrianglerighteq',
    u'\u22ee': u'\\vdots',
    u'\u22ef': u'\\cdots',
    u'\u22f0': u'\\upslopeellipsis',
    u'\u22f1': u'\\downslopeellipsis',
    u'\u2305': u'\\barwedge',
    u'\u2306': u'\\perspcorrespond',
    u'\u2308': u'\\lceil',
    u'\u2309': u'\\rceil',
    u'\u230a': u'\\lfloor',
    u'\u230b': u'\\rfloor',
    u'\u2315': u'\\recorder',
    u'\u2316': u'\\mathchar"2208',
    u'\u231c': u'\\ulcorner',
    u'\u231d': u'\\urcorner',
    u'\u231e': u'\\llcorner',
    u'\u231f': u'\\lrcorner',
    u'\u2322': u'\\frown',
    u'\u2323': u'\\smile',
    u'\u2329': u'\\langle',
    u'\u232a': u'\\rangle',
    u'\u233d': u'\\ElsevierGlyph{E838}',
    u'\u23a3': u'\\Elzdlcorn',
    u'\u23b0': u'\\lmoustache',
    u'\u23b1': u'\\rmoustache',
    u'\u2423': u'\\textvisiblespace',
    u'\u2460': u'\\ding{172}',
    u'\u2461': u'\\ding{173}',
    u'\u2462': u'\\ding{174}',
    u'\u2463': u'\\ding{175}',
    u'\u2464': u'\\ding{176}',
    u'\u2465': u'\\ding{177}',
    u'\u2466': u'\\ding{178}',
    u'\u2467': u'\\ding{179}',
    u'\u2468': u'\\ding{180}',
    u'\u2469': u'\\ding{181}',
    u'\u24c8': u'\\circledS',
    u'\u2506': u'\\Elzdshfnc',
    u'\u2519': u'\\Elzsqfnw',
    u'\u2571': u'\\diagup',
    u'\u25a0': u'\\ding{110}',
    u'\u25a1': u'\\square',
    u'\u25aa': u'\\blacksquare',
    u'\u25ad': u'\\fbox{~~}',
    u'\u25af': u'\\Elzvrecto',
    u'\u25b1': u'\\ElsevierGlyph{E381}',
    u'\u25b2': u'\\ding{115}',
    u'\u25b3': u'\\bigtriangleup',
    u'\u25b4': u'\\blacktriangle',
    u'\u25b5': u'\\vartriangle',
    u'\u25b8': u'\\blacktriangleright',
    u'\u25b9': u'\\triangleright',
    u'\u25bc': u'\\ding{116}',
    u'\u25bd': u'\\bigtriangledown',
    u'\u25be': u'\\blacktriangledown',
    u'\u25bf': u'\\triangledown',
    u'\u25c2': u'\\blacktriangleleft',
    u'\u25c3': u'\\triangleleft',
    u'\u25c6': u'\\ding{117}',
    u'\u25ca': u'\\lozenge',
    u'\u25cb': u'\\bigcirc',
    u'\u25cf': u'\\ding{108}',
    u'\u25d0': u'\\Elzcirfl',
    u'\u25d1': u'\\Elzcirfr',
    u'\u25d2': u'\\Elzcirfb',
    u'\u25d7': u'\\ding{119}',
    u'\u25d8': u'\\Elzrvbull',
    u'\u25e7': u'\\Elzsqfl',
    u'\u25e8': u'\\Elzsqfr',
    u'\u25ea': u'\\Elzsqfse',
    u'\u25ef': u'\\bigcirc',
    u'\u2605': u'\\ding{72}',
    u'\u2606': u'\\ding{73}',
    u'\u260e': u'\\ding{37}',
    u'\u261b': u'\\ding{42}',
    u'\u261e': u'\\ding{43}',
    u'\u263e': u'\\rightmoon',
    u'\u263f': u'\\mercury',
    u'\u2640': u'\\venus',
    u'\u2642': u'\\male',
    u'\u2643': u'\\jupiter',
    u'\u2644': u'\\saturn',
    u'\u2645': u'\\uranus',
    u'\u2646': u'\\neptune',
    u'\u2647': u'\\pluto',
    u'\u2648': u'\\aries',
    u'\u2649': u'\\taurus',
    u'\u264a': u'\\gemini',
    u'\u264b': u'\\cancer',
    u'\u264c': u'\\leo',
    u'\u264d': u'\\virgo',
    u'\u264e': u'\\libra',
    u'\u264f': u'\\scorpio',
    u'\u2650': u'\\sagittarius',
    u'\u2651': u'\\capricornus',
    u'\u2652': u'\\aquarius',
    u'\u2653': u'\\pisces',
    u'\u2660': u'\\ding{171}',
    u'\u2662': u'\\diamond',
    u'\u2663': u'\\ding{168}',
    u'\u2665': u'\\ding{170}',
    u'\u2666': u'\\ding{169}',
    u'\u2669': u'\\quarternote',
    u'\u266a': u'\\eighthnote',
    u'\u266d': u'\\flat',
    u'\u266e': u'\\natural',
    u'\u266f': u'\\sharp',
    u'\u2701': u'\\ding{33}',
    u'\u2702': u'\\ding{34}',
    u'\u2703': u'\\ding{35}',
    u'\u2704': u'\\ding{36}',
    u'\u2706': u'\\ding{38}',
    u'\u2707': u'\\ding{39}',
    u'\u2708': u'\\ding{40}',
    u'\u2709': u'\\ding{41}',
    u'\u270c': u'\\ding{44}',
    u'\u270d': u'\\ding{45}',
    u'\u270e': u'\\ding{46}',
    u'\u270f': u'\\ding{47}',
    u'\u2710': u'\\ding{48}',
    u'\u2711': u'\\ding{49}',
    u'\u2712': u'\\ding{50}',
    u'\u2713': u'\\ding{51}',
    u'\u2714': u'\\ding{52}',
    u'\u2715': u'\\ding{53}',
    u'\u2716': u'\\ding{54}',
    u'\u2717': u'\\ding{55}',
    u'\u2718': u'\\ding{56}',
    u'\u2719': u'\\ding{57}',
    u'\u271a': u'\\ding{58}',
    u'\u271b': u'\\ding{59}',
    u'\u271c': u'\\ding{60}',
    u'\u271d': u'\\ding{61}',
    u'\u271e': u'\\ding{62}',
    u'\u271f': u'\\ding{63}',
    u'\u2720': u'\\ding{64}',
    u'\u2721': u'\\ding{65}',
    u'\u2722': u'\\ding{66}',
    u'\u2723': u'\\ding{67}',
    u'\u2724': u'\\ding{68}',
    u'\u2725': u'\\ding{69}',
    u'\u2726': u'\\ding{70}',
    u'\u2727': u'\\ding{71}',
    u'\u2729': u'\\ding{73}',
    u'\u272a': u'\\ding{74}',
    u'\u272b': u'\\ding{75}',
    u'\u272c': u'\\ding{76}',
    u'\u272d': u'\\ding{77}',
    u'\u272e': u'\\ding{78}',
    u'\u272f': u'\\ding{79}',
    u'\u2730': u'\\ding{80}',
    u'\u2731': u'\\ding{81}',
    u'\u2732': u'\\ding{82}',
    u'\u2733': u'\\ding{83}',
    u'\u2734': u'\\ding{84}',
    u'\u2735': u'\\ding{85}',
    u'\u2736': u'\\ding{86}',
    u'\u2737': u'\\ding{87}',
    u'\u2738': u'\\ding{88}',
    u'\u2739': u'\\ding{89}',
    u'\u273a': u'\\ding{90}',
    u'\u273b': u'\\ding{91}',
    u'\u273c': u'\\ding{92}',
    u'\u273d': u'\\ding{93}',
    u'\u273e': u'\\ding{94}',
    u'\u273f': u'\\ding{95}',
    u'\u2740': u'\\ding{96}',
    u'\u2741': u'\\ding{97}',
    u'\u2742': u'\\ding{98}',
    u'\u2743': u'\\ding{99}',
    u'\u2744': u'\\ding{100}',
    u'\u2745': u'\\ding{101}',
    u'\u2746': u'\\ding{102}',
    u'\u2747': u'\\ding{103}',
    u'\u2748': u'\\ding{104}',
    u'\u2749': u'\\ding{105}',
    u'\u274a': u'\\ding{106}',
    u'\u274b': u'\\ding{107}',
    u'\u274d': u'\\ding{109}',
    u'\u274f': u'\\ding{111}',
    u'\u2750': u'\\ding{112}',
    u'\u2751': u'\\ding{113}',
    u'\u2752': u'\\ding{114}',
    u'\u2756': u'\\ding{118}',
    u'\u2758': u'\\ding{120}',
    u'\u2759': u'\\ding{121}',
    u'\u275a': u'\\ding{122}',
    u'\u275b': u'\\ding{123}',
    u'\u275c': u'\\ding{124}',
    u'\u275d': u'\\ding{125}',
    u'\u275e': u'\\ding{126}',
    u'\u2761': u'\\ding{161}',
    u'\u2762': u'\\ding{162}',
    u'\u2763': u'\\ding{163}',
    u'\u2764': u'\\ding{164}',
    u'\u2765': u'\\ding{165}',
    u'\u2766': u'\\ding{166}',
    u'\u2767': u'\\ding{167}',
    u'\u2776': u'\\ding{182}',
    u'\u2777': u'\\ding{183}',
    u'\u2778': u'\\ding{184}',
    u'\u2779': u'\\ding{185}',
    u'\u277a': u'\\ding{186}',
    u'\u277b': u'\\ding{187}',
    u'\u277c': u'\\ding{188}',
    u'\u277d': u'\\ding{189}',
    u'\u277e': u'\\ding{190}',
    u'\u277f': u'\\ding{191}',
    u'\u2780': u'\\ding{192}',
    u'\u2781': u'\\ding{193}',
    u'\u2782': u'\\ding{194}',
    u'\u2783': u'\\ding{195}',
    u'\u2784': u'\\ding{196}',
    u'\u2785': u'\\ding{197}',
    u'\u2786': u'\\ding{198}',
    u'\u2787': u'\\ding{199}',
    u'\u2788': u'\\ding{200}',
    u'\u2789': u'\\ding{201}',
    u'\u278a': u'\\ding{202}',
    u'\u278b': u'\\ding{203}',
    u'\u278c': u'\\ding{204}',
    u'\u278d': u'\\ding{205}',
    u'\u278e': u'\\ding{206}',
    u'\u278f': u'\\ding{207}',
    u'\u2790': u'\\ding{208}',
    u'\u2791': u'\\ding{209}',
    u'\u2792': u'\\ding{210}',
    u'\u2793': u'\\ding{211}',
    u'\u2794': u'\\ding{212}',
    u'\u2798': u'\\ding{216}',
    u'\u2799': u'\\ding{217}',
    u'\u279a': u'\\ding{218}',
    u'\u279b': u'\\ding{219}',
    u'\u279c': u'\\ding{220}',
    u'\u279d': u'\\ding{221}',
    u'\u279e': u'\\ding{222}',
    u'\u279f': u'\\ding{223}',
    u'\u27a0': u'\\ding{224}',
    u'\u27a1': u'\\ding{225}',
    u'\u27a2': u'\\ding{226}',
    u'\u27a3': u'\\ding{227}',
    u'\u27a4': u'\\ding{228}',
    u'\u27a5': u'\\ding{229}',
    u'\u27a6': u'\\ding{230}',
    u'\u27a7': u'\\ding{231}',
    u'\u27a8': u'\\ding{232}',
    u'\u27a9': u'\\ding{233}',
    u'\u27aa': u'\\ding{234}',
    u'\u27ab': u'\\ding{235}',
    u'\u27ac': u'\\ding{236}',
    u'\u27ad': u'\\ding{237}',
    u'\u27ae': u'\\ding{238}',
    u'\u27af': u'\\ding{239}',
    u'\u27b1': u'\\ding{241}',
    u'\u27b2': u'\\ding{242}',
    u'\u27b3': u'\\ding{243}',
    u'\u27b4': u'\\ding{244}',
    u'\u27b5': u'\\ding{245}',
    u'\u27b6': u'\\ding{246}',
    u'\u27b7': u'\\ding{247}',
    u'\u27b8': u'\\ding{248}',
    u'\u27b9': u'\\ding{249}',
    u'\u27ba': u'\\ding{250}',
    u'\u27bb': u'\\ding{251}',
    u'\u27bc': u'\\ding{252}',
    u'\u27bd': u'\\ding{253}',
    u'\u27be': u'\\ding{254}',
    u'\u27f5': u'\\longleftarrow',
    u'\u27f6': u'\\longrightarrow',
    u'\u27f7': u'\\longleftrightarrow',
    u'\u27f8': u'\\Longleftarrow',
    u'\u27f9': u'\\Longrightarrow',
    u'\u27fa': u'\\Longleftrightarrow',
    u'\u27fc': u'\\longmapsto',
    u'\u27ff': u'\\sim\\joinrel\\leadsto',
    u'\u2905': u'\\ElsevierGlyph{E212}',
    u'\u2912': u'\\UpArrowBar',
    u'\u2913': u'\\DownArrowBar',
    u'\u2923': u'\\ElsevierGlyph{E20C}',
    u'\u2924': u'\\ElsevierGlyph{E20D}',
    u'\u2925': u'\\ElsevierGlyph{E20B}',
    u'\u2926': u'\\ElsevierGlyph{E20A}',
    u'\u2927': u'\\ElsevierGlyph{E211}',
    u'\u2928': u'\\ElsevierGlyph{E20E}',
    u'\u2929': u'\\ElsevierGlyph{E20F}',
    u'\u292a': u'\\ElsevierGlyph{E210}',
    u'\u2933': u'\\ElsevierGlyph{E21C}',
    u'\u2933\u0338': u'\\ElsevierGlyph{E21D}',
    u'\u2936': u'\\ElsevierGlyph{E21A}',
    u'\u2937': u'\\ElsevierGlyph{E219}',
    u'\u2940': u'\\Elolarr',
    u'\u2941': u'\\Elorarr',
    u'\u2942': u'\\ElzRlarr',
    u'\u2944': u'\\ElzrLarr',
    u'\u2947': u'\\Elzrarrx',
    u'\u294e': u'\\LeftRightVector',
    u'\u294f': u'\\RightUpDownVector',
    u'\u2950': u'\\DownLeftRightVector',
    u'\u2951': u'\\LeftUpDownVector',
    u'\u2952': u'\\LeftVectorBar',
    u'\u2953': u'\\RightVectorBar',
    u'\u2954': u'\\RightUpVectorBar',
    u'\u2955': u'\\RightDownVectorBar',
    u'\u2956': u'\\DownLeftVectorBar',
    u'\u2957': u'\\DownRightVectorBar',
    u'\u2958': u'\\LeftUpVectorBar',
    u'\u2959': u'\\LeftDownVectorBar',
    u'\u295a': u'\\LeftTeeVector',
    u'\u295b': u'\\RightTeeVector',
    u'\u295c': u'\\RightUpTeeVector',
    u'\u295d': u'\\RightDownTeeVector',
    u'\u295e': u'\\DownLeftTeeVector',
    u'\u295f': u'\\DownRightTeeVector',
    u'\u2960': u'\\LeftUpTeeVector',
    u'\u2961': u'\\LeftDownTeeVector',
    u'\u296e': u'\\UpEquilibrium',
    u'\u296f': u'\\ReverseUpEquilibrium',
    u'\u2970': u'\\RoundImplies',
    u'\u297c': u'\\ElsevierGlyph{E214}',
    u'\u297d': u'\\ElsevierGlyph{E215}',
    u'\u2980': u'\\Elztfnc',
    u'\u2985': u'\\ElsevierGlyph{3018}',
    u'\u2986': u'\\Elroang',
    u'\u2993': u'<\\kern-0.58em(',
    u'\u2994': u'\\ElsevierGlyph{E291}',
    u'\u2999': u'\\Elzddfnc',
    u'\u299c': u'\\Angle',
    u'\u29a0': u'\\Elzlpargt',
    u'\u29b5': u'\\ElsevierGlyph{E260}',
    u'\u29b6': u'\\ElsevierGlyph{E61B}',
    u'\u29ca': u'\\ElzLap',
    u'\u29cb': u'\\Elzdefas',
    u'\u29cf': u'\\LeftTriangleBar',
    u'\u29cf\u0338': u'\\NotLeftTriangleBar',
    u'\u29d0': u'\\RightTriangleBar',
    u'\u29d0\u0338': u'\\NotRightTriangleBar',
    u'\u29dc': u'\\ElsevierGlyph{E372}',
    u'\u29eb': u'\\blacklozenge',
    u'\u29f4': u'\\RuleDelayed',
    u'\u2a04': u'\\Elxuplus',
    u'\u2a05': u'\\ElzThr',
    u'\u2a06': u'\\Elxsqcup',
    u'\u2a07': u'\\ElzInf',
    u'\u2a08': u'\\ElzSup',
    u'\u2a0d': u'\\ElzCint',
    u'\u2a0f': u'\\clockoint',
    u'\u2a10': u'\\ElsevierGlyph{E395}',
    u'\u2a16': u'\\sqrint',
    u'\u2a25': u'\\ElsevierGlyph{E25A}',
    u'\u2a2a': u'\\ElsevierGlyph{E25B}',
    u'\u2a2d': u'\\ElsevierGlyph{E25C}',
    u'\u2a2e': u'\\ElsevierGlyph{E25D}',
    u'\u2a2f': u'\\ElzTimes',
    u'\u2a34': u'\\ElsevierGlyph{E25E}',
    u'\u2a35': u'\\ElsevierGlyph{E25E}',
    u'\u2a3c': u'\\ElsevierGlyph{E259}',
    u'\u2a3f': u'\\amalg',
    u'\u2a53': u'\\ElzAnd',
    u'\u2a54': u'\\ElzOr',
    u'\u2a55': u'\\ElsevierGlyph{E36E}',
    u'\u2a56': u'\\ElOr',
    u'\u2a5e': u'\\perspcorrespond',
    u'\u2a5f': u'\\Elzminhat',
    u'\u2a63': u'\\ElsevierGlyph{225A}',
    u'\u2a6e': u'\\stackrel{*}{=}',
    u'\u2a75': u'\\Equal',
    u'\u2a7d': u'\\leqslant',
    u'\u2a7d\u0338': u'\\nleqslant',
    u'\u2a7e': u'\\geqslant',
    u'\u2a7e\u0338': u'\\ngeqslant',
    u'\u2a85': u'\\lessapprox',
    u'\u2a86': u'\\gtrapprox',
    u'\u2a87': u'\\lneq',
    u'\u2a88': u'\\gneq',
    u'\u2a89': u'\\lnapprox',
    u'\u2a8a': u'\\gnapprox',
    u'\u2a8b': u'\\lesseqqgtr',
    u'\u2a8c': u'\\gtreqqless',
    u'\u2a95': u'\\eqslantless',
    u'\u2a96': u'\\eqslantgtr',
    u'\u2a9d': u'\\Pisymbol{ppi020}{117}',
    u'\u2a9e': u'\\Pisymbol{ppi020}{105}',
    u'\u2aa1': u'\\NestedLessLess',
    u'\u2aa1\u0338': u'\\NotNestedLessLess',
    u'\u2aa2': u'\\NestedGreaterGreater',
    u'\u2aa2\u0338': u'\\NotNestedGreaterGreater',
    u'\u2aaf': u'\\preceq',
    u'\u2aaf\u0338': u'\\not\\preceq',
    u'\u2ab0': u'\\succeq',
    u'\u2ab0\u0338': u'\\not\\succeq',
    u'\u2ab5': u'\\precneqq',
    u'\u2ab6': u'\\succneqq',
    u'\u2ab7': u'\\precapprox',
    u'\u2ab8': u'\\succapprox',
    u'\u2ab9': u'\\precnapprox',
    u'\u2aba': u'\\succnapprox',
    u'\u2ac5': u'\\subseteqq',
    u'\u2ac5\u0338': u'\\nsubseteqq',
    u'\u2ac6': u'\\supseteqq',
    u'\u2ac6\u0338': u'\\nsupseteqq',
    u'\u2acb': u'\\subsetneqq',
    u'\u2acc': u'\\supsetneqq',
    u'\u2aeb': u'\\ElsevierGlyph{E30D}',
    u'\u2af6': u'\\Elztdcol',
    u'\u2afd': u'{{/}\\!\\!{/}}',
    u'\u2afd\u20e5': u'{\\rlap{\\textbackslash}{{/}\\!\\!{/}}}',
    u'\u300a': u'\\ElsevierGlyph{300A}',
    u'\u300b': u'\\ElsevierGlyph{300B}',
    u'\u3018': u'\\ElsevierGlyph{3018}',
    u'\u3019': u'\\ElsevierGlyph{3019}',
    u'\u301a': u'\\openbracketleft',
    u'\u301b': u'\\openbracketright',
    u'\ufb00': u'ff',
    u'\ufb01': u'fi',
    u'\ufb02': u'fl',
    u'\ufb03': u'ffi',
    u'\ufb04': u'ffl',
    u'\U0001d400': u'\\mathbf{A}',
    u'\U0001d401': u'\\mathbf{B}',
    u'\U0001d402': u'\\mathbf{C}',
    u'\U0001d403': u'\\mathbf{D}',
    u'\U0001d404': u'\\mathbf{E}',
    u'\U0001d405': u'\\mathbf{F}',
    u'\U0001d406': u'\\mathbf{G}',
    u'\U0001d407': u'\\mathbf{H}',
    u'\U0001d408': u'\\mathbf{I}',
    u'\U0001d409': u'\\mathbf{J}',
    u'\U0001d40a': u'\\mathbf{K}',
    u'\U0001d40b': u'\\mathbf{L}',
    u'\U0001d40c': u'\\mathbf{M}',
    u'\U0001d40d': u'\\mathbf{N}',
    u'\U0001d40e': u'\\mathbf{O}',
    u'\U0001d40f': u'\\mathbf{P}',
    u'\U0001d410': u'\\mathbf{Q}',
    u'\U0001d411': u'\\mathbf{R}',
    u'\U0001d412': u'\\mathbf{S}',
    u'\U0001d413': u'\\mathbf{T}',
    u'\U0001d414': u'\\mathbf{U}',
    u'\U0001d415': u'\\mathbf{V}',
    u'\U0001d416': u'\\mathbf{W}',
    u'\U0001d417': u'\\mathbf{X}',
    u'\U0001d418': u'\\mathbf{Y}',
    u'\U0001d419': u'\\mathbf{Z}',
    u'\U0001d41a': u'\\mathbf{a}',
    u'\U0001d41b': u'\\mathbf{b}',
    u'\U0001d41c': u'\\mathbf{c}',
    u'\U0001d41d': u'\\mathbf{d}',
    u'\U0001d41e': u'\\mathbf{e}',
    u'\U0001d41f': u'\\mathbf{f}',
    u'\U0001d420': u'\\mathbf{g}',
    u'\U0001d421': u'\\mathbf{h}',
    u'\U0001d422': u'\\mathbf{i}',
    u'\U0001d423': u'\\mathbf{j}',
    u'\U0001d424': u'\\mathbf{k}',
    u'\U0001d425': u'\\mathbf{l}',
    u'\U0001d426': u'\\mathbf{m}',
    u'\U0001d427': u'\\mathbf{n}',
    u'\U0001d428': u'\\mathbf{o}',
    u'\U0001d429': u'\\mathbf{p}',
    u'\U0001d42a': u'\\mathbf{q}',
    u'\U0001d42b': u'\\mathbf{r}',
    u'\U0001d42c': u'\\mathbf{s}',
    u'\U0001d42d': u'\\mathbf{t}',
    u'\U0001d42e': u'\\mathbf{u}',
    u'\U0001d42f': u'\\mathbf{v}',
    u'\U0001d430': u'\\mathbf{w}',
    u'\U0001d431': u'\\mathbf{x}',
    u'\U0001d432': u'\\mathbf{y}',
    u'\U0001d433': u'\\mathbf{z}',
    u'\U0001d434': u'\\mathsl{A}',
    u'\U0001d435': u'\\mathsl{B}',
    u'\U0001d436': u'\\mathsl{C}',
    u'\U0001d437': u'\\mathsl{D}',
    u'\U0001d438': u'\\mathsl{E}',
    u'\U0001d439': u'\\mathsl{F}',
    u'\U0001d43a': u'\\mathsl{G}',
    u'\U0001d43b': u'\\mathsl{H}',
    u'\U0001d43c': u'\\mathsl{I}',
    u'\U0001d43d': u'\\mathsl{J}',
    u'\U0001d43e': u'\\mathsl{K}',
    u'\U0001d43f': u'\\mathsl{L}',
    u'\U0001d440': u'\\mathsl{M}',
    u'\U0001d441': u'\\mathsl{N}',
    u'\U0001d442': u'\\mathsl{O}',
    u'\U0001d443': u'\\mathsl{P}',
    u'\U0001d444': u'\\mathsl{Q}',
    u'\U0001d445': u'\\mathsl{R}',
    u'\U0001d446': u'\\mathsl{S}',
    u'\U0001d447': u'\\mathsl{T}',
    u'\U0001d448': u'\\mathsl{U}',
    u'\U0001d449': u'\\mathsl{V}',
    u'\U0001d44a': u'\\mathsl{W}',
    u'\U0001d44b': u'\\mathsl{X}',
    u'\U0001d44c': u'\\mathsl{Y}',
    u'\U0001d44d': u'\\mathsl{Z}',
    u'\U0001d44e': u'\\mathsl{a}',
    u'\U0001d44f': u'\\mathsl{b}',
    u'\U0001d450': u'\\mathsl{c}',
    u'\U0001d451': u'\\mathsl{d}',
    u'\U0001d452': u'\\mathsl{e}',
    u'\U0001d453': u'\\mathsl{f}',
    u'\U0001d454': u'\\mathsl{g}',
    u'\U0001d456': u'\\mathsl{i}',
    u'\U0001d457': u'\\mathsl{j}',
    u'\U0001d458': u'\\mathsl{k}',
    u'\U0001d459': u'\\mathsl{l}',
    u'\U0001d45a': u'\\mathsl{m}',
    u'\U0001d45b': u'\\mathsl{n}',
    u'\U0001d45c': u'\\mathsl{o}',
    u'\U0001d45d': u'\\mathsl{p}',
    u'\U0001d45e': u'\\mathsl{q}',
    u'\U0001d45f': u'\\mathsl{r}',
    u'\U0001d460': u'\\mathsl{s}',
    u'\U0001d461': u'\\mathsl{t}',
    u'\U0001d462': u'\\mathsl{u}',
    u'\U0001d463': u'\\mathsl{v}',
    u'\U0001d464': u'\\mathsl{w}',
    u'\U0001d465': u'\\mathsl{x}',
    u'\U0001d466': u'\\mathsl{y}',
    u'\U0001d467': u'\\mathsl{z}',
    u'\U0001d468': u'\\mathbit{A}',
    u'\U0001d469': u'\\mathbit{B}',
    u'\U0001d46a': u'\\mathbit{C}',
    u'\U0001d46b': u'\\mathbit{D}',
    u'\U0001d46c': u'\\mathbit{E}',
    u'\U0001d46d': u'\\mathbit{F}',
    u'\U0001d46e': u'\\mathbit{G}',
    u'\U0001d46f': u'\\mathbit{H}',
    u'\U0001d470': u'\\mathbit{I}',
    u'\U0001d471': u'\\mathbit{J}',
    u'\U0001d472': u'\\mathbit{K}',
    u'\U0001d473': u'\\mathbit{L}',
    u'\U0001d474': u'\\mathbit{M}',
    u'\U0001d475': u'\\mathbit{N}',
    u'\U0001d476': u'\\mathbit{O}',
    u'\U0001d477': u'\\mathbit{P}',
    u'\U0001d478': u'\\mathbit{Q}',
    u'\U0001d479': u'\\mathbit{R}',
    u'\U0001d47a': u'\\mathbit{S}',
    u'\U0001d47b': u'\\mathbit{T}',
    u'\U0001d47c': u'\\mathbit{U}',
    u'\U0001d47d': u'\\mathbit{V}',
    u'\U0001d47e': u'\\mathbit{W}',
    u'\U0001d47f': u'\\mathbit{X}',
    u'\U0001d480': u'\\mathbit{Y}',
    u'\U0001d481': u'\\mathbit{Z}',
    u'\U0001d482': u'\\mathbit{a}',
    u'\U0001d483': u'\\mathbit{b}',
    u'\U0001d484': u'\\mathbit{c}',
    u'\U0001d485': u'\\mathbit{d}',
    u'\U0001d486': u'\\mathbit{e}',
    u'\U0001d487': u'\\mathbit{f}',
    u'\U0001d488': u'\\mathbit{g}',
    u'\U0001d489': u'\\mathbit{h}',
    u'\U0001d48a': u'\\mathbit{i}',
    u'\U0001d48b': u'\\mathbit{j}',
    u'\U0001d48c': u'\\mathbit{k}',
    u'\U0001d48d': u'\\mathbit{l}',
    u'\U0001d48e': u'\\mathbit{m}',
    u'\U0001d48f': u'\\mathbit{n}',
    u'\U0001d490': u'\\mathbit{o}',
    u'\U0001d491': u'\\mathbit{p}',
    u'\U0001d492': u'\\mathbit{q}',
    u'\U0001d493': u'\\mathbit{r}',
    u'\U0001d494': u'\\mathbit{s}',
    u'\U0001d495': u'\\mathbit{t}',
    u'\U0001d496': u'\\mathbit{u}',
    u'\U0001d497': u'\\mathbit{v}',
    u'\U0001d498': u'\\mathbit{w}',
    u'\U0001d499': u'\\mathbit{x}',
    u'\U0001d49a': u'\\mathbit{y}',
    u'\U0001d49b': u'\\mathbit{z}',
    u'\U0001d49c': u'\\mathscr{A}',
    u'\U0001d49e': u'\\mathscr{C}',
    u'\U0001d49f': u'\\mathscr{D}',
    u'\U0001d4a2': u'\\mathscr{G}',
    u'\U0001d4a5': u'\\mathscr{J}',
    u'\U0001d4a6': u'\\mathscr{K}',
    u'\U0001d4a9': u'\\mathscr{N}',
    u'\U0001d4aa': u'\\mathscr{O}',
    u'\U0001d4ab': u'\\mathscr{P}',
    u'\U0001d4ac': u'\\mathscr{Q}',
    u'\U0001d4ae': u'\\mathscr{S}',
    u'\U0001d4af': u'\\mathscr{T}',
    u'\U0001d4b0': u'\\mathscr{U}',
    u'\U0001d4b1': u'\\mathscr{V}',
    u'\U0001d4b2': u'\\mathscr{W}',
    u'\U0001d4b3': u'\\mathscr{X}',
    u'\U0001d4b4': u'\\mathscr{Y}',
    u'\U0001d4b5': u'\\mathscr{Z}',
    u'\U0001d4b6': u'\\mathscr{a}',
    u'\U0001d4b7': u'\\mathscr{b}',
    u'\U0001d4b8': u'\\mathscr{c}',
    u'\U0001d4b9': u'\\mathscr{d}',
    u'\U0001d4bb': u'\\mathscr{f}',
    u'\U0001d4bd': u'\\mathscr{h}',
    u'\U0001d4be': u'\\mathscr{i}',
    u'\U0001d4bf': u'\\mathscr{j}',
    u'\U0001d4c0': u'\\mathscr{k}',
    u'\U0001d4c1': u'\\mathscr{l}',
    u'\U0001d4c2': u'\\mathscr{m}',
    u'\U0001d4c3': u'\\mathscr{n}',
    u'\U0001d4c5': u'\\mathscr{p}',
    u'\U0001d4c6': u'\\mathscr{q}',
    u'\U0001d4c7': u'\\mathscr{r}',
    u'\U0001d4c8': u'\\mathscr{s}',
    u'\U0001d4c9': u'\\mathscr{t}',
    u'\U0001d4ca': u'\\mathscr{u}',
    u'\U0001d4cb': u'\\mathscr{v}',
    u'\U0001d4cc': u'\\mathscr{w}',
    u'\U0001d4cd': u'\\mathscr{x}',
    u'\U0001d4ce': u'\\mathscr{y}',
    u'\U0001d4cf': u'\\mathscr{z}',
    u'\U0001d4d0': u'\\mathmit{A}',
    u'\U0001d4d1': u'\\mathmit{B}',
    u'\U0001d4d2': u'\\mathmit{C}',
    u'\U0001d4d3': u'\\mathmit{D}',
    u'\U0001d4d4': u'\\mathmit{E}',
    u'\U0001d4d5': u'\\mathmit{F}',
    u'\U0001d4d6': u'\\mathmit{G}',
    u'\U0001d4d7': u'\\mathmit{H}',
    u'\U0001d4d8': u'\\mathmit{I}',
    u'\U0001d4d9': u'\\mathmit{J}',
    u'\U0001d4da': u'\\mathmit{K}',
    u'\U0001d4db': u'\\mathmit{L}',
    u'\U0001d4dc': u'\\mathmit{M}',
    u'\U0001d4dd': u'\\mathmit{N}',
    u'\U0001d4de': u'\\mathmit{O}',
    u'\U0001d4df': u'\\mathmit{P}',
    u'\U0001d4e0': u'\\mathmit{Q}',
    u'\U0001d4e1': u'\\mathmit{R}',
    u'\U0001d4e2': u'\\mathmit{S}',
    u'\U0001d4e3': u'\\mathmit{T}',
    u'\U0001d4e4': u'\\mathmit{U}',
    u'\U0001d4e5': u'\\mathmit{V}',
    u'\U0001d4e6': u'\\mathmit{W}',
    u'\U0001d4e7': u'\\mathmit{X}',
    u'\U0001d4e8': u'\\mathmit{Y}',
    u'\U0001d4e9': u'\\mathmit{Z}',
    u'\U0001d4ea': u'\\mathmit{a}',
    u'\U0001d4eb': u'\\mathmit{b}',
    u'\U0001d4ec': u'\\mathmit{c}',
    u'\U0001d4ed': u'\\mathmit{d}',
    u'\U0001d4ee': u'\\mathmit{e}',
    u'\U0001d4ef': u'\\mathmit{f}',
    u'\U0001d4f0': u'\\mathmit{g}',
    u'\U0001d4f1': u'\\mathmit{h}',
    u'\U0001d4f2': u'\\mathmit{i}',
    u'\U0001d4f3': u'\\mathmit{j}',
    u'\U0001d4f4': u'\\mathmit{k}',
    u'\U0001d4f5': u'\\mathmit{l}',
    u'\U0001d4f6': u'\\mathmit{m}',
    u'\U0001d4f7': u'\\mathmit{n}',
    u'\U0001d4f8': u'\\mathmit{o}',
    u'\U0001d4f9': u'\\mathmit{p}',
    u'\U0001d4fa': u'\\mathmit{q}',
    u'\U0001d4fb': u'\\mathmit{r}',
    u'\U0001d4fc': u'\\mathmit{s}',
    u'\U0001d4fd': u'\\mathmit{t}',
    u'\U0001d4fe': u'\\mathmit{u}',
    u'\U0001d4ff': u'\\mathmit{v}',
    u'\U0001d500': u'\\mathmit{w}',
    u'\U0001d501': u'\\mathmit{x}',
    u'\U0001d502': u'\\mathmit{y}',
    u'\U0001d503': u'\\mathmit{z}',
    u'\U0001d504': u'\\mathfrak{A}',
    u'\U0001d505': u'\\mathfrak{B}',
    u'\U0001d507': u'\\mathfrak{D}',
    u'\U0001d508': u'\\mathfrak{E}',
    u'\U0001d509': u'\\mathfrak{F}',
    u'\U0001d50a': u'\\mathfrak{G}',
    u'\U0001d50d': u'\\mathfrak{J}',
    u'\U0001d50e': u'\\mathfrak{K}',
    u'\U0001d50f': u'\\mathfrak{L}',
    u'\U0001d510': u'\\mathfrak{M}',
    u'\U0001d511': u'\\mathfrak{N}',
    u'\U0001d512': u'\\mathfrak{O}',
    u'\U0001d513': u'\\mathfrak{P}',
    u'\U0001d514': u'\\mathfrak{Q}',
    u'\U0001d516': u'\\mathfrak{S}',
    u'\U0001d517': u'\\mathfrak{T}',
    u'\U0001d518': u'\\mathfrak{U}',
    u'\U0001d519': u'\\mathfrak{V}',
    u'\U0001d51a': u'\\mathfrak{W}',
    u'\U0001d51b': u'\\mathfrak{X}',
    u'\U0001d51c': u'\\mathfrak{Y}',
    u'\U0001d51e': u'\\mathfrak{a}',
    u'\U0001d51f': u'\\mathfrak{b}',
    u'\U0001d520': u'\\mathfrak{c}',
    u'\U0001d521': u'\\mathfrak{d}',
    u'\U0001d522': u'\\mathfrak{e}',
    u'\U0001d523': u'\\mathfrak{f}',
    u'\U0001d524': u'\\mathfrak{g}',
    u'\U0001d525': u'\\mathfrak{h}',
    u'\U0001d526': u'\\mathfrak{i}',
    u'\U0001d527': u'\\mathfrak{j}',
    u'\U0001d528': u'\\mathfrak{k}',
    u'\U0001d529': u'\\mathfrak{l}',
    u'\U0001d52a': u'\\mathfrak{m}',
    u'\U0001d52b': u'\\mathfrak{n}',
    u'\U0001d52c': u'\\mathfrak{o}',
    u'\U0001d52d': u'\\mathfrak{p}',
    u'\U0001d52e': u'\\mathfrak{q}',
    u'\U0001d52f': u'\\mathfrak{r}',
    u'\U0001d530': u'\\mathfrak{s}',
    u'\U0001d531': u'\\mathfrak{t}',
    u'\U0001d532': u'\\mathfrak{u}',
    u'\U0001d533': u'\\mathfrak{v}',
    u'\U0001d534': u'\\mathfrak{w}',
    u'\U0001d535': u'\\mathfrak{x}',
    u'\U0001d536': u'\\mathfrak{y}',
    u'\U0001d537': u'\\mathfrak{z}',
    u'\U0001d538': u'\\mathbb{A}',
    u'\U0001d539': u'\\mathbb{B}',
    u'\U0001d53b': u'\\mathbb{D}',
    u'\U0001d53c': u'\\mathbb{E}',
    u'\U0001d53d': u'\\mathbb{F}',
    u'\U0001d53e': u'\\mathbb{G}',
    u'\U0001d540': u'\\mathbb{I}',
    u'\U0001d541': u'\\mathbb{J}',
    u'\U0001d542': u'\\mathbb{K}',
    u'\U0001d543': u'\\mathbb{L}',
    u'\U0001d544': u'\\mathbb{M}',
    u'\U0001d546': u'\\mathbb{O}',
    u'\U0001d54a': u'\\mathbb{S}',
    u'\U0001d54b': u'\\mathbb{T}',
    u'\U0001d54c': u'\\mathbb{U}',
    u'\U0001d54d': u'\\mathbb{V}',
    u'\U0001d54e': u'\\mathbb{W}',
    u'\U0001d54f': u'\\mathbb{X}',
    u'\U0001d550': u'\\mathbb{Y}',
    u'\U0001d552': u'\\mathbb{a}',
    u'\U0001d553': u'\\mathbb{b}',
    u'\U0001d554': u'\\mathbb{c}',
    u'\U0001d555': u'\\mathbb{d}',
    u'\U0001d556': u'\\mathbb{e}',
    u'\U0001d557': u'\\mathbb{f}',
    u'\U0001d558': u'\\mathbb{g}',
    u'\U0001d559': u'\\mathbb{h}',
    u'\U0001d55a': u'\\mathbb{i}',
    u'\U0001d55b': u'\\mathbb{j}',
    u'\U0001d55c': u'\\mathbb{k}',
    u'\U0001d55d': u'\\mathbb{l}',
    u'\U0001d55e': u'\\mathbb{m}',
    u'\U0001d55f': u'\\mathbb{n}',
    u'\U0001d560': u'\\mathbb{o}',
    u'\U0001d561': u'\\mathbb{p}',
    u'\U0001d562': u'\\mathbb{q}',
    u'\U0001d563': u'\\mathbb{r}',
    u'\U0001d564': u'\\mathbb{s}',
    u'\U0001d565': u'\\mathbb{t}',
    u'\U0001d566': u'\\mathbb{u}',
    u'\U0001d567': u'\\mathbb{v}',
    u'\U0001d568': u'\\mathbb{w}',
    u'\U0001d569': u'\\mathbb{x}',
    u'\U0001d56a': u'\\mathbb{y}',
    u'\U0001d56b': u'\\mathbb{z}',
    u'\U0001d56c': u'\\mathslbb{A}',
    u'\U0001d56d': u'\\mathslbb{B}',
    u'\U0001d56e': u'\\mathslbb{C}',
    u'\U0001d56f': u'\\mathslbb{D}',
    u'\U0001d570': u'\\mathslbb{E}',
    u'\U0001d571': u'\\mathslbb{F}',
    u'\U0001d572': u'\\mathslbb{G}',
    u'\U0001d573': u'\\mathslbb{H}',
    u'\U0001d574': u'\\mathslbb{I}',
    u'\U0001d575': u'\\mathslbb{J}',
    u'\U0001d576': u'\\mathslbb{K}',
    u'\U0001d577': u'\\mathslbb{L}',
    u'\U0001d578': u'\\mathslbb{M}',
    u'\U0001d579': u'\\mathslbb{N}',
    u'\U0001d57a': u'\\mathslbb{O}',
    u'\U0001d57b': u'\\mathslbb{P}',
    u'\U0001d57c': u'\\mathslbb{Q}',
    u'\U0001d57d': u'\\mathslbb{R}',
    u'\U0001d57e': u'\\mathslbb{S}',
    u'\U0001d57f': u'\\mathslbb{T}',
    u'\U0001d580': u'\\mathslbb{U}',
    u'\U0001d581': u'\\mathslbb{V}',
    u'\U0001d582': u'\\mathslbb{W}',
    u'\U0001d583': u'\\mathslbb{X}',
    u'\U0001d584': u'\\mathslbb{Y}',
    u'\U0001d585': u'\\mathslbb{Z}',
    u'\U0001d586': u'\\mathslbb{a}',
    u'\U0001d587': u'\\mathslbb{b}',
    u'\U0001d588': u'\\mathslbb{c}',
    u'\U0001d589': u'\\mathslbb{d}',
    u'\U0001d58a': u'\\mathslbb{e}',
    u'\U0001d58b': u'\\mathslbb{f}',
    u'\U0001d58c': u'\\mathslbb{g}',
    u'\U0001d58d': u'\\mathslbb{h}',
    u'\U0001d58e': u'\\mathslbb{i}',
    u'\U0001d58f': u'\\mathslbb{j}',
    u'\U0001d590': u'\\mathslbb{k}',
    u'\U0001d591': u'\\mathslbb{l}',
    u'\U0001d592': u'\\mathslbb{m}',
    u'\U0001d593': u'\\mathslbb{n}',
    u'\U0001d594': u'\\mathslbb{o}',
    u'\U0001d595': u'\\mathslbb{p}',
    u'\U0001d596': u'\\mathslbb{q}',
    u'\U0001d597': u'\\mathslbb{r}',
    u'\U0001d598': u'\\mathslbb{s}',
    u'\U0001d599': u'\\mathslbb{t}',
    u'\U0001d59a': u'\\mathslbb{u}',
    u'\U0001d59b': u'\\mathslbb{v}',
    u'\U0001d59c': u'\\mathslbb{w}',
    u'\U0001d59d': u'\\mathslbb{x}',
    u'\U0001d59e': u'\\mathslbb{y}',
    u'\U0001d59f': u'\\mathslbb{z}',
    u'\U0001d5a0': u'\\mathsf{A}',
    u'\U0001d5a1': u'\\mathsf{B}',
    u'\U0001d5a2': u'\\mathsf{C}',
    u'\U0001d5a3': u'\\mathsf{D}',
    u'\U0001d5a4': u'\\mathsf{E}',
    u'\U0001d5a5': u'\\mathsf{F}',
    u'\U0001d5a6': u'\\mathsf{G}',
    u'\U0001d5a7': u'\\mathsf{H}',
    u'\U0001d5a8': u'\\mathsf{I}',
    u'\U0001d5a9': u'\\mathsf{J}',
    u'\U0001d5aa': u'\\mathsf{K}',
    u'\U0001d5ab': u'\\mathsf{L}',
    u'\U0001d5ac': u'\\mathsf{M}',
    u'\U0001d5ad': u'\\mathsf{N}',
    u'\U0001d5ae': u'\\mathsf{O}',
    u'\U0001d5af': u'\\mathsf{P}',
    u'\U0001d5b0': u'\\mathsf{Q}',
    u'\U0001d5b1': u'\\mathsf{R}',
    u'\U0001d5b2': u'\\mathsf{S}',
    u'\U0001d5b3': u'\\mathsf{T}',
    u'\U0001d5b4': u'\\mathsf{U}',
    u'\U0001d5b5': u'\\mathsf{V}',
    u'\U0001d5b6': u'\\mathsf{W}',
    u'\U0001d5b7': u'\\mathsf{X}',
    u'\U0001d5b8': u'\\mathsf{Y}',
    u'\U0001d5b9': u'\\mathsf{Z}',
    u'\U0001d5ba': u'\\mathsf{a}',
    u'\U0001d5bb': u'\\mathsf{b}',
    u'\U0001d5bc': u'\\mathsf{c}',
    u'\U0001d5bd': u'\\mathsf{d}',
    u'\U0001d5be': u'\\mathsf{e}',
    u'\U0001d5bf': u'\\mathsf{f}',
    u'\U0001d5c0': u'\\mathsf{g}',
    u'\U0001d5c1': u'\\mathsf{h}',
    u'\U0001d5c2': u'\\mathsf{i}',
    u'\U0001d5c3': u'\\mathsf{j}',
    u'\U0001d5c4': u'\\mathsf{k}',
    u'\U0001d5c5': u'\\mathsf{l}',
    u'\U0001d5c6': u'\\mathsf{m}',
    u'\U0001d5c7': u'\\mathsf{n}',
    u'\U0001d5c8': u'\\mathsf{o}',
    u'\U0001d5c9': u'\\mathsf{p}',
    u'\U0001d5ca': u'\\mathsf{q}',
    u'\U0001d5cb': u'\\mathsf{r}',
    u'\U0001d5cc': u'\\mathsf{s}',
    u'\U0001d5cd': u'\\mathsf{t}',
    u'\U0001d5ce': u'\\mathsf{u}',
    u'\U0001d5cf': u'\\mathsf{v}',
    u'\U0001d5d0': u'\\mathsf{w}',
    u'\U0001d5d1': u'\\mathsf{x}',
    u'\U0001d5d2': u'\\mathsf{y}',
    u'\U0001d5d3': u'\\mathsf{z}',
    u'\U0001d5d4': u'\\mathsfbf{A}',
    u'\U0001d5d5': u'\\mathsfbf{B}',
    u'\U0001d5d6': u'\\mathsfbf{C}',
    u'\U0001d5d7': u'\\mathsfbf{D}',
    u'\U0001d5d8': u'\\mathsfbf{E}',
    u'\U0001d5d9': u'\\mathsfbf{F}',
    u'\U0001d5da': u'\\mathsfbf{G}',
    u'\U0001d5db': u'\\mathsfbf{H}',
    u'\U0001d5dc': u'\\mathsfbf{I}',
    u'\U0001d5dd': u'\\mathsfbf{J}',
    u'\U0001d5de': u'\\mathsfbf{K}',
    u'\U0001d5df': u'\\mathsfbf{L}',
    u'\U0001d5e0': u'\\mathsfbf{M}',
    u'\U0001d5e1': u'\\mathsfbf{N}',
    u'\U0001d5e2': u'\\mathsfbf{O}',
    u'\U0001d5e3': u'\\mathsfbf{P}',
    u'\U0001d5e4': u'\\mathsfbf{Q}',
    u'\U0001d5e5': u'\\mathsfbf{R}',
    u'\U0001d5e6': u'\\mathsfbf{S}',
    u'\U0001d5e7': u'\\mathsfbf{T}',
    u'\U0001d5e8': u'\\mathsfbf{U}',
    u'\U0001d5e9': u'\\mathsfbf{V}',
    u'\U0001d5ea': u'\\mathsfbf{W}',
    u'\U0001d5eb': u'\\mathsfbf{X}',
    u'\U0001d5ec': u'\\mathsfbf{Y}',
    u'\U0001d5ed': u'\\mathsfbf{Z}',
    u'\U0001d5ee': u'\\mathsfbf{a}',
    u'\U0001d5ef': u'\\mathsfbf{b}',
    u'\U0001d5f0': u'\\mathsfbf{c}',
    u'\U0001d5f1': u'\\mathsfbf{d}',
    u'\U0001d5f2': u'\\mathsfbf{e}',
    u'\U0001d5f3': u'\\mathsfbf{f}',
    u'\U0001d5f4': u'\\mathsfbf{g}',
    u'\U0001d5f5': u'\\mathsfbf{h}',
    u'\U0001d5f6': u'\\mathsfbf{i}',
    u'\U0001d5f7': u'\\mathsfbf{j}',
    u'\U0001d5f8': u'\\mathsfbf{k}',
    u'\U0001d5f9': u'\\mathsfbf{l}',
    u'\U0001d5fa': u'\\mathsfbf{m}',
    u'\U0001d5fb': u'\\mathsfbf{n}',
    u'\U0001d5fc': u'\\mathsfbf{o}',
    u'\U0001d5fd': u'\\mathsfbf{p}',
    u'\U0001d5fe': u'\\mathsfbf{q}',
    u'\U0001d5ff': u'\\mathsfbf{r}',
    u'\U0001d600': u'\\mathsfbf{s}',
    u'\U0001d601': u'\\mathsfbf{t}',
    u'\U0001d602': u'\\mathsfbf{u}',
    u'\U0001d603': u'\\mathsfbf{v}',
    u'\U0001d604': u'\\mathsfbf{w}',
    u'\U0001d605': u'\\mathsfbf{x}',
    u'\U0001d606': u'\\mathsfbf{y}',
    u'\U0001d607': u'\\mathsfbf{z}',
    u'\U0001d608': u'\\mathsfsl{A}',
    u'\U0001d609': u'\\mathsfsl{B}',
    u'\U0001d60a': u'\\mathsfsl{C}',
    u'\U0001d60b': u'\\mathsfsl{D}',
    u'\U0001d60c': u'\\mathsfsl{E}',
    u'\U0001d60d': u'\\mathsfsl{F}',
    u'\U0001d60e': u'\\mathsfsl{G}',
    u'\U0001d60f': u'\\mathsfsl{H}',
    u'\U0001d610': u'\\mathsfsl{I}',
    u'\U0001d611': u'\\mathsfsl{J}',
    u'\U0001d612': u'\\mathsfsl{K}',
    u'\U0001d613': u'\\mathsfsl{L}',
    u'\U0001d614': u'\\mathsfsl{M}',
    u'\U0001d615': u'\\mathsfsl{N}',
    u'\U0001d616': u'\\mathsfsl{O}',
    u'\U0001d617': u'\\mathsfsl{P}',
    u'\U0001d618': u'\\mathsfsl{Q}',
    u'\U0001d619': u'\\mathsfsl{R}',
    u'\U0001d61a': u'\\mathsfsl{S}',
    u'\U0001d61b': u'\\mathsfsl{T}',
    u'\U0001d61c': u'\\mathsfsl{U}',
    u'\U0001d61d': u'\\mathsfsl{V}',
    u'\U0001d61e': u'\\mathsfsl{W}',
    u'\U0001d61f': u'\\mathsfsl{X}',
    u'\U0001d620': u'\\mathsfsl{Y}',
    u'\U0001d621': u'\\mathsfsl{Z}',
    u'\U0001d622': u'\\mathsfsl{a}',
    u'\U0001d623': u'\\mathsfsl{b}',
    u'\U0001d624': u'\\mathsfsl{c}',
    u'\U0001d625': u'\\mathsfsl{d}',
    u'\U0001d626': u'\\mathsfsl{e}',
    u'\U0001d627': u'\\mathsfsl{f}',
    u'\U0001d628': u'\\mathsfsl{g}',
    u'\U0001d629': u'\\mathsfsl{h}',
    u'\U0001d62a': u'\\mathsfsl{i}',
    u'\U0001d62b': u'\\mathsfsl{j}',
    u'\U0001d62c': u'\\mathsfsl{k}',
    u'\U0001d62d': u'\\mathsfsl{l}',
    u'\U0001d62e': u'\\mathsfsl{m}',
    u'\U0001d62f': u'\\mathsfsl{n}',
    u'\U0001d630': u'\\mathsfsl{o}',
    u'\U0001d631': u'\\mathsfsl{p}',
    u'\U0001d632': u'\\mathsfsl{q}',
    u'\U0001d633': u'\\mathsfsl{r}',
    u'\U0001d634': u'\\mathsfsl{s}',
    u'\U0001d635': u'\\mathsfsl{t}',
    u'\U0001d636': u'\\mathsfsl{u}',
    u'\U0001d637': u'\\mathsfsl{v}',
    u'\U0001d638': u'\\mathsfsl{w}',
    u'\U0001d639': u'\\mathsfsl{x}',
    u'\U0001d63a': u'\\mathsfsl{y}',
    u'\U0001d63b': u'\\mathsfsl{z}',
    u'\U0001d63c': u'\\mathsfbfsl{A}',
    u'\U0001d63d': u'\\mathsfbfsl{B}',
    u'\U0001d63e': u'\\mathsfbfsl{C}',
    u'\U0001d63f': u'\\mathsfbfsl{D}',
    u'\U0001d640': u'\\mathsfbfsl{E}',
    u'\U0001d641': u'\\mathsfbfsl{F}',
    u'\U0001d642': u'\\mathsfbfsl{G}',
    u'\U0001d643': u'\\mathsfbfsl{H}',
    u'\U0001d644': u'\\mathsfbfsl{I}',
    u'\U0001d645': u'\\mathsfbfsl{J}',
    u'\U0001d646': u'\\mathsfbfsl{K}',
    u'\U0001d647': u'\\mathsfbfsl{L}',
    u'\U0001d648': u'\\mathsfbfsl{M}',
    u'\U0001d649': u'\\mathsfbfsl{N}',
    u'\U0001d64a': u'\\mathsfbfsl{O}',
    u'\U0001d64b': u'\\mathsfbfsl{P}',
    u'\U0001d64c': u'\\mathsfbfsl{Q}',
    u'\U0001d64d': u'\\mathsfbfsl{R}',
    u'\U0001d64e': u'\\mathsfbfsl{S}',
    u'\U0001d64f': u'\\mathsfbfsl{T}',
    u'\U0001d650': u'\\mathsfbfsl{U}',
    u'\U0001d651': u'\\mathsfbfsl{V}',
    u'\U0001d652': u'\\mathsfbfsl{W}',
    u'\U0001d653': u'\\mathsfbfsl{X}',
    u'\U0001d654': u'\\mathsfbfsl{Y}',
    u'\U0001d655': u'\\mathsfbfsl{Z}',
    u'\U0001d656': u'\\mathsfbfsl{a}',
    u'\U0001d657': u'\\mathsfbfsl{b}',
    u'\U0001d658': u'\\mathsfbfsl{c}',
    u'\U0001d659': u'\\mathsfbfsl{d}',
    u'\U0001d65a': u'\\mathsfbfsl{e}',
    u'\U0001d65b': u'\\mathsfbfsl{f}',
    u'\U0001d65c': u'\\mathsfbfsl{g}',
    u'\U0001d65d': u'\\mathsfbfsl{h}',
    u'\U0001d65e': u'\\mathsfbfsl{i}',
    u'\U0001d65f': u'\\mathsfbfsl{j}',
    u'\U0001d660': u'\\mathsfbfsl{k}',
    u'\U0001d661': u'\\mathsfbfsl{l}',
    u'\U0001d662': u'\\mathsfbfsl{m}',
    u'\U0001d663': u'\\mathsfbfsl{n}',
    u'\U0001d664': u'\\mathsfbfsl{o}',
    u'\U0001d665': u'\\mathsfbfsl{p}',
    u'\U0001d666': u'\\mathsfbfsl{q}',
    u'\U0001d667': u'\\mathsfbfsl{r}',
    u'\U0001d668': u'\\mathsfbfsl{s}',
    u'\U0001d669': u'\\mathsfbfsl{t}',
    u'\U0001d66a': u'\\mathsfbfsl{u}',
    u'\U0001d66b': u'\\mathsfbfsl{v}',
    u'\U0001d66c': u'\\mathsfbfsl{w}',
    u'\U0001d66d': u'\\mathsfbfsl{x}',
    u'\U0001d66e': u'\\mathsfbfsl{y}',
    u'\U0001d66f': u'\\mathsfbfsl{z}',
    u'\U0001d670': u'\\mathtt{A}',
    u'\U0001d671': u'\\mathtt{B}',
    u'\U0001d672': u'\\mathtt{C}',
    u'\U0001d673': u'\\mathtt{D}',
    u'\U0001d674': u'\\mathtt{E}',
    u'\U0001d675': u'\\mathtt{F}',
    u'\U0001d676': u'\\mathtt{G}',
    u'\U0001d677': u'\\mathtt{H}',
    u'\U0001d678': u'\\mathtt{I}',
    u'\U0001d679': u'\\mathtt{J}',
    u'\U0001d67a': u'\\mathtt{K}',
    u'\U0001d67b': u'\\mathtt{L}',
    u'\U0001d67c': u'\\mathtt{M}',
    u'\U0001d67d': u'\\mathtt{N}',
    u'\U0001d67e': u'\\mathtt{O}',
    u'\U0001d67f': u'\\mathtt{P}',
    u'\U0001d680': u'\\mathtt{Q}',
    u'\U0001d681': u'\\mathtt{R}',
    u'\U0001d682': u'\\mathtt{S}',
    u'\U0001d683': u'\\mathtt{T}',
    u'\U0001d684': u'\\mathtt{U}',
    u'\U0001d685': u'\\mathtt{V}',
    u'\U0001d686': u'\\mathtt{W}',
    u'\U0001d687': u'\\mathtt{X}',
    u'\U0001d688': u'\\mathtt{Y}',
    u'\U0001d689': u'\\mathtt{Z}',
    u'\U0001d68a': u'\\mathtt{a}',
    u'\U0001d68b': u'\\mathtt{b}',
    u'\U0001d68c': u'\\mathtt{c}',
    u'\U0001d68d': u'\\mathtt{d}',
    u'\U0001d68e': u'\\mathtt{e}',
    u'\U0001d68f': u'\\mathtt{f}',
    u'\U0001d690': u'\\mathtt{g}',
    u'\U0001d691': u'\\mathtt{h}',
    u'\U0001d692': u'\\mathtt{i}',
    u'\U0001d693': u'\\mathtt{j}',
    u'\U0001d694': u'\\mathtt{k}',
    u'\U0001d695': u'\\mathtt{l}',
    u'\U0001d696': u'\\mathtt{m}',
    u'\U0001d697': u'\\mathtt{n}',
    u'\U0001d698': u'\\mathtt{o}',
    u'\U0001d699': u'\\mathtt{p}',
    u'\U0001d69a': u'\\mathtt{q}',
    u'\U0001d69b': u'\\mathtt{r}',
    u'\U0001d69c': u'\\mathtt{s}',
    u'\U0001d69d': u'\\mathtt{t}',
    u'\U0001d69e': u'\\mathtt{u}',
    u'\U0001d69f': u'\\mathtt{v}',
    u'\U0001d6a0': u'\\mathtt{w}',
    u'\U0001d6a1': u'\\mathtt{x}',
    u'\U0001d6a2': u'\\mathtt{y}',
    u'\U0001d6a3': u'\\mathtt{z}',
    u'\U0001d6a8': u'\\mathbf{\\Alpha}',
    u'\U0001d6a9': u'\\mathbf{\\Beta}',
    u'\U0001d6aa': u'\\mathbf{\\Gamma}',
    u'\U0001d6ab': u'\\mathbf{\\Delta}',
    u'\U0001d6ac': u'\\mathbf{\\Epsilon}',
    u'\U0001d6ad': u'\\mathbf{\\Zeta}',
    u'\U0001d6ae': u'\\mathbf{\\Eta}',
    u'\U0001d6af': u'\\mathbf{\\Theta}',
    u'\U0001d6b0': u'\\mathbf{\\Iota}',
    u'\U0001d6b1': u'\\mathbf{\\Kappa}',
    u'\U0001d6b2': u'\\mathbf{\\Lambda}',
    u'\U0001d6b3': u'M',
    u'\U0001d6b4': u'N',
    u'\U0001d6b5': u'\\mathbf{\\Xi}',
    u'\U0001d6b6': u'O',
    u'\U0001d6b7': u'\\mathbf{\\Pi}',
    u'\U0001d6b8': u'\\mathbf{\\Rho}',
    u'\U0001d6b9': u'\\mathbf{\\vartheta}',
    u'\U0001d6ba': u'\\mathbf{\\Sigma}',
    u'\U0001d6bb': u'\\mathbf{\\Tau}',
    u'\U0001d6bc': u'\\mathbf{\\Upsilon}',
    u'\U0001d6bd': u'\\mathbf{\\Phi}',
    u'\U0001d6be': u'\\mathbf{\\Chi}',
    u'\U0001d6bf': u'\\mathbf{\\Psi}',
    u'\U0001d6c0': u'\\mathbf{\\Omega}',
    u'\U0001d6c1': u'\\mathbf{\\nabla}',
    u'\U0001d6c2': u'\\mathbf{\\Alpha}',
    u'\U0001d6c3': u'\\mathbf{\\Beta}',
    u'\U0001d6c4': u'\\mathbf{\\Gamma}',
    u'\U0001d6c5': u'\\mathbf{\\Delta}',
    u'\U0001d6c6': u'\\mathbf{\\Epsilon}',
    u'\U0001d6c7': u'\\mathbf{\\Zeta}',
    u'\U0001d6c8': u'\\mathbf{\\Eta}',
    u'\U0001d6c9': u'\\mathbf{\\theta}',
    u'\U0001d6ca': u'\\mathbf{\\Iota}',
    u'\U0001d6cb': u'\\mathbf{\\Kappa}',
    u'\U0001d6cc': u'\\mathbf{\\Lambda}',
    u'\U0001d6cd': u'M',
    u'\U0001d6ce': u'N',
    u'\U0001d6cf': u'\\mathbf{\\Xi}',
    u'\U0001d6d0': u'O',
    u'\U0001d6d1': u'\\mathbf{\\Pi}',
    u'\U0001d6d2': u'\\mathbf{\\Rho}',
    u'\U0001d6d3': u'\\mathbf{\\varsigma}',
    u'\U0001d6d4': u'\\mathbf{\\Sigma}',
    u'\U0001d6d5': u'\\mathbf{\\Tau}',
    u'\U0001d6d6': u'\\mathbf{\\Upsilon}',
    u'\U0001d6d7': u'\\mathbf{\\Phi}',
    u'\U0001d6d8': u'\\mathbf{\\Chi}',
    u'\U0001d6d9': u'\\mathbf{\\Psi}',
    u'\U0001d6da': u'\\mathbf{\\Omega}',
    u'\U0001d6db': u'\\partial',
    u'\U0001d6dc': u'\\in',
    u'\U0001d6dd': u'\\mathbf{\\vartheta}',
    u'\U0001d6de': u'\\mathbf{\\varkappa}',
    u'\U0001d6df': u'\\mathbf{\\phi}',
    u'\U0001d6e0': u'\\mathbf{\\varrho}',
    u'\U0001d6e1': u'\\mathbf{\\varpi}',
    u'\U0001d6e2': u'\\mathsl{\\Alpha}',
    u'\U0001d6e3': u'\\mathsl{\\Beta}',
    u'\U0001d6e4': u'\\mathsl{\\Gamma}',
    u'\U0001d6e5': u'\\mathsl{\\Delta}',
    u'\U0001d6e6': u'\\mathsl{\\Epsilon}',
    u'\U0001d6e7': u'\\mathsl{\\Zeta}',
    u'\U0001d6e8': u'\\mathsl{\\Eta}',
    u'\U0001d6e9': u'\\mathsl{\\Theta}',
    u'\U0001d6ea': u'\\mathsl{\\Iota}',
    u'\U0001d6eb': u'\\mathsl{\\Kappa}',
    u'\U0001d6ec': u'\\mathsl{\\Lambda}',
    u'\U0001d6ed': u'M',
    u'\U0001d6ee': u'N',
    u'\U0001d6ef': u'\\mathsl{\\Xi}',
    u'\U0001d6f0': u'O',
    u'\U0001d6f1': u'\\mathsl{\\Pi}',
    u'\U0001d6f2': u'\\mathsl{\\Rho}',
    u'\U0001d6f3': u'\\mathsl{\\vartheta}',
    u'\U0001d6f4': u'\\mathsl{\\Sigma}',
    u'\U0001d6f5': u'\\mathsl{\\Tau}',
    u'\U0001d6f6': u'\\mathsl{\\Upsilon}',
    u'\U0001d6f7': u'\\mathsl{\\Phi}',
    u'\U0001d6f8': u'\\mathsl{\\Chi}',
    u'\U0001d6f9': u'\\mathsl{\\Psi}',
    u'\U0001d6fa': u'\\mathsl{\\Omega}',
    u'\U0001d6fb': u'\\mathsl{\\nabla}',
    u'\U0001d6fc': u'\\mathsl{\\Alpha}',
    u'\U0001d6fd': u'\\mathsl{\\Beta}',
    u'\U0001d6fe': u'\\mathsl{\\Gamma}',
    u'\U0001d6ff': u'\\mathsl{\\Delta}',
    u'\U0001d700': u'\\mathsl{\\Epsilon}',
    u'\U0001d701': u'\\mathsl{\\Zeta}',
    u'\U0001d702': u'\\mathsl{\\Eta}',
    u'\U0001d703': u'\\mathsl{\\Theta}',
    u'\U0001d704': u'\\mathsl{\\Iota}',
    u'\U0001d705': u'\\mathsl{\\Kappa}',
    u'\U0001d706': u'\\mathsl{\\Lambda}',
    u'\U0001d707': u'M',
    u'\U0001d708': u'N',
    u'\U0001d709': u'\\mathsl{\\Xi}',
    u'\U0001d70a': u'O',
    u'\U0001d70b': u'\\mathsl{\\Pi}',
    u'\U0001d70c': u'\\mathsl{\\Rho}',
    u'\U0001d70d': u'\\mathsl{\\varsigma}',
    u'\U0001d70e': u'\\mathsl{\\Sigma}',
    u'\U0001d70f': u'\\mathsl{\\Tau}',
    u'\U0001d710': u'\\mathsl{\\Upsilon}',
    u'\U0001d711': u'\\mathsl{\\Phi}',
    u'\U0001d712': u'\\mathsl{\\Chi}',
    u'\U0001d713': u'\\mathsl{\\Psi}',
    u'\U0001d714': u'\\mathsl{\\Omega}',
    u'\U0001d715': u'\\partial',
    u'\U0001d716': u'\\in',
    u'\U0001d717': u'\\mathsl{\\vartheta}',
    u'\U0001d718': u'\\mathsl{\\varkappa}',
    u'\U0001d719': u'\\mathsl{\\phi}',
    u'\U0001d71a': u'\\mathsl{\\varrho}',
    u'\U0001d71b': u'\\mathsl{\\varpi}',
    u'\U0001d71c': u'\\mathbit{\\Alpha}',
    u'\U0001d71d': u'\\mathbit{\\Beta}',
    u'\U0001d71e': u'\\mathbit{\\Gamma}',
    u'\U0001d71f': u'\\mathbit{\\Delta}',
    u'\U0001d720': u'\\mathbit{\\Epsilon}',
    u'\U0001d721': u'\\mathbit{\\Zeta}',
    u'\U0001d722': u'\\mathbit{\\Eta}',
    u'\U0001d723': u'\\mathbit{\\Theta}',
    u'\U0001d724': u'\\mathbit{\\Iota}',
    u'\U0001d725': u'\\mathbit{\\Kappa}',
    u'\U0001d726': u'\\mathbit{\\Lambda}',
    u'\U0001d727': u'M',
    u'\U0001d728': u'N',
    u'\U0001d729': u'\\mathbit{\\Xi}',
    u'\U0001d72a': u'O',
    u'\U0001d72b': u'\\mathbit{\\Pi}',
    u'\U0001d72c': u'\\mathbit{\\Rho}',
    u'\U0001d72d': u'\\mathbit{O}',
    u'\U0001d72e': u'\\mathbit{\\Sigma}',
    u'\U0001d72f': u'\\mathbit{\\Tau}',
    u'\U0001d730': u'\\mathbit{\\Upsilon}',
    u'\U0001d731': u'\\mathbit{\\Phi}',
    u'\U0001d732': u'\\mathbit{\\Chi}',
    u'\U0001d733': u'\\mathbit{\\Psi}',
    u'\U0001d734': u'\\mathbit{\\Omega}',
    u'\U0001d735': u'\\mathbit{\\nabla}',
    u'\U0001d736': u'\\mathbit{\\Alpha}',
    u'\U0001d737': u'\\mathbit{\\Beta}',
    u'\U0001d738': u'\\mathbit{\\Gamma}',
    u'\U0001d739': u'\\mathbit{\\Delta}',
    u'\U0001d73a': u'\\mathbit{\\Epsilon}',
    u'\U0001d73b': u'\\mathbit{\\Zeta}',
    u'\U0001d73c': u'\\mathbit{\\Eta}',
    u'\U0001d73d': u'\\mathbit{\\Theta}',
    u'\U0001d73e': u'\\mathbit{\\Iota}',
    u'\U0001d73f': u'\\mathbit{\\Kappa}',
    u'\U0001d740': u'\\mathbit{\\Lambda}',
    u'\U0001d741': u'M',
    u'\U0001d742': u'N',
    u'\U0001d743': u'\\mathbit{\\Xi}',
    u'\U0001d744': u'O',
    u'\U0001d745': u'\\mathbit{\\Pi}',
    u'\U0001d746': u'\\mathbit{\\Rho}',
    u'\U0001d747': u'\\mathbit{\\varsigma}',
    u'\U0001d748': u'\\mathbit{\\Sigma}',
    u'\U0001d749': u'\\mathbit{\\Tau}',
    u'\U0001d74a': u'\\mathbit{\\Upsilon}',
    u'\U0001d74b': u'\\mathbit{\\Phi}',
    u'\U0001d74c': u'\\mathbit{\\Chi}',
    u'\U0001d74d': u'\\mathbit{\\Psi}',
    u'\U0001d74e': u'\\mathbit{\\Omega}',
    u'\U0001d74f': u'\\partial',
    u'\U0001d750': u'\\in',
    u'\U0001d751': u'\\mathbit{\\vartheta}',
    u'\U0001d752': u'\\mathbit{\\varkappa}',
    u'\U0001d753': u'\\mathbit{\\phi}',
    u'\U0001d754': u'\\mathbit{\\varrho}',
    u'\U0001d755': u'\\mathbit{\\varpi}',
    u'\U0001d756': u'\\mathsfbf{\\Alpha}',
    u'\U0001d757': u'\\mathsfbf{\\Beta}',
    u'\U0001d758': u'\\mathsfbf{\\Gamma}',
    u'\U0001d759': u'\\mathsfbf{\\Delta}',
    u'\U0001d75a': u'\\mathsfbf{\\Epsilon}',
    u'\U0001d75b': u'\\mathsfbf{\\Zeta}',
    u'\U0001d75c': u'\\mathsfbf{\\Eta}',
    u'\U0001d75d': u'\\mathsfbf{\\Theta}',
    u'\U0001d75e': u'\\mathsfbf{\\Iota}',
    u'\U0001d75f': u'\\mathsfbf{\\Kappa}',
    u'\U0001d760': u'\\mathsfbf{\\Lambda}',
    u'\U0001d761': u'M',
    u'\U0001d762': u'N',
    u'\U0001d763': u'\\mathsfbf{\\Xi}',
    u'\U0001d764': u'O',
    u'\U0001d765': u'\\mathsfbf{\\Pi}',
    u'\U0001d766': u'\\mathsfbf{\\Rho}',
    u'\U0001d767': u'\\mathsfbf{\\vartheta}',
    u'\U0001d768': u'\\mathsfbf{\\Sigma}',
    u'\U0001d769': u'\\mathsfbf{\\Tau}',
    u'\U0001d76a': u'\\mathsfbf{\\Upsilon}',
    u'\U0001d76b': u'\\mathsfbf{\\Phi}',
    u'\U0001d76c': u'\\mathsfbf{\\Chi}',
    u'\U0001d76d': u'\\mathsfbf{\\Psi}',
    u'\U0001d76e': u'\\mathsfbf{\\Omega}',
    u'\U0001d76f': u'\\mathsfbf{\\nabla}',
    u'\U0001d770': u'\\mathsfbf{\\Alpha}',
    u'\U0001d771': u'\\mathsfbf{\\Beta}',
    u'\U0001d772': u'\\mathsfbf{\\Gamma}',
    u'\U0001d773': u'\\mathsfbf{\\Delta}',
    u'\U0001d774': u'\\mathsfbf{\\Epsilon}',
    u'\U0001d775': u'\\mathsfbf{\\Zeta}',
    u'\U0001d776': u'\\mathsfbf{\\Eta}',
    u'\U0001d777': u'\\mathsfbf{\\Theta}',
    u'\U0001d778': u'\\mathsfbf{\\Iota}',
    u'\U0001d779': u'\\mathsfbf{\\Kappa}',
    u'\U0001d77a': u'\\mathsfbf{\\Lambda}',
    u'\U0001d77b': u'M',
    u'\U0001d77c': u'N',
    u'\U0001d77d': u'\\mathsfbf{\\Xi}',
    u'\U0001d77e': u'O',
    u'\U0001d77f': u'\\mathsfbf{\\Pi}',
    u'\U0001d780': u'\\mathsfbf{\\Rho}',
    u'\U0001d781': u'\\mathsfbf{\\varsigma}',
    u'\U0001d782': u'\\mathsfbf{\\Sigma}',
    u'\U0001d783': u'\\mathsfbf{\\Tau}',
    u'\U0001d784': u'\\mathsfbf{\\Upsilon}',
    u'\U0001d785': u'\\mathsfbf{\\Phi}',
    u'\U0001d786': u'\\mathsfbf{\\Chi}',
    u'\U0001d787': u'\\mathsfbf{\\Psi}',
    u'\U0001d788': u'\\mathsfbf{\\Omega}',
    u'\U0001d789': u'\\partial',
    u'\U0001d78a': u'\\in',
    u'\U0001d78b': u'\\mathsfbf{\\vartheta}',
    u'\U0001d78c': u'\\mathsfbf{\\varkappa}',
    u'\U0001d78d': u'\\mathsfbf{\\phi}',
    u'\U0001d78e': u'\\mathsfbf{\\varrho}',
    u'\U0001d78f': u'\\mathsfbf{\\varpi}',
    u'\U0001d790': u'\\mathsfbfsl{\\Alpha}',
    u'\U0001d791': u'\\mathsfbfsl{\\Beta}',
    u'\U0001d792': u'\\mathsfbfsl{\\Gamma}',
    u'\U0001d793': u'\\mathsfbfsl{\\Delta}',
    u'\U0001d794': u'\\mathsfbfsl{\\Epsilon}',
    u'\U0001d795': u'\\mathsfbfsl{\\Zeta}',
    u'\U0001d796': u'\\mathsfbfsl{\\Eta}',
    u'\U0001d797': u'\\mathsfbfsl{\\vartheta}',
    u'\U0001d798': u'\\mathsfbfsl{\\Iota}',
    u'\U0001d799': u'\\mathsfbfsl{\\Kappa}',
    u'\U0001d79a': u'\\mathsfbfsl{\\Lambda}',
    u'\U0001d79b': u'M',
    u'\U0001d79c': u'N',
    u'\U0001d79d': u'\\mathsfbfsl{\\Xi}',
    u'\U0001d79e': u'O',
    u'\U0001d79f': u'\\mathsfbfsl{\\Pi}',
    u'\U0001d7a0': u'\\mathsfbfsl{\\Rho}',
    u'\U0001d7a1': u'\\mathsfbfsl{\\vartheta}',
    u'\U0001d7a2': u'\\mathsfbfsl{\\Sigma}',
    u'\U0001d7a3': u'\\mathsfbfsl{\\Tau}',
    u'\U0001d7a4': u'\\mathsfbfsl{\\Upsilon}',
    u'\U0001d7a5': u'\\mathsfbfsl{\\Phi}',
    u'\U0001d7a6': u'\\mathsfbfsl{\\Chi}',
    u'\U0001d7a7': u'\\mathsfbfsl{\\Psi}',
    u'\U0001d7a8': u'\\mathsfbfsl{\\Omega}',
    u'\U0001d7a9': u'\\mathsfbfsl{\\nabla}',
    u'\U0001d7aa': u'\\mathsfbfsl{\\Alpha}',
    u'\U0001d7ab': u'\\mathsfbfsl{\\Beta}',
    u'\U0001d7ac': u'\\mathsfbfsl{\\Gamma}',
    u'\U0001d7ad': u'\\mathsfbfsl{\\Delta}',
    u'\U0001d7ae': u'\\mathsfbfsl{\\Epsilon}',
    u'\U0001d7af': u'\\mathsfbfsl{\\Zeta}',
    u'\U0001d7b0': u'\\mathsfbfsl{\\Eta}',
    u'\U0001d7b1': u'\\mathsfbfsl{\\vartheta}',
    u'\U0001d7b2': u'\\mathsfbfsl{\\Iota}',
    u'\U0001d7b3': u'\\mathsfbfsl{\\Kappa}',
    u'\U0001d7b4': u'\\mathsfbfsl{\\Lambda}',
    u'\U0001d7b5': u'M',
    u'\U0001d7b6': u'N',
    u'\U0001d7b7': u'\\mathsfbfsl{\\Xi}',
    u'\U0001d7b8': u'O',
    u'\U0001d7b9': u'\\mathsfbfsl{\\Pi}',
    u'\U0001d7ba': u'\\mathsfbfsl{\\Rho}',
    u'\U0001d7bb': u'\\mathsfbfsl{\\varsigma}',
    u'\U0001d7bc': u'\\mathsfbfsl{\\Sigma}',
    u'\U0001d7bd': u'\\mathsfbfsl{\\Tau}',
    u'\U0001d7be': u'\\mathsfbfsl{\\Upsilon}',
    u'\U0001d7bf': u'\\mathsfbfsl{\\Phi}',
    u'\U0001d7c0': u'\\mathsfbfsl{\\Chi}',
    u'\U0001d7c1': u'\\mathsfbfsl{\\Psi}',
    u'\U0001d7c2': u'\\mathsfbfsl{\\Omega}',
    u'\U0001d7c3': u'\\partial',
    u'\U0001d7c4': u'\\in',
    u'\U0001d7c5': u'\\mathsfbfsl{\\vartheta}',
    u'\U0001d7c6': u'\\mathsfbfsl{\\varkappa}',
    u'\U0001d7c7': u'\\mathsfbfsl{\\phi}',
    u'\U0001d7c8': u'\\mathsfbfsl{\\varrho}',
    u'\U0001d7c9': u'\\mathsfbfsl{\\varpi}',
    u'\U0001d7ce': u'\\mathbf{0}',
    u'\U0001d7cf': u'\\mathbf{1}',
    u'\U0001d7d0': u'\\mathbf{2}',
    u'\U0001d7d1': u'\\mathbf{3}',
    u'\U0001d7d2': u'\\mathbf{4}',
    u'\U0001d7d3': u'\\mathbf{5}',
    u'\U0001d7d4': u'\\mathbf{6}',
    u'\U0001d7d5': u'\\mathbf{7}',
    u'\U0001d7d6': u'\\mathbf{8}',
    u'\U0001d7d7': u'\\mathbf{9}',
    u'\U0001d7d8': u'\\mathbb{0}',
    u'\U0001d7d9': u'\\mathbb{1}',
    u'\U0001d7da': u'\\mathbb{2}',
    u'\U0001d7db': u'\\mathbb{3}',
    u'\U0001d7dc': u'\\mathbb{4}',
    u'\U0001d7dd': u'\\mathbb{5}',
    u'\U0001d7de': u'\\mathbb{6}',
    u'\U0001d7df': u'\\mathbb{7}',
    u'\U0001d7e0': u'\\mathbb{8}',
    u'\U0001d7e1': u'\\mathbb{9}',
    u'\U0001d7e2': u'\\mathsf{0}',
    u'\U0001d7e3': u'\\mathsf{1}',
    u'\U0001d7e4': u'\\mathsf{2}',
    u'\U0001d7e5': u'\\mathsf{3}',
    u'\U0001d7e6': u'\\mathsf{4}',
    u'\U0001d7e7': u'\\mathsf{5}',
    u'\U0001d7e8': u'\\mathsf{6}',
    u'\U0001d7e9': u'\\mathsf{7}',
    u'\U0001d7ea': u'\\mathsf{8}',
    u'\U0001d7eb': u'\\mathsf{9}',
    u'\U0001d7ec': u'\\mathsfbf{0}',
    u'\U0001d7ed': u'\\mathsfbf{1}',
    u'\U0001d7ee': u'\\mathsfbf{2}',
    u'\U0001d7ef': u'\\mathsfbf{3}',
    u'\U0001d7f0': u'\\mathsfbf{4}',
    u'\U0001d7f1': u'\\mathsfbf{5}',
    u'\U0001d7f2': u'\\mathsfbf{6}',
    u'\U0001d7f3': u'\\mathsfbf{7}',
    u'\U0001d7f4': u'\\mathsfbf{8}',
    u'\U0001d7f5': u'\\mathsfbf{9}',
    u'\U0001d7f6': u'\\mathtt{0}',
    u'\U0001d7f7': u'\\mathtt{1}',
    u'\U0001d7f8': u'\\mathtt{2}',
    u'\U0001d7f9': u'\\mathtt{3}',
    u'\U0001d7fa': u'\\mathtt{4}',
    u'\U0001d7fb': u'\\mathtt{5}',
    u'\U0001d7fc': u'\\mathtt{6}',
    u'\U0001d7fd': u'\\mathtt{7}',
    u'\U0001d7fe': u'\\mathtt{8}',
    u'\U0001d7ff': u'\\mathtt{9}'}


tex_to_unicode_map = {u"'": u'\u2019',
    u"''''": u'\u2057',
    u"'n": u'\u0149',
    u',': u'\u201a',
    u',,': u'\u201e',
    u'-': u'\u2212',
    u'.': u'\u2024',
    u'..': u'\u2025',
    u':=': u'\u2254',
    u'<\\kern-0.58em(': u'\u2993',
    u'=:': u'\u2255',
    u'IJ': u'\u0132',
    u'M': u'\U0001d7b5',
    u'N': u'\U0001d7b6',
    u'O': u'\U0001d7b8',
    u'\\"': u'\u0308',
    u'\\"{A}': u'\xc4',
    u'\\"{E}': u'\xcb',
    u'\\"{I}': u'\xcf',
    u'\\"{O}': u'\xd6',
    u'\\"{U}': u'\xdc',
    u'\\"{Y}': u'\u0178',
    u'\\"{\\i}': u'\xef',
    u'\\"{a}': u'\xe4',
    u'\\"{e}': u'\xeb',
    u'\\"{o}': u'\xf6',
    u'\\"{u}': u'\xfc',
    u'\\"{y}': u'\xff',
    u'\\#': u'#',
    u'\\%': u'%',
    u'\\&': u'&',
    u"\\'": u'\u0301',
    u"\\'{$\\alpha$}": u'\u03ac',
    u"\\'{A}": u'\u0386',
    u"\\'{C}": u'\u0106',
    u"\\'{E}": u'\u0388',
    u"\\'{H}": u'\u0389',
    u"\\'{I}": u'\xcd',
    u"\\'{L}": u'\u0139',
    u"\\'{N}": u'\u0143',
    u"\\'{O}": u'\xd3',
    u"\\'{R}": u'\u0154',
    u"\\'{S}": u'\u015a',
    u"\\'{U}": u'\xda',
    u"\\'{Y}": u'\xdd',
    u"\\'{Z}": u'\u0179',
    u"\\'{\\i}": u'\xed',
    u"\\'{a}": u'\xe1',
    u"\\'{c}": u'\u0107',
    u"\\'{e}": u'\xe9',
    u"\\'{g}": u'\u01f5',
    u"\\'{l}": u'\u013a',
    u"\\'{n}": u'\u0144',
    u"\\'{o}": u'\u03cc',
    u"\\'{r}": u'\u0155',
    u"\\'{s}": u'\u015b',
    u"\\'{u}": u'\xfa',
    u"\\'{y}": u'\xfd',
    u"\\'{z}": u'\u017a',
    u"\\'{}O": u'\u038c',
    u"\\'{}{I}": u'\u038a',
    u'\\-': u'\xad',
    u'\\.': u'\u0307',
    u'\\.{C}': u'\u010a',
    u'\\.{E}': u'\u0116',
    u'\\.{G}': u'\u0120',
    u'\\.{I}': u'\u0130',
    u'\\.{Z}': u'\u017b',
    u'\\.{c}': u'\u010b',
    u'\\.{e}': u'\u0117',
    u'\\.{g}': u'\u0121',
    u'\\.{z}': u'\u017c',
    u'\\;': u'\u2009\u200a\u200a',
    u'\\=': u'\u0304',
    u'\\={A}': u'\u0100',
    u'\\={E}': u'\u0112',
    u'\\={I}': u'\u012a',
    u'\\={O}': u'\u014c',
    u'\\={U}': u'\u016a',
    u'\\={\\i}': u'\u012b',
    u'\\={a}': u'\u0101',
    u'\\={e}': u'\u0113',
    u'\\={o}': u'\u014d',
    u'\\={u}': u'\u016b',
    u'\\AA': u'\u212b',
    u'\\AE': u'\xc6',
    u'\\Alpha': u'\u0391',
    u'\\Angle': u'\u299c',
    u'\\Beta': u'\u0392',
    u'\\Bumpeq': u'\u224e',
    u'\\Cap': u'\u22d2',
    u'\\Chi': u'\u03a7',
    u'\\Colon': u'\u2237',
    u'\\Cup': u'\u22d3',
    u'\\DH': u'\xd0',
    u'\\DJ': u'\u0110',
    u'\\Delta': u'\u0394',
    u'\\Digamma': u'\u03dc',
    u'\\DownArrowBar': u'\u2913',
    u'\\DownArrowUpArrow': u'\u21f5',
    u'\\DownLeftRightVector': u'\u2950',
    u'\\DownLeftTeeVector': u'\u295e',
    u'\\DownLeftVectorBar': u'\u2956',
    u'\\DownRightTeeVector': u'\u295f',
    u'\\DownRightVectorBar': u'\u2957',
    u'\\Downarrow': u'\u21d3',
    u'\\ElOr': u'\u2a56',
    u'\\Elolarr': u'\u2940',
    u'\\Elorarr': u'\u2941',
    u'\\Elroang': u'\u2986',
    u'\\ElsevierGlyph{2129}': u'\u2129',
    u'\\ElsevierGlyph{21B3}': u'\u21b3',
    u'\\ElsevierGlyph{2232}': u'\u2232',
    u'\\ElsevierGlyph{2233}': u'\u2233',
    u'\\ElsevierGlyph{2238}': u'\u2238',
    u'\\ElsevierGlyph{2242}': u'\u2242',
    u'\\ElsevierGlyph{225A}': u'\u2a63',
    u'\\ElsevierGlyph{225F}': u'\u225f',
    u'\\ElsevierGlyph{2274}': u'\u2274',
    u'\\ElsevierGlyph{2275}': u'\u2275',
    u'\\ElsevierGlyph{22C0}': u'\u22c0',
    u'\\ElsevierGlyph{22C1}': u'\u22c1',
    u'\\ElsevierGlyph{300A}': u'\u300a',
    u'\\ElsevierGlyph{300B}': u'\u300b',
    u'\\ElsevierGlyph{3018}': u'\u3018',
    u'\\ElsevierGlyph{3019}': u'\u3019',
    u'\\ElsevierGlyph{E20A}': u'\u2926',
    u'\\ElsevierGlyph{E20B}': u'\u2925',
    u'\\ElsevierGlyph{E20C}': u'\u2923',
    u'\\ElsevierGlyph{E20D}': u'\u2924',
    u'\\ElsevierGlyph{E20E}': u'\u2928',
    u'\\ElsevierGlyph{E20F}': u'\u2929',
    u'\\ElsevierGlyph{E210}': u'\u292a',
    u'\\ElsevierGlyph{E211}': u'\u2927',
    u'\\ElsevierGlyph{E212}': u'\u2905',
    u'\\ElsevierGlyph{E214}': u'\u297c',
    u'\\ElsevierGlyph{E215}': u'\u297d',
    u'\\ElsevierGlyph{E219}': u'\u2937',
    u'\\ElsevierGlyph{E21A}': u'\u2936',
    u'\\ElsevierGlyph{E21C}': u'\u2933',
    u'\\ElsevierGlyph{E21D}': u'\u2933\u0338',
    u'\\ElsevierGlyph{E259}': u'\u2a3c',
    u'\\ElsevierGlyph{E25A}': u'\u2a25',
    u'\\ElsevierGlyph{E25B}': u'\u2a2a',
    u'\\ElsevierGlyph{E25C}': u'\u2a2d',
    u'\\ElsevierGlyph{E25D}': u'\u2a2e',
    u'\\ElsevierGlyph{E25E}': u'\u2a35',
    u'\\ElsevierGlyph{E260}': u'\u29b5',
    u'\\ElsevierGlyph{E291}': u'\u2994',
    u'\\ElsevierGlyph{E30D}': u'\u2aeb',
    u'\\ElsevierGlyph{E36E}': u'\u2a55',
    u'\\ElsevierGlyph{E372}': u'\u29dc',
    u'\\ElsevierGlyph{E381}': u'\u25b1',
    u'\\ElsevierGlyph{E395}': u'\u2a10',
    u'\\ElsevierGlyph{E61B}': u'\u29b6',
    u'\\ElsevierGlyph{E838}': u'\u233d',
    u'\\Elxsqcup': u'\u2a06',
    u'\\Elxuplus': u'\u2a04',
    u'\\ElzAnd': u'\u2a53',
    u'\\ElzCint': u'\u2a0d',
    u'\\ElzInf': u'\u2a07',
    u'\\ElzLap': u'\u29ca',
    u'\\ElzOr': u'\u2a54',
    u'\\ElzRlarr': u'\u2942',
    u'\\ElzSup': u'\u2a08',
    u'\\ElzThr': u'\u2a05',
    u'\\ElzTimes': u'\u2a2f',
    u'\\Elzbar': u'\u0336',
    u'\\Elzbtdl': u'\u026c',
    u'\\Elzcirfb': u'\u25d2',
    u'\\Elzcirfl': u'\u25d0',
    u'\\Elzcirfr': u'\u25d1',
    u'\\Elzclomeg': u'\u0277',
    u'\\Elzddfnc': u'\u2999',
    u'\\Elzdefas': u'\u29cb',
    u'\\Elzdlcorn': u'\u23a3',
    u'\\Elzdshfnc': u'\u2506',
    u'\\Elzdyogh': u'\u02a4',
    u'\\Elzesh': u'\u0283',
    u'\\Elzfhr': u'\u027e',
    u'\\Elzglst': u'\u0294',
    u'\\Elzhlmrk': u'\u02d1',
    u'\\Elzinglst': u'\u0296',
    u'\\Elzinvv': u'\u028c',
    u'\\Elzinvw': u'\u028d',
    u'\\Elzlmrk': u'\u02d0',
    u'\\Elzlow': u'\u02d5',
    u'\\Elzlpargt': u'\u29a0',
    u'\\Elzltlmr': u'\u0271',
    u'\\Elzltln': u'\u0272',
    u'\\Elzminhat': u'\u2a5f',
    u'\\Elzopeno': u'\u0254',
    u'\\Elzpalh': u'\u0321',
    u'\\Elzpbgam': u'\u0264',
    u'\\Elzpgamma': u'\u0263',
    u'\\Elzpscrv': u'\u028b',
    u'\\Elzpupsil': u'\u028a',
    u'\\ElzrLarr': u'\u2944',
    u'\\Elzrais': u'\u02d4',
    u'\\Elzrarrx': u'\u2947',
    u'\\Elzreapos': u'\u201b',
    u'\\Elzreglst': u'\u0295',
    u'\\Elzrh': u'\u0322',
    u'\\Elzrl': u'\u027c',
    u'\\Elzrtld': u'\u0256',
    u'\\Elzrtll': u'\u026d',
    u'\\Elzrtln': u'\u0273',
    u'\\Elzrtlr': u'\u027d',
    u'\\Elzrtls': u'\u0282',
    u'\\Elzrtlt': u'\u0288',
    u'\\Elzrtlz': u'\u0290',
    u'\\Elzrttrnr': u'\u027b',
    u'\\Elzrvbull': u'\u25d8',
    u'\\Elzsbbrg': u'\u032a',
    u'\\Elzsblhr': u'\u02d3',
    u'\\Elzsbrhr': u'\u02d2',
    u'\\Elzschwa': u'\u0259',
    u'\\Elzsqfl': u'\u25e7',
    u'\\Elzsqfnw': u'\u2519',
    u'\\Elzsqfr': u'\u25e8',
    u'\\Elzsqfse': u'\u25ea',
    u'\\Elzsqspne': u'\u22e5',
    u'\\Elztdcol': u'\u2af6',
    u'\\Elztesh': u'\u02a7',
    u'\\Elztfnc': u'\u2980',
    u'\\Elztrna': u'\u0250',
    u'\\Elztrnh': u'\u0265',
    u'\\Elztrnm': u'\u026f',
    u'\\Elztrnmlr': u'\u0270',
    u'\\Elztrnr': u'\u0279',
    u'\\Elztrnrl': u'\u027a',
    u'\\Elztrnsa': u'\u0252',
    u'\\Elztrnt': u'\u0287',
    u'\\Elztrny': u'\u028e',
    u'\\Elzverti': u'\u02cc',
    u'\\Elzverts': u'\u02c8',
    u'\\Elzvrecto': u'\u25af',
    u'\\Elzxh': u'\u0127',
    u'\\Elzxl': u'\u0335',
    u'\\Elzxrat': u'\u211e',
    u'\\Elzyogh': u'\u0292',
    u'\\Epsilon': u'\u0395',
    u'\\Equal': u'\u2a75',
    u'\\Eta': u'\u0397',
    u'\\Gamma': u'\u0393',
    u'\\H': u'\u030b',
    u'\\H{O}': u'\u0150',
    u'\\H{U}': u'\u0170',
    u'\\H{o}': u'\u0151',
    u'\\H{u}': u'\u0171',
    u'\\H{}': u'\u02dd',
    u'\\Iota': u'\u0399',
    u'\\Kappa': u'\u039a',
    u'\\Koppa': u'\u03de',
    u'\\L': u'\u0141',
    u'\\Lambda': u'\u039b',
    u'\\LeftDownTeeVector': u'\u2961',
    u'\\LeftDownVectorBar': u'\u2959',
    u'\\LeftRightVector': u'\u294e',
    u'\\LeftTeeVector': u'\u295a',
    u'\\LeftTriangleBar': u'\u29cf',
    u'\\LeftUpDownVector': u'\u2951',
    u'\\LeftUpTeeVector': u'\u2960',
    u'\\LeftUpVectorBar': u'\u2958',
    u'\\LeftVectorBar': u'\u2952',
    u'\\Leftarrow': u'\u21d0',
    u'\\Leftrightarrow': u'\u21d4',
    u'\\Lleftarrow': u'\u21da',
    u'\\Longleftarrow': u'\u27f8',
    u'\\Longleftrightarrow': u'\u27fa',
    u'\\Longrightarrow': u'\u27f9',
    u'\\Lsh': u'\u21b0',
    u'\\NG': u'\u014a',
    u'\\NestedGreaterGreater': u'\u2aa2',
    u'\\NestedLessLess': u'\u2aa1',
    u'\\NotEqualTilde': u'\u2242\u0338',
    u'\\NotGreaterGreater': u'\u226b\u0338',
    u'\\NotHumpDownHump': u'\u224e\u0338',
    u'\\NotHumpEqual': u'\u224f\u0338',
    u'\\NotLeftTriangleBar': u'\u29cf\u0338',
    u'\\NotLessLess': u'\u226a\u0338',
    u'\\NotNestedGreaterGreater': u'\u2aa2\u0338',
    u'\\NotNestedLessLess': u'\u2aa1\u0338',
    u'\\NotPrecedesTilde': u'\u227e\u0338',
    u'\\NotRightTriangleBar': u'\u29d0\u0338',
    u'\\NotSquareSubset': u'\u228f\u0338',
    u'\\NotSquareSuperset': u'\u2290\u0338',
    u'\\NotSucceedsTilde': u'\u227f\u0338',
    u'\\O': u'\xd8',
    u'\\OE': u'\u0152',
    u'\\Omega': u'\u2126',
    u'\\Phi': u'\u03a6',
    u'\\Pi': u'\u03a0',
    u'\\Pisymbol{ppi020}{105}': u'\u2a9e',
    u'\\Pisymbol{ppi020}{117}': u'\u2a9d',
    u'\\Pisymbol{ppi022}{87}': u'\u03d0',
    u'\\Psi': u'\u03a8',
    u'\\ReverseUpEquilibrium': u'\u296f',
    u'\\Rho': u'\u03a1',
    u'\\RightDownTeeVector': u'\u295d',
    u'\\RightDownVectorBar': u'\u2955',
    u'\\RightTeeVector': u'\u295b',
    u'\\RightTriangleBar': u'\u29d0',
    u'\\RightUpDownVector': u'\u294f',
    u'\\RightUpTeeVector': u'\u295c',
    u'\\RightUpVectorBar': u'\u2954',
    u'\\RightVectorBar': u'\u2953',
    u'\\Rightarrow': u'\u21d2',
    u'\\RoundImplies': u'\u2970',
    u'\\Rrightarrow': u'\u21db',
    u'\\Rsh': u'\u21b1',
    u'\\RuleDelayed': u'\u29f4',
    u'\\Sampi': u'\u03e0',
    u'\\Sigma': u'\u03a3',
    u'\\Stigma': u'\u03da',
    u'\\Subset': u'\u22d0',
    u'\\Supset': u'\u22d1',
    u'\\TH': u'\xde',
    u'\\Tau': u'\u03a4',
    u'\\Theta': u'\u0398',
    u'\\UpArrowBar': u'\u2912',
    u'\\UpEquilibrium': u'\u296e',
    u'\\Uparrow': u'\u21d1',
    u'\\Updownarrow': u'\u21d5',
    u'\\Upsilon': u'\u03d2',
    u'\\VDash': u'\u22ab',
    u'\\Vdash': u'\u22a9',
    u'\\Vert': u'\u2016',
    u'\\Vvdash': u'\u22aa',
    u'\\Xi': u'\u039e',
    u'\\Zeta': u'\u0396',
    u'\\^': u'\u0302',
    u'\\^{A}': u'\xc2',
    u'\\^{C}': u'\u0108',
    u'\\^{E}': u'\xca',
    u'\\^{G}': u'\u011c',
    u'\\^{H}': u'\u0124',
    u'\\^{I}': u'\xce',
    u'\\^{J}': u'\u0134',
    u'\\^{O}': u'\xd4',
    u'\\^{S}': u'\u015c',
    u'\\^{U}': u'\xdb',
    u'\\^{W}': u'\u0174',
    u'\\^{Y}': u'\u0176',
    u'\\^{\\i}': u'\xee',
    u'\\^{\\j}': u'\u0135',
    u'\\^{a}': u'\xe2',
    u'\\^{c}': u'\u0109',
    u'\\^{e}': u'\xea',
    u'\\^{g}': u'\u011d',
    u'\\^{h}': u'\u0125',
    u'\\^{o}': u'\xf4',
    u'\\^{s}': u'\u015d',
    u'\\^{u}': u'\xfb',
    u'\\^{w}': u'\u0175',
    u'\\^{y}': u'\u0177',
    u'\\^{}': u'^',
    u'\\_': u'_',
    u'\\`': u'\u0300',
    u'\\`{A}': u'\xc0',
    u'\\`{E}': u'\xc8',
    u'\\`{I}': u'\xcc',
    u'\\`{O}': u'\xd2',
    u'\\`{U}': u'\xd9',
    u'\\`{\\i}': u'\xec',
    u'\\`{a}': u'\xe0',
    u'\\`{e}': u'\xe8',
    u'\\`{o}': u'\xf2',
    u'\\`{u}': u'\xf9',
    u'\\aa': u'\xe5',
    u'\\acute{\\ddot{\\iota}}': u'\u0390',
    u'\\acute{\\ddot{\\upsilon}}': u'\u03b0',
    u'\\acute{\\epsilon}': u'\u03ad',
    u'\\acute{\\eta}': u'\u03ae',
    u'\\acute{\\iota}': u'\u03af',
    u'\\acute{\\omega}': u'\u03ce',
    u'\\acute{\\upsilon}': u'\u03cd',
    u'\\ae': u'\xe6',
    u'\\aleph': u'\u2135',
    u'\\allequal': u'\u224c',
    u'\\alpha': u'\u03b1',
    u'\\amalg': u'\u2a3f',
    u'\\angle': u'\u2220',
    u'\\approx': u'\u2248',
    u'\\approxeq': u'\u224a',
    u'\\approxnotequal': u'\u2246',
    u'\\aquarius': u'\u2652',
    u'\\aries': u'\u2648',
    u'\\arrowwaveright': u'\u219d',
    u'\\ast': u'*',
    u'\\asymp': u'\u224d',
    u'\\backepsilon': u'\u03f6',
    u'\\backprime': u'\u2035',
    u'\\backsim': u'\u223d',
    u'\\backsimeq': u'\u22cd',
    u'\\barwedge': u'\u2305',
    u'\\because': u'\u2235',
    u'\\beta': u'\u03b2',
    u'\\beth': u'\u2136',
    u'\\between': u'\u226c',
    u'\\bigcap': u'\u22c2',
    u'\\bigcirc': u'\u25ef',
    u'\\bigcup': u'\u22c3',
    u'\\bigtriangledown': u'\u25bd',
    u'\\bigtriangleup': u'\u25b3',
    u'\\blacklozenge': u'\u29eb',
    u'\\blacksquare': u'\u25aa',
    u'\\blacktriangle': u'\u25b4',
    u'\\blacktriangledown': u'\u25be',
    u'\\blacktriangleleft': u'\u25c2',
    u'\\blacktriangleright': u'\u25b8',
    u'\\bowtie': u'\u22c8',
    u'\\boxdot': u'\u22a1',
    u'\\boxminus': u'\u229f',
    u'\\boxplus': u'\u229e',
    u'\\boxtimes': u'\u22a0',
    u'\\bullet': u'\u2219',
    u'\\bumpeq': u'\u224f',
    u'\\c': u'\u0327',
    u'\\cancer': u'\u264b',
    u'\\cap': u'\u2229',
    u'\\capricornus': u'\u2651',
    u'\\cdot': u'\u22c5',
    u'\\cdots': u'\u22ef',
    u'\\chi': u'\u03c7',
    u'\\circ': u'\u2218',
    u'\\circeq': u'\u2257',
    u'\\circlearrowleft': u'\u21ba',
    u'\\circlearrowright': u'\u21bb',
    u'\\circledS': u'\u24c8',
    u'\\circledast': u'\u229b',
    u'\\circledcirc': u'\u229a',
    u'\\circleddash': u'\u229d',
    u'\\clockoint': u'\u2a0f',
    u'\\clwintegral': u'\u2231',
    u'\\complement': u'\u2201',
    u'\\cong': u'\u2245',
    u'\\coprod': u'\u2210',
    u'\\cup': u'\u222a',
    u'\\curlyeqprec': u'\u22de',
    u'\\curlyeqsucc': u'\u22df',
    u'\\curlyvee': u'\u22ce',
    u'\\curlywedge': u'\u22cf',
    u'\\curvearrowleft': u'\u21b6',
    u'\\curvearrowright': u'\u21b7',
    u'\\cyrchar\\C': u'\u030f',
    u'\\cyrchar\\CYRA': u'\u0410',
    u'\\cyrchar\\CYRABHCH': u'\u04bc',
    u'\\cyrchar\\CYRABHCHDSC': u'\u04be',
    u'\\cyrchar\\CYRABHDZE': u'\u04e0',
    u'\\cyrchar\\CYRABHHA': u'\u04a8',
    u'\\cyrchar\\CYRAE': u'\u04d4',
    u'\\cyrchar\\CYRB': u'\u0411',
    u'\\cyrchar\\CYRBYUS': u'\u046a',
    u'\\cyrchar\\CYRC': u'\u0426',
    u'\\cyrchar\\CYRCH': u'\u0427',
    u'\\cyrchar\\CYRCHLDSC': u'\u04cb',
    u'\\cyrchar\\CYRCHRDSC': u'\u04b6',
    u'\\cyrchar\\CYRCHVCRS': u'\u04b8',
    u'\\cyrchar\\CYRD': u'\u0414',
    u'\\cyrchar\\CYRDJE': u'\u0402',
    u'\\cyrchar\\CYRDZE': u'\u0405',
    u'\\cyrchar\\CYRDZHE': u'\u040f',
    u'\\cyrchar\\CYRE': u'\u0415',
    u'\\cyrchar\\CYREREV': u'\u042d',
    u'\\cyrchar\\CYRERY': u'\u042b',
    u'\\cyrchar\\CYRF': u'\u0424',
    u'\\cyrchar\\CYRFITA': u'\u0472',
    u'\\cyrchar\\CYRG': u'\u0413',
    u'\\cyrchar\\CYRGHCRS': u'\u0492',
    u'\\cyrchar\\CYRGHK': u'\u0494',
    u'\\cyrchar\\CYRGUP': u'\u0490',
    u'\\cyrchar\\CYRH': u'\u0425',
    u'\\cyrchar\\CYRHDSC': u'\u04b2',
    u'\\cyrchar\\CYRHRDSN': u'\u042a',
    u'\\cyrchar\\CYRI': u'\u0418',
    u'\\cyrchar\\CYRIE': u'\u0404',
    u'\\cyrchar\\CYRII': u'\u0406',
    u'\\cyrchar\\CYRIOTBYUS': u'\u046c',
    u'\\cyrchar\\CYRIOTE': u'\u0464',
    u'\\cyrchar\\CYRIOTLYUS': u'\u0468',
    u'\\cyrchar\\CYRISHRT': u'\u0419',
    u'\\cyrchar\\CYRIZH': u'\u0474',
    u'\\cyrchar\\CYRJE': u'\u0408',
    u'\\cyrchar\\CYRK': u'\u041a',
    u'\\cyrchar\\CYRKBEAK': u'\u04a0',
    u'\\cyrchar\\CYRKDSC': u'\u049a',
    u'\\cyrchar\\CYRKHCRS': u'\u049e',
    u'\\cyrchar\\CYRKHK': u'\u04c3',
    u'\\cyrchar\\CYRKOPPA': u'\u0480',
    u'\\cyrchar\\CYRKSI': u'\u046e',
    u'\\cyrchar\\CYRKVCRS': u'\u049c',
    u'\\cyrchar\\CYRL': u'\u041b',
    u'\\cyrchar\\CYRLJE': u'\u0409',
    u'\\cyrchar\\CYRLYUS': u'\u0466',
    u'\\cyrchar\\CYRM': u'\u041c',
    u'\\cyrchar\\CYRN': u'\u041d',
    u'\\cyrchar\\CYRNDSC': u'\u04a2',
    u'\\cyrchar\\CYRNG': u'\u04a4',
    u'\\cyrchar\\CYRNHK': u'\u04c7',
    u'\\cyrchar\\CYRNJE': u'\u040a',
    u'\\cyrchar\\CYRO': u'\u041e',
    u'\\cyrchar\\CYROMEGA': u'\u0460',
    u'\\cyrchar\\CYROMEGARND': u'\u047a',
    u'\\cyrchar\\CYROMEGATITLO': u'\u047c',
    u'\\cyrchar\\CYROT': u'\u047e',
    u'\\cyrchar\\CYROTLD': u'\u04e8',
    u'\\cyrchar\\CYRP': u'\u041f',
    u'\\cyrchar\\CYRPHK': u'\u04a6',
    u'\\cyrchar\\CYRPSI': u'\u0470',
    u'\\cyrchar\\CYRR': u'\u0420',
    u'\\cyrchar\\CYRRTICK': u'\u048e',
    u'\\cyrchar\\CYRS': u'\u0421',
    u'\\cyrchar\\CYRSCHWA': u'\u04d8',
    u'\\cyrchar\\CYRSDSC': u'\u04aa',
    u'\\cyrchar\\CYRSEMISFTSN': u'\u048c',
    u'\\cyrchar\\CYRSFTSN': u'\u042c',
    u'\\cyrchar\\CYRSH': u'\u0428',
    u'\\cyrchar\\CYRSHCH': u'\u0429',
    u'\\cyrchar\\CYRSHHA': u'\u04ba',
    u'\\cyrchar\\CYRT': u'\u0422',
    u'\\cyrchar\\CYRTDSC': u'\u04ac',
    u'\\cyrchar\\CYRTETSE': u'\u04b4',
    u'\\cyrchar\\CYRTSHE': u'\u040b',
    u'\\cyrchar\\CYRU': u'\u0423',
    u'\\cyrchar\\CYRUK': u'\u0478',
    u'\\cyrchar\\CYRUSHRT': u'\u040e',
    u'\\cyrchar\\CYRV': u'\u0412',
    u'\\cyrchar\\CYRY': u'\u04ae',
    u'\\cyrchar\\CYRYA': u'\u042f',
    u'\\cyrchar\\CYRYAT': u'\u0462',
    u'\\cyrchar\\CYRYHCRS': u'\u04b0',
    u'\\cyrchar\\CYRYI': u'\u0407',
    u'\\cyrchar\\CYRYO': u'\u0401',
    u'\\cyrchar\\CYRYU': u'\u042e',
    u'\\cyrchar\\CYRZ': u'\u0417',
    u'\\cyrchar\\CYRZDSC': u'\u0498',
    u'\\cyrchar\\CYRZH': u'\u0416',
    u'\\cyrchar\\CYRZHDSC': u'\u0496',
    u'\\cyrchar\\CYRpalochka': u'\u04c0',
    u'\\cyrchar\\cyra': u'\u0430',
    u'\\cyrchar\\cyrabhch': u'\u04bd',
    u'\\cyrchar\\cyrabhchdsc': u'\u04bf',
    u'\\cyrchar\\cyrabhdze': u'\u04e1',
    u'\\cyrchar\\cyrabhha': u'\u04a9',
    u'\\cyrchar\\cyrae': u'\u04d5',
    u'\\cyrchar\\cyrb': u'\u0431',
    u'\\cyrchar\\cyrc': u'\u0446',
    u'\\cyrchar\\cyrch': u'\u0447',
    u'\\cyrchar\\cyrchldsc': u'\u04cc',
    u'\\cyrchar\\cyrchrdsc': u'\u04b7',
    u'\\cyrchar\\cyrchvcrs': u'\u04b9',
    u'\\cyrchar\\cyrd': u'\u0434',
    u'\\cyrchar\\cyrdje': u'\u0452',
    u'\\cyrchar\\cyrdze': u'\u0455',
    u'\\cyrchar\\cyrdzhe': u'\u045f',
    u'\\cyrchar\\cyre': u'\u0435',
    u'\\cyrchar\\cyrerev': u'\u044d',
    u'\\cyrchar\\cyrery': u'\u044b',
    u'\\cyrchar\\cyrf': u'\u0444',
    u'\\cyrchar\\cyrg': u'\u0433',
    u'\\cyrchar\\cyrghcrs': u'\u0493',
    u'\\cyrchar\\cyrghk': u'\u0495',
    u'\\cyrchar\\cyrgup': u'\u0491',
    u'\\cyrchar\\cyrh': u'\u0445',
    u'\\cyrchar\\cyrhdsc': u'\u04b3',
    u'\\cyrchar\\cyrhrdsn': u'\u044a',
    u'\\cyrchar\\cyrhundredthousands': u'\u0488',
    u'\\cyrchar\\cyri': u'\u0438',
    u'\\cyrchar\\cyrie': u'\u0454',
    u'\\cyrchar\\cyrii': u'\u0456',
    u'\\cyrchar\\cyriotbyus': u'\u046d',
    u'\\cyrchar\\cyriote': u'\u0465',
    u'\\cyrchar\\cyriotlyus': u'\u0469',
    u'\\cyrchar\\cyrishrt': u'\u0439',
    u'\\cyrchar\\cyrje': u'\u0458',
    u'\\cyrchar\\cyrk': u'\u043a',
    u'\\cyrchar\\cyrkbeak': u'\u04a1',
    u'\\cyrchar\\cyrkdsc': u'\u049b',
    u'\\cyrchar\\cyrkhcrs': u'\u049f',
    u'\\cyrchar\\cyrkhk': u'\u04c4',
    u'\\cyrchar\\cyrkoppa': u'\u0481',
    u'\\cyrchar\\cyrksi': u'\u046f',
    u'\\cyrchar\\cyrkvcrs': u'\u049d',
    u'\\cyrchar\\cyrl': u'\u043b',
    u'\\cyrchar\\cyrlje': u'\u0459',
    u'\\cyrchar\\cyrlyus': u'\u0467',
    u'\\cyrchar\\cyrm': u'\u043c',
    u'\\cyrchar\\cyrmillions': u'\u0489',
    u'\\cyrchar\\cyrn': u'\u043d',
    u'\\cyrchar\\cyrndsc': u'\u04a3',
    u'\\cyrchar\\cyrng': u'\u04a5',
    u'\\cyrchar\\cyrnhk': u'\u04c8',
    u'\\cyrchar\\cyrnje': u'\u045a',
    u'\\cyrchar\\cyro': u'\u043e',
    u'\\cyrchar\\cyromega': u'\u0461',
    u'\\cyrchar\\cyromegarnd': u'\u047b',
    u'\\cyrchar\\cyromegatitlo': u'\u047d',
    u'\\cyrchar\\cyrot': u'\u047f',
    u'\\cyrchar\\cyrotld': u'\u04e9',
    u'\\cyrchar\\cyrp': u'\u043f',
    u'\\cyrchar\\cyrphk': u'\u04a7',
    u'\\cyrchar\\cyrpsi': u'\u0471',
    u'\\cyrchar\\cyrr': u'\u0440',
    u'\\cyrchar\\cyrrtick': u'\u048f',
    u'\\cyrchar\\cyrs': u'\u0441',
    u'\\cyrchar\\cyrschwa': u'\u04d9',
    u'\\cyrchar\\cyrsdsc': u'\u04ab',
    u'\\cyrchar\\cyrsemisftsn': u'\u048d',
    u'\\cyrchar\\cyrsftsn': u'\u044c',
    u'\\cyrchar\\cyrsh': u'\u0448',
    u'\\cyrchar\\cyrshch': u'\u0449',
    u'\\cyrchar\\cyrshha': u'\u04bb',
    u'\\cyrchar\\cyrt': u'\u0442',
    u'\\cyrchar\\cyrtdsc': u'\u04ad',
    u'\\cyrchar\\cyrtetse': u'\u04b5',
    u'\\cyrchar\\cyrthousands': u'\u0482',
    u'\\cyrchar\\cyrtshe': u'\u045b',
    u'\\cyrchar\\cyru': u'\u0443',
    u'\\cyrchar\\cyruk': u'\u0479',
    u'\\cyrchar\\cyrushrt': u'\u045e',
    u'\\cyrchar\\cyrv': u'\u0432',
    u'\\cyrchar\\cyry': u'\u04af',
    u'\\cyrchar\\cyrya': u'\u044f',
    u'\\cyrchar\\cyryhcrs': u'\u04b1',
    u'\\cyrchar\\cyryi': u'\u0457',
    u'\\cyrchar\\cyryo': u'\u0451',
    u'\\cyrchar\\cyryu': u'\u044e',
    u'\\cyrchar\\cyrz': u'\u0437',
    u'\\cyrchar\\cyrzdsc': u'\u0499',
    u'\\cyrchar\\cyrzh': u'\u0436',
    u'\\cyrchar\\cyrzhdsc': u'\u0497',
    u'\\cyrchar\\textnumero': u'\u2116',
    u"\\cyrchar{\\'\\CYRG}": u'\u0403',
    u"\\cyrchar{\\'\\CYRK}": u'\u040c',
    u"\\cyrchar{\\'\\cyrg}": u'\u0453',
    u"\\cyrchar{\\'\\cyrk}": u'\u045c',
    u'\\c{C}': u'\xc7',
    u'\\c{G}': u'\u0122',
    u'\\c{K}': u'\u0136',
    u'\\c{L}': u'\u013b',
    u'\\c{N}': u'\u0145',
    u'\\c{R}': u'\u0156',
    u'\\c{S}': u'\u015e',
    u'\\c{T}': u'\u0162',
    u'\\c{c}': u'\xe7',
    u'\\c{g}': u'\u0123',
    u'\\c{k}': u'\u0137',
    u'\\c{l}': u'\u013c',
    u'\\c{n}': u'\u0146',
    u'\\c{r}': u'\u0157',
    u'\\c{s}': u'\u015f',
    u'\\c{t}': u'\u0163',
    u'\\c{}': u'\xb8',
    u'\\daleth': u'\u2138',
    u'\\dashv': u'\u22a3',
    u'\\dblarrowupdown': u'\u21c5',
    u'\\ddddot': u'\u20dc',
    u'\\dddot': u'\u20db',
    u'\\ddot{\\iota}': u'\u03ca',
    u'\\ddot{\\upsilon}': u'\u03cb',
    u'\\delta': u'\u03b4',
    u'\\dh': u'\xf0',
    u'\\diagup': u'\u2571',
    u'\\diamond': u'\u2662',
    u'\\digamma': u'\u03dd',
    u'\\ding{100}': u'\u2744',
    u'\\ding{101}': u'\u2745',
    u'\\ding{102}': u'\u2746',
    u'\\ding{103}': u'\u2747',
    u'\\ding{104}': u'\u2748',
    u'\\ding{105}': u'\u2749',
    u'\\ding{106}': u'\u274a',
    u'\\ding{107}': u'\u274b',
    u'\\ding{108}': u'\u25cf',
    u'\\ding{109}': u'\u274d',
    u'\\ding{110}': u'\u25a0',
    u'\\ding{111}': u'\u274f',
    u'\\ding{112}': u'\u2750',
    u'\\ding{113}': u'\u2751',
    u'\\ding{114}': u'\u2752',
    u'\\ding{115}': u'\u25b2',
    u'\\ding{116}': u'\u25bc',
    u'\\ding{117}': u'\u25c6',
    u'\\ding{118}': u'\u2756',
    u'\\ding{119}': u'\u25d7',
    u'\\ding{120}': u'\u2758',
    u'\\ding{121}': u'\u2759',
    u'\\ding{122}': u'\u275a',
    u'\\ding{123}': u'\u275b',
    u'\\ding{124}': u'\u275c',
    u'\\ding{125}': u'\u275d',
    u'\\ding{126}': u'\u275e',
    u'\\ding{161}': u'\u2761',
    u'\\ding{162}': u'\u2762',
    u'\\ding{163}': u'\u2763',
    u'\\ding{164}': u'\u2764',
    u'\\ding{165}': u'\u2765',
    u'\\ding{166}': u'\u2766',
    u'\\ding{167}': u'\u2767',
    u'\\ding{168}': u'\u2663',
    u'\\ding{169}': u'\u2666',
    u'\\ding{170}': u'\u2665',
    u'\\ding{171}': u'\u2660',
    u'\\ding{172}': u'\u2460',
    u'\\ding{173}': u'\u2461',
    u'\\ding{174}': u'\u2462',
    u'\\ding{175}': u'\u2463',
    u'\\ding{176}': u'\u2464',
    u'\\ding{177}': u'\u2465',
    u'\\ding{178}': u'\u2466',
    u'\\ding{179}': u'\u2467',
    u'\\ding{180}': u'\u2468',
    u'\\ding{181}': u'\u2469',
    u'\\ding{182}': u'\u2776',
    u'\\ding{183}': u'\u2777',
    u'\\ding{184}': u'\u2778',
    u'\\ding{185}': u'\u2779',
    u'\\ding{186}': u'\u277a',
    u'\\ding{187}': u'\u277b',
    u'\\ding{188}': u'\u277c',
    u'\\ding{189}': u'\u277d',
    u'\\ding{190}': u'\u277e',
    u'\\ding{191}': u'\u277f',
    u'\\ding{192}': u'\u2780',
    u'\\ding{193}': u'\u2781',
    u'\\ding{194}': u'\u2782',
    u'\\ding{195}': u'\u2783',
    u'\\ding{196}': u'\u2784',
    u'\\ding{197}': u'\u2785',
    u'\\ding{198}': u'\u2786',
    u'\\ding{199}': u'\u2787',
    u'\\ding{200}': u'\u2788',
    u'\\ding{201}': u'\u2789',
    u'\\ding{202}': u'\u278a',
    u'\\ding{203}': u'\u278b',
    u'\\ding{204}': u'\u278c',
    u'\\ding{205}': u'\u278d',
    u'\\ding{206}': u'\u278e',
    u'\\ding{207}': u'\u278f',
    u'\\ding{208}': u'\u2790',
    u'\\ding{209}': u'\u2791',
    u'\\ding{210}': u'\u2792',
    u'\\ding{211}': u'\u2793',
    u'\\ding{212}': u'\u2794',
    u'\\ding{216}': u'\u2798',
    u'\\ding{217}': u'\u2799',
    u'\\ding{218}': u'\u279a',
    u'\\ding{219}': u'\u279b',
    u'\\ding{220}': u'\u279c',
    u'\\ding{221}': u'\u279d',
    u'\\ding{222}': u'\u279e',
    u'\\ding{223}': u'\u279f',
    u'\\ding{224}': u'\u27a0',
    u'\\ding{225}': u'\u27a1',
    u'\\ding{226}': u'\u27a2',
    u'\\ding{227}': u'\u27a3',
    u'\\ding{228}': u'\u27a4',
    u'\\ding{229}': u'\u27a5',
    u'\\ding{230}': u'\u27a6',
    u'\\ding{231}': u'\u27a7',
    u'\\ding{232}': u'\u27a8',
    u'\\ding{233}': u'\u27a9',
    u'\\ding{234}': u'\u27aa',
    u'\\ding{235}': u'\u27ab',
    u'\\ding{236}': u'\u27ac',
    u'\\ding{237}': u'\u27ad',
    u'\\ding{238}': u'\u27ae',
    u'\\ding{239}': u'\u27af',
    u'\\ding{241}': u'\u27b1',
    u'\\ding{242}': u'\u27b2',
    u'\\ding{243}': u'\u27b3',
    u'\\ding{244}': u'\u27b4',
    u'\\ding{245}': u'\u27b5',
    u'\\ding{246}': u'\u27b6',
    u'\\ding{247}': u'\u27b7',
    u'\\ding{248}': u'\u27b8',
    u'\\ding{249}': u'\u27b9',
    u'\\ding{250}': u'\u27ba',
    u'\\ding{251}': u'\u27bb',
    u'\\ding{252}': u'\u27bc',
    u'\\ding{253}': u'\u27bd',
    u'\\ding{254}': u'\u27be',
    u'\\ding{33}': u'\u2701',
    u'\\ding{34}': u'\u2702',
    u'\\ding{35}': u'\u2703',
    u'\\ding{36}': u'\u2704',
    u'\\ding{37}': u'\u260e',
    u'\\ding{38}': u'\u2706',
    u'\\ding{39}': u'\u2707',
    u'\\ding{40}': u'\u2708',
    u'\\ding{41}': u'\u2709',
    u'\\ding{42}': u'\u261b',
    u'\\ding{43}': u'\u261e',
    u'\\ding{44}': u'\u270c',
    u'\\ding{45}': u'\u270d',
    u'\\ding{46}': u'\u270e',
    u'\\ding{47}': u'\u270f',
    u'\\ding{48}': u'\u2710',
    u'\\ding{49}': u'\u2711',
    u'\\ding{50}': u'\u2712',
    u'\\ding{51}': u'\u2713',
    u'\\ding{52}': u'\u2714',
    u'\\ding{53}': u'\u2715',
    u'\\ding{54}': u'\u2716',
    u'\\ding{55}': u'\u2717',
    u'\\ding{56}': u'\u2718',
    u'\\ding{57}': u'\u2719',
    u'\\ding{58}': u'\u271a',
    u'\\ding{59}': u'\u271b',
    u'\\ding{60}': u'\u271c',
    u'\\ding{61}': u'\u271d',
    u'\\ding{62}': u'\u271e',
    u'\\ding{63}': u'\u271f',
    u'\\ding{64}': u'\u2720',
    u'\\ding{65}': u'\u2721',
    u'\\ding{66}': u'\u2722',
    u'\\ding{67}': u'\u2723',
    u'\\ding{68}': u'\u2724',
    u'\\ding{69}': u'\u2725',
    u'\\ding{70}': u'\u2726',
    u'\\ding{71}': u'\u2727',
    u'\\ding{72}': u'\u2605',
    u'\\ding{73}': u'\u2729',
    u'\\ding{74}': u'\u272a',
    u'\\ding{75}': u'\u272b',
    u'\\ding{76}': u'\u272c',
    u'\\ding{77}': u'\u272d',
    u'\\ding{78}': u'\u272e',
    u'\\ding{79}': u'\u272f',
    u'\\ding{80}': u'\u2730',
    u'\\ding{81}': u'\u2731',
    u'\\ding{82}': u'\u2732',
    u'\\ding{83}': u'\u2733',
    u'\\ding{84}': u'\u2734',
    u'\\ding{85}': u'\u2735',
    u'\\ding{86}': u'\u2736',
    u'\\ding{87}': u'\u2737',
    u'\\ding{88}': u'\u2738',
    u'\\ding{89}': u'\u2739',
    u'\\ding{90}': u'\u273a',
    u'\\ding{91}': u'\u273b',
    u'\\ding{92}': u'\u273c',
    u'\\ding{93}': u'\u273d',
    u'\\ding{94}': u'\u273e',
    u'\\ding{95}': u'\u273f',
    u'\\ding{96}': u'\u2740',
    u'\\ding{97}': u'\u2741',
    u'\\ding{98}': u'\u2742',
    u'\\ding{99}': u'\u2743',
    u'\\div': u'\xf7',
    u'\\divideontimes': u'\u22c7',
    u'\\dj': u'\u0111',
    u'\\doteq': u'\u2250',
    u'\\doteqdot': u'\u2251',
    u'\\dotplus': u'\u2214',
    u'\\downarrow': u'\u2193',
    u'\\downdownarrows': u'\u21ca',
    u'\\downharpoonleft': u'\u21c3',
    u'\\downharpoonright': u'\u21c2',
    u'\\downslopeellipsis': u'\u22f1',
    u'\\eighthnote': u'\u266a',
    u'\\ensuremath{\\Elzpes}': u'\u20a7',
    u'\\epsilon': u'\u03b5',
    u'\\eqcirc': u'\u2256',
    u'\\eqslantgtr': u'\u2a96',
    u'\\eqslantless': u'\u2a95',
    u'\\equiv': u'\u2261',
    u'\\estimates': u'\u2259',
    u'\\eta': u'\u03b7',
    u'\\eth': u'\u01aa',
    u'\\exists': u'\u2203',
    u'\\fallingdotseq': u'\u2252',
    u'\\fbox{~~}': u'\u25ad',
    u'\\flat': u'\u266d',
    u'\\forall': u'\u2200',
    u'\\forcesextra': u'\u22a8',
    u'\\frown': u'\u2322',
    u'\\gamma': u'\u03b3',
    u'\\gemini': u'\u264a',
    u'\\geq': u'\u2265',
    u'\\geqq': u'\u2267',
    u'\\geqslant': u'\u2a7e',
    u'\\gg': u'\u226b',
    u'\\gimel': u'\u2137',
    u'\\gnapprox': u'\u2a8a',
    u'\\gneq': u'\u2a88',
    u'\\gneqq': u'\u2269',
    u'\\gnsim': u'\u22e7',
    u'\\greaterequivlnt': u'\u2273',
    u'\\gtrapprox': u'\u2a86',
    u'\\gtrdot': u'\u22d7',
    u'\\gtreqless': u'\u22db',
    u'\\gtreqqless': u'\u2a8c',
    u'\\gtrless': u'\u2277',
    u'\\guillemotleft': u'\xab',
    u'\\guillemotright': u'\xbb',
    u'\\guilsinglleft': u'\u2039',
    u'\\guilsinglright': u'\u203a',
    u'\\gvertneqq': u'\u2269\ufe00',
    u'\\hermitconjmatrix': u'\u22b9',
    u'\\homothetic': u'\u223b',
    u'\\hookleftarrow': u'\u21a9',
    u'\\hookrightarrow': u'\u21aa',
    u'\\hphantom{,}': u'\u2008',
    u'\\hphantom{0}': u'\u2007',
    u'\\hslash': u'\u210f',
    u'\\hspace{0.166em}': u'\u2006',
    u'\\hspace{0.167em}': u'\u2009',
    u'\\hspace{0.25em}': u'\u2005',
    u'\\hspace{0.33em}': u'\u2004',
    u'\\hspace{0.6em}': u'\u2002',
    u'\\hspace{1em}': u'\u2003',
    u'\\i': u'\u0131',
    u'\\image': u'\u22b7',
    u'\\in': u'\U0001d7c4',
    u'\\infty': u'\u221e',
    u'\\int': u'\u222b',
    u'\\int\\!\\int': u'\u222c',
    u'\\int\\!\\int\\!\\int': u'\u222d',
    u'\\intercal': u'\u22ba',
    u'\\iota': u'\u03b9',
    u'\\jupiter': u'\u2643',
    u'\\k': u'\u0328',
    u'\\kappa': u'\u03ba',
    u'\\k{A}': u'\u0104',
    u'\\k{E}': u'\u0118',
    u'\\k{I}': u'\u012e',
    u'\\k{U}': u'\u0172',
    u'\\k{a}': u'\u0105',
    u'\\k{e}': u'\u0119',
    u'\\k{i}': u'\u012f',
    u'\\k{u}': u'\u0173',
    u'\\k{}': u'\u02db',
    u'\\l': u'\u0142',
    u'\\lambda': u'\u03bb',
    u'\\langle': u'\u2329',
    u'\\lazysinv': u'\u223e',
    u'\\lbrace': u'{',
    u'\\lceil': u'\u2308',
    u'\\ldots': u'\u2026',
    u'\\leftarrow': u'\u2190',
    u'\\leftarrowtail': u'\u21a2',
    u'\\leftharpoondown': u'\u21bd',
    u'\\leftharpoonup': u'\u21bc',
    u'\\leftleftarrows': u'\u21c7',
    u'\\leftrightarrow': u'\u2194',
    u'\\leftrightarrows': u'\u21c6',
    u'\\leftrightharpoons': u'\u21cb',
    u'\\leftrightsquigarrow': u'\u21ad',
    u'\\leftthreetimes': u'\u22cb',
    u'\\leo': u'\u264c',
    u'\\leq': u'\u2264',
    u'\\leqq': u'\u2266',
    u'\\leqslant': u'\u2a7d',
    u'\\lessapprox': u'\u2a85',
    u'\\lessdot': u'\u22d6',
    u'\\lesseqgtr': u'\u22da',
    u'\\lesseqqgtr': u'\u2a8b',
    u'\\lessequivlnt': u'\u2272',
    u'\\lessgtr': u'\u2276',
    u'\\lfloor': u'\u230a',
    u'\\libra': u'\u264e',
    u'\\ll': u'\u226a',
    u'\\llcorner': u'\u231e',
    u'\\lmoustache': u'\u23b0',
    u'\\lnapprox': u'\u2a89',
    u'\\lneq': u'\u2a87',
    u'\\lneqq': u'\u2268',
    u'\\lnot': u'\xac',
    u'\\lnsim': u'\u22e6',
    u'\\longleftarrow': u'\u27f5',
    u'\\longleftrightarrow': u'\u27f7',
    u'\\longmapsto': u'\u27fc',
    u'\\longrightarrow': u'\u27f6',
    u'\\looparrowleft': u'\u21ab',
    u'\\looparrowright': u'\u21ac',
    u'\\lozenge': u'\u25ca',
    u'\\lrcorner': u'\u231f',
    u'\\ltimes': u'\u22c9',
    u'\\lvertneqq': u'\u2268\ufe00',
    u'\\male': u'\u2642',
    u'\\mapsto': u'\u21a6',
    u'\\mathbb{0}': u'\U0001d7d8',
    u'\\mathbb{1}': u'\U0001d7d9',
    u'\\mathbb{2}': u'\U0001d7da',
    u'\\mathbb{3}': u'\U0001d7db',
    u'\\mathbb{4}': u'\U0001d7dc',
    u'\\mathbb{5}': u'\U0001d7dd',
    u'\\mathbb{6}': u'\U0001d7de',
    u'\\mathbb{7}': u'\U0001d7df',
    u'\\mathbb{8}': u'\U0001d7e0',
    u'\\mathbb{9}': u'\U0001d7e1',
    u'\\mathbb{A}': u'\U0001d538',
    u'\\mathbb{B}': u'\U0001d539',
    u'\\mathbb{C}': u'\u2102',
    u'\\mathbb{D}': u'\U0001d53b',
    u'\\mathbb{E}': u'\U0001d53c',
    u'\\mathbb{F}': u'\U0001d53d',
    u'\\mathbb{G}': u'\U0001d53e',
    u'\\mathbb{H}': u'\u210d',
    u'\\mathbb{I}': u'\U0001d540',
    u'\\mathbb{J}': u'\U0001d541',
    u'\\mathbb{K}': u'\U0001d542',
    u'\\mathbb{L}': u'\U0001d543',
    u'\\mathbb{M}': u'\U0001d544',
    u'\\mathbb{N}': u'\u2115',
    u'\\mathbb{O}': u'\U0001d546',
    u'\\mathbb{P}': u'\u2119',
    u'\\mathbb{Q}': u'\u211a',
    u'\\mathbb{R}': u'\u211d',
    u'\\mathbb{S}': u'\U0001d54a',
    u'\\mathbb{T}': u'\U0001d54b',
    u'\\mathbb{U}': u'\U0001d54c',
    u'\\mathbb{V}': u'\U0001d54d',
    u'\\mathbb{W}': u'\U0001d54e',
    u'\\mathbb{X}': u'\U0001d54f',
    u'\\mathbb{Y}': u'\U0001d550',
    u'\\mathbb{Z}': u'\u2124',
    u'\\mathbb{a}': u'\U0001d552',
    u'\\mathbb{b}': u'\U0001d553',
    u'\\mathbb{c}': u'\U0001d554',
    u'\\mathbb{d}': u'\U0001d555',
    u'\\mathbb{e}': u'\U0001d556',
    u'\\mathbb{f}': u'\U0001d557',
    u'\\mathbb{g}': u'\U0001d558',
    u'\\mathbb{h}': u'\U0001d559',
    u'\\mathbb{i}': u'\U0001d55a',
    u'\\mathbb{j}': u'\U0001d55b',
    u'\\mathbb{k}': u'\U0001d55c',
    u'\\mathbb{l}': u'\U0001d55d',
    u'\\mathbb{m}': u'\U0001d55e',
    u'\\mathbb{n}': u'\U0001d55f',
    u'\\mathbb{o}': u'\U0001d560',
    u'\\mathbb{p}': u'\U0001d561',
    u'\\mathbb{q}': u'\U0001d562',
    u'\\mathbb{r}': u'\U0001d563',
    u'\\mathbb{s}': u'\U0001d564',
    u'\\mathbb{t}': u'\U0001d565',
    u'\\mathbb{u}': u'\U0001d566',
    u'\\mathbb{v}': u'\U0001d567',
    u'\\mathbb{w}': u'\U0001d568',
    u'\\mathbb{x}': u'\U0001d569',
    u'\\mathbb{y}': u'\U0001d56a',
    u'\\mathbb{z}': u'\U0001d56b',
    u'\\mathbf{0}': u'\U0001d7ce',
    u'\\mathbf{1}': u'\U0001d7cf',
    u'\\mathbf{2}': u'\U0001d7d0',
    u'\\mathbf{3}': u'\U0001d7d1',
    u'\\mathbf{4}': u'\U0001d7d2',
    u'\\mathbf{5}': u'\U0001d7d3',
    u'\\mathbf{6}': u'\U0001d7d4',
    u'\\mathbf{7}': u'\U0001d7d5',
    u'\\mathbf{8}': u'\U0001d7d6',
    u'\\mathbf{9}': u'\U0001d7d7',
    u'\\mathbf{A}': u'\U0001d400',
    u'\\mathbf{B}': u'\U0001d401',
    u'\\mathbf{C}': u'\U0001d402',
    u'\\mathbf{D}': u'\U0001d403',
    u'\\mathbf{E}': u'\U0001d404',
    u'\\mathbf{F}': u'\U0001d405',
    u'\\mathbf{G}': u'\U0001d406',
    u'\\mathbf{H}': u'\U0001d407',
    u'\\mathbf{I}': u'\U0001d408',
    u'\\mathbf{J}': u'\U0001d409',
    u'\\mathbf{K}': u'\U0001d40a',
    u'\\mathbf{L}': u'\U0001d40b',
    u'\\mathbf{M}': u'\U0001d40c',
    u'\\mathbf{N}': u'\U0001d40d',
    u'\\mathbf{O}': u'\U0001d40e',
    u'\\mathbf{P}': u'\U0001d40f',
    u'\\mathbf{Q}': u'\U0001d410',
    u'\\mathbf{R}': u'\U0001d411',
    u'\\mathbf{S}': u'\U0001d412',
    u'\\mathbf{T}': u'\U0001d413',
    u'\\mathbf{U}': u'\U0001d414',
    u'\\mathbf{V}': u'\U0001d415',
    u'\\mathbf{W}': u'\U0001d416',
    u'\\mathbf{X}': u'\U0001d417',
    u'\\mathbf{Y}': u'\U0001d418',
    u'\\mathbf{Z}': u'\U0001d419',
    u'\\mathbf{\\Alpha}': u'\U0001d6c2',
    u'\\mathbf{\\Beta}': u'\U0001d6c3',
    u'\\mathbf{\\Chi}': u'\U0001d6d8',
    u'\\mathbf{\\Delta}': u'\U0001d6c5',
    u'\\mathbf{\\Epsilon}': u'\U0001d6c6',
    u'\\mathbf{\\Eta}': u'\U0001d6c8',
    u'\\mathbf{\\Gamma}': u'\U0001d6c4',
    u'\\mathbf{\\Iota}': u'\U0001d6ca',
    u'\\mathbf{\\Kappa}': u'\U0001d6cb',
    u'\\mathbf{\\Lambda}': u'\U0001d6cc',
    u'\\mathbf{\\Omega}': u'\U0001d6da',
    u'\\mathbf{\\Phi}': u'\U0001d6d7',
    u'\\mathbf{\\Pi}': u'\U0001d6d1',
    u'\\mathbf{\\Psi}': u'\U0001d6d9',
    u'\\mathbf{\\Rho}': u'\U0001d6d2',
    u'\\mathbf{\\Sigma}': u'\U0001d6d4',
    u'\\mathbf{\\Tau}': u'\U0001d6d5',
    u'\\mathbf{\\Theta}': u'\U0001d6af',
    u'\\mathbf{\\Upsilon}': u'\U0001d6d6',
    u'\\mathbf{\\Xi}': u'\U0001d6cf',
    u'\\mathbf{\\Zeta}': u'\U0001d6c7',
    u'\\mathbf{\\nabla}': u'\U0001d6c1',
    u'\\mathbf{\\phi}': u'\U0001d6df',
    u'\\mathbf{\\theta}': u'\U0001d6c9',
    u'\\mathbf{\\varkappa}': u'\U0001d6de',
    u'\\mathbf{\\varpi}': u'\U0001d6e1',
    u'\\mathbf{\\varrho}': u'\U0001d6e0',
    u'\\mathbf{\\varsigma}': u'\U0001d6d3',
    u'\\mathbf{\\vartheta}': u'\U0001d6dd',
    u'\\mathbf{a}': u'\U0001d41a',
    u'\\mathbf{b}': u'\U0001d41b',
    u'\\mathbf{c}': u'\U0001d41c',
    u'\\mathbf{d}': u'\U0001d41d',
    u'\\mathbf{e}': u'\U0001d41e',
    u'\\mathbf{f}': u'\U0001d41f',
    u'\\mathbf{g}': u'\U0001d420',
    u'\\mathbf{h}': u'\U0001d421',
    u'\\mathbf{i}': u'\U0001d422',
    u'\\mathbf{j}': u'\U0001d423',
    u'\\mathbf{k}': u'\U0001d424',
    u'\\mathbf{l}': u'\U0001d425',
    u'\\mathbf{m}': u'\U0001d426',
    u'\\mathbf{n}': u'\U0001d427',
    u'\\mathbf{o}': u'\U0001d428',
    u'\\mathbf{p}': u'\U0001d429',
    u'\\mathbf{q}': u'\U0001d42a',
    u'\\mathbf{r}': u'\U0001d42b',
    u'\\mathbf{s}': u'\U0001d42c',
    u'\\mathbf{t}': u'\U0001d42d',
    u'\\mathbf{u}': u'\U0001d42e',
    u'\\mathbf{v}': u'\U0001d42f',
    u'\\mathbf{w}': u'\U0001d430',
    u'\\mathbf{x}': u'\U0001d431',
    u'\\mathbf{y}': u'\U0001d432',
    u'\\mathbf{z}': u'\U0001d433',
    u'\\mathbin{{:}\\!\\!{-}\\!\\!{:}}': u'\u223a',
    u'\\mathbit{A}': u'\U0001d468',
    u'\\mathbit{B}': u'\U0001d469',
    u'\\mathbit{C}': u'\U0001d46a',
    u'\\mathbit{D}': u'\U0001d46b',
    u'\\mathbit{E}': u'\U0001d46c',
    u'\\mathbit{F}': u'\U0001d46d',
    u'\\mathbit{G}': u'\U0001d46e',
    u'\\mathbit{H}': u'\U0001d46f',
    u'\\mathbit{I}': u'\U0001d470',
    u'\\mathbit{J}': u'\U0001d471',
    u'\\mathbit{K}': u'\U0001d472',
    u'\\mathbit{L}': u'\U0001d473',
    u'\\mathbit{M}': u'\U0001d474',
    u'\\mathbit{N}': u'\U0001d475',
    u'\\mathbit{O}': u'\U0001d72d',
    u'\\mathbit{P}': u'\U0001d477',
    u'\\mathbit{Q}': u'\U0001d478',
    u'\\mathbit{R}': u'\U0001d479',
    u'\\mathbit{S}': u'\U0001d47a',
    u'\\mathbit{T}': u'\U0001d47b',
    u'\\mathbit{U}': u'\U0001d47c',
    u'\\mathbit{V}': u'\U0001d47d',
    u'\\mathbit{W}': u'\U0001d47e',
    u'\\mathbit{X}': u'\U0001d47f',
    u'\\mathbit{Y}': u'\U0001d480',
    u'\\mathbit{Z}': u'\U0001d481',
    u'\\mathbit{\\Alpha}': u'\U0001d736',
    u'\\mathbit{\\Beta}': u'\U0001d737',
    u'\\mathbit{\\Chi}': u'\U0001d74c',
    u'\\mathbit{\\Delta}': u'\U0001d739',
    u'\\mathbit{\\Epsilon}': u'\U0001d73a',
    u'\\mathbit{\\Eta}': u'\U0001d73c',
    u'\\mathbit{\\Gamma}': u'\U0001d738',
    u'\\mathbit{\\Iota}': u'\U0001d73e',
    u'\\mathbit{\\Kappa}': u'\U0001d73f',
    u'\\mathbit{\\Lambda}': u'\U0001d740',
    u'\\mathbit{\\Omega}': u'\U0001d74e',
    u'\\mathbit{\\Phi}': u'\U0001d74b',
    u'\\mathbit{\\Pi}': u'\U0001d745',
    u'\\mathbit{\\Psi}': u'\U0001d74d',
    u'\\mathbit{\\Rho}': u'\U0001d746',
    u'\\mathbit{\\Sigma}': u'\U0001d748',
    u'\\mathbit{\\Tau}': u'\U0001d749',
    u'\\mathbit{\\Theta}': u'\U0001d73d',
    u'\\mathbit{\\Upsilon}': u'\U0001d74a',
    u'\\mathbit{\\Xi}': u'\U0001d743',
    u'\\mathbit{\\Zeta}': u'\U0001d73b',
    u'\\mathbit{\\nabla}': u'\U0001d735',
    u'\\mathbit{\\phi}': u'\U0001d753',
    u'\\mathbit{\\varkappa}': u'\U0001d752',
    u'\\mathbit{\\varpi}': u'\U0001d755',
    u'\\mathbit{\\varrho}': u'\U0001d754',
    u'\\mathbit{\\varsigma}': u'\U0001d747',
    u'\\mathbit{\\vartheta}': u'\U0001d751',
    u'\\mathbit{a}': u'\U0001d482',
    u'\\mathbit{b}': u'\U0001d483',
    u'\\mathbit{c}': u'\U0001d484',
    u'\\mathbit{d}': u'\U0001d485',
    u'\\mathbit{e}': u'\U0001d486',
    u'\\mathbit{f}': u'\U0001d487',
    u'\\mathbit{g}': u'\U0001d488',
    u'\\mathbit{h}': u'\U0001d489',
    u'\\mathbit{i}': u'\U0001d48a',
    u'\\mathbit{j}': u'\U0001d48b',
    u'\\mathbit{k}': u'\U0001d48c',
    u'\\mathbit{l}': u'\U0001d48d',
    u'\\mathbit{m}': u'\U0001d48e',
    u'\\mathbit{n}': u'\U0001d48f',
    u'\\mathbit{o}': u'\U0001d490',
    u'\\mathbit{p}': u'\U0001d491',
    u'\\mathbit{q}': u'\U0001d492',
    u'\\mathbit{r}': u'\U0001d493',
    u'\\mathbit{s}': u'\U0001d494',
    u'\\mathbit{t}': u'\U0001d495',
    u'\\mathbit{u}': u'\U0001d496',
    u'\\mathbit{v}': u'\U0001d497',
    u'\\mathbit{w}': u'\U0001d498',
    u'\\mathbit{x}': u'\U0001d499',
    u'\\mathbit{y}': u'\U0001d49a',
    u'\\mathbit{z}': u'\U0001d49b',
    u'\\mathchar"2208': u'\u2316',
    u'\\mathfrak{A}': u'\U0001d504',
    u'\\mathfrak{B}': u'\U0001d505',
    u'\\mathfrak{C}': u'\u212d',
    u'\\mathfrak{D}': u'\U0001d507',
    u'\\mathfrak{E}': u'\U0001d508',
    u'\\mathfrak{F}': u'\U0001d509',
    u'\\mathfrak{G}': u'\U0001d50a',
    u'\\mathfrak{H}': u'\u210c',
    u'\\mathfrak{I}': u'\u2111',
    u'\\mathfrak{J}': u'\U0001d50d',
    u'\\mathfrak{K}': u'\U0001d50e',
    u'\\mathfrak{L}': u'\U0001d50f',
    u'\\mathfrak{M}': u'\U0001d510',
    u'\\mathfrak{N}': u'\U0001d511',
    u'\\mathfrak{O}': u'\U0001d512',
    u'\\mathfrak{P}': u'\U0001d513',
    u'\\mathfrak{Q}': u'\U0001d514',
    u'\\mathfrak{R}': u'\u211c',
    u'\\mathfrak{S}': u'\U0001d516',
    u'\\mathfrak{T}': u'\U0001d517',
    u'\\mathfrak{U}': u'\U0001d518',
    u'\\mathfrak{V}': u'\U0001d519',
    u'\\mathfrak{W}': u'\U0001d51a',
    u'\\mathfrak{X}': u'\U0001d51b',
    u'\\mathfrak{Y}': u'\U0001d51c',
    u'\\mathfrak{Z}': u'\u2128',
    u'\\mathfrak{a}': u'\U0001d51e',
    u'\\mathfrak{b}': u'\U0001d51f',
    u'\\mathfrak{c}': u'\U0001d520',
    u'\\mathfrak{d}': u'\U0001d521',
    u'\\mathfrak{e}': u'\U0001d522',
    u'\\mathfrak{f}': u'\U0001d523',
    u'\\mathfrak{g}': u'\U0001d524',
    u'\\mathfrak{h}': u'\U0001d525',
    u'\\mathfrak{i}': u'\U0001d526',
    u'\\mathfrak{j}': u'\U0001d527',
    u'\\mathfrak{k}': u'\U0001d528',
    u'\\mathfrak{l}': u'\U0001d529',
    u'\\mathfrak{m}': u'\U0001d52a',
    u'\\mathfrak{n}': u'\U0001d52b',
    u'\\mathfrak{o}': u'\U0001d52c',
    u'\\mathfrak{p}': u'\U0001d52d',
    u'\\mathfrak{q}': u'\U0001d52e',
    u'\\mathfrak{r}': u'\U0001d52f',
    u'\\mathfrak{s}': u'\U0001d530',
    u'\\mathfrak{t}': u'\U0001d531',
    u'\\mathfrak{u}': u'\U0001d532',
    u'\\mathfrak{v}': u'\U0001d533',
    u'\\mathfrak{w}': u'\U0001d534',
    u'\\mathfrak{x}': u'\U0001d535',
    u'\\mathfrak{y}': u'\U0001d536',
    u'\\mathfrak{z}': u'\U0001d537',
    u'\\mathmit{A}': u'\U0001d4d0',
    u'\\mathmit{B}': u'\U0001d4d1',
    u'\\mathmit{C}': u'\U0001d4d2',
    u'\\mathmit{D}': u'\U0001d4d3',
    u'\\mathmit{E}': u'\U0001d4d4',
    u'\\mathmit{F}': u'\U0001d4d5',
    u'\\mathmit{G}': u'\U0001d4d6',
    u'\\mathmit{H}': u'\U0001d4d7',
    u'\\mathmit{I}': u'\U0001d4d8',
    u'\\mathmit{J}': u'\U0001d4d9',
    u'\\mathmit{K}': u'\U0001d4da',
    u'\\mathmit{L}': u'\U0001d4db',
    u'\\mathmit{M}': u'\U0001d4dc',
    u'\\mathmit{N}': u'\U0001d4dd',
    u'\\mathmit{O}': u'\U0001d4de',
    u'\\mathmit{P}': u'\U0001d4df',
    u'\\mathmit{Q}': u'\U0001d4e0',
    u'\\mathmit{R}': u'\U0001d4e1',
    u'\\mathmit{S}': u'\U0001d4e2',
    u'\\mathmit{T}': u'\U0001d4e3',
    u'\\mathmit{U}': u'\U0001d4e4',
    u'\\mathmit{V}': u'\U0001d4e5',
    u'\\mathmit{W}': u'\U0001d4e6',
    u'\\mathmit{X}': u'\U0001d4e7',
    u'\\mathmit{Y}': u'\U0001d4e8',
    u'\\mathmit{Z}': u'\U0001d4e9',
    u'\\mathmit{a}': u'\U0001d4ea',
    u'\\mathmit{b}': u'\U0001d4eb',
    u'\\mathmit{c}': u'\U0001d4ec',
    u'\\mathmit{d}': u'\U0001d4ed',
    u'\\mathmit{e}': u'\U0001d4ee',
    u'\\mathmit{f}': u'\U0001d4ef',
    u'\\mathmit{g}': u'\U0001d4f0',
    u'\\mathmit{h}': u'\U0001d4f1',
    u'\\mathmit{i}': u'\U0001d4f2',
    u'\\mathmit{j}': u'\U0001d4f3',
    u'\\mathmit{k}': u'\U0001d4f4',
    u'\\mathmit{l}': u'\U0001d4f5',
    u'\\mathmit{m}': u'\U0001d4f6',
    u'\\mathmit{n}': u'\U0001d4f7',
    u'\\mathmit{o}': u'\U0001d4f8',
    u'\\mathmit{p}': u'\U0001d4f9',
    u'\\mathmit{q}': u'\U0001d4fa',
    u'\\mathmit{r}': u'\U0001d4fb',
    u'\\mathmit{s}': u'\U0001d4fc',
    u'\\mathmit{t}': u'\U0001d4fd',
    u'\\mathmit{u}': u'\U0001d4fe',
    u'\\mathmit{v}': u'\U0001d4ff',
    u'\\mathmit{w}': u'\U0001d500',
    u'\\mathmit{x}': u'\U0001d501',
    u'\\mathmit{y}': u'\U0001d502',
    u'\\mathmit{z}': u'\U0001d503',
    u"\\mathrm{'Y}": u'\u038e',
    u"\\mathrm{'\\Omega}": u'\u038f',
    u'\\mathrm{\\ddot{I}}': u'\u03aa',
    u'\\mathrm{\\ddot{Y}}': u'\u03ab',
    u'\\mathrm{\\mu}': u'\xb5',
    u'\\mathscr{A}': u'\U0001d49c',
    u'\\mathscr{B}': u'\u212c',
    u'\\mathscr{C}': u'\U0001d49e',
    u'\\mathscr{D}': u'\U0001d49f',
    u'\\mathscr{E}': u'\u2130',
    u'\\mathscr{F}': u'\u2131',
    u'\\mathscr{G}': u'\U0001d4a2',
    u'\\mathscr{H}': u'\u210b',
    u'\\mathscr{I}': u'\u2110',
    u'\\mathscr{J}': u'\U0001d4a5',
    u'\\mathscr{K}': u'\U0001d4a6',
    u'\\mathscr{L}': u'\u2112',
    u'\\mathscr{M}': u'\u2133',
    u'\\mathscr{N}': u'\U0001d4a9',
    u'\\mathscr{O}': u'\U0001d4aa',
    u'\\mathscr{P}': u'\U0001d4ab',
    u'\\mathscr{Q}': u'\U0001d4ac',
    u'\\mathscr{R}': u'\u211b',
    u'\\mathscr{S}': u'\U0001d4ae',
    u'\\mathscr{T}': u'\U0001d4af',
    u'\\mathscr{U}': u'\U0001d4b0',
    u'\\mathscr{V}': u'\U0001d4b1',
    u'\\mathscr{W}': u'\U0001d4b2',
    u'\\mathscr{X}': u'\U0001d4b3',
    u'\\mathscr{Y}': u'\U0001d4b4',
    u'\\mathscr{Z}': u'\U0001d4b5',
    u'\\mathscr{a}': u'\U0001d4b6',
    u'\\mathscr{b}': u'\U0001d4b7',
    u'\\mathscr{c}': u'\U0001d4b8',
    u'\\mathscr{d}': u'\U0001d4b9',
    u'\\mathscr{e}': u'\u212f',
    u'\\mathscr{f}': u'\U0001d4bb',
    u'\\mathscr{g}': u'\u210a',
    u'\\mathscr{h}': u'\U0001d4bd',
    u'\\mathscr{i}': u'\U0001d4be',
    u'\\mathscr{j}': u'\U0001d4bf',
    u'\\mathscr{k}': u'\U0001d4c0',
    u'\\mathscr{l}': u'\U0001d4c1',
    u'\\mathscr{m}': u'\U0001d4c2',
    u'\\mathscr{n}': u'\U0001d4c3',
    u'\\mathscr{o}': u'\u2134',
    u'\\mathscr{p}': u'\U0001d4c5',
    u'\\mathscr{q}': u'\U0001d4c6',
    u'\\mathscr{r}': u'\U0001d4c7',
    u'\\mathscr{s}': u'\U0001d4c8',
    u'\\mathscr{t}': u'\U0001d4c9',
    u'\\mathscr{u}': u'\U0001d4ca',
    u'\\mathscr{v}': u'\U0001d4cb',
    u'\\mathscr{w}': u'\U0001d4cc',
    u'\\mathscr{x}': u'\U0001d4cd',
    u'\\mathscr{y}': u'\U0001d4ce',
    u'\\mathscr{z}': u'\U0001d4cf',
    u'\\mathsfbfsl{A}': u'\U0001d63c',
    u'\\mathsfbfsl{B}': u'\U0001d63d',
    u'\\mathsfbfsl{C}': u'\U0001d63e',
    u'\\mathsfbfsl{D}': u'\U0001d63f',
    u'\\mathsfbfsl{E}': u'\U0001d640',
    u'\\mathsfbfsl{F}': u'\U0001d641',
    u'\\mathsfbfsl{G}': u'\U0001d642',
    u'\\mathsfbfsl{H}': u'\U0001d643',
    u'\\mathsfbfsl{I}': u'\U0001d644',
    u'\\mathsfbfsl{J}': u'\U0001d645',
    u'\\mathsfbfsl{K}': u'\U0001d646',
    u'\\mathsfbfsl{L}': u'\U0001d647',
    u'\\mathsfbfsl{M}': u'\U0001d648',
    u'\\mathsfbfsl{N}': u'\U0001d649',
    u'\\mathsfbfsl{O}': u'\U0001d64a',
    u'\\mathsfbfsl{P}': u'\U0001d64b',
    u'\\mathsfbfsl{Q}': u'\U0001d64c',
    u'\\mathsfbfsl{R}': u'\U0001d64d',
    u'\\mathsfbfsl{S}': u'\U0001d64e',
    u'\\mathsfbfsl{T}': u'\U0001d64f',
    u'\\mathsfbfsl{U}': u'\U0001d650',
    u'\\mathsfbfsl{V}': u'\U0001d651',
    u'\\mathsfbfsl{W}': u'\U0001d652',
    u'\\mathsfbfsl{X}': u'\U0001d653',
    u'\\mathsfbfsl{Y}': u'\U0001d654',
    u'\\mathsfbfsl{Z}': u'\U0001d655',
    u'\\mathsfbfsl{\\Alpha}': u'\U0001d7aa',
    u'\\mathsfbfsl{\\Beta}': u'\U0001d7ab',
    u'\\mathsfbfsl{\\Chi}': u'\U0001d7c0',
    u'\\mathsfbfsl{\\Delta}': u'\U0001d7ad',
    u'\\mathsfbfsl{\\Epsilon}': u'\U0001d7ae',
    u'\\mathsfbfsl{\\Eta}': u'\U0001d7b0',
    u'\\mathsfbfsl{\\Gamma}': u'\U0001d7ac',
    u'\\mathsfbfsl{\\Iota}': u'\U0001d7b2',
    u'\\mathsfbfsl{\\Kappa}': u'\U0001d7b3',
    u'\\mathsfbfsl{\\Lambda}': u'\U0001d7b4',
    u'\\mathsfbfsl{\\Omega}': u'\U0001d7c2',
    u'\\mathsfbfsl{\\Phi}': u'\U0001d7bf',
    u'\\mathsfbfsl{\\Pi}': u'\U0001d7b9',
    u'\\mathsfbfsl{\\Psi}': u'\U0001d7c1',
    u'\\mathsfbfsl{\\Rho}': u'\U0001d7ba',
    u'\\mathsfbfsl{\\Sigma}': u'\U0001d7bc',
    u'\\mathsfbfsl{\\Tau}': u'\U0001d7bd',
    u'\\mathsfbfsl{\\Upsilon}': u'\U0001d7be',
    u'\\mathsfbfsl{\\Xi}': u'\U0001d7b7',
    u'\\mathsfbfsl{\\Zeta}': u'\U0001d7af',
    u'\\mathsfbfsl{\\nabla}': u'\U0001d7a9',
    u'\\mathsfbfsl{\\phi}': u'\U0001d7c7',
    u'\\mathsfbfsl{\\varkappa}': u'\U0001d7c6',
    u'\\mathsfbfsl{\\varpi}': u'\U0001d7c9',
    u'\\mathsfbfsl{\\varrho}': u'\U0001d7c8',
    u'\\mathsfbfsl{\\varsigma}': u'\U0001d7bb',
    u'\\mathsfbfsl{\\vartheta}': u'\U0001d7c5',
    u'\\mathsfbfsl{a}': u'\U0001d656',
    u'\\mathsfbfsl{b}': u'\U0001d657',
    u'\\mathsfbfsl{c}': u'\U0001d658',
    u'\\mathsfbfsl{d}': u'\U0001d659',
    u'\\mathsfbfsl{e}': u'\U0001d65a',
    u'\\mathsfbfsl{f}': u'\U0001d65b',
    u'\\mathsfbfsl{g}': u'\U0001d65c',
    u'\\mathsfbfsl{h}': u'\U0001d65d',
    u'\\mathsfbfsl{i}': u'\U0001d65e',
    u'\\mathsfbfsl{j}': u'\U0001d65f',
    u'\\mathsfbfsl{k}': u'\U0001d660',
    u'\\mathsfbfsl{l}': u'\U0001d661',
    u'\\mathsfbfsl{m}': u'\U0001d662',
    u'\\mathsfbfsl{n}': u'\U0001d663',
    u'\\mathsfbfsl{o}': u'\U0001d664',
    u'\\mathsfbfsl{p}': u'\U0001d665',
    u'\\mathsfbfsl{q}': u'\U0001d666',
    u'\\mathsfbfsl{r}': u'\U0001d667',
    u'\\mathsfbfsl{s}': u'\U0001d668',
    u'\\mathsfbfsl{t}': u'\U0001d669',
    u'\\mathsfbfsl{u}': u'\U0001d66a',
    u'\\mathsfbfsl{v}': u'\U0001d66b',
    u'\\mathsfbfsl{w}': u'\U0001d66c',
    u'\\mathsfbfsl{x}': u'\U0001d66d',
    u'\\mathsfbfsl{y}': u'\U0001d66e',
    u'\\mathsfbfsl{z}': u'\U0001d66f',
    u'\\mathsfbf{0}': u'\U0001d7ec',
    u'\\mathsfbf{1}': u'\U0001d7ed',
    u'\\mathsfbf{2}': u'\U0001d7ee',
    u'\\mathsfbf{3}': u'\U0001d7ef',
    u'\\mathsfbf{4}': u'\U0001d7f0',
    u'\\mathsfbf{5}': u'\U0001d7f1',
    u'\\mathsfbf{6}': u'\U0001d7f2',
    u'\\mathsfbf{7}': u'\U0001d7f3',
    u'\\mathsfbf{8}': u'\U0001d7f4',
    u'\\mathsfbf{9}': u'\U0001d7f5',
    u'\\mathsfbf{A}': u'\U0001d5d4',
    u'\\mathsfbf{B}': u'\U0001d5d5',
    u'\\mathsfbf{C}': u'\U0001d5d6',
    u'\\mathsfbf{D}': u'\U0001d5d7',
    u'\\mathsfbf{E}': u'\U0001d5d8',
    u'\\mathsfbf{F}': u'\U0001d5d9',
    u'\\mathsfbf{G}': u'\U0001d5da',
    u'\\mathsfbf{H}': u'\U0001d5db',
    u'\\mathsfbf{I}': u'\U0001d5dc',
    u'\\mathsfbf{J}': u'\U0001d5dd',
    u'\\mathsfbf{K}': u'\U0001d5de',
    u'\\mathsfbf{L}': u'\U0001d5df',
    u'\\mathsfbf{M}': u'\U0001d5e0',
    u'\\mathsfbf{N}': u'\U0001d5e1',
    u'\\mathsfbf{O}': u'\U0001d5e2',
    u'\\mathsfbf{P}': u'\U0001d5e3',
    u'\\mathsfbf{Q}': u'\U0001d5e4',
    u'\\mathsfbf{R}': u'\U0001d5e5',
    u'\\mathsfbf{S}': u'\U0001d5e6',
    u'\\mathsfbf{T}': u'\U0001d5e7',
    u'\\mathsfbf{U}': u'\U0001d5e8',
    u'\\mathsfbf{V}': u'\U0001d5e9',
    u'\\mathsfbf{W}': u'\U0001d5ea',
    u'\\mathsfbf{X}': u'\U0001d5eb',
    u'\\mathsfbf{Y}': u'\U0001d5ec',
    u'\\mathsfbf{Z}': u'\U0001d5ed',
    u'\\mathsfbf{\\Alpha}': u'\U0001d770',
    u'\\mathsfbf{\\Beta}': u'\U0001d771',
    u'\\mathsfbf{\\Chi}': u'\U0001d786',
    u'\\mathsfbf{\\Delta}': u'\U0001d773',
    u'\\mathsfbf{\\Epsilon}': u'\U0001d774',
    u'\\mathsfbf{\\Eta}': u'\U0001d776',
    u'\\mathsfbf{\\Gamma}': u'\U0001d772',
    u'\\mathsfbf{\\Iota}': u'\U0001d778',
    u'\\mathsfbf{\\Kappa}': u'\U0001d779',
    u'\\mathsfbf{\\Lambda}': u'\U0001d77a',
    u'\\mathsfbf{\\Omega}': u'\U0001d788',
    u'\\mathsfbf{\\Phi}': u'\U0001d785',
    u'\\mathsfbf{\\Pi}': u'\U0001d77f',
    u'\\mathsfbf{\\Psi}': u'\U0001d787',
    u'\\mathsfbf{\\Rho}': u'\U0001d780',
    u'\\mathsfbf{\\Sigma}': u'\U0001d782',
    u'\\mathsfbf{\\Tau}': u'\U0001d783',
    u'\\mathsfbf{\\Theta}': u'\U0001d777',
    u'\\mathsfbf{\\Upsilon}': u'\U0001d784',
    u'\\mathsfbf{\\Xi}': u'\U0001d77d',
    u'\\mathsfbf{\\Zeta}': u'\U0001d775',
    u'\\mathsfbf{\\nabla}': u'\U0001d76f',
    u'\\mathsfbf{\\phi}': u'\U0001d78d',
    u'\\mathsfbf{\\varkappa}': u'\U0001d78c',
    u'\\mathsfbf{\\varpi}': u'\U0001d78f',
    u'\\mathsfbf{\\varrho}': u'\U0001d78e',
    u'\\mathsfbf{\\varsigma}': u'\U0001d781',
    u'\\mathsfbf{\\vartheta}': u'\U0001d78b',
    u'\\mathsfbf{a}': u'\U0001d5ee',
    u'\\mathsfbf{b}': u'\U0001d5ef',
    u'\\mathsfbf{c}': u'\U0001d5f0',
    u'\\mathsfbf{d}': u'\U0001d5f1',
    u'\\mathsfbf{e}': u'\U0001d5f2',
    u'\\mathsfbf{f}': u'\U0001d5f3',
    u'\\mathsfbf{g}': u'\U0001d5f4',
    u'\\mathsfbf{h}': u'\U0001d5f5',
    u'\\mathsfbf{i}': u'\U0001d5f6',
    u'\\mathsfbf{j}': u'\U0001d5f7',
    u'\\mathsfbf{k}': u'\U0001d5f8',
    u'\\mathsfbf{l}': u'\U0001d5f9',
    u'\\mathsfbf{m}': u'\U0001d5fa',
    u'\\mathsfbf{n}': u'\U0001d5fb',
    u'\\mathsfbf{o}': u'\U0001d5fc',
    u'\\mathsfbf{p}': u'\U0001d5fd',
    u'\\mathsfbf{q}': u'\U0001d5fe',
    u'\\mathsfbf{r}': u'\U0001d5ff',
    u'\\mathsfbf{s}': u'\U0001d600',
    u'\\mathsfbf{t}': u'\U0001d601',
    u'\\mathsfbf{u}': u'\U0001d602',
    u'\\mathsfbf{v}': u'\U0001d603',
    u'\\mathsfbf{w}': u'\U0001d604',
    u'\\mathsfbf{x}': u'\U0001d605',
    u'\\mathsfbf{y}': u'\U0001d606',
    u'\\mathsfbf{z}': u'\U0001d607',
    u'\\mathsfsl{A}': u'\U0001d608',
    u'\\mathsfsl{B}': u'\U0001d609',
    u'\\mathsfsl{C}': u'\U0001d60a',
    u'\\mathsfsl{D}': u'\U0001d60b',
    u'\\mathsfsl{E}': u'\U0001d60c',
    u'\\mathsfsl{F}': u'\U0001d60d',
    u'\\mathsfsl{G}': u'\U0001d60e',
    u'\\mathsfsl{H}': u'\U0001d60f',
    u'\\mathsfsl{I}': u'\U0001d610',
    u'\\mathsfsl{J}': u'\U0001d611',
    u'\\mathsfsl{K}': u'\U0001d612',
    u'\\mathsfsl{L}': u'\U0001d613',
    u'\\mathsfsl{M}': u'\U0001d614',
    u'\\mathsfsl{N}': u'\U0001d615',
    u'\\mathsfsl{O}': u'\U0001d616',
    u'\\mathsfsl{P}': u'\U0001d617',
    u'\\mathsfsl{Q}': u'\U0001d618',
    u'\\mathsfsl{R}': u'\U0001d619',
    u'\\mathsfsl{S}': u'\U0001d61a',
    u'\\mathsfsl{T}': u'\U0001d61b',
    u'\\mathsfsl{U}': u'\U0001d61c',
    u'\\mathsfsl{V}': u'\U0001d61d',
    u'\\mathsfsl{W}': u'\U0001d61e',
    u'\\mathsfsl{X}': u'\U0001d61f',
    u'\\mathsfsl{Y}': u'\U0001d620',
    u'\\mathsfsl{Z}': u'\U0001d621',
    u'\\mathsfsl{a}': u'\U0001d622',
    u'\\mathsfsl{b}': u'\U0001d623',
    u'\\mathsfsl{c}': u'\U0001d624',
    u'\\mathsfsl{d}': u'\U0001d625',
    u'\\mathsfsl{e}': u'\U0001d626',
    u'\\mathsfsl{f}': u'\U0001d627',
    u'\\mathsfsl{g}': u'\U0001d628',
    u'\\mathsfsl{h}': u'\U0001d629',
    u'\\mathsfsl{i}': u'\U0001d62a',
    u'\\mathsfsl{j}': u'\U0001d62b',
    u'\\mathsfsl{k}': u'\U0001d62c',
    u'\\mathsfsl{l}': u'\U0001d62d',
    u'\\mathsfsl{m}': u'\U0001d62e',
    u'\\mathsfsl{n}': u'\U0001d62f',
    u'\\mathsfsl{o}': u'\U0001d630',
    u'\\mathsfsl{p}': u'\U0001d631',
    u'\\mathsfsl{q}': u'\U0001d632',
    u'\\mathsfsl{r}': u'\U0001d633',
    u'\\mathsfsl{s}': u'\U0001d634',
    u'\\mathsfsl{t}': u'\U0001d635',
    u'\\mathsfsl{u}': u'\U0001d636',
    u'\\mathsfsl{v}': u'\U0001d637',
    u'\\mathsfsl{w}': u'\U0001d638',
    u'\\mathsfsl{x}': u'\U0001d639',
    u'\\mathsfsl{y}': u'\U0001d63a',
    u'\\mathsfsl{z}': u'\U0001d63b',
    u'\\mathsf{0}': u'\U0001d7e2',
    u'\\mathsf{1}': u'\U0001d7e3',
    u'\\mathsf{2}': u'\U0001d7e4',
    u'\\mathsf{3}': u'\U0001d7e5',
    u'\\mathsf{4}': u'\U0001d7e6',
    u'\\mathsf{5}': u'\U0001d7e7',
    u'\\mathsf{6}': u'\U0001d7e8',
    u'\\mathsf{7}': u'\U0001d7e9',
    u'\\mathsf{8}': u'\U0001d7ea',
    u'\\mathsf{9}': u'\U0001d7eb',
    u'\\mathsf{A}': u'\U0001d5a0',
    u'\\mathsf{B}': u'\U0001d5a1',
    u'\\mathsf{C}': u'\U0001d5a2',
    u'\\mathsf{D}': u'\U0001d5a3',
    u'\\mathsf{E}': u'\U0001d5a4',
    u'\\mathsf{F}': u'\U0001d5a5',
    u'\\mathsf{G}': u'\U0001d5a6',
    u'\\mathsf{H}': u'\U0001d5a7',
    u'\\mathsf{I}': u'\U0001d5a8',
    u'\\mathsf{J}': u'\U0001d5a9',
    u'\\mathsf{K}': u'\U0001d5aa',
    u'\\mathsf{L}': u'\U0001d5ab',
    u'\\mathsf{M}': u'\U0001d5ac',
    u'\\mathsf{N}': u'\U0001d5ad',
    u'\\mathsf{O}': u'\U0001d5ae',
    u'\\mathsf{P}': u'\U0001d5af',
    u'\\mathsf{Q}': u'\U0001d5b0',
    u'\\mathsf{R}': u'\U0001d5b1',
    u'\\mathsf{S}': u'\U0001d5b2',
    u'\\mathsf{T}': u'\U0001d5b3',
    u'\\mathsf{U}': u'\U0001d5b4',
    u'\\mathsf{V}': u'\U0001d5b5',
    u'\\mathsf{W}': u'\U0001d5b6',
    u'\\mathsf{X}': u'\U0001d5b7',
    u'\\mathsf{Y}': u'\U0001d5b8',
    u'\\mathsf{Z}': u'\U0001d5b9',
    u'\\mathsf{a}': u'\U0001d5ba',
    u'\\mathsf{b}': u'\U0001d5bb',
    u'\\mathsf{c}': u'\U0001d5bc',
    u'\\mathsf{d}': u'\U0001d5bd',
    u'\\mathsf{e}': u'\U0001d5be',
    u'\\mathsf{f}': u'\U0001d5bf',
    u'\\mathsf{g}': u'\U0001d5c0',
    u'\\mathsf{h}': u'\U0001d5c1',
    u'\\mathsf{i}': u'\U0001d5c2',
    u'\\mathsf{j}': u'\U0001d5c3',
    u'\\mathsf{k}': u'\U0001d5c4',
    u'\\mathsf{l}': u'\U0001d5c5',
    u'\\mathsf{m}': u'\U0001d5c6',
    u'\\mathsf{n}': u'\U0001d5c7',
    u'\\mathsf{o}': u'\U0001d5c8',
    u'\\mathsf{p}': u'\U0001d5c9',
    u'\\mathsf{q}': u'\U0001d5ca',
    u'\\mathsf{r}': u'\U0001d5cb',
    u'\\mathsf{s}': u'\U0001d5cc',
    u'\\mathsf{t}': u'\U0001d5cd',
    u'\\mathsf{u}': u'\U0001d5ce',
    u'\\mathsf{v}': u'\U0001d5cf',
    u'\\mathsf{w}': u'\U0001d5d0',
    u'\\mathsf{x}': u'\U0001d5d1',
    u'\\mathsf{y}': u'\U0001d5d2',
    u'\\mathsf{z}': u'\U0001d5d3',
    u'\\mathslbb{A}': u'\U0001d56c',
    u'\\mathslbb{B}': u'\U0001d56d',
    u'\\mathslbb{C}': u'\U0001d56e',
    u'\\mathslbb{D}': u'\U0001d56f',
    u'\\mathslbb{E}': u'\U0001d570',
    u'\\mathslbb{F}': u'\U0001d571',
    u'\\mathslbb{G}': u'\U0001d572',
    u'\\mathslbb{H}': u'\U0001d573',
    u'\\mathslbb{I}': u'\U0001d574',
    u'\\mathslbb{J}': u'\U0001d575',
    u'\\mathslbb{K}': u'\U0001d576',
    u'\\mathslbb{L}': u'\U0001d577',
    u'\\mathslbb{M}': u'\U0001d578',
    u'\\mathslbb{N}': u'\U0001d579',
    u'\\mathslbb{O}': u'\U0001d57a',
    u'\\mathslbb{P}': u'\U0001d57b',
    u'\\mathslbb{Q}': u'\U0001d57c',
    u'\\mathslbb{R}': u'\U0001d57d',
    u'\\mathslbb{S}': u'\U0001d57e',
    u'\\mathslbb{T}': u'\U0001d57f',
    u'\\mathslbb{U}': u'\U0001d580',
    u'\\mathslbb{V}': u'\U0001d581',
    u'\\mathslbb{W}': u'\U0001d582',
    u'\\mathslbb{X}': u'\U0001d583',
    u'\\mathslbb{Y}': u'\U0001d584',
    u'\\mathslbb{Z}': u'\U0001d585',
    u'\\mathslbb{a}': u'\U0001d586',
    u'\\mathslbb{b}': u'\U0001d587',
    u'\\mathslbb{c}': u'\U0001d588',
    u'\\mathslbb{d}': u'\U0001d589',
    u'\\mathslbb{e}': u'\U0001d58a',
    u'\\mathslbb{f}': u'\U0001d58b',
    u'\\mathslbb{g}': u'\U0001d58c',
    u'\\mathslbb{h}': u'\U0001d58d',
    u'\\mathslbb{i}': u'\U0001d58e',
    u'\\mathslbb{j}': u'\U0001d58f',
    u'\\mathslbb{k}': u'\U0001d590',
    u'\\mathslbb{l}': u'\U0001d591',
    u'\\mathslbb{m}': u'\U0001d592',
    u'\\mathslbb{n}': u'\U0001d593',
    u'\\mathslbb{o}': u'\U0001d594',
    u'\\mathslbb{p}': u'\U0001d595',
    u'\\mathslbb{q}': u'\U0001d596',
    u'\\mathslbb{r}': u'\U0001d597',
    u'\\mathslbb{s}': u'\U0001d598',
    u'\\mathslbb{t}': u'\U0001d599',
    u'\\mathslbb{u}': u'\U0001d59a',
    u'\\mathslbb{v}': u'\U0001d59b',
    u'\\mathslbb{w}': u'\U0001d59c',
    u'\\mathslbb{x}': u'\U0001d59d',
    u'\\mathslbb{y}': u'\U0001d59e',
    u'\\mathslbb{z}': u'\U0001d59f',
    u'\\mathsl{A}': u'\U0001d434',
    u'\\mathsl{B}': u'\U0001d435',
    u'\\mathsl{C}': u'\U0001d436',
    u'\\mathsl{D}': u'\U0001d437',
    u'\\mathsl{E}': u'\U0001d438',
    u'\\mathsl{F}': u'\U0001d439',
    u'\\mathsl{G}': u'\U0001d43a',
    u'\\mathsl{H}': u'\U0001d43b',
    u'\\mathsl{I}': u'\U0001d43c',
    u'\\mathsl{J}': u'\U0001d43d',
    u'\\mathsl{K}': u'\U0001d43e',
    u'\\mathsl{L}': u'\U0001d43f',
    u'\\mathsl{M}': u'\U0001d440',
    u'\\mathsl{N}': u'\U0001d441',
    u'\\mathsl{O}': u'\U0001d442',
    u'\\mathsl{P}': u'\U0001d443',
    u'\\mathsl{Q}': u'\U0001d444',
    u'\\mathsl{R}': u'\U0001d445',
    u'\\mathsl{S}': u'\U0001d446',
    u'\\mathsl{T}': u'\U0001d447',
    u'\\mathsl{U}': u'\U0001d448',
    u'\\mathsl{V}': u'\U0001d449',
    u'\\mathsl{W}': u'\U0001d44a',
    u'\\mathsl{X}': u'\U0001d44b',
    u'\\mathsl{Y}': u'\U0001d44c',
    u'\\mathsl{Z}': u'\U0001d44d',
    u'\\mathsl{\\Alpha}': u'\U0001d6fc',
    u'\\mathsl{\\Beta}': u'\U0001d6fd',
    u'\\mathsl{\\Chi}': u'\U0001d712',
    u'\\mathsl{\\Delta}': u'\U0001d6ff',
    u'\\mathsl{\\Epsilon}': u'\U0001d700',
    u'\\mathsl{\\Eta}': u'\U0001d702',
    u'\\mathsl{\\Gamma}': u'\U0001d6fe',
    u'\\mathsl{\\Iota}': u'\U0001d704',
    u'\\mathsl{\\Kappa}': u'\U0001d705',
    u'\\mathsl{\\Lambda}': u'\U0001d706',
    u'\\mathsl{\\Omega}': u'\U0001d714',
    u'\\mathsl{\\Phi}': u'\U0001d711',
    u'\\mathsl{\\Pi}': u'\U0001d70b',
    u'\\mathsl{\\Psi}': u'\U0001d713',
    u'\\mathsl{\\Rho}': u'\U0001d70c',
    u'\\mathsl{\\Sigma}': u'\U0001d70e',
    u'\\mathsl{\\Tau}': u'\U0001d70f',
    u'\\mathsl{\\Theta}': u'\U0001d703',
    u'\\mathsl{\\Upsilon}': u'\U0001d710',
    u'\\mathsl{\\Xi}': u'\U0001d709',
    u'\\mathsl{\\Zeta}': u'\U0001d701',
    u'\\mathsl{\\nabla}': u'\U0001d6fb',
    u'\\mathsl{\\phi}': u'\U0001d719',
    u'\\mathsl{\\varkappa}': u'\U0001d718',
    u'\\mathsl{\\varpi}': u'\U0001d71b',
    u'\\mathsl{\\varrho}': u'\U0001d71a',
    u'\\mathsl{\\varsigma}': u'\U0001d70d',
    u'\\mathsl{\\vartheta}': u'\U0001d717',
    u'\\mathsl{a}': u'\U0001d44e',
    u'\\mathsl{b}': u'\U0001d44f',
    u'\\mathsl{c}': u'\U0001d450',
    u'\\mathsl{d}': u'\U0001d451',
    u'\\mathsl{e}': u'\U0001d452',
    u'\\mathsl{f}': u'\U0001d453',
    u'\\mathsl{g}': u'\U0001d454',
    u'\\mathsl{i}': u'\U0001d456',
    u'\\mathsl{j}': u'\U0001d457',
    u'\\mathsl{k}': u'\U0001d458',
    u'\\mathsl{l}': u'\U0001d459',
    u'\\mathsl{m}': u'\U0001d45a',
    u'\\mathsl{n}': u'\U0001d45b',
    u'\\mathsl{o}': u'\U0001d45c',
    u'\\mathsl{p}': u'\U0001d45d',
    u'\\mathsl{q}': u'\U0001d45e',
    u'\\mathsl{r}': u'\U0001d45f',
    u'\\mathsl{s}': u'\U0001d460',
    u'\\mathsl{t}': u'\U0001d461',
    u'\\mathsl{u}': u'\U0001d462',
    u'\\mathsl{v}': u'\U0001d463',
    u'\\mathsl{w}': u'\U0001d464',
    u'\\mathsl{x}': u'\U0001d465',
    u'\\mathsl{y}': u'\U0001d466',
    u'\\mathsl{z}': u'\U0001d467',
    u'\\mathtt{0}': u'\U0001d7f6',
    u'\\mathtt{1}': u'\U0001d7f7',
    u'\\mathtt{2}': u'\U0001d7f8',
    u'\\mathtt{3}': u'\U0001d7f9',
    u'\\mathtt{4}': u'\U0001d7fa',
    u'\\mathtt{5}': u'\U0001d7fb',
    u'\\mathtt{6}': u'\U0001d7fc',
    u'\\mathtt{7}': u'\U0001d7fd',
    u'\\mathtt{8}': u'\U0001d7fe',
    u'\\mathtt{9}': u'\U0001d7ff',
    u'\\mathtt{A}': u'\U0001d670',
    u'\\mathtt{B}': u'\U0001d671',
    u'\\mathtt{C}': u'\U0001d672',
    u'\\mathtt{D}': u'\U0001d673',
    u'\\mathtt{E}': u'\U0001d674',
    u'\\mathtt{F}': u'\U0001d675',
    u'\\mathtt{G}': u'\U0001d676',
    u'\\mathtt{H}': u'\U0001d677',
    u'\\mathtt{I}': u'\U0001d678',
    u'\\mathtt{J}': u'\U0001d679',
    u'\\mathtt{K}': u'\U0001d67a',
    u'\\mathtt{L}': u'\U0001d67b',
    u'\\mathtt{M}': u'\U0001d67c',
    u'\\mathtt{N}': u'\U0001d67d',
    u'\\mathtt{O}': u'\U0001d67e',
    u'\\mathtt{P}': u'\U0001d67f',
    u'\\mathtt{Q}': u'\U0001d680',
    u'\\mathtt{R}': u'\U0001d681',
    u'\\mathtt{S}': u'\U0001d682',
    u'\\mathtt{T}': u'\U0001d683',
    u'\\mathtt{U}': u'\U0001d684',
    u'\\mathtt{V}': u'\U0001d685',
    u'\\mathtt{W}': u'\U0001d686',
    u'\\mathtt{X}': u'\U0001d687',
    u'\\mathtt{Y}': u'\U0001d688',
    u'\\mathtt{Z}': u'\U0001d689',
    u'\\mathtt{a}': u'\U0001d68a',
    u'\\mathtt{b}': u'\U0001d68b',
    u'\\mathtt{c}': u'\U0001d68c',
    u'\\mathtt{d}': u'\U0001d68d',
    u'\\mathtt{e}': u'\U0001d68e',
    u'\\mathtt{f}': u'\U0001d68f',
    u'\\mathtt{g}': u'\U0001d690',
    u'\\mathtt{h}': u'\U0001d691',
    u'\\mathtt{i}': u'\U0001d692',
    u'\\mathtt{j}': u'\U0001d693',
    u'\\mathtt{k}': u'\U0001d694',
    u'\\mathtt{l}': u'\U0001d695',
    u'\\mathtt{m}': u'\U0001d696',
    u'\\mathtt{n}': u'\U0001d697',
    u'\\mathtt{o}': u'\U0001d698',
    u'\\mathtt{p}': u'\U0001d699',
    u'\\mathtt{q}': u'\U0001d69a',
    u'\\mathtt{r}': u'\U0001d69b',
    u'\\mathtt{s}': u'\U0001d69c',
    u'\\mathtt{t}': u'\U0001d69d',
    u'\\mathtt{u}': u'\U0001d69e',
    u'\\mathtt{v}': u'\U0001d69f',
    u'\\mathtt{w}': u'\U0001d6a0',
    u'\\mathtt{x}': u'\U0001d6a1',
    u'\\mathtt{y}': u'\U0001d6a2',
    u'\\mathtt{z}': u'\U0001d6a3',
    u'\\mbox{\\texteuro}': u'\u20ac',
    u'\\measuredangle': u'\u2221',
    u'\\mercury': u'\u263f',
    u'\\mho': u'\u2127',
    u'\\mid': u'\u2223',
    u'\\mkern1mu': u'\u200a',
    u'\\mkern4mu': u'\u205f',
    u'\\mp': u'\u2213',
    u'\\mu': u'\u03bc',
    u'\\multimap': u'\u22b8',
    u'\\nLeftarrow': u'\u21cd',
    u'\\nLeftrightarrow': u'\u21ce',
    u'\\nRightarrow': u'\u21cf',
    u'\\nVDash': u'\u22af',
    u'\\nVdash': u'\u22ae',
    u'\\nabla': u'\u2207',
    u'\\natural': u'\u266e',
    u'\\nearrow': u'\u2197',
    u'\\neptune': u'\u2646',
    u'\\nexists': u'\u2204',
    u'\\ng': u'\u014b',
    u'\\ngeqslant': u'\u2a7e\u0338',
    u'\\ni': u'\u220b',
    u'\\nleftarrow': u'\u219a',
    u'\\nleftrightarrow': u'\u21ae',
    u'\\nleqslant': u'\u2a7d\u0338',
    u'\\nmid': u'\u2224',
    u'\\nolinebreak': u'\u2060',
    u'\\not =': u'\u2260',
    u'\\not<': u'\u226e',
    u'\\not>': u'\u226f',
    u'\\not\\apid': u'\u224b\u0338',
    u'\\not\\approx': u'\u2249',
    u'\\not\\cong': u'\u2247',
    u'\\not\\doteq': u'\u2250\u0338',
    u'\\not\\equiv': u'\u2262',
    u'\\not\\geq': u'\u2271',
    u'\\not\\in': u'\u2209',
    u'\\not\\kern-0.3em\\times': u'\u226d',
    u'\\not\\leq': u'\u2270',
    u'\\not\\ni': u'\u220c',
    u'\\not\\prec': u'\u2280',
    u'\\not\\preceq': u'\u2aaf\u0338',
    u'\\not\\sim': u'\u2241',
    u'\\not\\simeq': u'\u2244',
    u'\\not\\sqsubseteq': u'\u22e2',
    u'\\not\\sqsupseteq': u'\u22e3',
    u'\\not\\subset': u'\u2284',
    u'\\not\\subseteq': u'\u2288',
    u'\\not\\succ': u'\u2281',
    u'\\not\\succeq': u'\u2ab0\u0338',
    u'\\not\\supset': u'\u2285',
    u'\\not\\supseteq': u'\u2289',
    u'\\notgreaterless': u'\u2279',
    u'\\notlessgreater': u'\u2278',
    u'\\nparallel': u'\u2226',
    u'\\nrightarrow': u'\u219b',
    u'\\nsubseteqq': u'\u2ac5\u0338',
    u'\\nsupseteqq': u'\u2ac6\u0338',
    u'\\ntriangleleft': u'\u22ea',
    u'\\ntrianglelefteq': u'\u22ec',
    u'\\ntriangleright': u'\u22eb',
    u'\\ntrianglerighteq': u'\u22ed',
    u'\\nu': u'\u03bd',
    u'\\nvDash': u'\u22ad',
    u'\\nvdash': u'\u22ac',
    u'\\nwarrow': u'\u2196',
    u'\\o': u'\xf8',
    u'\\odot': u'\u2299',
    u'\\oe': u'\u0153',
    u'\\oint': u'\u222e',
    u'\\omega': u'\u03c9',
    u'\\ominus': u'\u2296',
    u'\\openbracketleft': u'\u301a',
    u'\\openbracketright': u'\u301b',
    u'\\oplus': u'\u2295',
    u'\\original': u'\u22b6',
    u'\\oslash': u'\u2298',
    u'\\otimes': u'\u2297',
    u'\\parallel': u'\u2225',
    u'\\partial': u'\U0001d7c3',
    u'\\perp': u'\u22a5',
    u'\\perspcorrespond': u'\u2a5e',
    u'\\phi': u'\u03d5',
    u'\\pi': u'\u03c0',
    u'\\pisces': u'\u2653',
    u'\\pitchfork': u'\u22d4',
    u'\\pluto': u'\u2647',
    u'\\pm': u'\xb1',
    u'\\prec': u'\u227a',
    u'\\precapprox': u'\u2ab7',
    u'\\preccurlyeq': u'\u227c',
    u'\\precedesnotsimilar': u'\u22e8',
    u'\\preceq': u'\u2aaf',
    u'\\precnapprox': u'\u2ab9',
    u'\\precneqq': u'\u2ab5',
    u'\\prod': u'\u220f',
    u'\\propto': u'\u221d',
    u'\\psi': u'\u03c8',
    u'\\quarternote': u'\u2669',
    u'\\r': u'\u030a',
    u'\\rangle': u'\u232a',
    u'\\rbrace': u'}',
    u'\\rceil': u'\u2309',
    u'\\recorder': u'\u2315',
    u'\\rfloor': u'\u230b',
    u'\\rho': u'\u03c1',
    u'\\rightangle': u'\u221f',
    u'\\rightanglearc': u'\u22be',
    u'\\rightarrow': u'\u2192',
    u'\\rightarrowtail': u'\u21a3',
    u'\\rightharpoondown': u'\u21c1',
    u'\\rightharpoonup': u'\u21c0',
    u'\\rightleftarrows': u'\u21c4',
    u'\\rightleftharpoons': u'\u21cc',
    u'\\rightmoon': u'\u263e',
    u'\\rightrightarrows': u'\u21c9',
    u'\\rightsquigarrow': u'\u21dd',
    u'\\rightthreetimes': u'\u22cc',
    u'\\risingdotseq': u'\u2253',
    u'\\rmoustache': u'\u23b1',
    u'\\rtimes': u'\u22ca',
    u'\\rule{1em}{1pt}': u'\u2015',
    u'\\r{U}': u'\u016e',
    u'\\r{u}': u'\u016f',
    u'\\r{}': u'\u02da',
    u'\\sagittarius': u'\u2650',
    u'\\saturn': u'\u2644',
    u'\\scorpio': u'\u264f',
    u'\\searrow': u'\u2198',
    u'\\setminus': u'\u2216',
    u'\\sharp': u'\u266f',
    u'\\sigma': u'\u03c3',
    u'\\sim': u'\u223c',
    u'\\sim\\joinrel\\leadsto': u'\u27ff',
    u'\\simeq': u'\u2243',
    u'\\smile': u'\u2323',
    u'\\space': u' ',
    u'\\sphericalangle': u'\u2222',
    u'\\sqcap': u'\u2293',
    u'\\sqcup': u'\u2294',
    u'\\sqrint': u'\u2a16',
    u'\\sqsubset': u'\u228f',
    u'\\sqsubseteq': u'\u2291',
    u'\\sqsupset': u'\u2290',
    u'\\sqsupseteq': u'\u2292',
    u'\\square': u'\u25a1',
    u'\\ss': u'\xdf',
    u'\\stackrel{*}{=}': u'\u2a6e',
    u'\\star': u'\u22c6',
    u'\\starequal': u'\u225b',
    u'\\subset': u'\u2282',
    u'\\subseteq': u'\u2286',
    u'\\subseteqq': u'\u2ac5',
    u'\\subsetneq': u'\u228a',
    u'\\subsetneqq': u'\u2acb',
    u'\\succ': u'\u227b',
    u'\\succapprox': u'\u2ab8',
    u'\\succcurlyeq': u'\u227d',
    u'\\succeq': u'\u2ab0',
    u'\\succnapprox': u'\u2aba',
    u'\\succneqq': u'\u2ab6',
    u'\\succnsim': u'\u22e9',
    u'\\sum': u'\u2211',
    u'\\supset': u'\u2283',
    u'\\supseteq': u'\u2287',
    u'\\supseteqq': u'\u2ac6',
    u'\\supsetneq': u'\u228b',
    u'\\supsetneqq': u'\u2acc',
    u'\\surd': u'\u221a',
    u'\\surfintegral': u'\u222f',
    u'\\swarrow': u'\u2199',
    u'\\tau': u'\u03c4',
    u'\\taurus': u'\u2649',
    u'\\textTheta': u'\u03f4',
    u'\\textasciiacute': u'\xb4',
    u'\\textasciibreve': u'\u02d8',
    u'\\textasciicaron': u'\u02c7',
    u'\\textasciidieresis': u'\xa8',
    u'\\textasciigrave': u'`',
    u'\\textasciimacron': u'\xaf',
    u'\\textasciitilde': u'~',
    u'\\textbackslash': u'\\',
    u'\\textbrokenbar': u'\xa6',
    u'\\textbullet': u'\u2022',
    u'\\textcent': u'\xa2',
    u'\\textcopyright': u'\xa9',
    u'\\textcurrency': u'\xa4',
    u'\\textdagger': u'\u2020',
    u'\\textdaggerdbl': u'\u2021',
    u'\\textdegree': u'\xb0',
    u'\\textdollar': u'$',
    u'\\textdoublepipe': u'\u01c2',
    u'\\textemdash': u'\u2014',
    u'\\textendash': u'\u2013',
    u'\\textexclamdown': u'\xa1',
    u'\\textfrac{1}{3}': u'\u2153',
    u'\\textfrac{1}{5}': u'\u2155',
    u'\\textfrac{1}{6}': u'\u2159',
    u'\\textfrac{1}{8}': u'\u215b',
    u'\\textfrac{2}{3}': u'\u2154',
    u'\\textfrac{2}{5}': u'\u2156',
    u'\\textfrac{3}{5}': u'\u2157',
    u'\\textfrac{3}{8}': u'\u215c',
    u'\\textfrac{4}{5}': u'\u2158',
    u'\\textfrac{5}{6}': u'\u215a',
    u'\\textfrac{5}{8}': u'\u215d',
    u'\\textfrac{7}{8}': u'\u215e',
    u'\\texthvlig': u'\u0195',
    u'\\textnrleg': u'\u019e',
    u'\\textonehalf': u'\xbd',
    u'\\textonequarter': u'\xbc',
    u'\\textordfeminine': u'\xaa',
    u'\\textordmasculine': u'\xba',
    u'\\textparagraph': u'\xb6',
    u'\\textperiodcentered': u'\u02d9',
    u'\\textpertenthousand': u'\u2031',
    u'\\textperthousand': u'\u2030',
    u'\\textphi': u'\u0278',
    u'\\textquestiondown': u'\xbf',
    u'\\textquotedblleft': u'\u201c',
    u'\\textquotedblright': u'\u201d',
    u'\\textquotesingle': u"'",
    u'\\textregistered': u'\xae',
    u'\\textsection': u'\xa7',
    u'\\textsterling': u'\xa3',
    u'\\texttheta': u'\u03b8',
    u'\\textthreequarters': u'\xbe',
    u'\\texttildelow': u'\u02dc',
    u'\\texttimes': u'\xd7',
    u'\\texttrademark': u'\u2122',
    u'\\textturnk': u'\u029e',
    u'\\textvartheta': u'\u03d1',
    u'\\textvisiblespace': u'\u2423',
    u'\\textyen': u'\xa5',
    u'\\th': u'\xfe',
    u'\\therefore': u'\u2234',
    u'\\tildetrpl': u'\u224b',
    u'\\tone{11}': u'\u02e9',
    u'\\tone{22}': u'\u02e8',
    u'\\tone{33}': u'\u02e7',
    u'\\tone{44}': u'\u02e6',
    u'\\tone{55}': u'\u02e5',
    u'\\top': u'\u22a4',
    u'\\triangledown': u'\u25bf',
    u'\\triangleleft': u'\u25c3',
    u'\\trianglelefteq': u'\u22b4',
    u'\\triangleq': u'\u225c',
    u'\\triangleright': u'\u25b9',
    u'\\trianglerighteq': u'\u22b5',
    u'\\truestate': u'\u22a7',
    u'\\twoheadleftarrow': u'\u219e',
    u'\\twoheadrightarrow': u'\u21a0',
    u'\\u': u'\u0306',
    u'\\ulcorner': u'\u231c',
    u'\\uparrow': u'\u2191',
    u'\\updownarrow': u'\u2195',
    u'\\upharpoonleft': u'\u21bf',
    u'\\upharpoonright': u'\u21be',
    u'\\uplus': u'\u228e',
    u'\\upsilon': u'\u03c5',
    u'\\upslopeellipsis': u'\u22f0',
    u'\\upuparrows': u'\u21c8',
    u'\\uranus': u'\u2645',
    u'\\urcorner': u'\u231d',
    u'\\u{A}': u'\u0102',
    u'\\u{E}': u'\u0114',
    u'\\u{G}': u'\u011e',
    u'\\u{I}': u'\u012c',
    u'\\u{O}': u'\u014e',
    u'\\u{U}': u'\u016c',
    u'\\u{\\i}': u'\u012d',
    u'\\u{a}': u'\u0103',
    u'\\u{e}': u'\u0115',
    u'\\u{g}': u'\u011f',
    u'\\u{o}': u'\u014f',
    u'\\u{u}': u'\u016d',
    u'\\v': u'\u030c',
    u'\\varepsilon': u'\u025b',
    u'\\varkappa': u'\u03f0',
    u'\\varnothing': u'\u2205',
    u'\\varphi': u'\u03c6',
    u'\\varpi': u'\u03d6',
    u'\\varrho': u'\u03f1',
    u'\\varsigma': u'\u03c2',
    u'\\varsubsetneqq': u'\u228a\ufe00',
    u'\\varsupsetneq': u'\u228b\ufe00',
    u'\\vartriangle': u'\u25b5',
    u'\\vartriangleleft': u'\u22b2',
    u'\\vartriangleright': u'\u22b3',
    u'\\vdash': u'\u22a2',
    u'\\vdots': u'\u22ee',
    u'\\vee': u'\u2228',
    u'\\veebar': u'\u22bb',
    u'\\venus': u'\u2640',
    u'\\vert': u'|',
    u'\\verymuchgreater': u'\u22d9',
    u'\\verymuchless': u'\u22d8',
    u'\\virgo': u'\u264d',
    u'\\volintegral': u'\u2230',
    u'\\v{C}': u'\u010c',
    u'\\v{D}': u'\u010e',
    u'\\v{E}': u'\u011a',
    u'\\v{L}': u'\u013d',
    u'\\v{N}': u'\u0147',
    u'\\v{R}': u'\u0158',
    u'\\v{S}': u'\u0160',
    u'\\v{T}': u'\u0164',
    u'\\v{Z}': u'\u017d',
    u'\\v{c}': u'\u010d',
    u'\\v{d}': u'\u010f',
    u'\\v{e}': u'\u011b',
    u'\\v{l}': u'\u013e',
    u'\\v{n}': u'\u0148',
    u'\\v{r}': u'\u0159',
    u'\\v{s}': u'\u0161',
    u'\\v{t}': u'\u0165',
    u'\\v{z}': u'\u017e',
    u'\\wedge': u'\u2227',
    u'\\wp': u'\u2118',
    u'\\wr': u'\u2240',
    u'\\xi': u'\u03be',
    u'\\zeta': u'\u03b6',
    u'\\~': u'\u0303',
    u'\\~{A}': u'\xc3',
    u'\\~{I}': u'\u0128',
    u'\\~{N}': u'\xd1',
    u'\\~{O}': u'\xd5',
    u'\\~{U}': u'\u0168',
    u'\\~{\\i}': u'\u0129',
    u'\\~{a}': u'\xe3',
    u'\\~{n}': u'\xf1',
    u'\\~{o}': u'\xf5',
    u'\\~{u}': u'\u0169',
    u'`': u'\u2018',
    u'f': u'\u0192',
    u'ff': u'\ufb00',
    u'ffi': u'\ufb03',
    u'ffl': u'\ufb04',
    u'fi': u'\ufb01',
    u'fl': u'\ufb02',
    u'g': u'\u0261',
    u'ij': u'\u0133',
    u'o': u'\u03bf',
    u"{'''}": u'\u2034',
    u"{''}": u'\u2033',
    u"{'}": u'\u2032',
    u'{\\fontencoding{LECO}\\selectfont\\char177}': u'\u0311',
    u'{\\fontencoding{LECO}\\selectfont\\char184}': u'\u0318',
    u'{\\fontencoding{LECO}\\selectfont\\char185}': u'\u0319',
    u'{\\fontencoding{LECO}\\selectfont\\char203}': u'\u032b',
    u'{\\fontencoding{LECO}\\selectfont\\char207}': u'\u032f',
    u'{\\fontencoding{LECO}\\selectfont\\char215}': u'\u0337',
    u'{\\fontencoding{LECO}\\selectfont\\char216}': u'\u0338',
    u'{\\fontencoding{LECO}\\selectfont\\char218}': u'\u033a',
    u'{\\fontencoding{LECO}\\selectfont\\char219}': u'\u033b',
    u'{\\fontencoding{LECO}\\selectfont\\char220}': u'\u033c',
    u'{\\fontencoding{LECO}\\selectfont\\char221}': u'\u033d',
    u'{\\fontencoding{LECO}\\selectfont\\char225}': u'\u0361',
    u'{\\fontencoding{LEIP}\\selectfont\\char202}': u'\u027f',
    u'{\\fontencoding{LEIP}\\selectfont\\char61}': u'\u0258',
    u'{\\fontencoding{LELA}\\selectfont\\char195}': u'\u01ba',
    u'{\\fontencoding{LELA}\\selectfont\\char201}': u'\u013f',
    u'{\\fontencoding{LELA}\\selectfont\\char202}': u'\u0140',
    u'{\\fontencoding{LELA}\\selectfont\\char40}': u'\u0126',
    u'{\\fontencoding{LELA}\\selectfont\\char47}': u'\u0166',
    u'{\\fontencoding{LELA}\\selectfont\\char63}': u'\u0167',
    u'{\\fontencoding{LELA}\\selectfont\\char91}': u'\u0138',
    u'{\\rlap{\\textbackslash}{{/}\\!\\!{/}}}': u'\u2afd\u20e5',
    u'{^1}': u'\xb9',
    u'{^2}': u'\xb2',
    u'{^3}': u'\xb3',
    u'{_\\ast}': u'\u2217',
    u'{{/}\\!\\!{/}}': u'\u2afd',
    u'~': u'\xa0'}


def unicode_to_tex(s):
    '''
    Convert unicode string s to its' corressponding tex equation.
    '''
    return u''.join([unicode_to_tex_map.get(i, i) for i in s])


def tex_to_unicode(s):
    '''
    Convert latex equation to its' corresponding unicode string,
    the function will not deal with the logic of latex command.
    '''
    return u''.join([unicode_to_tex_map.get(i, i) for i in s])
