package unimr.vod.hmac;

/*
 * RED5 Open Source Flash Server - http://www.osflash.org/red5
 * 
 * Copyright (c) 2006-2008 by respective authors (see below). All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or modify it under the 
 * terms of the GNU Lesser General Public License as published by the Free Software 
 * Foundation; either version 2.1 of the License, or (at your option) any later 
 * version. 
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE. See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along 
 * with this library; if not, write to the Free Software Foundation, Inc., 
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 
 */

import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.io.UnsupportedEncodingException;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;

import org.red5.server.api.IScope;
import org.red5.server.api.Red5;
import org.red5.server.api.IConnection;
import org.red5.server.api.stream.IStreamPlaybackSecurity;
import org.red5.logging.Red5LoggerFactory;

import org.slf4j.Logger;


/**
 * @author Andreas Gabriel (gabriel@hrz.uni-marburg.de)
 * @author David Nax (naxd@hrz.uni-marburg.de)
 */

public class PlaybackSecurityHmac implements IStreamPlaybackSecurity {

	public static String BEAN_NAME = "streamPlaybackSecurity";

	/** value of sharedSecret can be set in red5-web.properties **/
	private static String sharedSecret = "";

	/** provider keyed-hashing algorithm; see RFC 2104 **/
	private static String provider = "HmacMD5";

	/** string keys evaluated from path by the getSignature method **/
	private static String signKeys[] = {"path","name","ip","expires"};

	protected Logger log = Red5LoggerFactory.getLogger( PlaybackSecurityHmac.class , "protectedVOD");

	public boolean isPlaybackAllowed(IScope scope, String name, int start, int length, boolean flushPlaylist) {


		// get the signature map
		Map<String, String> signMap = getSignMap(scope,name);

		if(!signMap.containsKey("signature")){
			log.info( "isPlaybackAllowed - false: remote signature empty");
			return false;
		}

		String localSignature = getSignature(signMap);

		log.debug("isPlaybackAllowed - localSignature \"{}\"" , new Object[]{localSignature});

		if(localSignature.length() == 0){
			log.info( "isPlaybackAllowed - false: local signature empty");
			return false;
		}

		String remoteSignature = signMap.get("signature");

		log.debug("isPlaybackAllowed - remoteSignature \"{}\"" , new Object[]{remoteSignature});

		// check signature
		if(!localSignature.contentEquals(remoteSignature)){
			log.info( "isPlaybackAllowed - false: remote signature wrong");
			return false;
		}

		// check TTL
		long expires = 0;
		try {
			/*expires = Long.valueOf(signMap.get("expires")).longValue();*/
			expires = Long.parseLong(signMap.get("expires"),16);
			
		}catch(NumberFormatException e){
			log.error("isPlaybackAllowed - expire date has wrong format or is empty");
			return false;
		}

		long now = (new Date()).getTime() / 1000;

		if (now > expires){
			log.info( "isPlaybackAllowed - false: remote signature stale (now: " + now + " expires: "+expires+")");
			return false;
		}

		log.debug( "isPlaybackAllowed - true");
		return true;
	}

	public void setSharedSecret( String value ){
		sharedSecret = value;
	}


	public void setProvider( String value ){
		provider = value;
	}

	private String getSignature(Map<String, String> signMap){

		Mac hmac;

		try {
			// Generate a key for the provider keyed-hashing algorithm; see RFC 2104
			hmac = Mac.getInstance(provider);

		} catch (NoSuchAlgorithmException e) {

			log.error( "getSignature - " + provider + " not supported");
			return "";
		}

		try {

			if(sharedSecret.length() == 0){
				log.error( "getSignature - sharedSecret empty");
				return "";				
			}
			hmac.init(new SecretKeySpec(sharedSecret.getBytes("UTF-8"),provider));
			for (String key: signKeys) {
				if(!signMap.containsKey(key)){
					log.debug("getSignature - signMap misses key: " + key);
					return "";
				}
				hmac.update(signMap.get(key).getBytes("UTF-8"));
			}
		} catch (InvalidKeyException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		} catch (UnsupportedEncodingException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		// return the hex digest
		return (byteArray2Hex(hmac.doFinal()));

	}

	private Map<String, String> getSignMap(IScope scope, String name){

		Map<String, String> signMap = new HashMap<String, String>();
		IConnection conn = Red5.getConnectionLocal();

		/**
		IStreamFilenameGenerator filenameGenerator = (IStreamFilenameGenerator) ScopeUtils.getScopeService(scope, IStreamFilenameGenerator.class);
        String filename = filenameGenerator.generateFilename(scope, name, GenerationType.PLAYBACK);
		 **/

		// client IP
		String ip = (String) conn.getRemoteAddress();
		signMap.put("ip", ip);

		// virtual path: context/signature/expires
		String virtualPath =  (String)  scope.getContextPath();
		final StringBuilder realPath = new StringBuilder();

		// extract real path, signature and expire date from virtual path
		String[] params = virtualPath.split("/");

		for (int i = 0; i < params.length; i++){
			if (i < params.length -2 ) {
				realPath.append(params[i]+"/");	
			}

			if ( i == params.length -2 ){        		
				signMap.put("signature", params[i]);
			}

			if ( i == params.length -1 ){
				signMap.put("expires", params[i]);
			}

		}

		// strip beginning slash from name
		if(name.startsWith("/")){
			name = name.substring(1);
		}

		signMap.put("path", realPath.toString());
		signMap.put("name", name);

		log.info( "getSignMap - " + signMap );

		return signMap;
	}

	private static char[] hex = {
		'0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'a', 'b', 'c', 'd', 'e', 'f'
	};

	private static String byteArray2Hex(byte[] b){
		StringBuffer sb = new StringBuffer();
		for (int i = 0; i < b.length; i++){
			int hbits = (b[i] & 0x000000f0) >> 4;
			int lbits = b[i] & 0x0000000f;
			sb.append("" + hex[hbits] + hex[lbits]);
		}
		return sb.toString();
	}

}