#!/usr/bin/env python
# -*- coding: utf-8 -*-

# Copyright 2013-2014, Nigel Small
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.


"""
An implementation of URIs from RFC 3986 (URI Generic Syntax).
"""


from __future__ import unicode_literals

import re


__all__ = ["general_delimiters", "subcomponent_delimiters",
           "reserved", "unreserved", "percent_encode", "percent_decode",
           "Authority", "Path", "Query", "URI"]


# RFC 3986 § 2.2.
general_delimiters = ":/?#[]@"
subcomponent_delimiters = "!$&'()*+,;="
reserved = general_delimiters + subcomponent_delimiters

# RFC 3986 § 2.3.
unreserved = ("ABCDEFGHIJKLMNOPQRSTUVWXYZ"
              "abcdefghijklmnopqrstuvwxyz"
              "0123456789-._~")


def percent_encode(data, safe=None):
    """ Percent encode a string of data, optionally keeping certain characters
    unencoded.

    """
    if data is None:
        return None
    if isinstance(data, (tuple, list, set)):
        return "&".join(
            percent_encode(value, safe)
            for value in data
        )
    if isinstance(data, dict):
        return "&".join(
            key + "=" + percent_encode(value, safe)
            for key, value in data.items()
        )
    if not safe:
        safe = ""
    try:
        chars = list(data)
    except TypeError:
        chars = list(str(data))
    for i, char in enumerate(chars):
        if char == "%" or (char not in unreserved and char not in safe):
            chars[i] = "".join("%" + hex(b)[2:].upper().zfill(2)
                               for b in bytearray(char, "utf-8"))
    return "".join(chars)


def percent_decode(data):
    """ Percent decode a string of data.

    """
    if data is None:
        return None
    percent_code = re.compile("(%[0-9A-Fa-f]{2})")
    try:
        bits = percent_code.split(data)
    except TypeError:
        bits = percent_code.split(str(data))
    out = bytearray()
    for bit in bits:
        if bit.startswith("%"):
            out.extend(bytearray([int(bit[1:], 16)]))
        else:
            out.extend(bytearray(bit, "utf-8"))
    return out.decode("utf-8")


class _Part(object):
    """ Internal base class for all URI component parts.
    """

    def __init__(self):
        pass

    def __repr__(self):
        return "{0}({1})".format(self.__class__.__name__, repr(self.string))

    def __str__(self):
        return self.string or ""

    def __bool__(self):
        return bool(self.string)

    def __nonzero__(self):
        return bool(self.string)

    def __len__(self):
        return len(str(self))

    def __iter__(self):
        return iter(self.string)

    @property
    def string(self):
        raise NotImplementedError()


class Authority(_Part):
    """ A host name plus optional port and user information detail.

    **Syntax**
        ``authority := [ user_info "@" ] host [ ":" port ]``

    .. seealso::
        `RFC 3986 § 3.2`_

    .. _`RFC 3986 § 3.2`: http://tools.ietf.org/html/rfc3986#section-3.2
    """

    @classmethod
    def __cast(cls, obj):
        if obj is None:
            return cls(None)
        elif isinstance(obj, cls):
            return obj
        else:
            return cls(str(obj))

    def __init__(self, string):
        super(Authority, self).__init__()
        if string is None:
            self.__user_info = None
            self.__host = None
            self.__port = None
        else:
            if "@" in string:
                self.__user_info, string = string.rpartition("@")[0::2]
                self.__user_info = percent_decode(self.__user_info)
            else:
                self.__user_info = None
            if ":" in string:
                self.__host, self.__port = string.rpartition(":")[0::2]
                self.__port = int(self.__port)
            else:
                self.__host = string
                self.__port = None

    def __eq__(self, other):
        other = self.__cast(other)
        return (self.__user_info == other.__user_info and
                self.__host == other.__host and
                self.__port == other.__port)

    def __ne__(self, other):
        other = self.__cast(other)
        return (self.__user_info != other.__user_info or
                self.__host != other.__host or
                self.__port != other.__port)

    def __hash__(self):
        return hash(self.string)

    @property
    def host(self):
        """ The host part of this authority component, an empty string if host
        is empty or :py:const:`None` if undefined.

        ::

            >>> Authority(None).host
            None
            >>> Authority("").host
            ''
            >>> Authority("example.com").host
            'example.com'
            >>> Authority("example.com:8080").host
            'example.com'
            >>> Authority("bob@example.com").host
            'example.com'
            >>> Authority("bob@example.com:8080").host
            'example.com'

        :return:
        """
        return self.__host

    @property
    def host_port(self):
        """ The host and port parts of this authority component or
        :py:const:`None` if undefined.

        ::

            >>> Authority(None).host_port
            None
            >>> Authority("").host_port
            ''
            >>> Authority("example.com").host_port
            'example.com'
            >>> Authority("example.com:8080").host_port
            'example.com:8080'
            >>> Authority("bob@example.com").host_port
            'example.com'
            >>> Authority("bob@example.com:8080").host_port
            'example.com:8080'

        :return:
        """
        u = [self.__host]
        if self.__port is not None:
            u += [":", str(self.__port)]
        return "".join(u)

    @property
    def port(self):
        """ The port part of this authority component or :py:const:`None` if
        undefined.

        ::

            >>> Authority(None).port
            None
            >>> Authority("").port
            None
            >>> Authority("example.com").port
            None
            >>> Authority("example.com:8080").port
            8080
            >>> Authority("bob@example.com").port
            None
            >>> Authority("bob@example.com:8080").port
            8080

        :return:
        """
        return self.__port

    @property
    def string(self):
        """ The full string value of this authority component or
        :`py:const:`None` if undefined.

        ::

            >>> Authority(None).string
            None
            >>> Authority("").string
            ''
            >>> Authority("example.com").string
            'example.com'
            >>> Authority("example.com:8080").string
            'example.com:8080'
            >>> Authority("bob@example.com").string
            'bob@example.com'
            >>> Authority("bob@example.com:8080").string
            'bob@example.com:8080'

        :return:
        """
        if self.__host is None:
            return None
        u = []
        if self.__user_info is not None:
            u += [percent_encode(self.__user_info), "@"]
        u += [self.__host]
        if self.__port is not None:
            u += [":", str(self.__port)]
        return "".join(u)
        
    @property
    def user_info(self):
        """ The user information part of this authority component or
        :py:const:`None` if undefined.

        ::

            >>> Authority(None).user_info
            None
            >>> Authority("").user_info
            None
            >>> Authority("example.com").user_info
            None
            >>> Authority("example.com:8080").user_info
            None
            >>> Authority("bob@example.com").user_info
            'bob'
            >>> Authority("bob@example.com:8080").user_info
            'bob'

        :return:
        """
        return self.__user_info


class Path(_Part):

    @classmethod
    def __cast(cls, obj):
        if obj is None:
            return cls(None)
        elif isinstance(obj, cls):
            return obj
        else:
            return cls(str(obj))

    def __init__(self, string):
        super(Path, self).__init__()
        if string is None:
            self.__segments = None
        else:
            self.__segments = list(map(percent_decode, string.split("/")))

    def __eq__(self, other):
        other = self.__cast(other)
        return self.__segments == other.__segments

    def __ne__(self, other):
        other = self.__cast(other)
        return self.__segments != other.__segments

    def __hash__(self):
        return hash(self.string)

    @property
    def string(self):
        if self.__segments is None:
            return None
        return "/".join(map(percent_encode, self.__segments))

    @property
    def segments(self):
        if self.__segments is None:
            return []
        else:
            return list(self.__segments)

    def __iter__(self):
        return iter(self.__segments)

    def remove_dot_segments(self):
        """ Implementation of RFC3986, section 5.2.4
        """
        inp = self.string
        out = ""
        while inp:
            if inp.startswith("../"):
                inp = inp[3:]
            elif inp.startswith("./"):
                inp = inp[2:]
            elif inp.startswith("/./"):
                inp = inp[2:]
            elif inp == "/.":
                inp = "/"
            elif inp.startswith("/../"):
                inp = inp[3:]
                out = out.rpartition("/")[0]
            elif inp == "/..":
                inp = "/"
                out = out.rpartition("/")[0]
            elif inp in (".", ".."):
                inp = ""
            else:
                if inp.startswith("/"):
                    inp = inp[1:]
                    out += "/"
                seg, slash, inp = inp.partition("/")
                out += seg
                inp = slash + inp
        return Path(out)

    def with_trailing_slash(self):
        if self.__segments is None:
            return self
        s = self.string
        if s.endswith("/"):
            return self
        else:
            return Path(s + "/")

    def without_trailing_slash(self):
        if self.__segments is None:
            return self
        s = self.string
        if s.endswith("/"):
            return Path(s[:-1])
        else:
            return self


class Query(_Part):

    @classmethod
    def __cast(cls, obj):
        if obj is None:
            return cls(None)
        elif isinstance(obj, cls):
            return obj
        else:
            return cls(str(obj))

    @classmethod
    def encode(cls, iterable):
        if iterable is None:
            return None
        bits = []
        if isinstance(iterable, dict):
            for key, value in iterable.items():
                if value is None:
                    bits.append(percent_encode(key))
                else:
                    bits.append(percent_encode(key) + "=" + percent_encode(value))
        else:
            for item in iterable:
                if isinstance(item, tuple) and len(item) == 2:
                    key, value = item
                    if value is None:
                        bits.append(percent_encode(key))
                    else:
                        bits.append(percent_encode(key) + "=" + percent_encode(value))
                else:
                    bits.append(percent_encode(item))
        return "&".join(bits)

    @classmethod
    def decode(cls, string):
        if string is None:
            return None
        data = []
        if string:
            bits = string.split("&")
            for bit in bits:
                if "=" in bit:
                    key, value = map(percent_decode, bit.partition("=")[0::2])
                else:
                    key, value = percent_decode(bit), None
                data.append((key, value))
        return data

    def __init__(self, string):
        super(Query, self).__init__()
        self.__query = Query.decode(string)
        if self.__query is None:
            self.__query_dict = None
        else:
            self.__query_dict = dict(self.__query)

    def __eq__(self, other):
        other = self.__cast(other)
        return self.__query == other.__query

    def __ne__(self, other):
        other = self.__cast(other)
        return self.__query != other.__query

    def __hash__(self):
        return hash(self.string)

    @property
    def string(self):
        return Query.encode(self.__query)

    def __iter__(self):
        if self.__query is None:
            return iter(())
        else:
            return iter(self.__query)

    def __getitem__(self, key):
        if self.__query_dict is None:
            raise KeyError(key)
        else:
            return self.__query_dict[key]


class URI(_Part):
    """ Uniform Resource Identifier.

    .. seealso::
        `RFC 3986`_

    .. _`RFC 3986`: http://tools.ietf.org/html/rfc3986
    """

    @classmethod
    def __cast(cls, obj):
        if obj is None:
            return cls(None)
        elif isinstance(obj, cls):
            return obj
        else:
            return cls(str(obj))

    def __init__(self, value):
        super(URI, self).__init__()
        try:
            if value.__uri__ is None:
                self.__scheme = None
                self.__authority = None
                self.__path = None
                self.__query = None
                self.__fragment = None
                return
        except AttributeError:
            pass
        if value is None:
            self.__scheme = None
            self.__authority = None
            self.__path = None
            self.__query = None
            self.__fragment = None
        else:
            try:
                value = str(value.__uri__)
            except AttributeError:
                value = str(value)
            # scheme
            if ":" in value:
                self.__scheme, value = value.partition(":")[0::2]
                self.__scheme = percent_decode(self.__scheme)
            else:
                self.__scheme = None
            # fragment
            if "#" in value:
                value, self.__fragment = value.partition("#")[0::2]
                self.__fragment = percent_decode(self.__fragment)
            else:
                self.__fragment = None
            # query
            if "?" in value:
                value, self.__query = value.partition("?")[0::2]
                self.__query = Query(self.__query)
            else:
                self.__query = None
            # hierarchical part
            if value.startswith("//"):
                value = value[2:]
                slash = value.find("/")
                if slash >= 0:
                    self.__authority = Authority(value[:slash])
                    self.__path = Path(value[slash:])
                else:
                    self.__authority = Authority(value)
                    self.__path = Path("")
            else:
                self.__authority = None
                self.__path = Path(value)

    def __eq__(self, other):
        other = self.__cast(other)
        return (self.__scheme == other.__scheme and
                self.__authority == other.__authority and
                self.__path == other.__path and
                self.__query == other.__query and
                self.__fragment == other.__fragment)

    def __ne__(self, other):
        other = self.__cast(other)
        return (self.__scheme != other.__scheme or
                self.__authority != other.__authority or
                self.__path != other.__path or
                self.__query != other.__query or
                self.__fragment != other.__fragment)

    def __hash__(self):
        return hash(self.string)

    @property
    def __uri__(self):
        return self.string

    @property
    def string(self):
        """ The full percent-encoded string value of this URI or
        :py:const:`None` if undefined.

        ::

            >>> URI(None).string
            None
            >>> URI("").string
            ''
            >>> URI("http://example.com").string
            'example.com'
            >>> URI("foo/bar").string
            'foo/bar'
            >>> URI("http://bob@example.com:8080/data/report.html?date=2000-12-25#summary").string
            'http://bob@example.com:8080/data/report.html?date=2000-12-25#summary'

        **Component Definition:**
        ::

            https://bob@example.com:8080/data/report.html?date=2000-12-25#summary
            \___________________________________________________________________/
                                             |
                                           string

        :rtype: percent-encoded string or :py:const:`None`

        .. note::
            Unlike ``string``, the ``__str__`` method will always return a
            string, even when the URI is undefined; in this case, an empty
            string is returned instead of :py:const:`None`.
        """
        if self.__path is None:
            return None
        u = []
        if self.__scheme is not None:
            u += [percent_encode(self.__scheme), ":"]
        if self.__authority is not None:
            u += ["//", str(self.__authority)]
        u += [str(self.__path)]
        if self.__query is not None:
            u += ["?", str(self.__query)]
        if self.__fragment is not None:
            u += ["#", percent_encode(self.__fragment)]
        return "".join(u)

    @property
    def scheme(self):
        """ The scheme part of this URI or :py:const:`None` if undefined.

        **Component Definition:**
        ::

            https://bob@example.com:8080/data/report.html?date=2000-12-25#summary
            \___/
              |
            scheme

        :rtype: unencoded string or :py:const:`None`
        """
        return self.__scheme

    @property
    def authority(self):
        """ The authority part of this URI or :py:const:`None` if undefined.

        **Component Definition:**
        ::

            https://bob@example.com:8080/data/report.html?date=2000-12-25#summary
                    \__________________/
                             |
                         authority

        :rtype: :py:class:`Authority <httpstream.uri.Authority>` instance or
            :py:const:`None`
        """
        return self.__authority

    @property
    def user_info(self):
        """ The user information part of this URI or :py:const:`None` if
        undefined.

        **Component Definition:**
        ::

            https://bob@example.com:8080/data/report.html?date=2000-12-25#summary
                    \_/
                     |
                 user_info

        :return: string value of user information part or :py:const:`None`
        :rtype: unencoded string or :py:const:`None`
        """
        if self.__authority is None:
            return None
        else:
            return self.__authority.user_info

    @property
    def host(self):
        """ The *host* part of this URI or :py:const:`None` if undefined.

        ::

            >>> URI(None).host
            None
            >>> URI("").host
            None
            >>> URI("http://example.com").host
            'example.com'
            >>> URI("http://example.com:8080/data").host
            'example.com'

        **Component Definition:**
        ::

            https://bob@example.com:8080/data/report.html?date=2000-12-25#summary
                        \_________/
                             |
                            host

        :return:
        :rtype: unencoded string or :py:const:`None`
        """
        if self.__authority is None:
            return None
        else:
            return self.__authority.host
        
    @property
    def port(self):
        """ The *port* part of this URI or :py:const:`None` if undefined.

        ::

            >>> URI(None).port
            None
            >>> URI("").port
            None
            >>> URI("http://example.com").port
            None
            >>> URI("http://example.com:8080/data").port
            8080

        **Component Definition:**
        ::

            https://bob@example.com:8080/data/report.html?date=2000-12-25#summary
                                    \__/
                                     |
                                    port

        :return:
        :rtype: integer or :py:const:`None`
        """
        if self.__authority is None:
            return None
        else:
            return self.__authority.port

    @property
    def host_port(self):
        """ The *host* and *port* parts of this URI separated by a colon or
        :py:const:`None` if both are undefined.

        ::

            >>> URI(None).host_port
            None
            >>> URI("").host_port
            None
            >>> URI("http://example.com").host_port
            'example.com'
            >>> URI("http://example.com:8080/data").host_port
            'example.com:8080'
            >>> URI("http://bob@example.com:8080/data").host_port
            'example.com:8080'

        **Component Definition:**
        ::

            https://bob@example.com:8080/data/report.html?date=2000-12-25#summary
                        \______________/
                               |
                           host_port

        :return:
        :rtype: percent-encoded string or :py:const:`None`
        """
        if self.__authority is None:
            return None
        else:
            return self.__authority.host_port

    @property
    def path(self):
        """ The *path* part of this URI or :py:const:`None` if undefined.

        **Component Definition:**
        ::

            https://bob@example.com:8080/data/report.html?date=2000-12-25#summary
                                        \_______________/
                                                |
                                               path

        :return:
        :rtype: :py:class:`Path <httpstream.uri.Path>` instance or
            :py:const:`None`
        """
        return self.__path

    @property
    def query(self):
        """ The *query* part of this URI or :py:const:`None` if undefined.

        **Component Definition:**
        ::

            https://bob@example.com:8080/data/report.html?date=2000-12-25#summary
                                                          \_____________/
                                                                 |
                                                               query

        :rtype: :py:class:`Query <httpstream.uri.Query>` instance or
            :py:const:`None`
        """
        return self.__query

    @property
    def fragment(self):
        """ The *fragment* part of this URI or :py:const:`None` if undefined.

        **Component Definition:**
        ::

            https://bob@example.com:8080/data/report.html?date=2000-12-25#summary
                                                                          \_____/
                                                                             |
                                                                          fragment

        :return:
        :rtype: unencoded string or :py:const:`None`
        """
        return self.__fragment

    @property
    def hierarchical_part(self):
        """ The authority and path parts of this URI or :py:const:`None` if
        undefined.

        **Component Definition:**
        ::

            https://bob@example.com:8080/data/report.html?date=2000-12-25#summary
                    \___________________________________/
                                      |
                              hierarchical_part

        :return: combined string values of authority and path parts or
            :py:const:`None`
        :rtype: percent-encoded string or :py:const:`None`
        """
        if self.__path is None:
            return None
        u = []
        if self.__authority is not None:
            u += ["//", str(self.__authority)]
        u += [str(self.__path)]
        return "".join(u)

    @property
    def absolute_path_reference(self):
        """ The path, query and fragment parts of this URI or :py:const:`None`
        if undefined.

        **Component Definition:**
        ::

            https://bob@example.com:8080/data/report.html?date=2000-12-25#summary
                                        \_______________________________________/
                                                            |
                                                  absolute_path_reference

        :return: combined string values of path, query and fragment parts or
            :py:const:`None`
        :rtype: percent-encoded string or :py:const:`None`
        """
        if self.__path is None:
            return None
        u = [str(self.__path)]
        if self.__query is not None:
            u += ["?", str(self.__query)]
        if self.__fragment is not None:
            u += ["#", percent_encode(self.__fragment)]
        return "".join(u)

    def _merge_path(self, relative_path_reference):
        if self.__authority is not None and not self.__path:
            return Path("/" + str(relative_path_reference))
        elif "/" in self.__path.string:
            segments = self.__path.segments
            segments[-1] = ""
            return Path("/".join(segments) + str(relative_path_reference))
        else:
            return relative_path_reference

    def resolve(self, reference, strict=True):
        """ Transform a reference relative to this URI to produce a full target
        URI.

        .. seealso::
            `RFC 3986 § 5.2.2`_

        .. _`RFC 3986 § 5.2.2`: http://tools.ietf.org/html/rfc3986#section-5.2.2
        """
        if reference is None:
            return None
        reference = self.__cast(reference)
        target = URI(None)
        if not strict and reference.__scheme == self.__scheme:
            reference_scheme = None
        else:
            reference_scheme = reference.__scheme
        if reference_scheme is not None:
            target.__scheme = reference_scheme
            target.__authority = reference.__authority
            target.__path = reference.__path.remove_dot_segments()
            target.__query = reference.__query
        else:
            if reference.__authority is not None:
                target.__authority = reference.__authority
                target.__path = reference.__path.remove_dot_segments()
                target.__query = reference.__query
            else:
                if not reference.path:
                    target.__path = self.__path
                    if reference.__query is not None:
                        target.__query = reference.__query
                    else:
                        target.__query = self.__query
                else:
                    if str(reference.__path).startswith("/"):
                        target.__path = reference.__path.remove_dot_segments()
                    else:
                        target.__path = self._merge_path(reference.__path)
                        target.__path = target.__path.remove_dot_segments()
                    target.__query = reference.__query
                target.__authority = self.__authority
            target.__scheme = self.__scheme
        target.__fragment = reference.__fragment
        return target
