"""Useful Script
Usage:
%s

--dry, -D        Display the commands on stdout, instead of executing them
"""

import os
import glob
import importlib
import inspect

class BaseCommand(object):
  def __init__(self, arguments):
    self.arguments = arguments

class CommandRouter(object):
  commands = {}
  modules_string = ""

  def build(self):
    self._load_modules()

    for command in self.commands:
      command = self.commands[command]
      try:
        self.modules_string += "  useful.py %s\n" % command.signature
      except AttributeError:
        # If a command has no signature, don't add its signature
        pass

    return self.doc_string()

  def doc_string(self):
    return __doc__  % self.modules_string

  def find(self, arguments):
    for command in self.commands:
      instance = self.commands[command](arguments)
    
      if instance.is_valid():
        return instance.run()
      else:
        # The arguments can't be processed by this command
        pass

    # If we've managed to get this far, the command signature in question
    # cannot be handled by any command
    raise NotImplemented("A matching command for the signature could not be found")

  def _load_modules(self):
    command_modules = [ os.path.basename(f)[:-3] for f in glob.glob(os.path.dirname(__file__)+"/*_commands.py")]

    for module_name in command_modules:
      module = importlib.import_module('usefulpy.logic.%s' % module_name)

      for name, obj in inspect.getmembers(module):
        if inspect.isclass(obj):
          self.commands[name] = obj