"""Useful Script

Usage:%s
"""

import os
import glob
import importlib
import inspect
import re
import docopt
from usefulpy.exceptions import NotMatched, InvalidCommand

class BaseCommand(object):
  def __init__(self, arguments={}):
    self.arguments = arguments

  def set_router(self, router):
    self.router = router

def requires_built(fn):
  def _wrapped(self, *args, **kwargs):
    self.build()
    return fn(self, *args, **kwargs)
  return _wrapped

class CommandRouter(object):
  commands = {}
  built = False

  def build(self):
    if self.built:
      return
    self._load_modules()
    for command in self.commands:
      command = self.commands[command]
    self.built = True

  @requires_built
  def find(self):
    for command in self.commands:
      instance = self.commands[command]()

      try:
        arguments = docopt.docopt(instance.__doc__, help=False)
      except docopt.DocoptExit:
        # The arguments can't be processed by this command
        continue
      except docopt.DocoptLanguageError as exception:
        raise InvalidCommand(instance.__class__, exception)

      instance = self.commands[command](arguments)
      
      # Pass the router to the command, in case it needs it (usually for introspective purposes)
      instance.set_router(self)
      return instance.run()

    # If we've managed to get this far, the command signature in question
    # cannot be handled by any command
    raise NotMatched()

  def list(self):
    usage = "\n  useful.py ".join([''] + self.all())
    return __doc__ % usage

  @requires_built
  def all(self):
    possible_commands = []
    regex = re.compile(r"\suseful\.py\s(.+)")

    for command in self.commands:
      for line in (self.commands[command].__doc__).split("\n"):
        result = regex.search(line)
        if result:
          possible_commands.append(result.group(1))
    return possible_commands

  def _load_modules(self):
    command_modules = [ os.path.basename(f)[:-3] for f in glob.glob(os.path.dirname(__file__)+"/*_commands.py")]

    for module_name in command_modules:
      module = importlib.import_module('usefulpy.logic.%s' % module_name)
      for name, obj in inspect.getmembers(module):
        if inspect.isclass(obj):
          self.commands[name] = obj