#!/usr/bin/python
# -*- coding: utf-8 -*-
"""This module contains two functions converting utf-8 characters
   to BibTeX encoding.

   * protect_caps encloses capital letters in braces (like this: {A})
   * utf8_to_bibtex converts a big bunch of utf8 encoded characters to
   BibTeX. The selection of characters is guided by LaTeX's T1 encoding."""

import re

def protect_caps(astring):
  """Protects capital letters by enclosing them in braces."""
  return re.sub(r'([A-Z])', r'{\1}', astring)

def utf8_to_bibtex(astring):
  """Converts special letters and accented letters from utf8 to BibTeX."""
  # Some interpunctation and special signs
  tmp = re.sub(r'\\', r'{\\textbackslash}', astring)
  tmp = re.sub(r'&', r'\&', tmp)
  tmp = re.sub(r'#', r'\#', tmp)
  tmp = re.sub(r'%', r'\%', tmp)
  tmp = re.sub(r'\$', r'\\\$', tmp)
  tmp = re.sub(r'~', r'\~{}', tmp)
  tmp = re.sub(r'<', r'{\\textless}', tmp)
  tmp = re.sub(r'>', r'{\\textgreater}', tmp)
  tmp = re.sub(r'_', r'\_', tmp)
  tmp = re.sub(r'\^', r'\\\^{}', tmp)
  tmp = re.sub(r'\|', r'{\\textbar}', tmp)
  tmp = re.sub(r'"', r'{\dq}', tmp) # Needs the babel package
  tmp = re.sub(r'£', r'{\pounds}', tmp)
  tmp = re.sub(r'©', r'{\copyright}', tmp)
  tmp = re.sub(r'§', r'{\S}', tmp)
  # A few special letters
  tmp = re.sub(r'Æ', r'{\AE}', tmp)
  tmp = re.sub(r'æ', r'{\ae}', tmp)
  tmp = re.sub(r'Ð', r'{\DH}', tmp)
  tmp = re.sub(r'ð', r'{\dh}', tmp)
  tmp = re.sub(r'Đ', r'{\DJ}', tmp)
  tmp = re.sub(r'đ', r'{\dj}', tmp)
  tmp = re.sub(r'ı', r'{\i}', tmp)
  tmp = re.sub(r'Ĳ', r'{\IJ}', tmp)
  tmp = re.sub(r'ĳ', r'{\ij}', tmp)
  tmp = re.sub(r'ȷ', r'{\j}', tmp)
  tmp = re.sub(r'Ł', r'{\L}', tmp)
  tmp = re.sub(r'ł', r'{\l}', tmp)
  tmp = re.sub(r'Ŋ', r'{\NG}', tmp)
  tmp = re.sub(r'ŋ', r'{\\ng}', tmp)
  tmp = re.sub(r'Ø', r'{\O}', tmp)
  tmp = re.sub(r'ø', r'{\o}', tmp)
  tmp = re.sub(r'Œ', r'{\OE}', tmp)
  tmp = re.sub(r'œ', r'{\oe}', tmp)
  tmp = re.sub(r'ẞ', r'{\SS}', tmp)  # LaTeX representation is "SS"
  tmp = re.sub(r'ß', r'{\ss}', tmp)
  tmp = re.sub(r'Þ', r'{\TH}', tmp)
  tmp = re.sub(r'þ', r'{\\th}', tmp)
  # With grave accent
  tmp = re.sub(r'À', r'{\`A}', tmp)
  tmp = re.sub(r'à', r'{\`a}', tmp)
  tmp = re.sub(r'È', r'{\`E}', tmp)
  tmp = re.sub(r'è', r'{\`e}', tmp)
  tmp = re.sub(r'Ì', r'{\`I}', tmp)
  tmp = re.sub(r'ì', r'{\`\i}', tmp)
  tmp = re.sub(r'Ò', r'{\`O}', tmp)
  tmp = re.sub(r'ò', r'{\`o}', tmp)
  tmp = re.sub(r'Ù', r'{\`U}', tmp)
  tmp = re.sub(r'ù', r'{\`u}', tmp)
  tmp = re.sub(r'Ỳ', r'{\`Y}', tmp)
  tmp = re.sub(r'ỳ', r'{\`y}', tmp)
  # With acute accent
  tmp = re.sub(r'Á', r"{\'A}", tmp)
  tmp = re.sub(r'á', r"{\'a}", tmp)
  tmp = re.sub(r'Ć', r"{\'C}", tmp)
  tmp = re.sub(r'ć', r"{\'c}", tmp)
  tmp = re.sub(r'É', r"{\'E}", tmp)
  tmp = re.sub(r'é', r"{\'e}", tmp)
  tmp = re.sub(r'Í', r"{\'I}", tmp)
  tmp = re.sub(r'í', r"{\'\i}", tmp)
  tmp = re.sub(r'Ḱ', r"{\'K}", tmp)
  tmp = re.sub(r'ḱ', r"{\'k}", tmp)
  tmp = re.sub(r'Ĺ', r"{\'L}", tmp)
  tmp = re.sub(r'ĺ', r"{\'l}", tmp)
  tmp = re.sub(r'Ń', r"{\'N}", tmp)
  tmp = re.sub(r'ń', r"{\'n}", tmp)
  tmp = re.sub(r'Ó', r"{\'O}", tmp)
  tmp = re.sub(r'ó', r"{\'o}", tmp)
  tmp = re.sub(r'Ŕ', r"{\'R}", tmp)
  tmp = re.sub(r'ŕ', r"{\'r}", tmp)
  tmp = re.sub(r'Ś', r"{\'S}", tmp)
  tmp = re.sub(r'ś', r"{\'s}", tmp)
  tmp = re.sub(r'Ú', r"{\'U}", tmp)
  tmp = re.sub(r'ú', r"{\'u}", tmp)
  tmp = re.sub(r'Ý', r"{\'Y}", tmp)
  tmp = re.sub(r'ý', r"{\'y}", tmp)
  tmp = re.sub(r'Ź', r"{\'Z}", tmp)
  tmp = re.sub(r'ź', r"{\'z}", tmp)
  # With double acute accent
  tmp = re.sub(r'Ő', r'{\H O}', tmp)
  tmp = re.sub(r'ő', r'{\H o}', tmp)
  tmp = re.sub(r'Ű', r'{\H U}', tmp)
  tmp = re.sub(r'ű', r'{\H u}', tmp)
  # With circumflex accent
  tmp = re.sub(r'Â', r'{\^A}', tmp)
  tmp = re.sub(r'â', r'{\^a}', tmp)
  tmp = re.sub(r'Ĉ', r'{\^C}', tmp)
  tmp = re.sub(r'ĉ', r'{\^c}', tmp)
  tmp = re.sub(r'Ê', r'{\^E}', tmp)
  tmp = re.sub(r'ê', r'{\^e}', tmp)
  tmp = re.sub(r'Ĝ', r'{\^G}', tmp)
  tmp = re.sub(r'ĝ', r'{\^g}', tmp)
  tmp = re.sub(r'Ĥ', r'{\^H}', tmp)
  tmp = re.sub(r'ĥ', r'{\^h}', tmp)
  tmp = re.sub(r'Î', r'{\^I}', tmp)
  tmp = re.sub(r'î', r'{\^\i}', tmp)
  tmp = re.sub(r'Ĵ', r'{\^J}', tmp)
  tmp = re.sub(r'ĵ', r'{\^\j}', tmp)
  tmp = re.sub(r'Ô', r'{\^O}', tmp)
  tmp = re.sub(r'ô', r'{\^o}', tmp)
  tmp = re.sub(r'Ŝ', r'{\^S}', tmp)
  tmp = re.sub(r'ŝ', r'{\^s}', tmp)
  tmp = re.sub(r'Û', r'{\^U}', tmp)
  tmp = re.sub(r'û', r'{\^u}', tmp)
  tmp = re.sub(r'Ŵ', r'{\^W}', tmp)
  tmp = re.sub(r'ŵ', r'{\^w}', tmp)
  tmp = re.sub(r'Ŷ', r'{\^Y}', tmp)
  tmp = re.sub(r'ŷ', r'{\^y}', tmp)
  # With hachek accent
  tmp = re.sub(r'Č', r'{\\v C}', tmp)
  tmp = re.sub(r'č', r'{\\v c}', tmp)
  tmp = re.sub(r'Ď', r'{\\v D}', tmp)
  tmp = re.sub(r'ď', r'{\\v d}', tmp) # Looks like d'. Same for L, l, and t
  tmp = re.sub(r'Ě', r'{\\v E}', tmp)
  tmp = re.sub(r'ě', r'{\\v e}', tmp)
  tmp = re.sub(r'Ľ', r'{\\v L}', tmp)
  tmp = re.sub(r'ľ', r'{\\v l}', tmp)
  tmp = re.sub(r'Ň', r'{\\v N}', tmp)
  tmp = re.sub(r'ň', r'{\\v n}', tmp)
  tmp = re.sub(r'Ř', r'{\\v R}', tmp)
  tmp = re.sub(r'ř', r'{\\v r}', tmp)
  tmp = re.sub(r'Š', r'{\\v S}', tmp)
  tmp = re.sub(r'š', r'{\\v s}', tmp)
  tmp = re.sub(r'Ť', r'{\\v T}', tmp)
  tmp = re.sub(r'ť', r'{\\v t}', tmp)
  tmp = re.sub(r'Ž', r'{\\v Z}', tmp)
  tmp = re.sub(r'ž', r'{\\v z}', tmp)
  # With dot above
  # Note: There are many more in Unicode, but only these few occur in
  # official languages in Europe
  tmp = re.sub(r'Ċ', r'{\.C}', tmp)
  tmp = re.sub(r'ċ', r'{\.c}', tmp)
  tmp = re.sub(r'Ė', r'{\.E}', tmp)
  tmp = re.sub(r'ė', r'{\.e}', tmp)
  tmp = re.sub(r'Ġ', r'{\.G}', tmp)
  tmp = re.sub(r'ġ', r'{\.g}', tmp)
  tmp = re.sub(r'İ', r'{\.I}', tmp)
  tmp = re.sub(r'Ż', r'{\.Z}', tmp)
  tmp = re.sub(r'ż', r'{\.z}', tmp)
  # With two dots above (trema/diaeresis/umlaut)
  tmp = re.sub(r'Ä', r'{\"A}', tmp)
  tmp = re.sub(r'ä', r'{\"a}', tmp)
  tmp = re.sub(r'Ë', r'{\"E}', tmp)
  tmp = re.sub(r'ë', r'{\"e}', tmp)
  tmp = re.sub(r'Ï', r'{\"I}', tmp)
  tmp = re.sub(r'ï', r'{\"\i}', tmp)
  tmp = re.sub(r'Ö', r'{\"O}', tmp)
  tmp = re.sub(r'ö', r'{\"o}', tmp)
  tmp = re.sub(r'Ü', r'{\"U}', tmp)
  tmp = re.sub(r'ü', r'{\"u}', tmp)
  tmp = re.sub(r'Ÿ', r'{\"Y}', tmp)
  tmp = re.sub(r'ÿ', r'{\"y}', tmp)
  # With tilde
  tmp = re.sub(r'Ã', r'{\~A}', tmp)
  tmp = re.sub(r'ã', r'{\~a}', tmp)
  tmp = re.sub(r'Ẽ', r'{\~E}', tmp)
  tmp = re.sub(r'ẽ', r'{\~e}', tmp)
  tmp = re.sub(r'Ĩ', r'{\~I}', tmp)
  tmp = re.sub(r'ĩ', r'{\~\i}', tmp)
  tmp = re.sub(r'Ñ', r'{\~N}', tmp)
  tmp = re.sub(r'ñ', r'{\~n}', tmp)
  tmp = re.sub(r'Õ', r'{\~O}', tmp)
  tmp = re.sub(r'õ', r'{\~o}', tmp)
  tmp = re.sub(r'Ũ', r'{\~U}', tmp)
  tmp = re.sub(r'ũ', r'{\~u}', tmp)
  tmp = re.sub(r'Ỹ', r'{\~Y}', tmp)
  tmp = re.sub(r'ỹ', r'{\~y}', tmp)
  # With bar above
  tmp = re.sub(r'Ā', r'{\=A}', tmp)
  tmp = re.sub(r'ā', r'{\=a}', tmp)
  tmp = re.sub(r'Ē', r'{\=E}', tmp)
  tmp = re.sub(r'ē', r'{\=e}', tmp)
  tmp = re.sub(r'Ī', r'{\=I}', tmp)
  tmp = re.sub(r'ī', r'{\=\i}', tmp)
  tmp = re.sub(r'Ō', r'{\=O}', tmp)
  tmp = re.sub(r'ō', r'{\=o}', tmp)
  tmp = re.sub(r'Ū', r'{\=U}', tmp)
  tmp = re.sub(r'ū', r'{\=u}', tmp)
  # With breve above
  tmp = re.sub(r'Ă', r'{\u A}', tmp)
  tmp = re.sub(r'ă', r'{\u a}', tmp)
  tmp = re.sub(r'Ĕ', r'{\u E}', tmp)
  tmp = re.sub(r'ĕ', r'{\u e}', tmp)
  tmp = re.sub(r'Ğ', r'{\u G}', tmp)
  tmp = re.sub(r'ğ', r'{\u g}', tmp)
  tmp = re.sub(r'Ĭ', r'{\u I}', tmp)
  tmp = re.sub(r'ĭ', r'{\u\i}', tmp)
  tmp = re.sub(r'Ŏ', r'{\u O}', tmp)
  tmp = re.sub(r'ŏ', r'{\u o}', tmp)
  tmp = re.sub(r'Ŭ', r'{\u U}', tmp)
  tmp = re.sub(r'ŭ', r'{\u u}', tmp)
  # With ring above 
  tmp = re.sub(r'Å', r'{\r A}', tmp) # We use generic LaTeX2e notation \r
  tmp = re.sub(r'å', r'{\r a}', tmp)
  tmp = re.sub(r'Ů', r'{\r U}', tmp)
  tmp = re.sub(r'ů', r'{\r u}', tmp)
  # With cedilla
  tmp = re.sub(r'Ç', r'{\c C}', tmp)
  tmp = re.sub(r'ç', r'{\c c}', tmp)
  tmp = re.sub(r'Ģ', r'{\c G}', tmp)
  tmp = re.sub(r'ģ', r'{\c g}', tmp)
  tmp = re.sub(r'Ķ', r'{\c K}', tmp)
  tmp = re.sub(r'ķ', r'{\c k}', tmp)
  tmp = re.sub(r'Ļ', r'{\c L}', tmp)
  tmp = re.sub(r'ļ', r'{\c l}', tmp)
  tmp = re.sub(r'Ņ', r'{\c N}', tmp)
  tmp = re.sub(r'ņ', r'{\c n}', tmp)
  tmp = re.sub(r'Ŗ', r'{\c R}', tmp)
  tmp = re.sub(r'ŗ', r'{\c r}', tmp)
  tmp = re.sub(r'Ş', r'{\c S}', tmp)
  tmp = re.sub(r'ş', r'{\c s}', tmp)
  tmp = re.sub(r'Ţ', r'{\c T}', tmp)
  tmp = re.sub(r'ţ', r'{\c t}', tmp)
  # With comma below: Map them to \c, because the T1 fontencoding does not 
  # offer anything better
  tmp = re.sub(r'Ș', r'{\c S}', tmp)
  tmp = re.sub(r'ș', r'{\c s}', tmp)
  tmp = re.sub(r'Ț', r'{\c T}', tmp)
  tmp = re.sub(r'ț', r'{\c t}', tmp)
  # With ogonek
  tmp = re.sub(r'Ą', r'{\k A}', tmp)
  tmp = re.sub(r'ą', r'{\k a}', tmp)
  tmp = re.sub(r'Ę', r'{\k E}', tmp)
  tmp = re.sub(r'ę', r'{\k e}', tmp)
  tmp = re.sub(r'Į', r'{\k I}', tmp)
  tmp = re.sub(r'į', r'{\k i}', tmp)
  tmp = re.sub(r'Ǫ', r'{\k O}', tmp)
  tmp = re.sub(r'ǫ', r'{\k o}', tmp)
  tmp = re.sub(r'Ų', r'{\k U}', tmp)
  tmp = re.sub(r'ų', r'{\k u}', tmp)
  #
  return tmp
