#!/usr/bin/env python3
import urllib, urllib.request, re, datetime
import sys, os, webbrowser
from bs4 import BeautifulSoup

class Warning():
	def __init__(self, postcode, country, lang):
		self.postcode = postcode
		self.country = country
		self.location = ''
		self.lang = lang
		
		self.time_from = None
		self.time_to   = None
		self.lastchange = None
		
		self.level = None
		self.watch = False
		self.type = ''
		self.text = ''
		
	def __repr__(self):
		return '<Warning%s Level %s about %s in %s-%s %s>' % (' (watch)' if self.watch else '', str(self.level), self.type.title(), self.country.upper(), str(self.postcode), self.location)

languages = {
	'de': ('warnstufe', 'orange', 'rot', 'violett', r'([0-9]+). ([a-z]+) ([0-9]+), ([0-9]+):([0-9]+)',
		lambda m: datetime.datetime(int(m.group(3)), ('januar','februar','märz','april','mai','juni','juli','august','september','oktober','november','dezember').index(m.group(2))+1, int(m.group(1)), int(m.group(4)), int(m.group(5)))
		),
	'en': ('level', 'orange', 'red', 'violet', r'([a-z]+) ([0-9]+), ([0-9]+) ([0-9]+):([0-9]+) (am|pm)',
		lambda m: datetime.datetime(int(m.group(3)), ('january','february','march','april','may','june','july','august','september','october','november','december').index(m.group(1))+1, int(m.group(2)), (int(m.group(4))+(12 if m.group(6)=='pm' else 0))%24, int(m.group(5)))
		),
	'fr': ('d’alerte', 'orange', 'rouge', 'violet', r'([0-9]+) ([a-z]+) ([0-9]+) - ([0-9]+):([0-9]+)',
		lambda m: datetime.datetime(int(m.group(3)), ('janvier','février','mars','avril','mai','juin','juillet','août','septembre','octobre','novembre','décembre').index(m.group(2))+1, int(m.group(1)), int(m.group(4)), int(m.group(5)))
		),
	'it': ('allarme', 'arancione', 'rosso', 'viola', r'([0-9]+). ([a-z]+) ([0-9]+), ([0-9]+):([0-9]+)',
		lambda m: datetime.datetime(int(m.group(3)), ('gennaio','febbraio','marzo','aprile','maggio','giugno','luglio','agosto','settembre','ottobre','novembre','dicembre').index(m.group(2))+1, int(m.group(1)), int(m.group(4)), int(m.group(5)))
		),
}

def warnings(postcode='21037', country='de', lang='de'):
	global languages
	if lang not in languages: return None # todo: es, pt, en, da, nl, lb, fi, se, sv, no 
	if country not in ('de', 'fr', 'es', 'pt', 'uk', 'ie', 'dk', 'nl', 'be', 'lu', 'it', 'ch', 'at', 'se', 'fn'): return None
	
	f = urllib.request.urlopen('http://www.unwetterzentrale.de/uwz/getwarning_de.php?plz=%s&uwz=UWZ-%s&lang=%s' % (str(postcode), country.upper(), lang.lower()))
	soup = BeautifulSoup(f.read().decode('utf-8'))
	
	title = soup.find(id="content").find('h1')
	allwarnings = title.find_next_siblings()
	if not allwarnings: return None # Unknown location
	allwarnings = allwarnings[0:allwarnings.index(title.find_next_sibling('p'))]
	
	warnings = []
	warning = []
	while allwarnings:
		elem = allwarnings.pop(0)
		if elem.name == 'div':
			warning.append(elem)
		elif elem.name == 'br' and warning:
			warnings.append(warning)
			warning = []
		
		
	parse_warning = {
		'sturm': 'storm',
		'regen': 'rain',
		'schnee': 'snow',
		'gewitter': 'thunderstorm',
		'glatteisregen': 'sleet'
	}
	parse_notice = {
		'strassenglaette': 'slipperiness',
		'temperatur': 'temperature'
	}
	
	r = []
	while warnings:
		w = warnings.pop(0)
		warning = Warning(postcode, country, lang)
		warning.location = w[0].find('b').text
		mytype, level = w[1].find('img')['src'].split('/')[-1][:-4].split('-')
		
		if level == 'gelb':
			title = w[1].find('div').find('div').find('div').text
			warning.level = languages[lang].index(title.lower().split(languages[lang][0])[1].strip().split(' ')[0])
			warning.watch = True
		else:
			warning.level = ('orange', 'rot', 'violett').index(level.lower())+1
				
		if mytype in parse_warning:
			warning.type = parse_warning[mytype]
		elif mytype in parse_notice:
			warning.level = 0
			warning.type = parse_notice[mytype]
			
		bolds = w[1].find_all('b')
		a = re.search(languages[lang][4], bolds[0].text.lower())
		b = re.search(languages[lang][4], bolds[1].text.lower())
		c = re.search(languages[lang][4], w[3].text.lower())
		if a is not None: warning.time_from  = languages[lang][5](a)
		if b is not None: warning.time_to    = languages[lang][5](b)
		if c is not None: warning.lastchange = languages[lang][5](c)
		r.append(warning)
	
	return r

try:
	from gi.repository import AppIndicator3 as appindicator
except:
	pass
from gi.repository import Gtk as gtk
from gi.repository import GLib
import json

class Tray:	
	countries = ('de', 'fr', 'es', 'pt', 'uk', 'ie', 'dk', 'nl', 'be', 'lu', 'it', 'ch', 'at', 'se', 'fn')
	
	lang = 'en'
	
	def __init__(self):
		self.noclicks = False
		
		self.bywifi = False
		self.plz = None
		self.country = None
		
		self.warnings = None
		self.menuwarnings = []
		
		self.interval = 10
		self.timeoutid = None
		
		try:
			data = json.load(open(os.path.expanduser('~/.uwztray.json'), 'r'))
		except(FileNotFoundError):
			pass
		else:
			self.bywifi = data['bywifi']
			self.interval = data['interval']
			if not self.bywifi:
				self.plz = data['plz']
				self.country = data['country']
		
		self.path = os.path.join(os.path.dirname(os.path.abspath(__file__)), 'icons')
		if not os.path.isdir(self.path):
			self.path = os.sep+os.path.join('usr', 'share', 'uwz', 'icons')
		
		self.menu = gtk.Menu()
		
		self.menu_title = gtk.MenuItem('Loading…')
		self.menu_title.connect("activate", self.show_page)
		self.menu_title.show()
		self.menu.append(self.menu_title)
		
		sep0 = gtk.SeparatorMenuItem()
		sep0.show()
		self.menu.append(sep0)
		
		self.loc_set = gtk.CheckMenuItem('Set Location')
		self.loc_set.set_active(0)
		self.loc_set.connect("activate", self.set_location_manually)
		self.loc_set.show()
		self.menu.append(self.loc_set)
		
		self.loc_wifi = gtk.CheckMenuItem('Location by Wifi')
		self.loc_wifi.set_active(1)
		self.loc_wifi.connect("activate", self.set_location_by_wifi)
		self.loc_wifi.show()
		self.menu.append(self.loc_wifi)
		
		sep1 = gtk.SeparatorMenuItem()
		sep1.show()
		self.menu.append(sep1)
		
		self.reload_set = gtk.MenuItem('Set Reload Interval')
		self.reload_set.connect("activate", self.set_reload_interval)
		self.reload_set.show()
		self.menu.append(self.reload_set)

		self.reload_now = gtk.MenuItem('Reload now')
		self.loc_set.connect("activate", self.check)
		self.reload_now.show()
		self.menu.append(self.reload_now)
		
		sep1 = gtk.SeparatorMenuItem()
		sep1.show()
		self.menu.append(sep1)
		
		exit = gtk.MenuItem("Exit")
		exit.connect("activate", self.menu_exit)
		exit.show()
		self.menu.append(exit)

		self.menu.show()
	
		if 'appindicator' in globals():
			self.indicator = appindicator.Indicator.new("UWZ", os.path.join(self.path, 'warnings-0.svg'), appindicator.IndicatorCategory.APPLICATION_STATUS)
			self.indicator.set_status(appindicator.IndicatorStatus.ACTIVE)
			self.indicator.set_menu(self.menu)
		else:
			self.staticon = gtk.StatusIcon()
			self.staticon.set_from_file(os.path.join(self.path, 'warnings-0.svg'))
			self.staticon.connect("popup_menu", self.trayicon_popup)
			self.staticon.set_visible(True)
		
		self.check()
		try:
			gtk.main()
		except(KeyboardInterrupt):
			pass
	
	def trayicon_popup(self, widget, button, time, data = None):
		self.menu.popup(None, None, lambda w,x: self.staticon.position_menu(self.menu, self.staticon), self.staticon, 3, time)		
		
	def set_location_by_wifi(self, w):
		if self.noclicks: return
		
		dialog = gtk.MessageDialog(None, 0, gtk.MessageType.QUESTION, gtk.ButtonsType.OK_CANCEL, 'This means that Wifi Access Points near you will be regulary trnasmitted to Google.')
		response = dialog.run()
		dialog.destroy()
		if response != gtk.ResponseType.OK:
			return
		
		self.bywifi = True
		self.plz = None
		self.country = None
		
		self.check()
		self.update_menu_settings()
		self.save_settings()
		
	def set_location_manually(self, w):
		if self.noclicks: return
		
		dialog = gtk.MessageDialog(None, 0, gtk.MessageType.QUESTION, gtk.ButtonsType.OK_CANCEL, 'Please specify country and post code:')
		
		hbox = gtk.HBox.new(False, 8)
		dialog.get_message_area().pack_end(hbox, False, False, 0)
		
		country_store = gtk.ListStore(str, str)
		for l in self.countries:
			country_store.append([l.lower(), l.upper()])
			
		combobox = gtk.ComboBox.new_with_model(country_store)
		combobox.set_entry_text_column(1)
		renderer = gtk.CellRendererText()
		combobox.set_active(0 if self.country is None else self.countries.index(self.country))
		combobox.pack_start(renderer, True)
		combobox.add_attribute(renderer, 'text', 1)
		hbox.pack_start(combobox, False, True, 0)
		
		plzentry = gtk.Entry.new()
		if self.plz is not None:
			plzentry.set_text(self.plz)
		hbox.pack_start(plzentry, False, False, 0)
		hbox.show_all()
	
		response = dialog.run()
		if response == gtk.ResponseType.OK:
			newplz = plzentry.get_text().strip()
			self.bywifi = False
			print(newplz)
			if newplz:
				self.country = self.countries[combobox.get_active()]
				self.plz = newplz
			else:
				self.country = None
				self.plz = None
		dialog.destroy()
		
		self.check()
		self.update_menu_settings()
		self.save_settings()
		
	def set_reload_interval(self, w):
		if self.noclicks: return
		
		dialog = gtk.MessageDialog(None, 0, gtk.MessageType.QUESTION, gtk.ButtonsType.OK_CANCEL, 'Interval in Minutes:')
		
		adjustment = gtk.Adjustment(self.interval, 1, 60, 1, 5, 0)
		spinbutton = gtk.SpinButton()
		spinbutton.set_adjustment(adjustment)
		spinbutton.set_value(self.interval)
        
		dialog.get_message_area().pack_end(spinbutton, False, False, 0)
		spinbutton.show()
	
		response = dialog.run()
		if response == gtk.ResponseType.OK:
			self.interval = spinbutton.get_value()
		dialog.destroy()
		
		self.save_settings()
		self.check()
		
	def show_page(self, w):
		if self.plz is not None and self.country is not None:
			webbrowser.open('http://www.unwetterzentrale.de/uwz/getwarning_de.php?plz=%s&uwz=UWZ-%s&lang=%s' % (str(self.plz), self.country.upper(), self.lang))
		
	def save_settings(self):
		json.dump({'bywifi': True, 'interval': self.interval} if self.bywifi else {'bywifi': False, 'interval': self.interval, 'plz': self.plz, 'country': self.country}, open(os.path.expanduser('~/.uwztray.json'), 'w'))
		
	def update_menu_settings(self):
		self.noclicks = True
		self.loc_set.set_active((not self.bywifi) and self.plz is not None)
		self.loc_wifi.set_active(self.bywifi)
		if self.plz is not None:
			self.menu_title.set_label('UWZ: %s-%s' % (self.country.upper(), self.plz))
		else:
			self.menu_title.set_label('UWZ: No Location')
		self.noclicks = False
		
	def locate(self):
		if self.bywifi:
			import geolocate
			from geopy import geocoders
			rawdata = geocoders.Nominatim().reverse('%f,%f' % geolocate.locate()[0:2]).raw
			self.plz = rawdata['address']['postcode']
			self.country = rawdata['address']['country_code']
		
	def check(self, w=None):
		if self.timeoutid is not None:
			try:
				GLib.source_remove(self.timeoutid)
			except:
				pass
		self.locate()
		if self.plz is not None and self.country is not None:
			self.warnings = warnings(self.plz, self.country)
		else:
			self.warnings = None
		self.update_menu_settings()
		self.update_indicator()
		self.timeoutid = GLib.timeout_add_seconds(self.interval*60, self.check)
		return False
		
	def update_indicator(self):
		if self.warnings is None:
			filename = os.path.join(self.path, 'warnings-error.svg')
		else:	
			level = -1
			for warning in self.warnings:
				if warning.watch and 0 > level:
					level = 0
				elif warning.level > 0 and warning.level > level:
					level = warning.level
			
			if level == -1:
				level = ''
			else:
				level = str(level)
			
			filename = os.path.join(self.path, 'warnings-%s%d.svg' % (level, len(self.warnings)))
		
		if 'appindicator' in globals():
			self.indicator.set_icon(filename)
		else:
			self.staticon.set_from_file(filename)
			
		self.reload_now.set_label('Reload now (last: %s)' % datetime.datetime.now().strftime('%H:%M'))
		
		while self.menuwarnings:
			self.menu.remove(self.menuwarnings.pop())
			
		if self.warnings is None:
			tmp = gtk.MenuItem('Unsupported Location')
			tmp.connect("activate", self.show_page)
			tmp.show()
			self.menuwarnings.append(tmp)
			self.menu.insert(tmp, 1)
		elif not self.warnings:
			tmp = gtk.MenuItem('No warnings')
			tmp.connect("activate", self.show_page)
			tmp.show()
			self.menuwarnings.append(tmp)
			self.menu.insert(tmp, 1)
		else:
			now = datetime.datetime.now()
			i = 1
			for warning in self.warnings:
				if warning.time_from > now:
					mytime = warning.time_from
					timestr = 'starting %s'
				else:
					mytime = warning.time_to
					timestr = 'until %s'	
				timestr = timestr % mytime.strftime(('%d.%m. ' if mytime-now > datetime.timedelta(hours=20) else '')+'%H:%M')
				
				if warning.level == 0:
					text = 'Notice: %s %s' % (warning.type, timestr)
				elif warning.watch:
					text = 'Watch Level %d: %s %s' % (warning.level, warning.type, timestr)
				else:
					text = 'Warning Level %d: %s %s' % (warning.level, warning.type, timestr)
				
				tmp = gtk.MenuItem(text)
				tmp.connect("activate", self.show_page)
				tmp.show()
				self.menuwarnings.append(tmp)
				self.menu.insert(tmp, i)
				i += 1
			
		return
		
	def menu_exit(self, foo=None):
		sys.exit(0)

if __name__ == '__main__':
    Tray()
