# Copyright (C) 2011-2013 Versile AS
# 
# This file is part of Versile Python.
# 
# Versile Python is free software: you can redistribute it and/or
# modify it under the terms of the GNU Affero General Public License
# as published by the Free Software Foundation, either version 3 of
# the License, or (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Affero General Public License for more details.
# 
# You should have received a copy of the GNU Affero General Public
# License along with this program.  If not, see
# <http://www.gnu.org/licenses/>.
#
# Other Usage
# Alternatively, this file may be used in accordance with the terms
# and conditions contained in a signed written agreement between you
# and Versile AS.
#
# Versile Python implements Versile Platform which is a copyrighted
# specification that is not part of this software.  Modification of
# the software is subject to Versile Platform licensing, see
# https://versile.com/ for details. Distribution of unmodified
# versions released by Versile AS is not subject to Versile Platform
# licensing.
#

"""Framework for cryptography methods including :term:`VCA` support."""


import base64
import re
from threading import RLock

from versile.internal import _b2s, _s2b,  _vexport, _b_ord, _b_chr, _pyver
from versile.common.iface import abstract
from versile.common.util import VByteBuffer, VObjectIdentifier, VLockable
from versile.common.util import netbytes_to_posint, posint_to_netbytes
from versile.common.util import bytes_to_posint
from versile.crypto.math import is_prime

__all__ = ['VAsymmetricKey', 'VBlockCipher', 'VBlockTransform', 'VCrypto',
           'VCryptoException', 'VDefaultCrypto', 'VHash', 'VKey',
           'VKeyFactory', 'VNumCipher', 'VNumTransform', 'VProxyCrypto',
           'VRSAKeyFactory', 'VDecentralIdentityScheme',
           'VDecentralIdentitySchemeA']
__all__ = _vexport(__all__)


class VCryptoException(Exception):
    """Exception for crypto operations."""
    def __init__(self, *args):
        super(VCryptoException, self).__init__(*args)

        
@abstract
class VCrypto(object):
    """Provider of cryptographic methods such as hash methods and ciphers.
    
    For general information about crypto providers see :ref:`lib_crypto`\ .
    
    .. note::
    
        The :class:`VCrypto` class is abstract. For complete
        implementations see
        e.g. :class:`versile.crypto.local.VLocalCrypto`\ .
    
    Implements getattribute aliasing so that attributes which are not
    explicitly defined are searched (in order) among hash types, block
    ciphers and number ciphers.
    
    """
    
    # Lock for retreiving default provider
    _default_provider_lock = RLock()

    def __getattr__(self, attr):
        """Retreives cryptography classes or objects generated by provider.

        Overloaded to return hash type, block cipher, num cipher,
        transform or decentral key scheme with matching name (in that
        order).
        
        :param attr: attribute name
        :returns:    hash method class or cipher object
        
        """
        if attr in self.hash_types:
            return self.hash_cls(attr)
        elif attr in self.block_ciphers:
            return self.block_cipher(attr)
        elif attr[:4] == 'num_' and attr[4:] in self.num_ciphers:
            return self.num_cipher(attr[4:])
        elif attr in self.transforms:
            return self.transform(attr)
        elif attr in self.decentral_key_schemes:
            return self.decentral_key_scheme(attr)
        else:
            raise AttributeError('No attribute %s' % attr)

    @classmethod
    def default(cls, lazy=True):
        """Returns a default global crypto provider.

        :param lazy: if True will lazy-create a default provider
        :type  lazy: bool
        :returns:    crypto provider
        :rtype:      :class:`VCrypto`
        :raises:     :exc:`VCryptoException`

        Raises an exception if no provider is set and is not lazy-generated.
        
        """
        with cls._default_provider_lock:
            try:
                provider = getattr(cls, '_default_provider')
            except AttributeError:
                if lazy:
                    provider = VDefaultCrypto()
                    cls._default_provider = provider
                else:
                    raise VCryptoException('No provider set')
            return provider

    @classmethod
    def set_default(cls, provider):
        """Sets the default global crypto provider.

        :param provider: new provider
        :type  provider: :class:`VCrypto`
        
        Replaces any previous provider set
        
        """
        with cls._default_provider_lock:
            cls._default_provider = provider

    @classmethod
    def lazy(cls, provider=None):
        """Returns provider, global provider, or a new default provider.

        :param provider: provider (or None)
        :type  provider: :class:`VCrypto`
        :returns:        crypto provider
        :rtype:          :class:`VCrypto`

        If *provider* is set then provider is returned, otherwise the
        default global provider is returned (with lazy-creation).
        
        """
        if provider:
            return provider
        else:
            return cls.default(lazy=True)

    @property
    def hash_types(self):
        """Holds a tuple of the names of supported hash functions.

        Default holds an empty tuple, derived classes can overload.
        
        """
        return tuple()

    def hash_cls(self, hash_name):
        """Returns a hash class for the given hash type.

        :param hash_name: hash method name
        :type  hash_name: unicode
        :returns:         hash class
        :rtype:           :class:`VHash` class
        :raises:          :exc:`VCryptoException`

        Default raises an exception, derived classes should override.
        
        """
        raise VCryptoException('Hash class not implemented.')

    @property
    def block_ciphers(self):
        """Holds a tuple of the names of supported block ciphers.

        Default holds an empty tuple, derived classes can overload.
        
        """
        return tuple()

    def block_cipher(self, cipher_name):
        """Returns a block cipher for the given cipher name.

        :param cipher_name: block cipher name
        :type  cipher_name: unicode
        :returns:           cipher object
        :rtype:             :class:`VBlockCipher`
        :raises:            :exc:`VCryptoException`

        Default raises an exception, derived classes should override.
        
        """
        raise VCryptoException('Block cipher not implemented.')

    @property
    def num_ciphers(self):
        """Holds a tuple of the names of supported num ciphers.

        Default holds an empty tuple, derived classes can overload.
        
        """
        return tuple()
    
    def num_cipher(self, cipher_name):
        """Returns a num cipher for the given cipher name.

        :param cipher_name: num cipher name
        :type  cipher_name: unicode
        :returns:           cipher object
        :rtype:             :class:`VNumCipher`
        :raises:            :exc:`VCryptoException`

        Default raises an exception, derived classes should override.
        
        """
        raise VCryptoException('Number cipher not implemented.')

    @property
    def transforms(self):
        """Holds a tuple of transforms supported by the provider.

        This 'transforms' mechanism is intended for general transforms
        such as message digest encoding, which may be extended by 3rd
        party modules or rely on 3rd party libraries that may not
        always be available.
        
        Default holds an empty tuple, derived classes can overload.
        
        """
        return tuple()
    
    def transform(self, transform_name):
        """Returns a transform for the given transform name.

        :param transform_name: transform name
        :type  transform_name: unicode
        :returns:              a transform
        :rtype:                callable
        :raises:               :exc:`VCryptoException`
        
        The 'transform' mechanism is intended for general transforms
        such as message digest encoding, which may be extended by 3rd
        party modules or rely on 3rd party libraries that may not
        always be available.

        The format for the returned callable (which may be a callable
        object) is specific to each particular transform name; please
        refer to documentation of the individual transforms for
        details.
        
        Default raises an exception, derived classes should override.
        
        """
        return tuple()
    
    @property
    def decentral_key_schemes(self):
        """Holds a tuple of names of supported decentral key schemes.

        Default holds an empty tuple, derived classes can overload.
        
        """
        return tuple()

    def decentral_key_scheme(self, name):
        """Returns a decentral key scheme for the given scheme name.

        :param name: decentral key scheme name
        :type  name: unicode
        :returns:    scheme object
        :rtype:      :class:`VDecentralIdentityScheme`
        :raises:     :exc:`VCryptoException`
        
        Default raises an exception, derived classes should override.
        
        """
        raise VCryptoException('Scheme not implemented.')

    @abstract
    def import_ascii_key(self, keydata):
        """Imports a key from a native ASCII format

        :param keydata: key in native ASCII representation
        :type  keydata: bytes
        :returns:       :class:`VKey`
        
        The keydata parameter should have the appropriate standard
        ASCII representation defined in :term:`VP` VCrypto
        specifications.

        .. note::

            This method imports from a native format. In order to
            import from standards based formats, other mechanisms must
            be used. See e.g. :class:`versile.crypto.x509.VX509Crypto`
            for importing PKCS formatted RSA keys.
        
        """
        raise NotImplementedError
    
    
class VProxyCrypto(VCrypto, VLockable):
    """Proxy for combining multiple crypto providers into one.

    The proxy is set up on a set of providers. Those providers are
    searched for hash types and ciphers in the order they were
    registered with the proxy, and the first match is returned.

    Use of a proxy provider makes it possible to e.g. combine
    providers built around an optimized crypto library (such as
    pycrypto) as a default provider, with fallback provider(s) such as
    :class:`versile.crypto.local.VLocalCrypto` for features
    not implemented by that provider.
    
    :param providers: crypto provider(s)
    :type  providers: :class:`VCrypto`

    The proxy object searches its providers in the order they are
    passed to the constructor.

    """

    def __init__(self, *providers):
        VLockable.__init__(self)
        self._providers = list(providers)

    def append_provider(self, provider, lazy_add=True):
        """Appends a crypto provider to the current set.

        :param provider: provider to add
        :type  provider: :class:`VCrypto`
        :param lazy_add: if True, only add if no other provider with same type
        :type  lazy_add: bool
        
        """
        with self:
            if lazy_add:
                tp = type(provider)
                for p in self._providers:
                    if type(p) is tp:
                        return
            self._providers.append(provider)                

    @property
    def providers(self):
        """Holds a tuple of registered providers."""
        with self:
            return tuple(self._providers)
    
    @property
    def hash_types(self):
        with self:
            s = set()
            for p in self._providers:
                for h_t in p.hash_types:
                    s.add(h_t)
            return tuple(s)

    def hash_cls(self, hash_name):
        with self:
            for p in self._providers:
                if hash_name in p.hash_types:
                    return p.hash_cls(hash_name)
            else:
                raise VCryptoException('Hash type not implemented')
        
    @property
    def block_ciphers(self):
        with self:
            s = set()
            for p in self._providers:
                for cipher in p.block_ciphers:
                    s.add(cipher)
            return tuple(s)

    def block_cipher(self, cipher_name):
        with self:
            for p in self._providers:
                if cipher_name in p.block_ciphers:
                    return p.block_cipher(cipher_name)
            else:
                raise VCryptoException('Cipher not implemented')
    
    @property
    def num_ciphers(self):
        with self:
            s = set()
            for p in self._providers:
                for cipher in p.num_ciphers:
                    s.add(cipher)
            return tuple(s)

    def num_cipher(self, cipher_name):
        with self:
            for p in self._providers:
                if cipher_name in p.num_ciphers:
                    return p.num_cipher(cipher_name)
            else:
                raise VCryptoException('Cipher not implemented')
    
    @property
    def transforms(self):
        with self:
            s = set()
            for p in self._providers:
                for t in p.transforms:
                    s.add(t)
            return tuple(s)
    
    def transform(self, transform_name):
        with self:
            for p in self._providers:
                if transform_name in p.transforms:
                    return p.transform(transform_name)
            else:
                raise VCryptoException('Transform not implemented')

    @property
    def decentral_key_schemes(self):
        with self:
            s = set()
            for p in self._providers:
                for scheme in p.decentral_key_schemes:
                    s.add(scheme)
            return tuple(s)

    def decentral_key_scheme(self, name):
        with self:
            for p in self._providers:
                if name in p.decentral_key_schemes:
                    return p.decentral_key_scheme(name)
            else:
                raise VCryptoException('Decentral key scheme not implemented')
        

class VDefaultCrypto(VProxyCrypto):
    """A default crypto provider.
    
    The default provider instantiates available providers and sets up
    as a proxy provider, using 3rd party libraries when possible.
    
    The current order that providers are added is:
    
    #. :class:`versile.crypto.pycrypto.PyCrypto`
    #. :class:`versile.crypto.local.VLocalCrypto`
    
    """

    def __init__(self):
        providers = []

        # Tries to import provider for optimized 3rd party ciphers
        try:
            from versile.crypto.pycrypto import PyCrypto
        except ImportError:
            pass
        else:
            providers.append(PyCrypto())
        
        # Imports X.509 services provider
        from versile.crypto.x509 import VX509Crypto
        p = VX509Crypto()
        providers.append(VX509Crypto())
        
        # Imports local crypto provider
        from versile.crypto.local import VLocalCrypto
        providers.append(VLocalCrypto())
        super(VDefaultCrypto, self).__init__(*providers)


@abstract
class VHash(object):
    """Hash method implementation for computing hash digests.
    
    Implements a hash function with methods for adding input data to
    the processor and retreiving a digest for all input data which has
    been fed into the hasher.

    :param data: initial input data to the hasher
    :type  data: bytes

    .. note::

        This is an abstract base class which should not be directly
        instantiated. Hash classes should normally be created by
        calling a factory method on a :class:`VCrypto`\ .

    """
    
    def __init__(self, data=None):
        if data is not None:
            self.update(data)

    @abstract
    @classmethod
    def name(cls):
        """Returns the hash method name which constructs this hash class.

        :returns: hash class name
        :rtype:   unicode

        """
        raise NotImplementedError()
    
    @abstract
    @classmethod
    def oid(cls):
        """Returns an Object Identifier which constructs this hash class.
        
        :returns: OID
        :rtype:   :class:`versile.common.util.VObjectIdentifier`
        
        See `Object Identifier
        <http://en.wikipedia.org/wiki/Object_identifier>`__\
        info. Returns None if OID is not supported for this hash.
        
        """
        raise NotImplementedError()
    
    @abstract
    @classmethod
    def digest_size(cls):
        """Returns the size of the digests computed by this hash method.
        
        :returns: number of bytes for hash digests
        :rtype:   int
        
        """
        raise NotImplementedError()
        
    @abstract
    def update(self, data):
        """Feeds (additional) input data into the hash function.

        :param data: input data to the hash function
        :type  data: bytes
        
        """
        raise NotImplementedError()

    @abstract
    def digest(self):
        """Returns a hash digest of all input data provided.
        
        :returns: hash function digest
        :rtype:   bytes
        
        """
        raise NotImplementedError()
    
    @classmethod
    def hmac(cls, secret, message):
        """Generates and returns a :term:`HMAC`
        
        :param secret:  secret key
        :type  secret:  bytes
        :param message: message
        :type  message: bytes
        :returns:       message authentication code
        :rtype:         bytes
        
        Implements :term:`HMAC` algorithm defined by :rfc:`2104`\ .
        
        """
        blocksize = cls.digest_size()
        if len(secret) > blocksize:
            secret = cls(secret).digest()
        elif len(secret) < blocksize:
            secret += (blocksize-len(secret))*b'\x00'
        
        i_pad = blocksize*b'\x36'
        inner = b''.join(_s2b(_b_chr(_b_ord(a) ^ _b_ord(b))) 
                         for a, b in zip(secret, i_pad))
        i_hash = cls(inner)
        i_hash.update(message)
        i_digest = i_hash.digest()
        
        o_pad= blocksize*b'\x5c'
        outer = b''.join(_s2b(_b_chr(_b_ord(a) ^ _b_ord(b))) 
                         for a, b in zip(secret, o_pad))
        o_hash = cls(outer)
        o_hash.update(i_digest)
        return o_hash.digest()
    
    
@abstract
class VBlockCipher(object):
    """A block cipher for plaintext encryption and ciphertext decryption.
    
    The cipher operates on blocks of data. The block size of a
    plaintext block can be retreived by calling :meth:`blocksize` and
    for the ciphertext block size :meth:`c_blocksize`\ .

    :param name:       cipher name which constructs the cipher
    :type  name:       unicode
    :param modes:      names of supported block chain modes
    :type  modes:      list<unicode>
    :param symmetric:  if True then the cipher is symmetric
    :type  symmetric:  bool

    .. note::

        This is an abstract base class which should not be directly
        instantiated. Block ciphers should normally be created by
        calling a factory method on a :class:`VCrypto`\ .
    
    """

    def __init__(self, name, modes, symmetric):
        self.__name = name
        self.__modes = tuple(modes)
        self.__symmetric = bool(symmetric)

    @property
    def name(self):
        """Holds the name (unicode) which constructs this cipher."""
        return self.__name
    
    @abstract
    def blocksize(self, key=None):
        """Returns plaintext blocksize for the provided key.

        :param key: encryption key
        :type  key: :class:`VKey`
        :returns:   plaintext blocksize
        :rtype:     int
        :raises:    :exc:`VCryptoException`
        
        Ciphers which have fixed blocksize may return that blocksize
        without inspecting the key.
        
        """
        raise NotImplementedError()

    @abstract
    def c_blocksize(self, key):
        """Returns ciphertext blocksize for the provided key.

        :param key: decryption key
        :type  key: :class:`VKey`
        :returns:   ciphertext blocksize
        :rtype:     int
        :raises:    :exc:`VCryptoException`
        
        Ciphers which have fixed blocksize may return that blocksize
        without inspecting the key.
        
        """
        raise NotImplementedError()

    @property
    def modes(self):
        """Holds a tuple of mode names (tuple<unicode>) for this cipher."""
        return self.__modes
    
    @property
    def symmetric(self):
        """Is True if cipher is symmetric, otherwise False."""
        return self.__symmetric

    @abstract
    def encrypter(self, key, iv=None, mode='cbc'):
        """Generates a block transform for cipher encryption.
        
        :param key:  encryption key
        :type  key:  :class:`VKey`
        :param iv:   initialization vector
        :type  iv:   bytes
        :param mode: name of the block chain mode
        :type  mode: unicode
        :returns:    block transform for cipher encryption
        :rtype:      :class:`VBlockTransform`
        
        The initialization vector must have the same length as the
        cipher's plaintext blocksize. If iv is None, then an
        initialization vector is generated with only zero-bytes.
        
        """
        raise NotImplementedError()        
        
    def msg_encrypter(self, hash_cls, pad_source, key, iv=None, mode='cbc',
                      mac_secret=b''):
        """Generates an encryption block transform with message encapsulation.
        
        :param hash_cls:     hash class for message integrity
        :type  hash_cls:     :class:`versile.crypto.VHash`
        :param pad_provider: message padding data provider
        :type  pad_provider: callable
        :param mac_secret:   secret data for package authentication
        :type  mac_secret:   bytes
        :returns:            message encrypter
        :rtype: :class:`versile.crypto.algorithm.message.VMessageEncrypter`
        
        Remaining arguments are similar to :meth:`encrypter`\
        . Protects encrypted plaintext with the plaintext message
        format for data integrity validation defined in :term:`VP`
        Crypto specifications..
        
        """
        from versile.crypto.algorithm.message import VMessageEncrypter
        encrypter = self.encrypter(key, iv, mode)
        return VMessageEncrypter(encrypter, hash_cls, pad_source, mac_secret)
        
    @abstract
    def decrypter(self, key, iv=None, mode='cbc'):
        """Generates a block transform for cipher decryption.

        Arguments and return values are similar to :meth:`encrypter`\ .
        
        """
        raise NotImplementedError()        

    def msg_decrypter(self, hash_cls, key, iv=None, mode='cbc',
                      mac_secret=b''):
        """Generates and returns message decrypter.
        
        :returns: message decrypter
        :rtype:   :class:`versile.crypto.algorithm.message.VMessageDecrypter`

        Arguments are similar to :meth:`msg_encrypter`\ .

        """
        from versile.crypto.algorithm.message import VMessageDecrypter
        decrypter = self.decrypter(key, iv, mode)
        return VMessageDecrypter(decrypter, hash_cls, mac_secret)

    @abstract
    @property
    def key_factory(self):
        """Holds a :class:`VKeyFactory` for this cipher."""
        raise NotImplementedError()
        

@abstract
class VKeyFactory(object):
    """Factory class for generating or importing cipher keys.
    
    :param min_len:  min key length (in bytes)
    :type  min_len:  int
    :param max_len:  max key length (in bytes)
    :type  max_len:  int
    :param size_inc: key size must satisfy (key_len % size_inc == 0)
    :type  size_inc: int
    
    .. note::

        This is an abstract base class which should not be directly
        instantiated. It is normally available as a property on a
        cipher object.

    .. automethod:: _decode_ascii
    
    """

    def __init__(self, min_len, max_len, size_inc):
        self.__min_len = min_len
        self.__max_len = max_len
        self.__size_inc = size_inc

    @abstract
    def generate(self, source, length, p=None):
        """Generates and returns a cipher key.
        
        :param source: a provider of input byte data to the generator
        :type  source: callable
        :param length: key length in number of bytes
        :type  length: int
        :param p:      argument for probabilistic key generation
        :type  p:      int
        :returns:      generated key
        :rtype:        :class:`VKey`
        
        *source* is a function which provides input data to the key
        generator, such as a
        :class:`versile.crypto.rand.VByteGenerator`\ . It takes a
        number of bytes as an input and provides a bytes object of
        that length as an output. Typically *source* should be a
        source of cryptographically strong random data, however it is
        also possible to use pseudo-random generators in order to
        create a key that can be re-created from the same
        pseudo-random generator seed.

        Some implementations with fixed-length keys may require a
        specific *length* argument, e.g. AES256 keys are always 32
        bytes. When appropriate, implementations should implement a
        default value for length which creates a key of the
        appropriate size for the cipher.
        
        The *p* argument is ignored for key generation algorithms
        which are not probabilistic. For algorithms that do use this
        parameter, they should generate a key such that the
        probability that the key is not a valid key is less than
        2**(-\ *p*\ ).
        
        """
        raise NotImplementedError()
    
    @abstract
    def load(self, keydata):
        """Generates a key from provided keydata.
        
        :param keydata: key data
        :returns:       loaded (reconstructed) key
        :rtype:         :class:`VKey`
        
        The keydata parameter should have the appropriate native
        library key format for the given key. A key's native data
        representation is available from its :attr:`VKey.keydata`
        property.
        
        """
        raise NotImplementedError()

    @abstract
    def import_ascii(self, keydata):
        """Imports a key from a native :term:`VP` ASCII format.

        :param keydata: native ASCII key representation
        :type  keydata: bytes
        :returns:       reconstructed key
        :raises:        :exc:`VCryptoException`

        The keydata parameter should have the appropriate standard
        ASCII representation defined in :term:`VP` VCrypto
        specifications.

        .. note::

            This method imports from a native format. In order to
            import from standards based formats, other mechanisms must
            be used. See e.g. :class:`versile.crypto.x509.VX509Crypto`
            for importing PKCS formatted RSA keys.
        
        """
        raise NotImplementedError()

    @classmethod
    def _decode_ascii(cls, block):
        """Decodes an ASCII block.

        :param block: standard ASCII block representation
        :type  block: bytes
        :returns:     (block_name, block_data)
        :raises:      VCryptoException

        Internal convenience method for derived classes implementing
        ascii key import.
        
        """
        block = block.strip()
        m = re.match(b'-----BEGIN ', block)
        if not m:
            raise VCryptoException()
        block = block[m.end():]
        m = re.search(b'-----', block)
        if not m:
            raise VCryptoException()
        block_name = block[:m.start()]
        block = block[m.end():]
        block = block.strip()
        end = b'-----END ' + block_name + b'-----'
        if not block.endswith(end):
            raise VCryptoException()
        block = block[:(-len(end))]        
        try:
            if _pyver == 2:
                decoded = _s2b(base64.decodestring(_b2s(block)))
            else:
                decoded = base64.decodebytes(block)
        except:
            raise VCryptoException()
        else:
            return (block_name, decoded)

    def _decode_numbers(self, block):
        result = []
        while block:
            num, num_read = netbytes_to_posint(block)
            if num is None:
                raise VCryptoException()
            result.append(num)
            block = block[num_read:]
        return result
        
    def constraints(self):
        """Returns a tuple of key parameters for valid keys.

        :returns: (min_size, max_size, size_increment)
        
        Elements of returned value have the same meaning as the
        :class:`VKeyFactory` constructor.
        
        """
        return (self.__min_len, self.__max_len, self.__size_inc)


@abstract
class VRSAKeyFactory(VKeyFactory):
    """Base key factory for RSA keys.

    .. note::

        This is an abstract base class which should not be directly
        instantiated. It is normally available as a property on a
        cipher object.

    """
    
    def extract_prime(self, source, length, p, callback=None, advance=False):
        """Generates prime numbers of given length.

        :param source:    provider of input byte data to the generator
        :type  source:    callable
        :param length:    number of bytes for the prime
        :type  length:    int
        :param p:         parameter for probabilistic primality test 
        :type  p:         int
        :param callback:  callback function for each generator iteration
        :type  callback:  callable
        :param advance:   if True fetch new random data for each failed prime
        :type  advance:   bool
        :returns:         (source_index, offset, prime)
        
        *source* works similar to :meth:`VKeyFactory.generate`\ .

        The *p* argument is used to define primes should be generated
        which have probability <= 2**(-4*p) of being non-prime

        The *callback* function (if provided) is called as
        *callback(n)* for each iteration the generator performs to try
        to identify a prime, where *n* is the number of generator
        iterations that have passed.

        If *advance* is True, for each candidate prime number which is
        evaluated to be non-prime, random data is pulled to generate a
        new candidate prime number. If *advance* is False, the next
        candidate is instead generated by adding 2 to the previous
        candidate number, and no new random data is pulled.

        .. note::

            *advance*\ =False effectively reduces the cryptographic
            strength of the generated prime by some bits, as the
            distribution of generated primes is not uniform - assuming
            an attacker has knowledge of what are the
            higher-probability prime candidates (which is still not
            very likely). Security may still be as good though because
            pulling large quantities of random data may deplete the
            entropy pool of the underlying random-data generating
            subsystem.
        
        Returned prime numbers always have the two highest bits and
        the lowest bit of its byte representation set. This implies
        the effective bits of the prime is three bits less than the
        total number of bits in the byte representation.
        
        """
        if callback:
            class C:
                def __init__(self):
                    self.iterations = 0
                def cback(self, *args):
                    self.iterations += 1
                    callback(self.iterations)
            cback = C().cback
        else:
            def cback(*args):
                return None

        source_index = offset = 0

        data = source(length)
        num = self.primedata_to_number(data)
        while True:
            is_pr = is_prime(num, p)
            cback()
            if is_pr:
                break
            else:
                if advance:
                    data = source(length)
                    num = self.primedata_to_number(data)
                    source_index += length
                else:
                    offset += 2
                    num += 2
        return (source_index, offset, num)
                
    @classmethod
    def primedata_to_number(self, prime_data):
        """Returns a candidate prime number associated with byte data.
        
        :param prime_data: data which may represents a prime
        :type  prime_data: bytes
        :returns:          number associated with prime_data
        :rtype:            int, long
        
        Sets the two most significant bits and the least significant
        bit of prime_data, before converting to an int or long.
        
        .. warning::

            This method does not perform any type of primality check,
            and it is the responsibility of the caller to check
            whether the returned number represents a prime.
        
        """
        bval = [_b_ord(c) for c in prime_data]
        bval[0] |= 0xc0
        bval[-1] |= 0x01
        if _pyver == 2:
            data = b''.join([_s2b(_b_chr(b)) for b in bval])
        else:
            data = bytes(bval)
        return bytes_to_posint(data)
                
    @abstract
    @classmethod
    def from_primes(cls, p, q):
        """Generates an RSA cipher key from two provided primes.

        :param p:  a prime
        :type  p:  int, long
        :param q:  a prime
        :type  q:  int, long
        :returns:  RSA key generated from p, q
        :rtype:    :class:`VAsymmetricKey`
        :raises:   :exc:`VCryptoException`

        .. warning::

            This method does not perform any type of primality check,
            and it is the responsibility of the caller to check
            whether the returned number represents a prime.

        The method may raise an exception if it detects p and q do not
        form a valid key. However, this is not a reliable test and
        should not be used as a test that p, q are primes (which is a
        requirement for a generally valid key).

        """
        raise NotImplementedError()


@abstract
class VKey(object):
    """An encryption and/or decryption key for a cipher.

    For a symmetric cipher the key is always a complete key which can
    be used both for encryption and decryption. For asymmetric ciphers
    it may be a complete key (e.g. RSA keypair) or a partial key
    (e.g. RSA public key).

    :param name: cipher name for this key
    :type  name: unicode

    .. automethod:: _encode_ascii
    
    """

    def __init__(self, name):
        self.__name = name

    @abstract
    def export_ascii(self):
        """Exports the key in a native :term:`VP` ASCII format.
        
        :returns: ASCII representation of the key
        :rtype:   bytes
        
        .. note::
        
            This method exports to a native format. In order to export
            to standards based formats, other mechanisms must be
            used. See e.g. :class:`versile.crypto.x509.VX509Crypto`
            for exporting PKCS formatted RSA keys.
            
        """
        raise NotImplementedError()

    def _encode_ascii(self, name, numbers):
        """Internal convenience method for exporting as ASCII.

        :param name:    cipher name
        :type  name:    unicode
        :param numbers: keydata number components
        :type  numbers: list<int>
        
        Can be used internally by implementations of derived classes.
        
        """
        header = b'-----BEGIN ' + name + b'-----\n'
        footer = b'-----END ' + name + b'-----\n'
        num_data = b''.join([posint_to_netbytes(i) for i in numbers])
        if _pyver == 2:
            num_data = _s2b(base64.encodestring(_b2s(num_data)))
        else:
            num_data = base64.encodebytes(num_data)
        data = b''.join((header, num_data, footer))
        data = b'\r\n'.join(data.split(b'\n'))
        return data        

    @property
    def cipher_name(self):
        """The name (unicode) of the cipher associated with this key."""
        return self.__name

    @abstract
    @property
    def keydata(self):
        """Keydata for the library native keydata representation."""
        raise NotImplementedError()
    

class VAsymmetricKey(VKey):
    """A cipher key of an asymmetric cipher.
    
    Asymmetric ciphers use different key components for encryption and
    decryption. A 'public' key component is required for encryption,
    and a 'private' key component is required for decryption. If a key
    holds both components, it is considered to be a 'keypair'.
    
    """
    
    @abstract
    @property
    def has_private(self):
        """True if the key includes the private key component."""
        raise NotImplementedError()

    @abstract
    @property
    def has_public(self):
        """True if the key includes the public key component."""
        raise NotImplementedError()

    @property
    def encrypts(self):
        """True if the key can be used for encryption."""
        return self.has_public

    @property
    def decrypts(self):
        """ True if the key can be used for decryption."""
        return self.has_private
        
    @abstract
    @property
    def private(self):
        """The private key component of this key.
        
        Raises :exc:`VCryptoException` if a private key cannot be extracted.
        
        """
        raise NotImplementedError()

    @abstract
    @property
    def public(self):
        """The public key component of this key.
        
        Raises :exc:`VCryptoException` if a private key cannot be extracted.
        
        """
        raise NotImplementedError()
    
    @abstract
    def merge_with(self, key):
        """Merge with another key component and return the result.
        
        :param key: another key component to merge with
        :type  key: :class:`VKey`
        
        Raises :exc:`VCryptoException` if the keys cannot be combined into
        a complete keypair.

        .. note::

            This method would typically be used to combine a public
            key with a private key to form a complete keypair.
            
        """
        raise NotImplementedError()

        
class VBlockTransform(object):
    """Transform operating on byte blocks.

    Performs block transformation on blocks of data, such as a block
    cipher encryption or decryption. Transformation may depend on the
    previously transformed data, which is always the case for ciphers
    with block chain modes are employed.

    :param blocksize: block size of transform input data
    :type  blocksize: int

    Note that input blocksize may not necessarily be the same as
    the output blocksize.

    .. automethod:: __call__
    .. automethod:: _transform
    
    """

    def __init__(self, blocksize):
        self.__blocksize = blocksize
    
    @property
    def blocksize(self):
        """Blocksize of transform byte data input."""
        return self.__blocksize
    
    def padding(self, data, padding_source):
        """Creates padding for aligning input data with transform block size.

        :param data:           input data
        :type  data:           bytes
        :param padding_source: function which generates padding bytes
        :type  padding_source: callable        
        :returns:              generated padding
        :rtype:                bytes

        Creates a bytes object with the minimum number of padding data
        extracted from padding_source such that the length of the
        input data plus the length of the padding is a multiple of the
        transform's input blocksize.

        *padding_source* should be a callable which takes a number of
        bytes as input and returns a bytes object with that number of
        bytes, such as a :class:`versile.crypto.rand.VByteGenerator`\
        .
        
        This method is mainly intended for internal use by derived
        classes implementing a block transform.
        
        """
        non_aligned = len(data) % self.__blocksize
        if non_aligned == 0:
            return b''
        else:
            pad_len = self.__blocksize - non_aligned
            return padding_source(pad_len)
    
    def __call__(self, data, padding_source=None):
        """See :meth:`transform`\ ."""
        return self.transform(data, padding_source=padding_source)
        
    def transform(self, data, padding_source=None):
        """Performs block transform on data and returns the result.

        :param data:           input data to transform
        :type  data:           bytes
        :param padding_source: source of padding bytes (or None)
        :type  padding_source: callable
        :returns:              transformation output
        :rtype:                bytes
        :raises:               :exc:`VCryptoException`
        
        If *data* is not aligned to the transform's blocksize then
        padding data is added to the input if *padding_source* was
        set, otherwise raises an exception.
        
        If *data* is larger than the transformation input data
        blocksize, the method splits up the data into blocks of that
        size (using padding as needed on the last block), performs
        transformation on each block, and returns a concatenation of
        the transformed blocks.

        If input data is empty, the return value is also empty.
        
        """
        if not data:
            return b''
        len_data = len(data)
        aligned = (len_data % self.__blocksize == 0)
        if not aligned:
            if not padding_source:
                raise VCryptoException('not block aligned, padding needed')
            else:
                data = b''.join((data, self.padding(data, padding_source)))
        return self._transform(data)
    
    @abstract
    def _transform(self, data):
        """Transform data, which must be aligned to blocksize.
        
        Used internally by the transform object. External code should
        call :meth:`transform`\ .
        
        """
        raise NotImplementedError


class VNumCipher(object):
    """A cipher which encrypts/decrypts integers.
    
    Use for implementing number transformation algorithms such as RSA,
    which natively operates on integers.

    The class assumes that the cipher operates on a set of integers
    starting and zero and up to some maximum number; this is not
    generally true for \"any\" number transform, however it is the
    typical case for algorithms involving modular arithmetics, and it
    is a design constraint for this class.

    A number cipher can be converted to a block cipher by calling
    :meth.`block_cipher`\ .
    
    :param name:      the name which constructs this cipher
    :type  name:      unicode
    :param symmetric: True if this is a symmetric cipher
    :type  symmetric: bool
    
    """

    def __init__(self, name, symmetric):
        self.__name = name
        self.__symmetric = bool(symmetric)

    @property
    def name(self):
        """The name (unicode) which constructs this cipher."""
        return self.__name
    
    @property
    def symmetric(self):
        """True if this cipher is symmetric, otherwise False."""
        return self.__symmetric

    @abstract
    def encrypter(self, key):
        """Generates a transform object for number encryption.
        
        :param key:  encryption key
        :type  key:  :class:`VKey`
        :returns:    transformation object for cipher encryption
        :rtype:      :class:`VNumTransform`
        
        """
        raise NotImplementedError()        
        
    @abstract
    def decrypter(self, key):
        """Generates transform object for number decryption.
        
        :param key:  decryption key
        :type  key:  :class:`VKey`
        :returns:    transformation object for cipher decryption
        :rtype:      :class:`VNumTransform`
        
        """
        raise NotImplementedError()        

    def block_cipher(self, cipher_name=None):
        """Create a block cipher for this num cipher.

        :param cipher_name: name of the constructed cipher
        :type  cipher_name: unicode
        :returns:           block cipher
        :rtype:             :class:`VBlockCipher`

        If cipher name is None then the same cipher name as the number
        cipher is used.
        
        """
        from versile.crypto.algorithm.numblock import VNumBlockCipher
        return VNumBlockCipher(self, cipher_name)

    @abstract
    @property
    def key_factory(self):
        """Key factory for this cipher."""
        raise NotImplementedError()        


class VNumTransform(object):
    """Integer transform.

    Performs number transformation on integers, such as a number
    cipher encryption or decryption transforms. The transformation may
    depend on previously transformed data.

    .. automethod:: __call__
    
    """

    def __call__(self, num):
        """See :meth:`transform`\ ."""
        return self.transform(num)
    
    @abstract
    def transform(self, num):        
        """Performs a transform and returns transformed value.
        
        :param num:  input number to transform
        :type  num:  int, long
        :returns:    transform output
        :rtype:      int, long

        Input data to the transform must be 0 <= num <= :attr:`max_number`
                
        """
        raise NotImplementedError()

    @abstract
    @property
    def max_number(self):
        """The largest number which can be transformed."""
        raise NotImplementedError()


@abstract
class VDecentralIdentityScheme(object):
    """Implementation of a decentral key scheme.
    
    .. automethod:: __call__
    
    """

    def __call__(self, bits, *args, **kargs):
        """Proxy for :meth:`generate`\ ."""
        return self.generate(bits, *args, **kargs)

    def generate(self, bits, *args):
        """Generates and returns a decentral key from input data.

        :param bits:  number of bits in decentral key
        :type  bits:  int
        :param args:  input arguments for key generation
        :type  args:  (unicode,)
        :returns:     decentral key
        :rtype:       :class:`VAsymmetricKey`
        :raises:      :exc:`VCryptoException`

        Input arguments *args* are specific to the individual key
        schemes.
        
        """
    
    def raw_generate(self, bits, sec_id_data):
        """Generates and returns a decentral key from a 'raw' sec_id_data.
        
        :param    bits:        number of bits in decentral key
        :type     bits:        int
        :param    sec_id_data: the raw secret phrase to generate from
        :type     sec_id_data: unicode
        :returns:              decentral key
        :rtype:                :class:`VAsymmetricKey`
        :raises:               :exc:`VCryptoException`

        The 'raw' sec_id_data should be composed according to the
        standard defined by scheme. It should however not include any
        scheme name postfix defined by the scheme; this should be
        added internally by this method.
        
        """
        raise NotImplementedError()

    @abstract
    @classmethod
    def name(cls):
        """Returns the name of the decentral key scheme for this class.

        :returns: hash class name
        :rtype:   unicode

        """
        raise NotImplementedError()


@abstract
class VDecentralIdentitySchemeA(VDecentralIdentityScheme):
    """Implementation of a decentral key scheme."""
    
    def generate(self, bits, purpose, personal, passphrase):
        """Generates and returns a decentral key for the scheme.
        
        :param    bits:       number of bits in decentral key
        :type     bits:       int
        :param    purpose:    label for identity purpose
        :type     purpose:    unicode
        :param    personal:   some personal information
        :type     personal:   unicode
        :param    passphrase: passphrase for the key
        :type     passphrase: unicode
        :returns:             decentral key
        :rtype:               :class:`VAsymmetricKey`
        :raises:              :exc:`VCryptoException`
        
        """
        if len(passphrase) < 10:
            raise VCryptoException('Passphrase must be minimum 10 characters')
        sec_id_data = ''.join(('Purpose:', purpose, ':Personal:', personal,
                               ':Passphrase:', passphrase))
        return self.raw_generate(bits, sec_id_data)
