from urllib.request import urlopen
from urllib.error import URLError
from urllib.parse import urlencode
from bs4 import BeautifulSoup

class Text:
    """Represent generated text.

    :ivar title: A title of the text.
    :ivar description: A description of the text.

    :ivar content: An actual content of the text. Paragraphs separated
        by two newline characters.

    :ivar topics: A list of topics used for generating the text.
    :ivar source: A source of the text from a service.

    """

    def __init__(self, topics, source):
        self.topics = topics
        self.source = source
        bs = BeautifulSoup(source)
        self.title = bs.find('strong').text[7:-1]
        self.description = bs.find('div').text
        paragraphs = []
        for tag in bs.find_all('p'):
            text = ' '.join(tag.get_text().split())
            paragraphs.append(text)
        self.content = '\n\n'.join(paragraphs)

class Slogan:
    """Represent generated slogan.

    :ivar brand: A brand for wich the slogan was generated.
    :ivar content: A content of the slogan.

    """

    def __init__(self, brand, content):
        self.brand = brand
        self.content = content

class Answer(Slogan):
    """Represent generated answer.

    :ivar question: A question to which the answer was given.
    :ivar content: A content of the answer.

    """

    def __init__(self, question, content):
        self.question = question
        self.content = content

class ConnectionError(URLError):
    pass

class Connection:

    def request(self, url, parms=None):
        if parms is not None:
            parms = urlencode(parms)
            url = '{0}?{1}'.format(url, parms)
        try:
            data = urlopen(url)
            return data.read().decode()
        except URLError as e:
            raise ConnectionError(e)

class Vesna:
    """Represent http://vesna.yandex.ru/ web service."""

    text_url = 'http://vesna.yandex.ru/referats/write/'
    slogan_url = 'http://vesna.yandex.ru/creator/write/'
    answer_url = 'http://vesna.yandex.ru/gadalka/write/'
    text_topics = (
        'astronomy',
        'geology',
        'gyroscope',
        'literature',
        'marketing',
        'mathematics',
        'music',
        'polit',
        'agrobiologia',
        'law',
        'psychology',
        'geography',
        'physics',
        'philosophy',
        'chemistry',
        'estetica',
    )
    """A list of available topics for text generation."""

    def __init__(self, connection=None):
        """Setup network connection.

        :parms connection: Optional instance of the Connection class or
            similar.

        """
        if connection is None:
            self.connection = Connection()
        else:
            self.connection = connection

    def text(self, topics):
        """Generate text using one or several topics.

        :param topics: A list of topics used to generate text.
        :type topics: list
        :returns: A text represented by the Text class instance.
        :raises: ConnectionError

        """
        topics = list(set(self.text_topics) & set(topics))
        if len(topics) == 0:
            return None
        topics_str = ' '.join(topics)
        source = self.connection.request(self.text_url, {'t': topics_str})
        return Text(topics, source)

    def slogan(self, brand=''):
        """Generate slogan for some brand.

        :param brand: A brand to generate slogan for.
        :type brand: str
        :returns: A slogan represented by the Slogan class instance.
        :raises: ConnectionError

        """
        brand = str(brand).strip()
        parms = {'t': brand} if brand else None
        data = self.connection.request(self.slogan_url, parms)
        return Slogan(brand, data)

    def answer(self, question=''):
        """Generate answer for some question.

        :param question: Question to generate answer for.
        :type question: str
        :returns: An answer represented by the Answer class instance.
        :raises: ConnectionError

        """
        question = str(question).strip()
        parms = {'t': question} if question else None
        data = self.connection.request(self.answer_url, parms)
        return Answer(question, data)
