import os
import os.path as path
import viset.util
from viset.util import quietprint, isurl, isfile, iscached, incache
import pylab
import hashlib
import viset.dataset
import time
import socket
import urlparse

class Async(object):
    """Mirrors Python's AsyncResult class"""
    OBJ = None    
    def successful():
        return (self.OBJ is not None)
    def get():
        raise IOError('this should be overloaded')
    def wait(timeout):
        socket.setdefaulttimeout(timeout)  # do not set globally        
    def ready():
        return (self.OBJ is not None)


class File(Async):
    URL = None
    CACHEFILE = None
    CACHEDIR = None
    OBJ = None
    VERBOSE = False

    def fetch(self):
        if isfile(self.CACHEFILE):
            pass
        elif isurl(str(self.URL)):
            self.CACHEFILE = path.join(self.CACHEDIR, hashlib.sha1(self.URL).hexdigest() + os.path.splitext(self.URL)[1])
            if not isfile(self.CACHEFILE):
                quietprint('[viset.types]: fetching \'' + str(self.URL) + '\'' + ' to cached file \'' + self.CACHEFILE + '\'', self.VERBOSE)             
                viset.download.download(self.URL, self.CACHEFILE, verbose = self.VERBOSE)                        
        else:
            IOError('invalid URL')
            
    def pack(self):
        # serialize
        pass
        
    def get(self):
        try:
            self.fetch()
        except:
            quietprint('[viset.types]: download failed...skipping', self.VERBOSE)                                 
            time.sleep(1)  # for python ctrl-c
            return None
        try:
            self.unpack()
        except:
            quietprint('[viset.types]: unpack failed...clearing cache and refetching', self.VERBOSE)                                             
            os.remove(self.CACHEFILE)
            self.fetch()
            self.unpack()
        return self.OBJ

    
class Image(File):
    def __init__(self, impath, cachedir=None, verbose=True, imcodec='universal', imreadfunc=None):
        self.VERBOSE = verbose
        if isfile(impath):
            self.CACHEFILE = impath
        elif iscached(impath, cache=cachedir):
            self.CACHEFILE = incache(impath, cache=cachedir)
        elif isfile(urlparse.urldefrag(impath)[0]):
            self.CACHEFILE = impath            
        elif iscached(str(urlparse.urldefrag(impath)[0]), cache=cachedir):
            self.CACHEFILE = incache(impath, cache=cachedir)       
        elif isurl(impath):
            self.URL = impath
        else:
            raise IOError('invalid image path format')
        self.imread = self.codec(imcodec, imreadfunc)
        
    def __repr__(self):
        if self.URL is not None:
            URI = URL
        else:
            URI = self.CACHEFILE
        return str('<viset.image: obj=' + str(type(self.OBJ)) + ', cached=' + str(self.CACHEFILE is not None) + ', URI=\'' + str(URI) + '\'>')        

    def codec(self, imcodec, imreadfunc=None):
        if imcodec == 'universal':
            return pylab.imread
        elif imcodec == 'mnist':
            return viset.mnist.imread
        elif imreadfunc is not None:
            return imreadfunc
        else:
            raise ValueError('invalid image codec "' + str(imcodec) + '"')
    
    def unpack(self):
        # convert cached file to object (unserialize)
        quietprint('[viset.types]: reading image \'' + self.CACHEFILE + '\' to numpy.ndarray object', self.VERBOSE) 
        self.OBJ = self.imread(self.CACHEFILE)

        
