"""Classes to create and manage vision datasets"""

import tables
import sys
import os
from os import path
import shutil
import viset.download
import viset.util
from viset.util import ishdf5, isfile, isarchive, quietprint
import numpy as np 
import urllib
import socket
import hashlib
import time
import random
from itertools import imap, islice, count
import urlparse
import viset.views 
import datetime
from viset.cache import Cache
from viset.views import ImageView
import viset.views

class Viset(object):
    _cache = Cache()  # cache object
    _obj = None # pytables object
    _dburl = None   # file location
    _verbose = None  
    _async = None
    _createviews = [ImageView()]    
    
    def __init__(self, dburl=None, task=None, strategy=None, kfold=0, datastep=1, async=False, verbose=False):
        self._verbose = verbose            
        self._cache._verbose = verbose
        self._dburl = dburl
        self._async = async
        if dburl is not None:
            self._obj = self.open(dburl)
        else:
            # empty initialization for subclass create
            pass
        
    def __getattr__(self, viewname):
        """Return static and dynamic attributes"""
        if viewname == 'cache':
            return self._cache
        for node in self._obj.root:
            if node._v_attrs.viewname == viewname:
                viewmodule = sys.modules['viset.views']
                viewclass = getattr(viewmodule, node._v_attrs.viewclass)
                return viewclass(node, self)
        raise AttributeError, viewname
        
    def __del__(self):
        if self._obj is not None:
            self.close()
        
    def __str__(self):
        return self.summary()

    def flush(self):
        self._obj.flush()

    def summary(self):
        print 'Database URL: ' + self._dburl
        print 'Database name: ' + str(self.name())            
        print 'Database views: ' + str(self.views())
        print 'Number of images: ' + str(len(self.image))
        print 'Number of annotation views: ' + str(len(self.annotation))
        print 'Database verbosity: ' + str(self._verbose)        
        print 'HDF5 file structure: ' 
        print '  ' + "\n  ".join(str(self._obj).split("\n"))
        print 'Cache directory: ' + self._cache._cacheroot
        print 'Database local file: ' + self._cache.get(self._dburl)
        return str()
        
    def open(self, dburl):
        # Database location resolution
        if not type(dburl) is str:
            raise IOError('database filename must be a string')
        elif not ishdf5(dburl):
            raise IOError('invalid database file extension - must be HDF5 (.h5)')                
        elif viset.util.isfile(dburl):
            dbfile = dburl
        else:
            dbfile = self._cache.get(dburl)        
        if not viset.util.isfile(dbfile):
            raise IOError('Invalid database file "' + str(dbfile) + '"')
            
        # Open in read only mode
        try:
            quietprint('Opening viset "' + str(dbfile) + '"', self._verbose)
            obj = tables.open_file(dbfile, mode = "r")
            if not self.isviset(obj):
                raise IOError()        
        except:
            print 'Invalid database file "' + str(dbfile) + '"'
            raise 
        return obj

    def close(self):
        if self._obj is not None:
            self._obj.close()
            del(self._obj)    

    def create(self, dbname, createviews=None):
        dbfile = self._cache.abspath(dbname + '.h5')
        quietprint('[viset.dataset.create]: Creating viset "' + dbfile + '"', self._verbose)                

        # HDF5 file 
        obj = tables.open_file(dbfile, mode = "w", title = dbname)

        # View objects
        if createviews is None:
            createviews = self._createviews
        for view in createviews:
            quietprint('[viset.dataset.create]: Creating view "' + str(view) + '"', self._verbose)                            
            view.create(obj)

        # Metadata
        obj.root._v_attrs.viset = 'http://www.visym.com'
        obj.root._v_attrs.viset_version = 0
        obj.root._v_attrs.viset_name = dbname

        # Return newly created database 
        obj.flush()
        self._obj = obj
        self.dburl = dbfile
        return (self, dbfile)
        
    def name(self):
        return self._obj.root._v_attrs.viset_name        
  
    def version(self):
        return self._obj.root._v_attrs.viset_version

    def isviset(self, obj):
        try: 
            if obj.root._v_attrs.viset == 'http://www.visym.com':
                return True
            else:
                return False
        except:
            # does not contain required attribute
            return False
        
    def views(self):
        """List of views in database"""    
        viewlist = []
        for node in self._obj.root._f_walknodes():
            viewlist.append(str(node._v_pathname).replace('/','.'))
        return viewlist

    def urlquery(self, url):
        """query database by using querystring format 'http://domain.com/mydb.h5?table=x&row=y&field=z'"""
        if not viset.util.isurl(url):
            ValueError('Invalid URL \'' + url + '\'')
        p = urlparse.urlsplit(url)
        p = urlparse.parse_qs(p[3])
        if 'table' in p:
            table = p['table'][0]
        else:
            table = None
        if 'row' in p:
            row = int(p['row'][0])
        else:
            row = None
        if 'field' in p:
            field = int(p['field'][0])
        else: 
            field = None
        return self.query(table, row, field)
        
    def query(self, table, row, field=None):
        """database query by table[row][field]"""
        if field is None:
            return self._obj.getNode(table)[row] 
        else:
            return self._obj.getNode(table)[row][field] 
   
    
class CategorizationViset(Viset):
    _createviews = [viset.views.ImageView()] + [viset.views.CategorizationAnnotationView()]

class DetectionViset(Viset):
    _createviews = [viset.views.ImageView()] + [viset.views.DetectionAnnotationView()]

class SegmentationViset(Viset):
    _createviews = [viset.views.ImageView()] + [viset.views.SegmentationAnnotationView()]
    


