"""Classes to create and manage vision datasets"""

import tables
import sys
import os
from os import path
import shutil
import viset.download
import viset.util
from viset.util import ishdf5, isfile, isarchive, quietprint
import numpy as np 
import urllib
import socket
import hashlib
import time
import random
from itertools import imap, islice, count
import urlparse
import datetime
from viset.cache import Cache
import viset.stream 
from viset.stream import ImageStream

class Viset(object):
    _cache = Cache()  # cache object
    _obj = None # pytables object
    _dburl = None   # file location
    _verbose = None  
    _async = None
    _createviews = [ImageStream()]    
    
    def __init__(self, dburl, mode='r', async=False, verbose=False):
        self._verbose = verbose            
        self._cache._verbose = verbose
        self._async = async
        (self._obj, self._dburl) = self._open(dburl, mode)
        
    def __getattr__(self, viewname):
        """Return static and dynamic attributes"""
        if viewname == 'cache':
            return self._cache
        for node in self._obj.root:
            if node._v_attrs.viewname == viewname:
                viewmodule = sys.modules['viset.stream']
                viewclass = getattr(viewmodule, node._v_attrs.viewclass)
                return viewclass(node, self)
        raise AttributeError, viewname
        
    def __del__(self):
        self.close()
        
    def __str__(self):
        return self._summary()

    def flush(self):
        self._obj.flush()

    def _summary(self):
        print 'Database URL: ' + self._dburl
        print 'Database name: ' + str(self.name())            
        print 'Database views: ' + str(self.views())
        print 'Number of images: ' + str(len(self.image))
        print 'Number of annotation views: ' + str(len(self.annotation))
        print 'Database verbosity: ' + str(self._verbose)        
        print 'HDF5 file structure: ' 
        print '  ' + "\n  ".join(str(self._obj).split("\n"))
        print 'Cache directory: ' + self._cache._cacheroot
        print 'Database local file: ' + self._cache.get(self._dburl)
        return str()
        
    def _open(self, dburl, mode='r'):
        if mode in ['r', 'w+']:
            # Database location resolution
            if not type(dburl) is str:
                raise IOError('database filename must be a string')
            elif not ishdf5(dburl):
                raise IOError('invalid database file extension - must be HDF5 (.h5)')                
            elif viset.util.isfile(dburl):
                dbfile = dburl
            else:
                dbfile = self._cache.get(dburl)        
            if not viset.util.isfile(dbfile):
                raise IOError('Invalid database file "' + str(dburl) + '"')

            # Open hdf5 file in mode
            try:
                quietprint('Opening viset "' + str(dbfile) + '"', self._verbose)
                obj = tables.open_file(dbfile, mode = mode)
                if not self._isviset(obj):
                    raise IOError()        
            except:
                print 'Invalid database file "' + str(dbfile) + '"'
                raise 
            return (obj, dbfile)
        elif mode in 'w':
            return self._create(path.basename(dburl), createviews=self._createviews)
        else:
            raise IOError('Visets can only be opened in read "r",  append "w+" or write "w", mode, not "' + str(mode) + '"')                   
        
    def close(self):
        if self._obj is not None:
            self._obj.close()

    def _create(self, dbname, createviews=[ImageStream()]):
        """Create a new viset with the provided name and views"""
        if not viset.util.ishdf5(dbname):
            dbname = dbname + '.h5'
        dbfile = self._cache.abspath(dbname)            
        quietprint('[viset.create]: Creating viset "' + dbfile + '"', self._verbose)                

        # HDF5 file 
        obj = tables.open_file(dbfile, mode = "w", title = dbname)

        # View objects
        for view in createviews:
            quietprint('[viset.create]: Creating view "' + str(view) + '"', self._verbose)                            
            view._create(obj)

        # Metadata
        obj.root._v_attrs.viset = 'http://www.visym.com'
        obj.root._v_attrs.viset_version = 0
        obj.root._v_attrs.viset_name = dbname

        # Return newly created database 
        obj.flush()
        return (obj, dbfile)
        
    def name(self):
        return self._obj.root._v_attrs.viset_name        

    def abspath(self):
        """Return the file path to the database file"""
        return self._cache.get(self._dburl)
  
    def version(self):
        return self._obj.root._v_attrs.viset_version

    def _isviset(self, obj):
        try: 
            if obj.root._v_attrs.viset == 'http://www.visym.com':
                return True
            else:
                return False
        except:
            # does not contain required attribute
            return False
        
    def views(self):
        """List of views in database"""    
        viewlist = []
        for node in self._obj.root._f_walknodes():
            viewlist.append(str(node._v_pathname).replace('/','.'))
        return viewlist

    ## def urlquery(self, url):
    ##     """query database by using querystring format 'http://domain.com/mydb.h5?table=x&row=y&field=z'"""
    ##     if not viset.util.isurl(url):
    ##         ValueError('Invalid URL \'' + url + '\'')
    ##     p = urlparse.urlsplit(url)
    ##     p = urlparse.parse_qs(p[3])
    ##     if 'table' in p:
    ##         table = p['table'][0]
    ##     else:
    ##         table = None
    ##     if 'row' in p:
    ##         row = int(p['row'][0])
    ##     else:
    ##         row = None
    ##     if 'field' in p:
    ##         field = int(p['field'][0])
    ##     else: 
    ##         field = None
    ##     return self.query(table, row, field)
        
    ## def query(self, table, row, field=None):
    ##     """database query by table[row][field]"""
    ##     if field is None:
    ##         return self._obj.getNode(table)[row] 
    ##     else:
    ##         return self._obj.getNode(table)[row][field] 
   
    
class CategorizationViset(Viset): 
    _createviews = Viset._createviews + [viset.stream.CategorizationAnnotationStream(), viset.stream.CategorizationOutputGroup()]

class DetectionViset(Viset):
    _createviews = Viset._createviews + [viset.stream.DetectionAnnotationStream()]

class SegmentationViset(Viset):
    _createviews = Viset._createviews + [viset.stream.SegmentationAnnotationStream()]
    
class CategorizationDetectionViset(Viset):
    # FIXME: there must be a better way
    _createviews = Viset._createviews + [viset.stream.DetectionAnnotationStream(), viset.stream.CategorizationAnnotationStream()]

class CategorizationDetectionSegmentationViset(Viset):
    # FIXME: there must be a better way
    _createviews = Viset._createviews + [viset.stream.DetectionAnnotationStream(), viset.stream.CategorizationAnnotationStream(), viset.stream.SegmentationAnnotationStream()]
    
    


