##
#
# Copyright 2012 Ghent University
# Copyright 2009-2012 Stijn De Weirdt
#
# This file is part of VSC-tools,
# originally created by the HPC team of Ghent University (http://ugent.be/hpc/en),
# with support of Ghent University (http://ugent.be/hpc),
# the Flemish Supercomputer Centre (VSC) (https://vscentrum.be/nl/en),
# the Hercules foundation (http://www.herculesstichting.be/in_English)
# and the Department of Economy, Science and Innovation (EWI) (http://www.ewi-vlaanderen.be/en).
#
# http://github.com/hpcugent/VSC-tools
#
# VSC-tools is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation v2.
#
# VSC-tools is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with VSC-tools. If not, see <http://www.gnu.org/licenses/>.
##
"""
module with various convenience functions and classes to deal with date, time and timezone
"""

import calendar
import operator
import re
import time as _time
from datetime import tzinfo, timedelta, datetime, date

class FancyMonth:
    """Convenience class for month math"""
    def __init__(self, tmpdate=None, year=None, month=None, day=None):
        """Initialise the month based on first day of month of tmpdate"""

        if tmpdate is None:
            tmpdate = date.today()

        if day is None:
            day = tmpdate.day
        if month is None:
            month = tmpdate.month
        if year is None:
            year = tmpdate.year

        self.date = date(year, month, day)


        self.first = None
        self.last = None
        self.nrdays = None

        # when calculating deltas, include non-full months
        #   eg when True, nr of months between last day of month
        #   and first day of following month is 2
        self.include = True

        self.set_details()

    def set_details(self):
        """Get first/last day of the month of date"""
        c = calendar.Calendar()
        self.nrdays = len([x for x in c.itermonthdays(self.date.year, self.date.month) if x > 0])

        self.first = date(self.date.year, self.date.month, 1)

        self.last = date(self.date.year, self.date.month, self.nrdays)

    def get_start_end(self, otherdate):
        """Return tuple date and otherdate ordered oldest first"""
        if self.date > otherdate:
            start = otherdate
            end = self.date
        else:
            start = self.date
            end = otherdate

        return start, end

    def number(self, otherdate):
        """Calculate the number of months between this month (date actually) and otherdate
        """
        if self.include == False:
            msg = "number: include=False not implemented"
            raise(Exception(msg))
        else:
            startdate, enddate = self.get_start_end(otherdate)

            if startdate == enddate:
                nr = 0
            else:
                nr = (enddate.year - startdate.year) * 12 + enddate.month - startdate.month + 1

        return nr

    def get_other(self, shift= -1):
        """Return month that is shifted shift months: negative integer is in past, positive is in future"""
        new = self.date.year * 12 + self.date.month - 1 + shift
        return self.__class__(date(new // 12, new % 12 + 1, 01))

    def interval(self, otherdate):
        """Return time ordered list of months between date and otherdate"""
        if self.include == False:
            msg = "interval: include=False not implemented"
            raise(Exception(msg))
        else:
            nr = self.number(otherdate)
            startdate, enddate = self.get_start_end(otherdate)

            start = self.__class__(startdate)
            all_dates = [ start.get_other(m)  for m in range(nr)]

        return all_dates

    def parser(self, txt):
        """Based on strings, return date: eg BEGINTHIS returns first day of the current month"""
        supportedtime = ('BEGIN', 'END',)
        supportedshift = ['THIS', 'LAST', 'NEXT']
        regtxt = r"^(%s)(%s)?" % ('|'.join(supportedtime), '|'.join(supportedshift))

        reseervedregexp = re.compile(regtxt)
        reg = reseervedregexp.search(txt)
        if not reg:
            msg = "parse: no match for regexp %s for txt %s" % (regtxt, txt)
            raise(Exception(msg))

        shifttxt = reg.group(2)
        if shifttxt is None or shifttxt == 'THIS':
            shift = 0
        elif shifttxt == 'LAST':
            shift = -1
        elif shifttxt == 'NEXT':
            shift = 1
        else:
            msg = "parse: unknown shift %s (supported: %s)" % (shifttxt, supportedshift)
            raise(Exception(msg))

        nm = self.get_other(shift)

        timetxt = reg.group(1)
        if timetxt == 'BEGIN':
            res = nm.first
        elif timetxt == 'END':
            res = nm.last
        else:
            msg = "parse: unknown time %s (supported: %s)" % (timetxt, supportedtime)
            raise(Exception(msg))

        return res

def date_parser(txt):
    """Parse txt: date YYYY-MM-DD in datetime.date
        also (BEGIN|END)(THIS|LAST|NEXT)MONTH
             (BEGIN | END)(JANUARY | FEBRUARY | MARCH | APRIL | MAY | JUNE | JULY | AUGUST | SEPTEMBER | OCTOBER | NOVEMBER | DECEMBER)
    """

    reserveddate = ('TODAY',)
    testsupportedmonths = [txt.endswith(calendar.month_name[x].upper()) for x in range(1, 13)]

    if txt.endswith('MONTH'):
        m = FancyMonth()
        res = m.parser(txt)
    elif reduce(operator.or_, testsupportedmonths):  # TODO replace with any()
        m = FancyMonth(month=testsupportedmonths.index(True) + 1)
        res = m.parser(txt)
    elif txt in reserveddate:
        if txt in ('TODAY',):
            m = FancyMonth()
            res = m.date
        else:
            msg = 'dateparser: unimplemented reservedword %s' % txt
            raise(Exception(msg))
    else:
        try:
            datetuple = [int(x) for x in txt.split("-")]
            res = date(*datetuple)
        except:
            msg = ("dateparser: failed on '%s' date txt expects a YYYY-MM-DD format or "
                   "reserved words %s") % (txt, ','.join(reserveddate))
            raise(Exception(msg))

    return res

def datetime_parser(txt):
    """Parse txt: tmpdate YYYY-MM-DD HH:MM:SS.mmmmmm in datetime.datetime
        - date part is parsed with date_parser
    """
    tmpts = txt.split(" ")
    tmpdate = date_parser(tmpts[0])

    datetuple = [tmpdate.year, tmpdate.month, tmpdate.day]
    if len(tmpts) > 1:
        ## add hour and minutes
        datetuple.extend([int(x) for x in tmpts[1].split(':')[:2]])

        try:
            sects = tmpts[1].split(':')[2].split('.')
        except:
            sects = [0]
        ## add seconds
        datetuple.append(int(sects[0]))
        if len(sects) > 1:
            ## add microseconds
            datetuple.append(int(float('.%s' % sects[1]) * 10 ** 6))

    res = datetime(*datetuple)

    return res

#
# example code from http://docs.python.org/library/datetime.html
# Implements Local, the local timezone
#

ZERO = timedelta(0)
HOUR = timedelta(hours=1)

# A UTC class.

class UTC(tzinfo):
    """UTC"""

    def utcoffset(self, dt):
        return ZERO

    def tzname(self, dt):
        return "UTC"

    def dst(self, dt):
        return ZERO

utc = UTC()

# A class building tzinfo objects for fixed-offset time zones.
# Note that FixedOffset(0, "UTC") is a different way to build a
# UTC tzinfo object.

class FixedOffset(tzinfo):
    """Fixed offset in minutes east from UTC."""
    def __init__(self, offset, name):
        self.__offset = timedelta(minutes=offset)
        self.__name = name

    def utcoffset(self, dt):
        return self.__offset

    def tzname(self, dt):
        return self.__name

    def dst(self, dt):
        return ZERO

# A class capturing the platform's idea of local time.

STDOFFSET = timedelta(seconds= -_time.timezone)
if _time.daylight:
    DSTOFFSET = timedelta(seconds= -_time.altzone)
else:
    DSTOFFSET = STDOFFSET

DSTDIFF = DSTOFFSET - STDOFFSET

class LocalTimezone(tzinfo):

    def utcoffset(self, dt):
        if self._isdst(dt):
            return DSTOFFSET
        else:
            return STDOFFSET

    def dst(self, dt):
        if self._isdst(dt):
            return DSTDIFF
        else:
            return ZERO

    def tzname(self, dt):
        return _time.tzname[self._isdst(dt)]

    def _isdst(self, dt):
        tt = (dt.year, dt.month, dt.day,
              dt.hour, dt.minute, dt.second,
              dt.weekday(), 0, 0)
        stamp = _time.mktime(tt)
        tt = _time.localtime(stamp)
        return tt.tm_isdst > 0

Local = LocalTimezone()



